/*****************************************************************

Copyright (c) 2001 Carsten Pfeiffer <pfeiffer@kde.org>
              2001 Matthias Elter   <elter@kde.org>
              2001 Martijn Klingens <mklingens@yahoo.com>

              KShutDown Branch:
              2005 Stephen Ellwood
              2005 Konrad Twardowski <kdtonline@poczta.onet.pl>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

// TODO: 0.9: update handbook

#include "lockout.h"

#include <qlayout.h>
#include <qtoolbutton.h>
#include <qtooltip.h>

#include <dcopclient.h>
#include <kapplication.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kprocess.h>

// TODO: 0.9: more buttons (reboot, logout, etc)

// plugin

extern "C"
{
	KDE_EXPORT KPanelApplet *init(QWidget *parent, const QString& configFile)
	{
		// share i18n with KShutDown
		KGlobal::locale()->insertCatalogue("kshutdown");

		return new Lockout(configFile, parent);
	}
}

// public

Lockout::Lockout(const QString& configFile, QWidget *parent)
	: KPanelApplet(
		configFile,
		KPanelApplet::Normal,
		0, // no standard actions
		parent,
		"kshutdownlockout"
	)
{
	// read configuration
	KConfig *conf = config();
	conf->setGroup("kshutdownlockout");
	_showLockButton = conf->readBoolEntry("ShowLockButton", false);
	_transparent = conf->readBoolEntry("Transparent", true);

	// init widget
	setBackgroundOrigin(AncestorOrigin);
	setFrameStyle(Panel | Sunken);

	// init layout
	_layout = new QBoxLayout(
		this,
		(orientation() == Horizontal)
			? QBoxLayout::TopToBottom
			: QBoxLayout::LeftToRight
	);
	_layout->setAutoAdd(true);
	_layout->setMargin(0);
	_layout->setSpacing(0);

	// init buttons
	_logoutButton = addButton("kshutdown", i18n("Run KShutDown"));
	connect(_logoutButton, SIGNAL(clicked()), SLOT(slotLogout()));
	_lockButton = addButton("lock", i18n("Lock Session"));
	connect(_lockButton, SIGNAL(clicked()), SLOT(slotLock()));
	if (_showLockButton)
		_lockButton->show();
	else
		_lockButton->hide();

// TODO: 0.9: Kiosk

	if (!kapp->dcopClient()->isAttached())
		kapp->dcopClient()->attach();
	connect(kapp, SIGNAL(iconChanged(int)), SLOT(slotIconChanged()));
}

Lockout::~Lockout()
{
	KGlobal::locale()->removeCatalogue("kshutdown");
}

int Lockout::heightForWidth(int width) const
{
	checkLayout(width);

	return sizeHint().height();
}

int Lockout::widthForHeight(int height) const
{
	checkLayout(height);

	return sizeHint().width();
}

// protected

bool Lockout::eventFilter(QObject */* o */, QEvent *e)
{
	if (e->type() != QEvent::MouseButtonPress)
		return false;

	QMouseEvent *mouseEvent = static_cast<QMouseEvent *>(e);

	if (mouseEvent->button() != QMouseEvent::RightButton)
		return false;

	if (!kapp->authorizeKAction("kicker_rmb"))
		return false;

	KConfig *conf = config();
	conf->setGroup("kshutdownlockout");

	KPopupMenu *popup = new KPopupMenu();
	popup->insertTitle("KShutDown");
	// run kshutdown
	popup->insertItem(
		SmallIcon("kshutdown"), i18n("&Run KShutDown"),
		this, SLOT(slotLogout())
	);
	// lock session
	popup->insertItem(
		SmallIcon("lock"), i18n("&Lock Session"),
		this, SLOT(slotLock())
	);
	// -
	popup->insertSeparator();
	// transparent
	int transparentItem = popup->insertItem(i18n("&Transparent"));
	popup->connectItem(transparentItem, this, SLOT(slotTransparent()));
	popup->setItemChecked(transparentItem, _transparent);
	// show lock button
	int showLockButtonItem = popup->insertItem(i18n("&Show Lock Button"));
	popup->connectItem(showLockButtonItem, this, SLOT(slotShowLockButton()));
	popup->setItemChecked(showLockButtonItem, _showLockButton);
	// configure kshutdown
	popup->insertItem(
		SmallIcon("configure"), i18n("&Configure: KShutDown..."),
		this, SLOT(slotConfigure())
	);
	popup->exec(mouseEvent->globalPos());
	delete popup;

	return true;
}

void Lockout::mouseDoubleClickEvent(QMouseEvent* e)
{
	propagateMouseEvent(e);
}

void Lockout::mouseMoveEvent(QMouseEvent* e)
{
	propagateMouseEvent(e);
}

void Lockout::mousePressEvent(QMouseEvent* e)
{
	propagateMouseEvent(e);
}

void Lockout::mouseReleaseEvent(QMouseEvent* e)
{
	propagateMouseEvent(e);
}

// private

QToolButton *Lockout::addButton(const QString &icon, const QString &text)
{
	QToolButton *b = new QToolButton(this, icon); // icon = widget name
	b->installEventFilter(this);
	b->setAutoRaise(_transparent);
	b->setBackgroundMode(X11ParentRelative);
	b->setPixmap(SmallIcon(icon));
	b->setMinimumSize(b->pixmap()->size());
	b->setSizePolicy(QSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding));
	QToolTip::add(b, text);

	return b;
}

void Lockout::checkLayout(int height) const
{
	// the -2 is due to kicker allowing us a width/height of 42 and the buttons
	// having a size of 44. So we rather cut those 2 pixels instead of changing
	// direction and wasting a lot of space.
	QSize hint = minimumSizeHint();
	QBoxLayout::Direction direction = _layout->direction();

	if (
		(direction == QBoxLayout::LeftToRight) &&
		(
			((orientation() == Vertical) && (hint.width() - 2 >= height)) ||
			((orientation() == Horizontal) && (hint.width() - 2 < height))
		)
	)
	{
		_layout->setDirection(QBoxLayout::TopToBottom);
	}
	else if (
		(direction == QBoxLayout::TopToBottom) &&
		(
			((orientation() == Vertical) && (hint.height() - 2 < height)) ||
			((orientation() == Horizontal) && (hint.height() - 2 >= height))
		)
	)
	{
		_layout->setDirection(QBoxLayout::LeftToRight);
	}
}

void Lockout::propagateMouseEvent(const QMouseEvent* e) const
{
	if (isTopLevel())
		return;

	QMouseEvent mouseEvent(
		e->type(),
		mapTo(topLevelWidget(), e->pos()),
		e->globalPos(),
		e->button(),
		e->state()
	);
	QApplication::sendEvent(topLevelWidget(), &mouseEvent);
}

void Lockout::runKShutDown() const
{
	KProcess *p = new KProcess();
	if (!p)
	{
		KMessageBox::error(0, i18n("Could not run KShutDown!"));

		return;
	}
	*p << "kshutdown";
	*p << "--init";
	if (!p->start(KProcess::Block))
		KMessageBox::error(0, i18n("Could not run KShutDown!"));
	delete p;
}

// private slots

void Lockout::slotConfigure()
{
	DCOPClient *client = kapp->dcopClient();
	if (!client->isApplicationRegistered("kshutdown"))
		runKShutDown();
	client->send("kshutdown", "KShutdownIface", "configure()", "");
}

void Lockout::slotIconChanged()
{
	_lockButton->setPixmap(SmallIcon("lock"));
	_logoutButton->setPixmap(SmallIcon("kshutdown"));
}

void Lockout::slotLock()
{
	DCOPClient *client = kapp->dcopClient();
	if (!client->isApplicationRegistered("kshutdown"))
		runKShutDown();
	client->send("kshutdown", "KShutdownIface", "lockScreen()", "");
}

void Lockout::slotLogout()
{
	DCOPClient *client = kapp->dcopClient();
	if (!client->isApplicationRegistered("kshutdown"))
		runKShutDown();
	client->send("kshutdown", "KShutdownIface", "makeVisible()", "");
}

void Lockout::slotShowLockButton()
{
	_showLockButton = !_showLockButton;
	if (_showLockButton)
		_lockButton->show();
	else
		_lockButton->hide();

	// save config
	KConfig* conf = config();
	conf->setGroup("kshutdownlockout");
	conf->writeEntry("ShowLockButton", _showLockButton);
	conf->sync();
}

void Lockout::slotTransparent()
{
	_transparent = !_transparent;
	_lockButton->setAutoRaise(_transparent);
	_logoutButton->setAutoRaise(_transparent);

	// save config
	KConfig* conf = config();
	conf->setGroup("kshutdownlockout");
	conf->writeEntry("Transparent", _transparent);
	conf->sync();
}
