/*
	msystemtray.cpp - A system tray icon
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "configuration.h"
#include "miscutils.h"
#include "mmainwindow.h"
#include "msystemtray.h"

#include <qtimer.h>

#include <kaction.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>
#include <kpopupmenu.h>
#include <kwin.h>

const double
	KS_TRAY_FADE_CHANGE = 0.05f,
	KS_TRAY_FADE_MAX = 0.7f;
const int KS_TRAY_FADE_TIMEOUT = 50;
const QColor KS_TRAY_FADE_COLOR = Qt::yellow;

MSystemTray::Mode MSystemTray::_mode = Always;
MSystemTray *MSystemTray::_instance = 0;

// public

MSystemTray::~MSystemTray()
{
	if (_flashIcon)
		delete _flashIcon;
	if (_saveIcon)
		delete _saveIcon;
}

void MSystemTray::flashIcon()
{
	if (_saveIcon)
		delete _saveIcon;
	_saveIcon = new QPixmap();
	*_saveIcon = *pixmap();
	_flashCount = 0;
	setPixmap(*_flashIcon);
	_flashTimer->start(200);
}

void MSystemTray::setActive(const bool yes)
{
	// kdDebug() << "MSystemTray::setActive: " << yes << endl;

	if (_active == yes)
		return;

	_active = yes;
	_incFade = _active;
	_fadeTimer->start(KS_TRAY_FADE_TIMEOUT);
}

void MSystemTray::setMode(const Mode mode)
{
	// kdDebug() << "MSystemTray::setMode: " << mode << endl;

	_mode = mode;
	switch (_mode)
	{
		case Always:
			ks_tray; // init instance
			if (ks_actions->active())
				_instance->setActive(true);
			_instance->show();
			MiscUtils::setAutostart(true);
			break;
		case IfActive:
			if (ks_actions->active())
			{
				ks_tray; // init instance
				_instance->setActive(true);
				_instance->show();
				MiscUtils::setAutostart(true);
			}
			else
			{
				delete _instance;
				_instance = 0;
				MiscUtils::setAutostart(false);
			}
			break;
		case Never:
			delete _instance;
			_instance = 0;
			MiscUtils::setAutostart(false);
			break;
	}
}

// protected

void MSystemTray::mousePressEvent(QMouseEvent *e)
{
	// middle mouse button - display actions menu
// TODO: 2.0: configure middle button action
	if (e->button() == MidButton)
		contextMenu()->popup(e->globalPos());
	else
		KSystemTray::mousePressEvent(e);
}

// private

MSystemTray::MSystemTray()
	: KSystemTray(ks_main, "MSystemTray"),
	_active(false),
	_incFade(true),
	_fadeValue(0.0f),
	_flashCount(0),
	_saveIcon(0)
{
	setPixmap(SmallIcon("exit", KIcon::SizeSmallMedium)); // 22x22

// TODO: 2.0: mouse wheel: decrease/increase end time by 5min. (?)

	// init fade timer
	_fadeTimer = new QTimer(this);
	connect(_fadeTimer, SIGNAL(timeout()), SLOT(slotFade()));

	// init flash timer
	_flashTimer = new QTimer(this);
	connect(_flashTimer, SIGNAL(timeout()), SLOT(slotFlashTimeout()));

	// init icons
	KPixmap pm(SmallIcon("exit", KIcon::SizeSmallMedium));
	_flashIcon = new KPixmap(KPixmapEffect::fade(pm, 0.7f, white));

	show();

	// add some items to the context menu
	KPopupMenu *pm_systemTrayMenu = contextMenu();
	pm_systemTrayMenu->setCaption("KShutDown");
	// id 0 = title
	ks_main->shutDownAction()->plug(pm_systemTrayMenu, 1);
	ks_main->rebootAction()->plug(pm_systemTrayMenu, 2);
	ks_main->lockScreenAction()->plug(pm_systemTrayMenu, 3);
	ks_main->logoutAction()->plug(pm_systemTrayMenu, 4);
	pm_systemTrayMenu->insertSeparator(5);
	ks_main->cancelAction()->plug(pm_systemTrayMenu, 6);
	pm_systemTrayMenu->insertSeparator(7);
	ks_main->configureKShutDownAction()->plug(pm_systemTrayMenu, 8);
	pm_systemTrayMenu->insertTearOffHandle();

	// quit signal
	connect(
		this, SIGNAL(quitSelected()),
		ks_main, SLOT(slotQuit()));
}

// private slots

void MSystemTray::slotFade()
{
	if (_incFade)
	{
		_fadeValue += KS_TRAY_FADE_CHANGE;
		if (_fadeValue > KS_TRAY_FADE_MAX)
		{
			_fadeTimer->stop();
			_fadeValue = KS_TRAY_FADE_MAX;
		}
	}
	else
	{
		_fadeValue -= KS_TRAY_FADE_CHANGE;
		if (_fadeValue < 0.0f)
		{
			_fadeTimer->stop();
			_fadeValue = 0.0f;
		}
	}
	KPixmap p(SmallIcon("exit", KIcon::SizeSmallMedium));
	setPixmap(KPixmapEffect::fade(p, _fadeValue, KS_TRAY_FADE_COLOR));
}

void MSystemTray::slotFlashTimeout()
{
	_flashCount++;

	if ((_flashCount % 2) == 0)
		setPixmap(*_saveIcon);
	else
		setPixmap(*_flashIcon);

	if (_flashCount == 6)
		_flashTimer->stop();
}
