//
//   File : kvi_string.cpp (/usr/build/NEW_kvirc/kvirc/src/kvicore/kvi_string.cpp)
//   Last major modification : Fri Mar 19 1999 03:20:45 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

//#warning "__paranoia_sanity_check is still defined!"
//#define __paranoia_sanity_check if(strlen(m_ptr)!=m_len)debug("PARANOIA : strlen(m_ptr) = %d and m_len = %d",strlen(m_ptr),m_len);
//#define __paranoia_sanity_check __range_valid(strlen(m_ptr)==((uint)m_len));

#define _KVI_STRING_CPP_
#include "kvi_string.h"

#include "kvi_memmove.h"
#include "kvi_malloc.h"


bool kvi_matchWildExpr(register const char *m1,register const char *m2)
{
	//Matches two regular expressions containging wildcards

	//          s1
	//          m1
	// mask1 : *xor
	// mask2 : xorand*xor
	//         m2
	//          s2

	//                        s2
	//                       m2
	//                       | 
	// XorT!xor@111.111.111.11
	//
	// *!*@*.net
	//      |
	//      m1
	//      s1
	//

	if(!(m1 && m2 && (*m1)))return false;
	const char * savePos1 = 0;
	const char * savePos2 = m2;
	while(*m1){ //loop managed by m1 (initially first mask)
		if(*m1=='*'){
			//Found a wildcard in m1
			savePos1 = ++m1;            //move to the next char and save the position...this is our jolly
			if(!*savePos1)return true;  //last was a wildcard , matches everything ahead...
			savePos2 = m2+1;            //next return state for the second string
			continue;                   //and return
		}
		if(!(*m2))return false;         //m2 finished and we had something to match here!
		if(tolower(*m1)==tolower(*m2)){
			//chars matched
			m1++;                       //Go ahead in the two strings
			m2++;                       //
			if((!(*m1)) && *m2 && savePos1){
				//m1 finished , but m2 not yet and we have a savePosition for m1 (there was a wildcard)...
				//retry matching the string following the * from the savePos2 (one char ahead last time)
				m1 = savePos1;          //back to char after wildcard
				m2 = savePos2;          //back to last savePos2
				savePos2++;             //next savePos2 will be next char
			}
		} else {
			if(*m2 == '*'){
				//A wlidcard in the second string
				//Invert the game : mask1 <-> mask2
				//mask2 now leads the game...
				savePos1 = m1;          //aux
				m1 = m2;                //...swap
				m2 = savePos1;          //...swap
				savePos1 = m1;          //sync save pos1
				savePos2 = m2 + 1;      //sync save pos2
				continue;               //...and again
			}
			if(savePos1){               //Have a jolly man...allow not matching...
				m1 = savePos1;          //go back to char after wildcard...need to rematch...
				m2 = savePos2;          //back to last savePos2
				savePos2++;             //and set next savePos2
			} else return false;        //No previous wildcards...not matched!
		}
	}
	return (!(*m2));                     //m1 surely finished , so for the match , m2 must be finished too

}

bool kvi_matchWildExprCS(register const char *m1,register const char *m2)
{
	if(!(m1 && m2 && (*m1)))return false;
	const char * savePos1 = 0;
	const char * savePos2 = m2;
	while(*m1){ //loop managed by m1 (initially first mask)
		if(*m1=='*'){
			//Found a wildcard in m1
			savePos1 = ++m1;            //move to the next char and save the position...this is our jolly
			if(!*savePos1)return true;  //last was a wildcard , matches everything ahead...
			savePos2 = m2+1;            //next return state for the second string
			continue;                   //and return
		}
		if(!(*m2))return false;         //m2 finished and we had something to match here!
		if((*m1)==(*m2)){
			//chars matched
			m1++;                       //Go ahead in the two strings
			m2++;                       //
			if((!(*m1)) && *m2 && savePos1){
				//m1 finished , but m2 not yet and we have a savePosition for m1 (there was a wildcard)...
				//retry matching the string following the * from the savePos2 (one char ahead last time)
				m1 = savePos1;          //back to char after wildcard
				m2 = savePos2;          //back to last savePos2
				savePos2++;             //next savePos2 will be next char
			}
		} else {
			if(*m2 == '*'){
				//A wlidcard in the second string
				//Invert the game : mask1 <-> mask2
				//mask2 now leads the game...
				savePos1 = m1;          //aux
				m1 = m2;                //...swap
				m2 = savePos1;          //...swap
				savePos1 = m1;          //sync save pos1
				savePos2 = m2 + 1;      //sync save pos2
				continue;               //...and again
			}
			if(savePos1){               //Have a jolly man...allow not matching...
				m1 = savePos1;          //go back to char after wildcard...need to rematch...
				m2 = savePos2;          //back to last savePos2
				savePos2++;             //and set next savePos2
			} else return false;        //No previous wildcards...not matched!
		}
	}
	return (!(*m2));                     //m1 surely finished , so for the match , m2 must be finished too

}
const char * kvi_extractToken(KviStr &str,const char *aux_ptr,char sep)
{
	__range_valid(aux_ptr);
	while(*aux_ptr && (*aux_ptr == sep))aux_ptr++;
	const char *p=aux_ptr;
	while(*p && (*p != sep))p++;
	str.m_len=p-aux_ptr;
	str.m_ptr = (char *)kvi_realloc(str.m_ptr,str.m_len+1);
	kvi_fastmove(str.m_ptr,aux_ptr,str.m_len);
	*(str.m_ptr+str.m_len)='\0';
	while(*p && (*p == sep))p++;
	return p;
}

const char * kvi_extractUpTo(KviStr &str,const char *aux_ptr,char sep)
{
	__range_valid(aux_ptr);
	const char *p=aux_ptr;
	while(*p && (*p != sep))p++;
	str.m_len=p-aux_ptr;
	str.m_ptr = (char *)kvi_realloc(str.m_ptr,str.m_len+1);
	kvi_fastmove(str.m_ptr,aux_ptr,str.m_len);
	*(str.m_ptr+str.m_len)='\0';
	return p;
}

int kvi_vsnprintf(char *buffer,int len,const char *fmt,va_list list)
{
	__range_valid(fmt);
	__range_valid(buffer);
	__range_valid(len > 0); //printing 0 characters is senseless
	register char *p = buffer;
	char *argString;
	long argValue;
	unsigned long argUValue;
	//9999999999999999999999999999999\0
	char numberBuffer[32]; //enough ? 10 is enough for 32bit unsigned int...
	char *pNumBuf;
	unsigned int tmp;

	for (p=buffer ; *fmt ; ++fmt) {
		if(len < 1)return (-1); //not enough space ... (in fact this could be len < 2 for the terminator)
		//copy up to a '%'
		if (*fmt != '%') {
			*p++ = *fmt;
			--len;
			continue;
		}
		++fmt; //skip this '%'
		switch(*fmt){
			case 's': //string
				argString = va_arg(list,char *);
				if(!argString){
					argString = "[!NULL!]";
					argValue = 8;
				} else argValue = strlen(argString);
				//check for space...
				if(len <= argValue)return (-1); //not enough space for buffer and terminator
				while(*argString)*p++ = *argString++;
				len -= argValue;
				continue;
			case 'd': //signed integer
				argValue = va_arg(list,int);
				if(argValue < 0){ //negative integer
					*p++ = '-';
					if(--len == 0)return (-1);
					argValue = -argValue; //need to have it positive
					// most negative integer exception (avoid completly senseless (non digit) resposnes)
					if(argValue < 0)argValue = 0;  //we get -0 here
				}
				//write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argValue / 10;
					*pNumBuf++ = argValue - (tmp * 10) + '0';
				} while((argValue = tmp));
				//copy now....
				argUValue = pNumBuf - numberBuffer; //length of the number string
				if(((uint)len) <= argUValue)return (-1); //not enough space for number and terminator
				do { *p++ = *--pNumBuf; } while(pNumBuf != numberBuffer);
				len -= argUValue;
				continue;
			case 'u': //unsigned integer
				argUValue = va_arg(list,unsigned int); //many implementations place int here
				//write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argUValue / 10;
					*pNumBuf++ = argUValue - (tmp * 10) + '0';
				} while((argUValue = tmp));
				//copy now....
				argValue = pNumBuf - numberBuffer; //length of the number string
				if(len <= argValue)return (-1); //not enough space for number and terminator
				do { *p++ = *--pNumBuf; } while(pNumBuf != numberBuffer);
				len -= argValue;
				continue;
			case 'c': //char
				//
				// I'm not sure about this...
				// In the linux kernel source the
				// unsigned char is extracted from an integer type.
				// We assume that gcc stacks a char argument
				// as sizeof(int) bytes value.
				// Is this always true ?
				//
				*p++ = (char)va_arg(list,int);
				--len;
				continue;
			default: //a normal percent
				*p++ = '%';  //write it
				if(--len == 0)return (-1); //not enough space for next char or terminator
				if(*fmt){        //this if is just in case that we have a % at the end of the string.
					*p++ = *fmt; //and write this char
					--len;
				}
				continue;
		}
	}
	if(len < 1)return (-1); //missing space for terminator
	*p = '\0';
	return p-buffer;
}

//
// Nearly the same as the above function...
//

int kvi_irc_vsnprintf(char *buffer,const char *fmt,va_list list,bool *bTruncated)
{
	__range_valid(fmt);
	__range_valid(buffer);
	register char *p = buffer;
	char *argString;
	long argValue;
	unsigned long argUValue;
	char numberBuffer[64]; //enough ? 10 is enough for 32bit unsigned int...
	char *pNumBuf;
	unsigned int tmp;
	*bTruncated = false;
	int len = 512;

	for (p=buffer ; *fmt ; ++fmt) {
		if(len < 3)goto truncate;
		//copy up to a '%'
		if (*fmt != '%') {
			*p++ = *fmt;
			--len;
			continue;
		}
		++fmt; //skip this '%'
		switch(*fmt){
			case 's': //string
				argString = va_arg(list,char *);
				if(!argString)argString = "[!NULL!]";
				//check for space...
				while(*argString){
					*p++ = *argString++;
					if(--len < 3)goto truncate;
				}
				continue;
			case 'd': //signed integer
				argValue = va_arg(list,int);
				if(argValue < 0){ //negative integer
					*p++ = '-';
					if(--len < 3)goto truncate; //place just for CRLF
					argValue = -argValue; //need to have it positive
					if(argValue < 0)argValue = 0; // -0 (hack the exception)
				}
				//write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argValue / 10;
					*pNumBuf++ = argValue - (tmp * 10) + '0';
				} while((argValue = tmp));
				//copy now....
				do {
					*p++ = *--pNumBuf;
					if(--len < 3)goto truncate;
				} while(pNumBuf != numberBuffer);
				continue;
			case 'u': //unsigned integer
				argUValue = va_arg(list,unsigned int); //many implementations place int here
				//write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argUValue / 10;
					*pNumBuf++ = argUValue - (tmp * 10) + '0';
				} while((argUValue = tmp));
				//copy now....
				if(--len < 3)goto truncate; //no place for digits
				do {
					*p++ = *--pNumBuf;
					if(--len < 3)goto truncate;
				} while(pNumBuf != numberBuffer);
				continue;
			case 'c': //char
				*p++ = (char)va_arg(list,int);
				--len;
				continue;
			default: //a normal percent
				*p++ = '%';  //write it
				if(--len < 3)goto truncate; //not enough space for next char
				if(*fmt){        //this if is just in case that we have a % at the end of the string.
					*p++ = *fmt; //and write this char
					--len;
				}
				continue;
		}
	}
	//succesfull finish
	__range_valid(len >= 2);
	*p++ = '\r';
	*p   = '\n';
	return ((p-buffer)+1);
truncate:
	__range_valid(len >= 2);
	*bTruncated = true;
	*p++ = '\r';
	*p   = '\n';
	return ((p-buffer)+1);
}

#ifndef COMPILE_i386_ASM_CODE

bool kvi_strEqualCS(const char *str1,const char *str2)
{
	__range_valid(str1);
	__range_valid(str2);
	register unsigned char *s1 = (unsigned char *)str1;
	register unsigned char *s2 = (unsigned char *)str2;
	while(*s1)if(*s1++ != *s2++)return false;
	return (*s1 == *s2);
}

bool kvi_strEqualCSN(const char *str1,const char *str2,int len)
{
	__range_valid(str1);
	__range_valid(str2);
	__range_valid(len >= 0);
	register unsigned char *s1 = (unsigned char *)str1;
	register unsigned char *s2 = (unsigned char *)str2;
	while(len-- && *s1)if(*s1++ != *s2++)return false;
	return (len < 0);
}

#endif

bool kvi_strEqualCIN(const char *str1,const char *str2,int len)
{
	__range_valid(str1);
	__range_valid(str2);
	__range_valid(len >= 0);
	register unsigned char *s1 = (unsigned char *)str1;
	register unsigned char *s2 = (unsigned char *)str2;
	while(len-- && *s1)if(tolower(*s1++) != tolower(*s2++))return false;
	return (len < 0);
}

bool kvi_strEqualCI(const char *str1,const char *str2)
{
	__range_valid(str1);
	__range_valid(str2);
	register unsigned char *s1 = (unsigned char *)str1;
	register unsigned char *s2 = (unsigned char *)str2;
	while(*s1)if(tolower(*s1++) != tolower(*s2++))return false;
	return (*s1 == *s2);
}

//note that greater here means that come AFTER in the alphabetic order
int kvi_strcmpCI(const char *str1,const char *str2)
{
	//abcd abce
	__range_valid(str1);
	__range_valid(str2);
	register unsigned char *s1 = (unsigned char *)str1;
	register unsigned char *s2 = (unsigned char *)str2;
	int diff;
	unsigned char rightchar;
	while(!(diff=(rightchar=tolower(*s2++)) - tolower(*s1++)))if(!rightchar)break;
    return diff; //diff is nonzero or end of both was reached (it is positive if *s2 > *s1
}

int kvi_strcmpCS(const char *str1,const char *str2)
{
	//abcd abce
	__range_valid(str1);
	__range_valid(str2);
	register unsigned char *s1 = (unsigned char *)str1;
	register unsigned char *s2 = (unsigned char *)str2;
	int diff;
	while(!(diff=(*s2)-(*s1++)))if(!*s2++)break;
    return diff; //diff is nonzero or end of both was reached (it is positive if *s2 > *s1
}

int kvi_strMatchRevCS(const char *str1, const char *str2, int index)
{
	__range_valid(str1);
	__range_valid(str2);
	register char *s1=(char *)str1;
	register char *s2=(char *)str2;

	int curlen=strlen(str1);
	int diff;

	if (index<0 || index >= curlen) index = curlen-1;

	s1+=index;
	while (*s2) s2++;
	s2--;

// now start comparing
        while (1) {
                /* in this case, we have str1 = "lo" and str2 = "hello" */
                if (s1<str1 && !(s2<str2)) return 256;
                if (s2<str2) return 0;
                if ((diff=(*s1)-(*s2))) return diff;
                s1--;
                s2--;
                }

}   

KviStr::KviStr()
{
	m_ptr = (char *)kvi_malloc(1);
	*m_ptr = '\0';
	m_len = 0;
}

KviStr::KviStr(const char *str)
{
	//Deep copy constructor
	if(str){
		//Deep copy
		m_len = strlen(str);
		m_ptr = (char *)kvi_malloc(m_len+1);
		kvi_fastmove(m_ptr,str,m_len+1);
	} else {
		m_ptr = (char *)kvi_malloc(1);
		*m_ptr = '\0';
		m_len = 0;
	}
//	__paranoia_sanity_check;
}

KviStr::KviStr(const char *str,int len)
{
	__range_valid(str);
	__range_valid(len <= ((int)strlen(str)));
	__range_valid(len >= 0);
	m_len = len;
	m_ptr = (char *)kvi_malloc(m_len+1);
	kvi_fastmove(m_ptr,str,m_len);
	*(m_ptr+m_len) = '\0';
//	__paranoia_sanity_check;
}

KviStr::KviStr(const char *bg,const char *end)
{
	__range_valid(bg);
	__range_valid(end);
	__range_valid(bg <= end);
	m_len = end-bg;
	m_ptr = (char *)kvi_malloc(m_len +1);
	kvi_fastmove(m_ptr,bg,m_len);
	*(m_ptr + m_len)='\0';
//	__paranoia_sanity_check;
}

KviStr::KviStr(KviFormatConstructorTag tag,const char *fmt,...)
{
	m_ptr=(char *)kvi_malloc(256);
	//First try
	va_list list;
	va_start(list,fmt);
	//print...with max 256 chars
	m_len=kvi_vsnprintf(m_ptr,256,fmt,list);
	va_end(list);

	//check if we failed
	if(m_len < 0){
		//yes , failed....
		int dummy=256;
		do{ //we failed , so retry with 256 more chars
			dummy+=256;
			//realloc
			m_ptr=(char *)kvi_realloc(m_ptr,dummy);
			//print...
			va_start(list,fmt);
			m_len=kvi_vsnprintf(m_ptr,dummy,fmt,list);
			va_end(list);
		} while(m_len < 0);
	}
	//done...
	//now m_len is the length of the written string not including the terminator...
	//perfect! :)
	m_ptr=(char *)kvi_realloc(m_ptr,m_len+1);
//	__paranoia_sanity_check;
}

KviStr::KviStr(const KviStr &str)
{
	__range_valid(str.m_ptr);
	m_len = str.m_len;
	m_ptr = (char *)kvi_malloc(m_len+1);
	kvi_fastmove(m_ptr,str.m_ptr,m_len+1);
//	__paranoia_sanity_check;
}

KviStr::KviStr(const QString &str)
{
#ifdef COMPILE_USE_LOCAL_8BIT_ENCODING
	QCString tmp = str.local8Bit();
	const char *ptr = tmp.data();
#else
	const char *ptr=str.latin1();
#endif
	if(ptr){
		m_len = strlen(ptr);
		m_ptr = (char *)kvi_malloc(m_len+1);
		kvi_fastmove(m_ptr,ptr,m_len+1);
	} else {
		m_ptr = (char *)kvi_malloc(1);
		*m_ptr = '\0';
		m_len = 0;
	}
//	__paranoia_sanity_check;
}

KviStr::KviStr(char c,int fillLen)
{
	__range_valid(fillLen >= 0);
	m_len = fillLen;
	m_ptr = (char *)kvi_malloc(m_len+1);
	register char *p=m_ptr;
	while(fillLen--)*p++=c;
	*p='\0';
//	__paranoia_sanity_check;
}

KviStr::~KviStr()
{
	kvi_free(m_ptr);
}

KviStr & KviStr::operator=(const KviStr &str)
{
	__range_valid(str.m_ptr);
	__range_valid(str.m_ptr != m_ptr);
	m_len = str.m_len;
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
	kvi_fastmove(m_ptr,str.m_ptr,m_len+1);
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::operator=(const char *str)
{
//	__range_valid(str);
	if(str){
		m_len = strlen(str);
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
		kvi_memmove(m_ptr,str,m_len+1);
	} else {
		m_ptr = (char *)kvi_realloc(m_ptr,1);
		*m_ptr = '\0';
		m_len = 0;
	}
//	__paranoia_sanity_check;
	return (*this);
}

bool KviStr::hasNonWhiteSpaceData() const
{
	const char * aux = m_ptr;
	while(*aux)
	{
		if(((*aux) != ' ') && ((*aux) != '\t'))return true;
		aux++;
	}
	return false;
}

static char digits[16] = { '0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F' };

void KviStr::bufferToHex(char *buffer,int len)
{
	__range_valid(buffer);
	m_len = (len * 2);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len + 1);
	char * aux = m_ptr;
	while(len)
	{
		*aux = digits[(unsigned int)(((unsigned char)(*buffer)) / 16)];
		aux++;
		*aux = digits[(unsigned int)(((unsigned char)(*buffer)) % 16)];
		aux++;
		len--;
		buffer++;
	}
	*(m_ptr+m_len) = '\0';
}

static char get_decimal_from_hex_digit_char(char dgt)
{
	if((dgt >= '0') && (dgt <= '9'))return (dgt - '0');
	if((dgt >= 'A') && (dgt <= 'F'))return (10 + (dgt - 'A'));
	if((dgt >= 'a') && (dgt <= 'f'))return (10 + (dgt - 'a'));
	return 0;
}

int KviStr::hexToBuffer(char ** buffer,bool bNullToNewlines)
{
	int len;
	if(m_len % 2)len = (m_len / 2) + 1;
	else len = (m_len / 2);
	*buffer = (char *)kvi_malloc(len);

	char * ptr = *buffer;

	char * aux = m_ptr;
	while(*aux)
	{
		*ptr = get_decimal_from_hex_digit_char(*aux) * 16;
		aux++;
		if(*aux)
		{
			*ptr += get_decimal_from_hex_digit_char(*aux);
			aux++;
		}
		if(bNullToNewlines)if(!(*ptr))*ptr = '\n';
		ptr++;
	}
	return len;
}

KviStr & KviStr::setStr(const char *str,int len)
{
	int alen = strlen(str);
	if((len < 0) || (len > alen))m_len = alen;
	else m_len = len;
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
	kvi_memmove(m_ptr,str,m_len);
	*(m_ptr+m_len) = '\0';
	return (*this);
}

KviStr & KviStr::operator=(const QString &str)
{
#ifdef COMPILE_USE_LOCAL_8BIT_ENCODING
	QCString tmp = str.local8Bit();
	const char *p = tmp.data();
#else
	const char *p=str.latin1();
#endif
	if(p){
		m_len = strlen(p);
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
		kvi_fastmove(m_ptr,p,m_len+1);
	} else {
		m_ptr = (char *)kvi_realloc(m_ptr,1);
		*m_ptr = '\0';
		m_len = 0;
	}
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::operator=(char c)
{
	m_len = 1;
	m_ptr = (char *)kvi_realloc(m_ptr,2);
	*m_ptr = c;
	*(m_ptr+1)='\0';
//	__paranoia_sanity_check;
	return (*this);
}

void KviStr::append(char c)
{
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+2);
	*(m_ptr+m_len)=c;
	m_len++;
	*(m_ptr+m_len)='\0';
//	__paranoia_sanity_check;
}

void KviStr::append(const KviStr &str)
{
	__range_valid(str.m_ptr);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+str.m_len+1);
	kvi_fastmove((m_ptr+m_len),str.m_ptr,str.m_len+1);
	m_len += str.m_len;
//	__paranoia_sanity_check;
}

void KviStr::append(const char *str)
{
	if(!str)return;
	int len = strlen(str);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+len+1);
	kvi_fastmove((m_ptr+m_len),str,len+1);
	m_len += len;
//	__paranoia_sanity_check;
}

void KviStr::append(const QString &str)
{
#ifdef COMPILE_USE_LOCAL_8BIT_ENCODING
	QCString tmp = str.local8Bit();
	const char *ptr = tmp.data();
#else
	const char *ptr = str.latin1();
#endif
	if(!ptr)return;
	int len = strlen(ptr);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+len+1);
	kvi_fastmove((m_ptr+m_len),ptr,len+1);
	m_len += len;
//	__paranoia_sanity_check;
}

void KviStr::append(const char *str,int len)
{
	__range_valid(str);
//	__range_valid(len <= ((int)strlen(str)));
	__range_valid(len >= 0);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+len+1);
	kvi_fastmove((m_ptr+m_len),str,len);
	m_len += len;
	*(m_ptr + m_len)='\0';
//	__paranoia_sanity_check;
}

void KviStr::append(KviFormatConstructorTag tag,const char *fmt,...)
{
	int auxLen;
	m_ptr=(char *)kvi_realloc(m_ptr,m_len + 256);
	//First try
	va_list list;
	va_start(list,fmt);
	//print...with max 256 chars
	auxLen =kvi_vsnprintf(m_ptr + m_len,256,fmt,list);
	va_end(list);

	//check if we failed
	if(auxLen < 0){
		//yes , failed....
		int dummy=256;
		do{ //we failed , so retry with 256 more chars
			dummy+=256;
			//realloc
			m_ptr=(char *)kvi_realloc(m_ptr,m_len + dummy);
			//print...
			va_start(list,fmt);
			auxLen=kvi_vsnprintf(m_ptr + m_len,dummy,fmt,list);
			va_end(list);
		} while(auxLen < 0);
	}
	m_len += auxLen;
	//done...
	//now m_len is the length of the written string not including the terminator...
	//perfect! :)
	m_ptr=(char *)kvi_realloc(m_ptr,m_len+1);
//	__paranoia_sanity_check;
}

void KviStr::extractFromString(const char *begin,const char *end)
{
	__range_valid(begin);
	__range_valid(end);
	__range_valid(end >= begin);
	m_len = end-begin;
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
	kvi_fastmove(m_ptr,begin,m_len);
	*(m_ptr + m_len)='\0';
//	__paranoia_sanity_check;
}

void KviStr::prepend(const KviStr &str)
{
	__range_valid(str.m_ptr);
	__range_valid(str.m_ptr != m_ptr);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+str.m_len+1);
	kvi_memmove((m_ptr+str.m_len),m_ptr,m_len+1); //move self
	kvi_fastmove(m_ptr,str.m_ptr,str.m_len);
	m_len += str.m_len;
//	__paranoia_sanity_check;
}

void KviStr::prepend(const char *str)
{
	if(!str)return;
	int len = strlen(str);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+len+1);
	kvi_memmove((m_ptr+len),m_ptr,m_len+1); //move self
	kvi_fastmove(m_ptr,str,len);
	m_len += len;
//	__paranoia_sanity_check;
}

void KviStr::prepend(const char *str,int len)
{
	__range_valid(str);
	__range_valid(len <= ((int)strlen(str)));
	__range_valid(len >= 0);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+len+1);
	kvi_memmove((m_ptr+len),m_ptr,m_len+1); //move self
	kvi_fastmove(m_ptr,str,len);
	m_len += len;
//	__paranoia_sanity_check;
}

void KviStr::toUpper()
{
	register char *p=m_ptr;
	while(*p)*p++=toupper(*p);
}

void KviStr::toLower()
{
	register char *p=m_ptr;
	while(*p)*p++=tolower(*p);
}

KviStr KviStr::upper() const
{
	KviStr tmp(*this);
	tmp.toUpper();
	return tmp;
}

KviStr KviStr::lower() const
{
	KviStr tmp(*this);
	tmp.toLower();
	return tmp;
}

KviStr KviStr::left(int maxLen)
{
	if(maxLen <= 0)
	{
		KviStr empty;
		return empty;
	}
	if(maxLen > m_len)maxLen=m_len;
	KviStr str(m_ptr,maxLen);
	return str;
}

KviStr KviStr::right(int maxLen)
{
	if(maxLen <= 0)
	{
		KviStr empty;
		return empty;
	}
	if(maxLen > m_len)maxLen=m_len;
	KviStr str((m_ptr+(m_len-maxLen)),maxLen);
	return str;
}

KviStr KviStr::middle(int idx,int maxLen)
{
	__range_valid(maxLen >= 0);
	__range_valid(idx >= 0);
	if((maxLen <= 0) || (idx < 0)){ //max len negative...invalid params
		KviStr ret;
		return ret;
	}
	if((maxLen + idx) <= m_len){ //valid params
		KviStr str(m_ptr+idx,maxLen);
		return str;
	}
	if(idx < m_len){ //string shorter than requested
		KviStr str(m_ptr+idx);
		return str;
	}
	// idx out of bounds
	KviStr ret;
	return ret;
}

KviStr & KviStr::insert(int idx,const char *data)
{
	__range_valid(data);
	if(idx <= m_len){
		int len = strlen(data);
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+len+1);
		kvi_memmove(m_ptr+idx+len,m_ptr+idx,(m_len - idx)+1);
		kvi_fastmove(m_ptr+idx,data,len);
		m_len+=len;
	}
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::insert(int idx,char c)
{
	if(idx <= m_len){
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+2);
		kvi_memmove(m_ptr+idx+1,m_ptr+idx,(m_len - idx)+1);
		m_len++;
		*(m_ptr + idx) = c;
	}
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::replaceAll(char c,const char *str)
{
	int idx = findFirstIdx(c);
	KviStr tmp;
	while(idx >= 0){
		if(idx > 0)tmp += left(idx);
		cutLeft(idx+1);
		tmp.append(str);
		idx = findFirstIdx(c);
	}
	tmp.append(*this);
	// Now copy
	m_len = tmp.m_len;
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
	kvi_fastmove(m_ptr,tmp.m_ptr,m_len+1);
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::replaceAll(char *toFind,const char *str,bool bCaseS)
{
	int len = strlen(toFind);
	int idx = findFirstIdx(toFind,bCaseS);
	KviStr tmp;
	while(idx >= 0){
		if(idx > 0)tmp += left(idx);
		cutLeft(idx+len);
		tmp.append(str);
		idx = findFirstIdx(toFind,bCaseS);
	}
	tmp.append(*this);
	// Now copy
	m_len = tmp.m_len;
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
	kvi_fastmove(m_ptr,tmp.m_ptr,m_len+1);
//	__paranoia_sanity_check;
	return (*this);
}

int KviStr::contains(char c,bool caseS)
{
	register char *p = m_ptr;
	int cnt=0;
	if(caseS){
		while(*p){
			if(*p == c)cnt++;
			p++;
		}
	} else {
		char b=tolower(c);
		while(*p){
			if(tolower(*p) == b)cnt++;
			p++;
		}
	}
	return cnt;
}

int KviStr::contains(const char *str,bool caseS)
{
	__range_valid(str);
	register char *p = m_ptr;
	int cnt=0;
	int len = strlen(str);
	if(caseS){
		while(*p){
			if(*p == *str){
				if(kvi_strEqualCSN(p,str,len))cnt++;
			}
			p++;
		}
	} else {
		while(*p){
			char c = tolower(*str);
			if(tolower(*p) == c){
				if(kvi_strEqualCIN(p,str,len))cnt++;
			}
			p++;
		}
	}
	return cnt;
}

KviStr & KviStr::setNum(long num)
{
	char numberBuffer[30];
	bool bNegative = false;
	long tmp;
	register char *p;
	register char *pNumBuf = numberBuffer;

	// somebody can explain me why 	-(-2147483648) = -2147483648 ? (2^31)
	// it is like signed char x = 128 ---> 10000000 that is signed -0 (!?)
	// mmmmh...or it is assumed to be -128 (a number rappresentation exception)
	// at least on my machine it happens...

	// found the solution by myself today...
	//
	// ABS(3)              Linux Programmer's Manual              ABS(3)
	// NAME
	//        abs - computes the absolute value of an integer.
	// ...
	// DESCRIPTION
	//        The abs() function computes the absolute value of the integer argument j.
	// RETURN VALUE
	//        Returns the absolute value of the integer argument.
	// CONFORMING TO
	//        SVID 3, POSIX, BSD 4.3, ISO 9899
	// NOTE ##################################################################################
	//        Trying to take the absolute value of the most negative integer is not defined.
	// #######################################################################################

	// so should i use temporaneous doubles to make calculations ?

	if(num < 0){ //negative integer
		bNegative = true;
		num = -num; //need to have it positive
		if(num < 0){ // 2^31 exception
			// We need to avoid absurd responses like ".(./),." :)
			num = 0; // we get a negative zero here...it is still an exception
		}
	}

	//write the number in a temporary buffer (at least '0')
	do {
		tmp = num / 10;
		*pNumBuf++ = num - (tmp * 10) + '0';
	} while((num = tmp));

	//copy now....
	m_len = pNumBuf - numberBuffer; //length of the number string
	if(bNegative){
		m_len++;
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
		p=m_ptr;
		*p++='-';
	} else {
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
		p=m_ptr;
	}
	do { *p++ = *--pNumBuf; } while(pNumBuf != numberBuffer);
	*(m_ptr + m_len)='\0';
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::setNum(unsigned long num)
{
	char numberBuffer[30];
	unsigned long tmp;
	register char *p;
	register char *pNumBuf = numberBuffer;

	//write the number in a temporary buffer (at least '0')
	do {
		tmp = num / 10;
		*pNumBuf++ = num - (tmp * 10) + '0';
	} while((num = tmp));

	//copy now....
	m_len = pNumBuf - numberBuffer; //length of the number string
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
	p=m_ptr;
	do { *p++ = *--pNumBuf; } while(pNumBuf != numberBuffer);
	*(m_ptr + m_len)='\0';
//	__paranoia_sanity_check;
	return (*this);
}

long KviStr::toLong(bool *bOk) const
{
	long result = 0;
	if(bOk)*bOk = false;
	register char *p=m_ptr;
	bool bNeg = false;
	while(isspace(*p))p++; //skip spaces
	if(*p == '-'){
		bNeg = true;
		p++;
	} else {
		if(*p == '+')p++;
	}
	if(isdigit(*p)){                      //point to something interesting ?
		do{
			result = (result * 10) + (*p - '0');
			p++;
		} while(isdigit(*p));
		if(bNeg)result = -result;
		while(isspace(*p))p++;        //skip trailing spaces
		if(*p)return 0;               //if this is not the end...die.
		if(bOk)*bOk = true;
		return result;
	}
	return 0;
}

unsigned long KviStr::toULong(bool *bOk) const
{
	unsigned long result = 0;
	if(bOk)*bOk = false;
	register char *p=m_ptr;
	while(isspace(*p))p++; //skip spaces
	if(isdigit(*p)){                      //point to something interesting ?
		do{
			result = (result * 10) + (*p - '0');
			p++;
		} while(isdigit(*p));
		while(isspace(*p))p++;        //skip trailing spaces
		if(*p)return 0;               //if this is not the end...die.
		if(bOk)*bOk = true;
		return result;
	}
	return 0;
}

long KviStr::toLongExt(bool *bOk,int base)
{
	if(m_len == 0){
		if(bOk)*bOk = false;
		return 0;
	}
	char * endptr;
	long result = strtol(m_ptr,&endptr,base);
	if(*endptr){
		// must be whitespaces , otherwise there is trailing garbage inside
		while(isspace(*endptr) && (*endptr))endptr++;
		if(*endptr){
			// still not at the end
			// trailing garbage not allowed
			if(bOk)*bOk = false;
			return result;
		}
	}
	if(bOk)*bOk = true;
	return result;
}

//
//working code , but unused in kvirc
//
//unsigned long KviStr::toULongExt(bool *bOk = 0,int base = 0)
//{
//	if(m_len == 0){
//		if(bOk)*bOk = false;
//		return 0;
//	}
//	char * endptr;
//	unsigned long result = strtoul(m_ptr,&endptr,base);
//	if(*endptr != '\0'){
//		if(bOk)*bOk = false;
//	}
//	return result;	
//}

KviStr & KviStr::cutLeft(int len)
{
	__range_valid(len >= 0);
//	__paranoia_sanity_check;
	if(len <= m_len){
		m_len -= len;
		kvi_memmove(m_ptr,m_ptr+len,m_len+1);
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
	} else {
		m_ptr = (char *)kvi_realloc(m_ptr,1);
		*m_ptr = '\0';
		m_len = 0;
	}
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::cutRight(int len)
{
	__range_valid(len >= 0);
//	__paranoia_sanity_check;
	if(len <= m_len){
		m_len -= len;
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
		*(m_ptr +m_len)='\0';
	} else {
		m_ptr = (char *)kvi_realloc(m_ptr,1);
		*m_ptr = '\0';
		m_len = 0;
	}
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::cut(int idx,int len)
{
	__range_valid(idx >= 0);
	__range_valid(len >= 0);
	if(idx < m_len){
		// idx = 3 len = 3 m_len = 10
		// 0123456789
		// abcdefghij
		//    ^  ^
		//   p1  p2
		char * p1 = m_ptr+idx;
		if(len + idx > m_len)len = m_len - idx;
		char * p2 = p1+len;
		kvi_memmove(p1,p2,(m_len - (len+idx)) +1);
		m_len -= len;
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
	}
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::cutToFirst(char c,bool bIncluded)
{
	int idx = findFirstIdx(c);
	if(idx != -1)cutLeft(bIncluded ? idx + 1 : idx);
	return (*this);
}

KviStr & KviStr::cutFromFirst(char c,bool bIncluded)
{
	int idx = findFirstIdx(c);
	if(idx != -1)cutRight(bIncluded ? (m_len - idx) : (m_len - (idx + 1)));
	return (*this);
}

KviStr & KviStr::cutToLast(char c,bool bIncluded)
{
	int idx = findLastIdx(c);
	if(idx != -1)cutLeft(bIncluded ? idx + 1 : idx);
	return (*this);
}

KviStr & KviStr::setLen(int len)
{
	__range_valid(len >= 0);
	m_ptr = (char *)kvi_realloc(m_ptr,len+1);
	*(m_ptr+len)='\0';
	m_len = len;
	return (*this);
}

KviStr & KviStr::stripLeftWhiteSpace()
{
	register char *p=m_ptr;
	while(isspace(*p))p++;
	m_len -= (p-m_ptr);
	kvi_memmove(m_ptr,p,m_len+1);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
//	__paranoia_sanity_check;
	return (*this);
}

KviStr & KviStr::stripLeft(char c)
{
	__range_valid(c != '\0');
	register char *p=m_ptr;
	while(*p == c)p++;
	m_len -= (p-m_ptr);
	kvi_memmove(m_ptr,p,m_len+1);
	m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
//	__paranoia_sanity_check;
	return (*this);
}

bool KviStr::getToken(KviStr & str,char sep)
{
	__range_valid(str.m_ptr);
	__range_valid(str.m_ptr != m_ptr);
	register char *p=m_ptr;
	//skip to the end
	while(*p && (*p != sep))p++;
	//0123456789
	//abcd xyz
	//^   ^
	str.m_len = p-m_ptr;
	str.m_ptr = (char *)kvi_realloc(str.m_ptr,str.m_len+1);
	kvi_fastmove(str.m_ptr,m_ptr,str.m_len);
	*(str.m_ptr + str.m_len)='\0';
	while(*p && (*p == sep))p++;
	cutLeft(p-m_ptr);
//	__paranoia_sanity_check;
	return (m_len != 0);
}

bool KviStr::getLine(KviStr &str)
{
	__range_valid(str.m_ptr);
	__range_valid(str.m_ptr != m_ptr);
	if(m_len == 0)return false;
	register char *p=m_ptr;
	//skip to the end
	while(*p && (*p != '\n'))p++;
	//0123456789
	//abcd xyz
	//^   ^
	str.m_len = p-m_ptr;
	str.m_ptr = (char *)kvi_realloc(str.m_ptr,str.m_len+1);
	kvi_fastmove(str.m_ptr,m_ptr,str.m_len);
	*(str.m_ptr + str.m_len)='\0';
	p++;
	cutLeft(p-m_ptr);
//	__paranoia_sanity_check;
	return true;
}

KviStr KviStr::getToken(char sep)
{
	register char *p=m_ptr;
	while(*p && (*p != sep))p++;
	KviStr ret(m_ptr,p);
	while(*p && (*p == sep))p++;
	cutLeft(p-m_ptr);
	return ret;
}

KviStr & KviStr::sprintf(const char *fmt,...)
{
	m_ptr=(char *)kvi_realloc(m_ptr,256);
	//First try
	va_list list;
	va_start(list,fmt);
	//print...with max 256 chars
	m_len=kvi_vsnprintf(m_ptr,256,fmt,list);
	va_end(list);

	//check if we failed
	if(m_len < 0){
		//yes , failed....
		int dummy=256;
		do{ //we failed , so retry with 256 more chars
			dummy+=256;
			//realloc
			m_ptr=(char *)kvi_realloc(m_ptr,dummy);
			//print...
			va_start(list,fmt);
			m_len=kvi_vsnprintf(m_ptr,dummy,fmt,list);
			va_end(list);
		} while(m_len < 0);
	}
	//done...
	//now m_len is the length of the written string not including the terminator...
	//perfect! :)
	m_ptr=(char *)kvi_realloc(m_ptr,m_len+1);
//	__paranoia_sanity_check;
	return (*this);
}

//================= findFirstIdx ==================//

int KviStr::findFirstIdx(char c)
{
	register char *p=m_ptr;
	while(*p && (*p != c))p++;
	return (*p ? p-m_ptr : -1);
}

//================= findFirstIdx ==================//

int KviStr::findFirstIdx(const char *str,bool caseS)
{
	__range_valid(str);
	register char *p=m_ptr;
	int len = strlen(str);
	if(caseS){
		for(;;){
			while(*p && (*p != *str))p++;
			if(*p){
				if(kvi_strEqualCSN(str,p,len))return (p-m_ptr);
				else p++;
			} else return -1;
		}
	} else {
		for(;;){
			char tmp = toupper(*str);
			while(*p && (toupper(*p) != tmp))p++;
			if(*p){
				if(kvi_strEqualCIN(str,p,len))return (p-m_ptr);
				else p++;
			} else return -1;
		}
	}
}

//================= findLastIdx ==================//

int KviStr::findLastIdx(char c)
{
	//Empty string ?
	if(m_len < 1)return -1;
	//p points to the last character in the string
	register char *p=((m_ptr+m_len)-1);
	//go back until we find a match or we run to the first char in the string.
	while((*p != c) && (p > m_ptr))p--;
	//if *p == c --> matched , else we are at the beginning of the string.
	return ((*p == c)? p-m_ptr : -1);
}

//================= findLastIdx ==================//

int KviStr::findLastIdx(const char *str,bool caseS)
{
	__range_valid(str);
	//Calc the len of the searched string
	int len = strlen(str);
	//Too long ?
	if(m_len < len)return -1;
	//p points to the last character in the string
	register char *p=((m_ptr+m_len)-1);
	if(caseS){
		for(;;){
			//go back until we find a character that mathes or we run to the first char.
			while((*p != *str) && (p > m_ptr))p--;
			if(*p == *str){
				//maybe occurence....
				if(kvi_strEqualCSN(str,p,len))return (p-m_ptr);
				else {
					//Nope...continue if there is more data to check...
					if(p == m_ptr)return -1;
					p--;
				}
			} else return -1; //Beginning of the string
		}
	} else {
		// case insensitive
		for(;;){
			//go back until we find a character that mathes or we run to the first char.
			char tmp = toupper(*str);
			while((toupper(*p) != tmp) && (p > m_ptr))p--;
			if(toupper(*p) == tmp){
				//maybe occurence....
				if(kvi_strEqualCIN(str,p,len))return (p-m_ptr);
				else {
					//Nope...continue if there is more data to check...
					if(p == m_ptr)return -1;
					p--;
				}
			} else return -1; //Beginning of the string
		}

	}
}

//================= stripWhiteSpace ==================//

KviStr & KviStr::stripWhiteSpace()
{
	// 0123456789
	//    abcd   0
	// ^        ^
	// left   right
	register char *left=m_ptr;
	register char *right=m_ptr+m_len-1;
	// skip initial spaces
	while(isspace(*left))left++;
	if(*left){
		// valid string , left points to first non-space
		while((right >= left) && isspace(*right))right--;
		// 0123456789
		//    abcd   0
		//    ^  ^
		// left   right	
		m_len = (right - left)+1;
		kvi_memmove(m_ptr,left,m_len);
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
		*(m_ptr+m_len)='\0';
	} else {
		m_ptr = (char *)kvi_realloc(m_ptr,1);
		*m_ptr = '\0';
		m_len = 0;
	}
	return (*this);
}

//================= stripSpace ==================//

KviStr & KviStr::stripSpace()
{
	// 0123456789
	//    abcd   0
	// ^        ^
	// left   right
	register char *left=m_ptr;
	register char *right=m_ptr+m_len-1;
	// skip initial spaces
	while((*left == ' ') || (*left == '\t'))left++;
	if(*left){
		// valid string , left points to first non-space
		while((right >= left) && ((*right == ' ') || (*right == '\t')))right--;
		// 0123456789
		//    abcd   0
		//    ^  ^
		// left   right	
		m_len = (right - left)+1;
		kvi_memmove(m_ptr,left,m_len);
		m_ptr = (char *)kvi_realloc(m_ptr,m_len+1);
		*(m_ptr+m_len)='\0';
	} else {
		m_ptr = (char *)kvi_realloc(m_ptr,1);
		*m_ptr = '\0';
		m_len = 0;
	}
	return (*this);
}


bool KviStr::isNum()
{
	register char *p=m_ptr;
	while(isspace(*p))p++;
	if(*p=='-')p++;
	if(!isdigit(*p))return false;
	while(isdigit(*p))p++;
	while(isspace(*p))p++;
	return (*p=='\0');
}

bool KviStr::isUnsignedNum()
{
	register char *p=m_ptr;
	while(isspace(*p))p++;
	if(!isdigit(*p))return false;
	while(isdigit(*p))p++;
	while(isspace(*p))p++;
	return (*p=='\0');
}

//void KviStr::pointerToBitString(const void * ptr)
//{
//	m_len = (sizeof(void *) * 8);
//	m_ptr = kvi_realloc(m_ptr,m_len + 1);
//	for(int i=0;i < m_len;i++)
//	{
//		m_ptr[i] = (ptr & 1) ? '1' : '0';
//		ptr >> 1;
//	}
//	m_ptr[i] = '\0';
//}
//
//void * KviStr::bitStringToPointer()
//{
//	if(m_len != (sizeof(void *) * 8))return 0;
//	const char * aux = m_ptr;
//	void * ptr = 0;
//	for(int i=m_len - 1;i >= 0;i--)
//	{
//		if(m_ptr[i] == '1')ptr &= 1;
//		else if(m_ptr[i] !='0')return 0;
//		ptr << 1;
//	}
//	return ptr;
//}


