#ifndef _KVI_IRCUSER_H_INCLUDED_
#define _KVI_IRCUSER_H_INCLUDED_

//
//   File : kvi_ircuser.h (/usr/build/NEW_kvirc/kvirc/kvilib/kvi_ircuser.h)
//   Last major modification : Fri Jan 8 1999 19:50:35 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_string.h"

/**
* @short Irc user mask abstraction
*/

class KviIrcUser{
	friend class KviIrcUserList;
	friend class KviIrcUserChanList;
public:
	KviIrcUser();
	/**
	* Constructs this KviIrcUser object from a mask.<br>
	* @see KviIrcUser::setMask
	*/
	KviIrcUser(const char *szMask,char c=' ');
	/**
	* Deep copy of data.
	*/
	KviIrcUser(const KviIrcUser &ircUser);
	/**
	* Kill user :)
	*/
	~KviIrcUser();
private:
	char *m_nick_ptr;
	char *m_user_ptr;
	char *m_host_ptr;
public:
	/**
	* Sets the nick for this user.<br>
	* If szNick is NULL or it points to an empty string the nick is set to "*".<br>
	* WARNING : The szNick string MUST be null-terminated (if the szNick pointer is not NULL).
	*/
	void setNick(const char *szNick);
	/**
	* Sets the username for this user.<br>
	* If szUsername is NULL or it points to an empty string the username is set to "*".<br>
	* WARNING : The szUsername string MUST be null-terminated (if the szUsername pointer is not NULL).
	*/
	void setUsername(const char *szUsername);
	/**
	* Sets the host for this user.<br>
	* If szHost is NULL or it points to an empty string the host is set to "*".<br>
	* WARNING : The szHost string MUST be null-terminated (if the szHost pointer is not NULL).
	*/
	void setHost(const char *szHost);
	/**
	* Sets the host , nick and username extracting it from an irc mask:<br>
	* nick!user@host<br>
	* The mask is terminated by end-of string null character or a character equal to c in the string.<br>
	* Returns the pointer to the end of the mask in the szMask string.(c or null-terminator)<br>
	* WARNING : the szMask pointer can NOT be NULL
	*/
	const char * setMask(const char *szMask,char c=' ');
	/**
	* Sets the host , nick and username extracting it from an userhost mask:<br>
	* nick[*]=<+|->user@host<br>
	* The mask is terminated by end-of string null char or a space character.<br>
	* Returns the pointer to the next non-space char in the szMask string or to the null-terminator<br>
	* If there are no more masks avaiable.
	* WARNING : the szMask pointer can NOT be NULL
	*/
	const char *setUserhostMask(const char *szMask);
	/**
	* Returns the nick of this user.<br>
	* In the worst case you get a string == "*"<br>
	* (inline)
	*/
	const char *nick() const;
	/**
	* Returns the username of this user.<br>
	* (inline)
	*/
	const char *username() const;
	/**
	* Returns the host of this user.<br>
	* (inline)
	*/
	const char *host() const;
	/**
	* Returns true if the nick of this user is known (non '*').<br>
	* (inline)
	*/
	bool hasNick() const;
	/**
	* Returns true if the username of this user is known (non '*').<br>
	* (inline)
	*/
	bool hasUsername() const;
	/**
	* Returns true if the host of this user is known (non '*').<br>
	* (inline)
	*/
	bool hasHost() const;
	/**
	* Returns true if a full irc mask can be constructed from the known data.<br>
	* (inline)
	*/
	bool hasMask() const;
	/**
	* Retuns true if the user's host is a numeric Ip (has three dots and only numbers).
	* false otherwise
	*/
	bool hasNumericHost() const;
	/**
	* Retuns in szMask the specified (if possible) mask of this user.<br>
	* If the host or username are not known , the mask may contain less information
	* than requested.<br>
	* Mask types:<br>
	* 0 : nick!user@machine.host.top (nick!user@XXX.XXX.XXX.XXX) (default)<br>
	* 1 : nick!user@*.host.top       (nick!user@XXX.XXX.XXX.*)<br>
	* 2 : nick!user@*<br>
	* 3 : nick!*@machine.host.top    (nick!user@XXX.XXX.XXX.XXX)<br>
	* 4 : nick!*@*.host.top          (nick!user@XXX.XXX.XXX.*)<br>
	* 5 : nick!*@*<br>
	* 6 : *!user@machine.host.top    (*!user@XXX.XXX.XXX.XXX)<br>
	* 7 : *!user@*.host.top          (*!user@XXX.XXX.XXX.*)<br>
	* 8 : *!user@*<br>
	* 9 : *!*@machine.host.top       (*!*@XXX.XXX.XXX.XXX)<br>
	* 10: *!*@*.host.top             (*!*@XXX.XXX.XXX.*)<br>
	* If some data is missing , these types may change:<br>
	* For example , if hostname is missing , the mask type 3 may be reduced to type 5
	*/
	void mask(KviStr &szMask,int mask_type=0);
	/**
	* Returns true if this user matches the mask szMask
	*/
	bool matches(const char *szMask);
	/**
	* Matches irc masks.<br>
	* (static member)
	* WARNING : use kvi_matchWildExpr instead
	*/
//	static bool matchMasks(register const char *m1,register const char *m2);
private:
	int getIpDomainMaskLen();
	char *getHostDomainMaskPtr();
};

inline bool KviIrcUser::hasNick() const
{ return ((*m_nick_ptr != '*')||(*(m_nick_ptr+1) != '\0')); }
inline bool KviIrcUser::hasUsername() const
{ return ((*m_user_ptr != '*')||(*(m_user_ptr+1) != '\0')); }
inline bool KviIrcUser::hasHost() const
{ return ((*m_host_ptr != '*')||(*(m_host_ptr+1) != '\0')); }
inline bool KviIrcUser::hasMask() const
{ return  ( ( (*m_nick_ptr != '*') || (*(m_nick_ptr+1) != '\0') ) &&
            ( (*m_user_ptr != '*') || (*(m_user_ptr+1) != '\0') ) &&
			( (*m_host_ptr != '*') || (*(m_host_ptr+1) != '\0') ) ); }
inline const char * KviIrcUser::nick() const
{ return m_nick_ptr; }
inline const char * KviIrcUser::username() const
{ return m_user_ptr; }
inline const char * KviIrcUser::host() const
{ return m_host_ptr; }



#endif //_KVI_IRCUSER_H_INCLUDED_
