//
//   File : kvi_userlist.cpp (/usr/build/NEW_kvirc/kvirc/src/kvilib/kvi_userlist.cpp)
//   Last major modification : Tue Mar 23 1999 04:21:11 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//
// Lavoro sporco....(dirty work)
//

//#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_userlist.h"
#include "kvi_string.h"

/* FIXME : We want a hash table here for 2.1.0!!! */

//===========================================================================//
// Master list
//

KviIrcUserList::KviIrcUserList()
{
	m_pHead = 0;
	m_pTail = 0;
}

KviIrcUserList::~KviIrcUserList()
{
	// All items should have been removed before this list is destroyed.
	__range_invalid(m_pHead);
	clearList();
}

void KviIrcUserList::clearList()
{
	while(m_pHead)removeNode(m_pHead);
	__range_valid(m_pTail == 0);
}

void KviIrcUserList::removeNode(KviIrcUserListNode * node)
{
	// Internal : removes a node from the list
	// and deletes it!
	__range_valid(node);
	__range_valid(node->pUser->hasNick());
	__range_valid(findNode(node->pUser->nick()));

	if(node == m_pHead){
		// Head node
		if(node->next){
			// has a next one , it becomes the new head node.
			node->next->prev = 0;
			m_pHead = node->next;
		} else {
			// the only node in the list (must be the tail node too!)
			__range_valid(m_pTail == node);
			m_pTail = 0;
			m_pHead = 0;
		}
	} else {
		// in the middle , or the last one
		__range_valid(node->prev);
		if(node->next){
			// Is in the middle , make necessary connections
			node->next->prev = node->prev;
			node->prev->next = node->next;
		} else {
			// The last one. Previous becomes new last
			m_pTail = node->prev;
			node->prev->next = 0;
		}
	}
	// Do not forget to delete it :)
	delete node->pUser;
	delete node;
}

void KviIrcUserList::insertNode(KviIrcUserListNode * node)
{
	// Insert a node in the list respecting the alphabetic order.
	__range_valid(node);
	__range_valid(node->pUser->hasNick());
	__range_valid(!findNode(node->pUser->nick()));

	if(m_pHead == 0){
		// First node in the list
		m_pHead = node;
		m_pTail = node;
		node->prev = 0;
		node->next = 0;
	} else {
		// There are other nodes , walk the list...
		KviIrcUserListNode * cur = m_pHead;

		do{
			// Walk until we find a greater one
			if(kvi_strcmpCI(node->pUser->m_nick_ptr,cur->pUser->m_nick_ptr) > 0){
				//cur is greater...insert before
				node->next = cur;
				node->prev = cur->prev;
				if(node->prev)node->prev->next = node;
				else m_pHead = node; //no prev item
				cur->prev = node;
				return;
			}
#ifdef _KVI_DEBUG_CLASS_NAME_
			if(!cur->next){ //debug
				__range_valid(cur == m_pTail);
			}
#endif
			cur = cur->next;
		} while(cur);

		//ran to the end!
		m_pTail->next = node;
		node->prev = m_pTail;
		node->next = 0;
		m_pTail = node;	
	}
}

KviIrcUserListNode * KviIrcUserList::findNode(const char *nick)
{
	// Lookup a nickname in the list , and find the related node.
	KviIrcUserListNode * cur = m_pHead;
	int result;
	while(cur){
		result = kvi_strcmpCI(nick,cur->pUser->m_nick_ptr);
		if(result == 0)return cur;
		if(result > 0)return 0; //can not be...the cur is greater...
		cur = cur->next;
	}
	return 0; //not found
}

void KviIrcUserList::getFirstFiveNicksToUpdate(KviStr &buffer)
{
	buffer = "";
	KviIrcUserListNode * cur = m_pHead;
	int iCount = 0;
	while(cur && iCount < 5){
		if(!(cur->pUser->hasHost())){
			buffer.append(cur->pUser->nick());
			if(iCount != 4)buffer.append(' ');
			iCount++;
		}
		cur = cur->next;
	}
}

KviIrcUser * KviIrcUserList::findUser(const KviIrcUser &user)
{
	// Same as above , but return the nick structure pointer
	KviIrcUserListNode * node = findNode(user.nick());
	if(node)return node->pUser;
	return 0;
}

KviIrcUser * KviIrcUserList::findUser(const char *nick)
{
	// Same as above , but return the nick structure pointer (overload)
	KviIrcUserListNode * node = findNode(nick);
	if(node)return node->pUser;
	return 0;
}

KviIrcUserListNode *  KviIrcUserList::addUser(const KviIrcUser &user)
{
	// Add a nick to the list
	KviIrcUserListNode * node = findNode(user.nick());
	if(node){
		__range_valid(node->nRefs > 0);
		node->nRefs++;
		//Update user data
		if(!node->pUser->hasHost())node->pUser->setHost(user.host());
		if(!node->pUser->hasUsername())node->pUser->setUsername(user.username());
	} else {
		node = new KviIrcUserListNode;
		node->pUser = new KviIrcUser(user);
		node->nRefs = 1;
		insertNode(node);
	}
	return node;
}

bool KviIrcUserList::updateUser(const KviIrcUser &user)
{
	// Update an user entry in the list
	KviIrcUserListNode * node = findNode(user.nick());
//	__range_valid(node); //sanity check for invalid calls
	if(node){
		if(!node->pUser->hasHost())node->pUser->setHost(user.host());
		if(!node->pUser->hasUsername())node->pUser->setUsername(user.username());
	}
	return node != 0;
}

//Unused
//void KviIrcUserList::killUser(KviIrcUser &user)
//{
//	KviIrcUserListNode * node = findNode(user.nick());
//	__range_valid(node);
//	if(node){
//		if(node->nRefs < 2)removeNode(node);
//		else node->nRefs--;
//	}
//}

void KviIrcUserList::killUserByNode(KviIrcUserListNode * node)
{
	// Removes a node from the list
	// or decreases its reference count (if it is > 1)
	__range_valid(node);
	__range_valid(node->pUser);
	__range_valid(node->pUser->hasNick());
	if(node){
		if(node->nRefs < 2){
			__range_valid(node->nRefs == 1);
			// Only one reference , kill the node
#ifdef _KVI_DEBUG_CHECK_RANGE_
			// Sanity check for the debug version
			KviStr nick = node->pUser->nick();
			removeNode(node);
			__range_invalid(findNode(nick.ptr()));
#else
			removeNode(node);
#endif
		} else node->nRefs--;
	}
}

//===========================================================================//
// Client list
//

KviIrcUserChanList::KviIrcUserChanList(KviIrcUserList * pList)
{
	m_pGlobalList = pList;
	m_pHead       = 0;
	m_pTail       = 0;
	m_pCur        = 0;
	m_iCount      = 0;
	m_iOwnerCount = 0;
	m_iOpCount    = 0;
	m_iHalfOpCount = 0;
	m_iVoiceCount = 0;
	m_iUserOpCount = 0;
}

KviIrcUserChanList::~KviIrcUserChanList()
{
	clearList();
}

void KviIrcUserChanList::clearList()
{
//	debug("CLEAR LIST");
	__range_valid((m_pHead && m_pTail)||(!(m_pHead || m_pTail)));
	while(m_pHead){
//		debug("Deleting %s",m_pHead->pNode->pUser->nick());
		part(*(m_pHead->pNode->pUser));
	}
	__range_valid(m_pHead == m_pTail);
	__range_valid(m_pTail == 0);
}

KviIrcUserChanData * KviIrcUserChanList::join(const KviIrcUser &user,char bOp,char bVoice,char bHalfOp,char bUserOp,char bOwner)
{
	// Adds an user to this list
	// Adding it to the global one too
	KviIrcUserChanData * data = findData(user.nick());
	if(data){
		// Alredy in the list...double join ?
		__debug_1arg("Double join for user %s",user.nick());
		// update the fields only
		if(data->qFlag != 0){
			if(bOwner == 0){
				m_iOwnerCount--;
				data->qFlag = bOwner;
			}
		} else {
			if(bOwner != 0){
				m_iOwnerCount++;
				data->qFlag = bOwner;
			}
		}
		if(data->oFlag != 0){
			if(bOp == 0){
				m_iOpCount--;
				data->oFlag = bOp;
			}
		} else {
			if(bOp != 0){
				m_iOpCount++;
				data->oFlag = bOp;
			}
		}
		if(data->hFlag != 0){
			if(bHalfOp == 0){
				m_iHalfOpCount--;
				data->hFlag = bHalfOp;
			}
		} else {
			if(bHalfOp != 0){
				m_iOpCount++;
				data->hFlag = bHalfOp;
			}
		}
		if(data->vFlag != 0){
			if(bVoice == 0){
				m_iVoiceCount--;
				data->vFlag = bVoice;
			}
		} else {
			if(bVoice != 0){
				m_iVoiceCount++;
				data->vFlag = bVoice;
			}
		}
		if(data->uFlag != 0){
			if(bUserOp == 0){
				m_iUserOpCount--;
				data->uFlag = bUserOp;
			}
		} else {
			if(bUserOp != 0){
				m_iUserOpCount++;
				data->uFlag = bUserOp;
			}
		}
		return data;
	}
	data = new KviIrcUserChanData;
	data->qFlag = bOwner;
	data->oFlag = bOp;
	data->hFlag = bHalfOp;
	data->vFlag = bVoice;
	data->uFlag = bUserOp;
	data->extFlag = 0; //Set it to zero now...(this flag can be used by the external classes.KviListBox uses it to keep track of the selected items)
	data->pNode = m_pGlobalList->addUser(user);
	__range_valid(data->pNode);
	insertData(data);
	return data;
}

bool KviIrcUserChanList::part(const KviIrcUser &user)
{
	return part(user.nick());
}

bool KviIrcUserChanList::part(const char *nick)
{
	KviIrcUserChanData * data = findData(nick);
	__range_valid(data);
	if(!data)return false;
	__range_valid(data->pNode == m_pGlobalList->findNode(nick));
	__range_valid(data->pNode);
	// It deletes the whole data->pNode structure
	m_pGlobalList->killUserByNode(data->pNode);
	// Remove data deleting it
	removeData(data);
//	__range_valid(m_pGlobalList->findNode(nick) == 0); //untrue
	return true;
}

bool KviIrcUserChanList::select(const KviIrcUser &user)
{
	return select(user.nick());
}

bool KviIrcUserChanList::select(const char *nick)
{
	KviIrcUserChanData * data = findData(nick);
	if(!data)return false;
	data->extFlag = 1;
	return true;
}

bool KviIrcUserChanList::deselect(const KviIrcUser &user)
{
	return deselect(user.nick());
}

bool KviIrcUserChanList::deselect(const char *nick)
{
	KviIrcUserChanData * data = findData(nick);
	if(!data)return false;
	data->extFlag = 0;
	return true;
}

bool KviIrcUserChanList::nickChange(const KviIrcUser &nicker,const char *newNick)
{
	KviIrcUserChanData * data = findData(nicker.nick());
	if(!data)return false;
	__range_valid(data->pNode == m_pGlobalList->findNode(nicker.nick()));
	__range_valid(data->pNode);
	char qFlag = data->qFlag;
	char oFlag = data->oFlag;
	char hFlag = data->hFlag;
	char vFlag = data->vFlag;
	char uFlag = data->uFlag;
	// Now like part...
	m_pGlobalList->killUserByNode(data->pNode);
	removeData(data);
//	__range_valid(m_pGlobalList->findNode(nicker.nick()) == 0);<--- may be false (if the nicker is on two channels and this is the first one that calls nickChange())
	KviIrcUser newNicker(nicker);
	newNicker.setNick(newNick);
	join(newNicker,oFlag,vFlag,hFlag,uFlag,qFlag);
	return true;
}


KviIrcUser * KviIrcUserChanList::findUser(const KviIrcUser &user)
{
	return findUser(user.nick());
}

KviIrcUser * KviIrcUserChanList::findUser(const char *nick)
{
	KviIrcUserChanData * data = findData(nick);
	if(!data)return 0;
	__range_valid(data->pNode);
	__range_valid(data->pNode->pUser);
	return data->pNode->pUser;
}

int KviIrcUserChanList::findUserPosition(const char *nick)
{
	KviIrcUserChanData * cur = m_pHead;
	int item = 0;
	int result;
	while(cur){
		// Compare
		result = kvi_strcmpCI(nick,cur->pNode->pUser->m_nick_ptr);
		if(result = 0)return item;
//		if(result > 0)return -1; //can not be...the cur is greater...
		item++;
		cur = cur->next;
	}
	return item; //not found
}

KviIrcUserChanData * KviIrcUserChanList::findData(const KviIrcUser &user)
{
	return findData(user.nick());
}

KviIrcUserChanData * KviIrcUserChanList::findData(const char *nick)
{
	// Find a data entry in the list
	KviIrcUserChanData * cur = m_pHead;
	int result;
	while(cur){
		// Compare
		result = kvi_strcmpCI(nick,cur->pNode->pUser->m_nick_ptr);
		if(result == 0)return cur; // Found (Case insensitive)
		if(result > 0){
			//the cur is greater...
			if(cur->qFlag != 0){
				// We were looking in the ops block
				// skip all the ops and continue
				while(cur){
					if(cur->qFlag == 0)break;
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr,nick)); //sanity : must not be here , if it is , the order is wrong!
					cur = cur->next;
				}
			} else if(cur->oFlag != 0){
				// We were looking in the ops block
				// skip all the ops and continue
				while(cur){
					if(cur->oFlag == 0)break;
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr,nick)); //sanity : must not be here , if it is , the order is wrong!
					cur = cur->next;
				}
			} else if(cur->hFlag != 0){
				// We were looking in the halfop block
				// skip all the halfops and continue
				while(cur){
					if(cur->hFlag == 0)break;
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr,nick)); //sanity : must not be here , if it is , the order is wrong!
					cur = cur->next;
				}
			} else if(cur->vFlag != 0){
				// We were looking in the voiced block
				// skip all the voiced and continue
				while(cur){
					if(cur->vFlag == 0)break;
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr,nick)); //sanity : must not be here , if it is , the order is wrong!
					cur = cur->next;
				}
			} else if(cur->uFlag != 0){
				// We were looking in the userop block
				// skip all the userops and continue
				while(cur){
					if(cur->uFlag == 0)break;
					__range_invalid(kvi_strEqualCI(cur->pNode->pUser->m_nick_ptr,nick)); //sanity : must not be here , if it is , the order is wrong!
					cur = cur->next;
				}
			} else return 0; //(normal user) cannot be...
		} else cur = cur->next;
	}
	// Ran up to the end...
	return 0; //not found
}

bool KviIrcUserChanList::owner(const char *nick,char bOwner)
{
	KviIrcUserChanData * data = findData(nick);
	__range_valid(data);
	if(!data)return false;
	if(data->qFlag != bOwner){
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->qFlag = bOwner;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::op(const char *nick,char bOp)
{
	KviIrcUserChanData * data = findData(nick);
	__range_valid(data);
	if(!data)return false;
	if(data->oFlag != bOp){
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->oFlag = bOp;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::halfop(const char *nick,char bHalfOp)
{
	KviIrcUserChanData * data = findData(nick);
	__range_valid(data);
	if(!data)return false;
	if(data->hFlag != bHalfOp){
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->hFlag = bHalfOp;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::voice(const char *nick,char bVoice)
{
	KviIrcUserChanData * data = findData(nick);
	__range_valid(data);
	if(!data)return false;
	if(data->vFlag != bVoice){
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->vFlag = bVoice;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::userop(const char *nick,char bUserOp)
{
	KviIrcUserChanData * data = findData(nick);
	__range_valid(data);
	if(!data)return false;
	if(data->uFlag != bUserOp){
		// Adjust the list order
		removeDataNoDelete(data);
		__range_invalid(findData(nick));
		data->uFlag = bUserOp;
		insertData(data);
		__range_valid(findData(nick));
	}
	return true;
}

bool KviIrcUserChanList::isOwner(const char *nick)
{
	KviIrcUserChanData * data = findData(nick);
	if(!data)return false;
	return (data->qFlag != 0);
}


bool KviIrcUserChanList::isOp(const char *nick)
{
	KviIrcUserChanData * data = findData(nick);
	if(!data)return false;
	return (data->oFlag != 0);
}

bool KviIrcUserChanList::isHalfOp(const char *nick)
{
	KviIrcUserChanData * data = findData(nick);
	if(!data)return false;
	return (data->hFlag != 0);
}

bool KviIrcUserChanList::isVoice(const char *nick)
{
	KviIrcUserChanData * data = findData(nick);
	if(!data)return false;
	return (data->vFlag != 0);
}

bool KviIrcUserChanList::isUserOp(const char *nick)
{
	KviIrcUserChanData * data = findData(nick);
	if(!data)return false;
	return (data->uFlag != 0);
}

void KviIrcUserChanList::insertData(KviIrcUserChanData * data)
{
	// Inserts a data node in the local list
	__range_valid(data->pNode);
	__range_valid(data->pNode->pUser);
	__range_valid(data->pNode->pUser->hasNick());
	__range_invalid(findData(data->pNode->pUser->nick()));
	// Increase the count
	m_iCount++;
	if(m_pHead){
		// Have items inside
		if(data->qFlag != 0){
			insertOwnerData(data);
			m_iOwnerCount++;
		} else if(data->oFlag != 0){
			insertOpData(data);//isop
			m_iOpCount++;
		} else if(data->hFlag != 0){
			insertHalfOpData(data);
			m_iHalfOpCount++;
		} else if(data->vFlag != 0){
			insertVoiceData(data);
			m_iVoiceCount++;
		} else if(data->uFlag != 0){
			insertUserOpData(data);
			m_iUserOpCount++;
		} else insertNormalData(data);
	} else {
		// First in the list
		m_pHead = data;
		m_pTail = data;
		data->next = 0;
		data->prev = 0;
		if(data->qFlag != 0)m_iOwnerCount++;
		else if(data->oFlag != 0)m_iOpCount++;
		else if(data->hFlag != 0)m_iHalfOpCount++;
		else if(data->vFlag != 0)m_iVoiceCount++;
		else if(data->uFlag != 0)m_iUserOpCount++;
	}
}

void KviIrcUserChanList::insertOwnerData(KviIrcUserChanData * data)
{
	//Walk the op list...
	__range_valid(m_pHead);
	__range_valid(data->qFlag != 0);
	KviIrcUserChanData * cur = m_pHead;
	while(cur){
		if((kvi_strcmpCI(data->pNode->pUser->m_nick_ptr,cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->qFlag == 0))
		{
			//cur is greater or is non-op ...insert before
			data->next = cur;
			data->prev = cur->prev;
			if(data->prev)data->prev->next = data;
			else m_pHead = data; //no prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	//ran to the end of the list! (only ops inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}


void KviIrcUserChanList::insertOpData(KviIrcUserChanData * data)
{
	//Walk the op list...
	__range_valid(m_pHead);
	__range_valid(data->oFlag != 0);
	KviIrcUserChanData * cur = m_pHead;
	while(cur){ //skip the owners
		if(cur->qFlag == 0)break;
		cur = cur->next;
	}
	while(cur){
		if((kvi_strcmpCI(data->pNode->pUser->m_nick_ptr,cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->oFlag == 0))
		{
			//cur is greater or is non-op ...insert before
			data->next = cur;
			data->prev = cur->prev;
			if(data->prev)data->prev->next = data;
			else m_pHead = data; //no prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	//ran to the end of the list! (only ops inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::insertHalfOpData(KviIrcUserChanData * data)
{
	//Walk the halfop list...
	__range_valid(m_pHead);
	__range_valid(data->hFlag != 0);
	KviIrcUserChanData * cur = m_pHead;
	while(cur){ //skip the owners
		if(cur->qFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the ops
		if(cur->oFlag == 0)break;
		cur = cur->next;
	}
	while(cur){
		if((kvi_strcmpCI(data->pNode->pUser->m_nick_ptr,cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->hFlag == 0))
		{
			//cur is greater or is non-halfop ...insert before
			data->next = cur;
			data->prev = cur->prev;
			if(data->prev)data->prev->next = data;
			else m_pHead = data; //no prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	//ran to the end of the list! (only ops and halfops inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::insertVoiceData(KviIrcUserChanData * data)
{
	//Walk the voice list...
	__range_valid(m_pHead);
	__range_valid(data->vFlag != 0);
	KviIrcUserChanData * cur = m_pHead;
	while(cur){ //skip the owners
		if(cur->qFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the ops
		if(cur->oFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the halfops
		if(cur->hFlag == 0)break;
		cur = cur->next;
	}
	// Now pointing to the first non op or to the end of the list
	while(cur){
		// First non op
		if((kvi_strcmpCI(data->pNode->pUser->m_nick_ptr,cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->vFlag == 0)){
			//cur is greater or is non-voice ...insert before
			data->next = cur;
			data->prev = cur->prev;
			if(data->prev)data->prev->next = data;
			else m_pHead = data; //no prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	//ran to the end of the list! (only ops, halfops, and voiced inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
} /* BUGHERE - MAYBE -Trisk */

void KviIrcUserChanList::insertUserOpData(KviIrcUserChanData * data)
{
	//Walk the userop list...
	__range_valid(m_pHead);
	__range_valid(data->uFlag != 0);
	KviIrcUserChanData * cur = m_pHead;
	while(cur){ //skip the owners
		if(cur->qFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the ops
		if(cur->oFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the halfops
		if(cur->hFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the voiced ones
		if(cur->vFlag == 0)break;
		cur = cur->next;
	}
	while(cur){
		if((kvi_strcmpCI(data->pNode->pUser->m_nick_ptr,cur->pNode->pUser->m_nick_ptr) > 0) ||
			(cur->uFlag == 0))
		{
			//cur is greater or is non-userop ...insert before
			data->next = cur;
			data->prev = cur->prev;
			if(data->prev)data->prev->next = data;
			else m_pHead = data; //no prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	//ran to the end of the list! (only ops, halfops, voiced, and userops inside)
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::insertNormalData(KviIrcUserChanData * data)
{
	//Walk the voice list...
	__range_valid(m_pHead);
	__range_valid((data->uFlag == 0)&&(data->vFlag == 0)&&(data->hFlag == 0)&&(data->oFlag == 0)&&(data->qFlag == 0));
	KviIrcUserChanData * cur = m_pHead;
	while(cur){ //skip the owners
		if(cur->qFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the ops
		if(cur->oFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the halfops
		if(cur->hFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the voiced ones
		if(cur->vFlag == 0)break;
		cur = cur->next;
	}
	while(cur){ //skip the userops
		if(cur->uFlag == 0)break;
		cur = cur->next;
	}
	// Now pointing to the first non op-voice or to the end of the list
	while(cur){
		// First non op-voice
		if(kvi_strcmpCI(data->pNode->pUser->m_nick_ptr,cur->pNode->pUser->m_nick_ptr) > 0){
			//cur is greater ...insert before
			data->next = cur;
			data->prev = cur->prev;
			if(data->prev)data->prev->next = data;
			else m_pHead = data; //no prev item
			cur->prev = data;
			return;
		}
		cur = cur->next;
	}
	//ran to the end of the list!
	m_pTail->next = data;
	data->prev = m_pTail;
	data->next = 0;
	m_pTail = data;
}

void KviIrcUserChanList::removeDataNoDelete(KviIrcUserChanData * data)
{
	// Removes just the data node from the list , nothing else
	__range_valid(data);
	__range_valid(data->pNode);
//	__range_valid(data->pNode->pUser); <---the node may not exist anymore!!! (part deletes it before)

	m_iCount--; //One less

	if(data->qFlag != 0)m_iOwnerCount--;
	else if(data->oFlag != 0)m_iOpCount--; //Was op ?
	else if(data->hFlag != 0)m_iHalfOpCount--; //Was halfop ?
	else if(data->vFlag != 0)m_iVoiceCount--; //Was voice ?
	else if(data->uFlag != 0)m_iUserOpCount--; //Was userop ?

	if(data == m_pHead){
		// was the first item in the list
		if(data->next){
			// and has a next one
			data->next->prev = 0;
			m_pHead = data->next;
		} else {
			// and was the only one
			__range_valid(m_pTail == data);
			m_pTail = 0;
			m_pHead = 0;
		}
	} else {
		// somewhere in the middle or the last one
		__range_valid(data->prev);
		if(data->next){
			// in the middle
			data->next->prev = data->prev;
			data->prev->next = data->next;
		} else {
			// was the last
			m_pTail = data->prev;
			data->prev->next = 0;
		}
	}
}

void KviIrcUserChanList::removeData(KviIrcUserChanData * data)
{
	removeDataNoDelete(data);
	delete data;
}
/*
void KviIrcUserChanList::dump()
{
	// dirty work...clean it up!
#ifdef _KVI_DEBUG_CHECK_RANGE_
	debug("Client user list dump:");
	debug("This ptr = %u",this);
	debug("Items    = %d",m_iCount);
	debug("Ops      = %d",m_iOpCount);
	debug("Voiced   = %d",m_iVoiceCount);
	debug("Items dump:");
	KviIrcUserChanData * cur = m_pHead;
	int index = 1;
	while(cur){
		debug("--%d) ptr: %u, node_ptr: %u, user_ptr: %u",index,cur,cur->pNode,cur->pNode->pUser);
		debug("  %s!%s@%s : o(%d) v(%d)",cur->pNode->pUser->nick(),
			cur->pNode->pUser->username(),cur->pNode->pUser->host(),cur->oFlag,cur->vFlag);
		cur = cur->next;
		index++;
	}
	debug("End of client user list dump");
#endif
}
*/
