//
//   File : kvi_chaneditors.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_chaneditors.cpp)
//   Last major modification : Sat May 1 1999 01:42:53 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#include "kvi_chaneditors.h"
#include "kvi_channel.h"
#include "kvi_locale.h"
#include "kvi_app.h"

#include <qlayout.h>
#include <qlineedit.h>

KviTopicEditor::KviTopicEditor(QWidget * parent)
:KviLineEdit(parent,"topic_editor")
{
	setFocusPolicy(QWidget::ClickFocus);
	m_pHistory = new QList<KviStr>;
	m_pHistory->setAutoDelete(true);
	m_iCurHistoryIdx = -1;
}

KviTopicEditor::~KviTopicEditor()
{
	delete m_pHistory;
}

void KviTopicEditor::keyPressEvent(QKeyEvent *e)
{
	KviStr m_szTextBuffer = QLineEdit::text();
	KviStr *pHist = new KviStr(m_szTextBuffer);
	switch(e->key()){
		case Qt::Key_Up:
			if(m_pHistory->count() > 0){
				if(m_iCurHistoryIdx < 0){
					m_szSaveTextBuffer = m_szTextBuffer;
					m_szTextBuffer = m_pHistory->at(0)->ptr();
					m_iCurHistoryIdx = 0;
				} else if(m_iCurHistoryIdx >= (int)(m_pHistory->count()-1)){
					m_szTextBuffer = m_szSaveTextBuffer;
					m_iCurHistoryIdx = -1;
				} else {
					m_iCurHistoryIdx++;
					m_szTextBuffer = m_pHistory->at(m_iCurHistoryIdx)->ptr();
				}
				QLineEdit::setText(QString(m_szTextBuffer.ptr()));
			}
			break;
		case Qt::Key_Down:
			if(m_pHistory->count() > 0){
				if(m_iCurHistoryIdx < 0){
					m_szSaveTextBuffer = m_szTextBuffer;
					m_szTextBuffer = m_pHistory->at(m_pHistory->count()-1)->ptr();
					m_iCurHistoryIdx = m_pHistory->count()-1;
				} else if(m_iCurHistoryIdx == 0){
					m_szTextBuffer=m_szSaveTextBuffer;
					m_iCurHistoryIdx = -1;
				} else {
					m_iCurHistoryIdx--;
					m_szTextBuffer = m_pHistory->at(m_iCurHistoryIdx)->ptr();
				}
				QLineEdit::setText(QString(m_szTextBuffer.ptr()));
			}
			break;
		case Qt::Key_Enter:
		case Qt::Key_Return:
			if(m_pHistory->count() > 0){
				if(!kvi_strEqualCI(m_szTextBuffer.ptr(),m_pHistory->at(0)->ptr()))
					m_pHistory->insert(0,pHist);
			} else m_pHistory->insert(0,pHist);
			if(m_pHistory->count() > 10)m_pHistory->removeLast();
			emit returnPressed();
			e->ignore();
			break;
		default:
			KviLineEdit::keyPressEvent(e);
			break;
	}
}

void KviTopicEditor::focusOutEvent(QFocusEvent *)
{
	hide();
}

void KviTopicEditor::connectMe(QObject * receiver,const char *slot)
{
	disconnect(this,SIGNAL(returnPressed()),0,0);
	connect(this,SIGNAL(returnPressed()),receiver,slot);
}


KviModeDescriptionEntry KviModeEditor::modeTable[6]=
{
	{ 'i' , _i19n_("i - Invite-only")      },
	{ 'm' , _i19n_("m - Moderated")        },
	{ 'n' , _i19n_("n - No external messages") },
	{ 'p' , _i19n_("p - Private")          },
	{ 's' , _i19n_("s - Secret")           },
	{ 't' , _i19n_("t - Topic change limited") }
};

KviModeEditor::KviModeEditor(QWidget * parent)
:QFrame(parent)
{
	setFrameStyle(QFrame::Panel|QFrame::Sunken);

	QGridLayout * g = new QGridLayout(this,13,2,2,2);

	QLabel * l = new QLabel(__tr("<b>Channel Modes</b>"),this);
	g->addMultiCellWidget(l,0,0,0,1);

	QFrame * f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f,1,1,0,1);

	for(int i=0;i<6;i++){
		m_pCheck[i] = new QCheckBox(_i18n_translateNoText(modeTable[i].description),this);
		g->addMultiCellWidget(m_pCheck[i],i + 2,i + 2,0,1);
		m_pCheck[i]->setFocusPolicy(QWidget::ClickFocus);
	}

	m_pLimitCheck = new QCheckBox(__tr("l - User limit:"),this);
	g->addWidget(m_pLimitCheck,8,0);

	m_pLimitEdit = new QLineEdit(this);
	g->addWidget(m_pLimitEdit,8,1);

	m_pKeyCheck = new QCheckBox(__tr("k - Key:"),this);
	g->addWidget(m_pKeyCheck,9,0);

	m_pKeyEdit = new QLineEdit(this);
	g->addWidget(m_pKeyEdit,9,1);

	l = new QLabel(this);
	g->addMultiCellWidget(l,10,10,0,1);

	m_pCommit = new QPushButton(__tr("&OK"),this);
	m_pCommit->setDefault(true);
	g->addMultiCellWidget(m_pCommit,11,11,0,1);
	connect(m_pCommit,SIGNAL(clicked()),this,SLOT(commitChanges()));

	QPushButton * b = new QPushButton(__tr("&Cancel"),this);
	g->addMultiCellWidget(b,12,12,0,1);
	connect(b,SIGNAL(clicked()),this,SLOT(cancel()));

	g->setRowStretch(10,1);
}

KviModeEditor::~KviModeEditor()
{
}

void KviModeEditor::setCanCommit(bool bCanCommit)
{
	m_pCommit->setEnabled(bCanCommit);
}

void KviModeEditor::commitChanges()
{
	hide();
	KviStr minus = "-";
	KviStr plus  = "+";
	KviStr args;
	KviStr additionalModeChange;

	for(int i=0;i<6;i++){
		if(m_bInitState[i]){ //It was set at the beginning
			if(!m_pCheck[i]->isChecked()){ //and was unset
				minus += modeTable[i].flag;
			}
		} else { //It was NOT set at the beginning
			if(m_pCheck[i]->isChecked()){ //and now is set
				plus += modeTable[i].flag;
			}
		}
	}

	KviStr curLimit = m_pLimitEdit->text();
	curLimit.stripWhiteSpace();
	bool bOk = false;
	unsigned int uLimit = curLimit.toUInt(&bOk);

	if(m_szInitLimit.hasData()) // there was a limit before
	{
		if(m_pLimitCheck->isChecked()) // we still want a limit
		{
			if(bOk && (uLimit > 0))
			{
				unsigned int oldLimit = m_szInitLimit.toUInt(&bOk);
				if(oldLimit != uLimit)
				{
					plus.append('l');
					args = curLimit;
				}
			}
		} else minus.append('l'); // remove the l mode
	} else {
		if(m_pLimitCheck->isChecked()) // we want a limit now
		{
			if(bOk && (uLimit > 0))
			{
				plus.append('l');
				args = curLimit;
			}
		}
	}

	KviStr curKey = m_pKeyEdit->text();
	curKey.stripWhiteSpace();

	if(m_szInitKey.hasData()) // there was a key before
	{
		if(m_pKeyCheck->isChecked()) // we still want a key
		{
			if(!kvi_strEqualCS(m_szInitKey.ptr(),curKey.ptr()))
			{
				// the key has been changed!
				minus.append('k');
				if(args.hasData())args.append(' ');
				args.append(m_szInitKey);
				additionalModeChange.sprintf("+k %s",curKey.ptr());
			}
		} else { // we don't want it anymore
			minus.append('k');
			if(args.hasData())args.append(' ');
			args.append(m_szInitKey);
		}
	} else { // there was no key before
		if(m_pKeyCheck->isChecked() && curKey.hasData()) // and we want it now
		{
			plus.append('k');
			if(args.hasData())args.append(' ');
			args.append(curKey);
		}
	}

	KviStr tmp;
	if(minus.len() > 1)tmp+=minus;
	if(plus.len() > 1)tmp+=plus;
	if(args.hasData()){
		tmp.append(' ');
		tmp.append(args);
	}
	emit modeChanged(tmp.ptr(),additionalModeChange.ptr());
}

void KviModeEditor::cancel()
{
	emit modeChanged("","");
}

void KviModeEditor::doEdit(KviStr &flags,KviStr &limit,KviStr &key,QObject *receiver,const char *slot)
{
	disconnect(this,SIGNAL(modeChanged(const char *,const char *)),0,0);
	connect(this,SIGNAL(modeChanged(const char *,const char *)),receiver,slot);

	for(int i=0;i<6;i++){
		m_bInitState[i] = flags.findFirstIdx(modeTable[i].flag) != -1;
		m_pCheck[i]->setChecked(m_bInitState[i]);
	}

	if(limit.hasData())
	{
		m_pLimitCheck->setChecked(true);
		m_pLimitEdit->setText(limit.ptr());
		m_szInitLimit = limit;
	}

	if(key.hasData())
	{
		m_pKeyCheck->setChecked(true);
		m_pKeyEdit->setText(key.ptr());
		m_szInitKey = key;
	}

	show();
	setFocus();
}


KviMaskEditor::KviMaskEditor(QWidget * parent)
:QFrame(parent)
{
	setFrameStyle(QFrame::WinPanel|QFrame::Raised);
	setFocusPolicy(QWidget::ClickFocus);

	QGridLayout *g = new QGridLayout(this,5,1,2,2);

	m_pLabel = new QLabel(__tr(""),this);
	g->addWidget(m_pLabel,0,0);


	m_pMaskBox = new QListView(this);
	m_pMaskBox->setFocusPolicy(QWidget::ClickFocus);
	m_pMaskBox->setFocusProxy(this);
	m_pMaskBox->setFrameStyle(QFrame::WinPanel|QFrame::Sunken);
	m_pMaskBox->addColumn(_CHAR_2_QSTRING(__tr(" - Mask - List - ")));
	m_pMaskBox->addColumn(_CHAR_2_QSTRING(__tr("Set by")));
	m_pMaskBox->addColumn(_CHAR_2_QSTRING(__tr("Set at")));
	m_pMaskBox->setMultiSelection(true);
	m_pMaskBox->setAllColumnsShowFocus(true);
	g->addWidget(m_pMaskBox,1,0);

	m_pRemoveMask  = new QPushButton(__tr("&Remove"),this);
	m_pRemoveMask->setFocusPolicy(QWidget::ClickFocus);
	m_pRemoveMask->setFocusProxy(this);
	connect(m_pRemoveMask,SIGNAL(clicked()),this,SLOT(removeMaskClicked()));
	g->addWidget(m_pRemoveMask,2,0);

	m_pCommit = new QPushButton(__tr("&OK"),this);
	m_pCommit->setFocusPolicy(QWidget::ClickFocus);
	m_pCommit->setFocusProxy(this);
	connect(m_pCommit,SIGNAL(clicked()),this,SLOT(commitChanges()));
	g->addWidget(m_pCommit,3,0);

	QPushButton * b = new QPushButton(__tr("&Cancel"),this);
	b->setFocusPolicy(QWidget::ClickFocus);
	b->setFocusProxy(this);
	connect(b,SIGNAL(clicked()),this,SLOT(cancelClicked()));
	g->addWidget(b,4,0);
}

KviMaskEditor::~KviMaskEditor()
{
}

void KviMaskEditor::doEdit(QObject * receiver,const char * slot,QList<KviBanMask> * maskList,const char *masksLabel,bool canCommit)
{
	disconnect(this,SIGNAL(masksEditCompleted(const char *)),0,0);
	connect(this,SIGNAL(masksEditCompleted(const char *)),receiver,slot);
	KviStr mLabel(KviStr::Format,"<b>%s List</b>",masksLabel);
	m_pLabel->setText(mLabel.ptr());
	setMasks(maskList,masksLabel);
	m_pCommit->setEnabled(canCommit);
	m_pRemoveMask->setEnabled(canCommit && (m_pMaskBox->childCount() > 0));
	m_szMasksRemoved = "";
	show();
	setFocus();
}

void KviMaskEditor::setMasks(QList<KviBanMask> * maskList,const char *masksLabel)
{
	m_pMaskBox->clear();
	m_pMaskBox->setColumnText(0,_CHAR_2_QSTRING(masksLabel));
	QListViewItem *i = 0;
	for(KviBanMask * b=maskList->first();b;b=maskList->next()){
		i = new QListViewItem(m_pMaskBox,_CHAR_2_QSTRING(b->mask.ptr()),_CHAR_2_QSTRING(b->setBy.ptr()),_CHAR_2_QSTRING(b->setAt.ptr()));
	}
}

void KviMaskEditor::setCanCommit(bool bCanCommit)
{
	m_pCommit->setEnabled(bCanCommit);
	m_pRemoveMask->setEnabled(bCanCommit && (m_pMaskBox->childCount() > 0));
}

void KviMaskEditor::removeMaskClicked()
{
	QListViewItem * it = m_pMaskBox->firstChild();
	while(it){
		if(it->isSelected()){
			if(m_szMasksRemoved.hasData())m_szMasksRemoved += " ";
			m_szMasksRemoved+=it->text(0);
			QListViewItem * tmp = it;
			it = it->nextSibling();
			delete tmp;
		} else it = it->nextSibling();
	}
	m_pRemoveMask->setEnabled(m_pMaskBox->childCount() > 0);
}

void KviMaskEditor::commitChanges()
{
	emit masksEditCompleted(m_szMasksRemoved.ptr());
}

void KviMaskEditor::cancelClicked()
{
	emit masksEditCompleted("");
}

#include "m_kvi_chaneditors.moc"
