//
//   File : kvi_list.cpp (/usr/build/KVICVS/kvirc/src/kvirc/kvi_list.cpp)
//   Last major modification : Sun Sep 12 1999 16:34:09 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_app.h"
#include "kvi_defines.h"
#include "kvi_list.h"
#include "kvi_ircview.h"
#include "kvi_frame.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_fileutils.h"
#include "kvi_ircsocket.h"

#include <qlayout.h>
#include <qvaluelist.h>
#include <qframe.h>
#include <qregexp.h>
#include <qfile.h>
#include <qdatetime.h>

#include <ctype.h>

//Declared in kvi_app.cpp and managed by KviApp class
//extern QPixmap * g_pWindowIcon[KVI_WND_NUM_ICONS];
extern QPixmap * g_pixViewOut[KVI_OUT_NUM_IMAGES];

KviListWindow::KviListWindow(KviFrame * lpFrm)
:KviWindow(KVI_LIST_WINDOW_NAME,KVI_WND_TYPE_LIST,lpFrm)
{
	m_pChanList = new QList<KviChanListEntry>;
	m_pChanList->setAutoDelete(true);

	m_pVerSplitter   = new QSplitter(QSplitter::Vertical,this);
	m_pSplitter      = new QSplitter(QSplitter::Horizontal,m_pVerSplitter);
	m_pListView      = new QListView(m_pSplitter);
	m_pListView->addColumn(_CHAR_2_QSTRING(__tr("Channel")));
	m_pListView->addColumn(_CHAR_2_QSTRING(__tr("Users")));
	m_pListView->addColumn(_CHAR_2_QSTRING(__tr("Topic")));
	m_pListView->setAllColumnsShowFocus(true);
	connect(m_pListView,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(joinChannel(QListViewItem *)));
	m_pView          = new KviIrcView(m_pVerSplitter,lpFrm,this);

	m_pPanel         = new QWidget(m_pSplitter);


	QGridLayout *g   = new QGridLayout(m_pPanel,12,2,5,4);

	QLabel *l        = new QLabel(_CHAR_2_QSTRING(__tr("/LIST command parameters:")),m_pPanel);
	g->addMultiCellWidget(l,0,0,0,1);
	m_pRequestList   = new QPushButton(_CHAR_2_QSTRING(__tr("&Request New List")),m_pPanel);
	connect(m_pRequestList,SIGNAL(clicked()),this,SLOT(requestList()));
	g->addMultiCellWidget(m_pRequestList,2,2,0,1);

	QFrame *r        = new QFrame(m_pPanel);
	r->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	r->setMinimumHeight(r->frameWidth() * 2);
	g->addMultiCellWidget(r,3,3,0,1);

	l                = new QLabel(_CHAR_2_QSTRING(__tr("Local filter:")),m_pPanel);
	g->addMultiCellWidget(l,4,4,0,1);

	l                = new QLabel(_CHAR_2_QSTRING(__tr("Channel mask:")),m_pPanel);
	g->addWidget(l,5,0);

	l                = new QLabel(_CHAR_2_QSTRING(__tr("Topic mask:")),m_pPanel);
	g->addWidget(l,6,0);

	l                = new QLabel(_CHAR_2_QSTRING(__tr("Minimum users:")),m_pPanel);
	g->addWidget(l,7,0);

	l                = new QLabel(_CHAR_2_QSTRING(__tr("Maximum users:")),m_pPanel);
	g->addWidget(l,8,0);

	r                = new QFrame(m_pPanel);
	r->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	r->setMinimumHeight(r->frameWidth() * 2);
	g->addMultiCellWidget(r,10,10,0,1);

	m_pDisplayedItems = new QLabel(m_pPanel);
	g->addMultiCellWidget(m_pDisplayedItems,11,11,0,1);
	m_pApplyFilter   = new QPushButton(_CHAR_2_QSTRING(__tr("&Apply Filter")),m_pPanel);
	connect(m_pApplyFilter,SIGNAL(clicked()),this,SLOT(applyFilter()));
	g->addMultiCellWidget(m_pApplyFilter,9,9,0,1);

	updateDisplayedItems();

	setFocusHandler(m_pListView);

	// These object MUST be created AFTER the call to setFocusHandler

	m_pListParams    = new QLineEdit(m_pPanel);
	m_pListParams->setFocusPolicy(ClickFocus);
	g->addMultiCellWidget(m_pListParams,1,1,0,1);
	m_pChannelRegexp = new QLineEdit(m_pPanel);
	m_pChannelRegexp->setFocusPolicy(ClickFocus);
	m_pChannelRegexp->setText("*");
	g->addWidget(m_pChannelRegexp,5,1);
	m_pTopicRegexp = new QLineEdit(m_pPanel);
	m_pTopicRegexp->setFocusPolicy(ClickFocus);
	m_pTopicRegexp->setText("*");
	g->addWidget(m_pTopicRegexp,6,1);
	m_pMinUsers = new QLineEdit(m_pPanel);
	m_pMinUsers->setFocusPolicy(ClickFocus);
	m_pMinUsers->setText("0");
	g->addWidget(m_pMinUsers,7,1);
	m_pMaxUsers = new QLineEdit(m_pPanel);
	m_pMaxUsers->setFocusPolicy(ClickFocus);
	m_pMaxUsers->setText("-1");
	g->addWidget(m_pMaxUsers,8,1);

	m_szLastServer = "???";
	m_szLastDate   = "???";
	m_bInRequest = false;

	m_pChanListLoadTimer = 0;
	m_pChanListData = 0;

	loadChanList();
}

KviListWindow::~KviListWindow()
{
	if(m_pChanListLoadTimer)
	{
		m_pChanListLoadTimer->stop();
		delete m_pChanListLoadTimer;
	}
	saveChanList();
	clearChanList();
	delete m_pChanList;
}

void KviListWindow::joinChannel(QListViewItem *it)
{
	if(!it)return;
	KviStr chan = it->text(0);
	if(!m_pFrm->m_pSocket->sendFmtData("JOIN %s",chan.ptr()))
	{
		outputNoFmt(KVI_OUT_ERROR,__tr("Not connected to a server : Cannot join channel"));
	} else output(KVI_OUT_INTERNAL,__tr("Sent JOIN %s message : Waiting for reply..."),chan.ptr());
}

void KviListWindow::loadChanList()
{
	clearChanList();
	m_pListView->clear();

	KviStr szName;
	g_pApp->getLocalKVircDirectory(szName,KviApp::Config,"kvi.chanlist.conf");
	if(!kvi_fileExists(szName.ptr()))
	{
		outputNoFmt(KVI_OUT_INTERNAL,__tr("No saved channel list; you must request one from your server."));
		return;
	}

	if(!kvi_loadFile(szName.ptr(),m_szChanListData))
	{
		output(KVI_OUT_ERROR,__tr("Unable to load the channel list (%s)"),szName.ptr());
		return;
	}

	m_pChanListData = m_szChanListData.ptr();

	outputNoFmt(KVI_OUT_INTERNAL,__tr("Loading the most recent channel list. This may take a while."));

	m_pRequestList->setEnabled(false);
	m_pApplyFilter->setEnabled(false);

	m_bInRequest =true; //Avoid the "Resetting..." message

	m_pChanListLoadTimer = new QTimer();
	connect(m_pChanListLoadTimer,SIGNAL(timeout()),this,SLOT(loadChanListChunk()));
	m_pChanListLoadTimer->start(50);
}

void KviListWindow::loadChanListChunk()
{
	__range_valid(m_pChanListData);
	m_pListView->setUpdatesEnabled(false);

	int n = 50;

	KviStr line;

	while((*m_pChanListData) && n){
		m_pChanListData = kvi_extractToken(line,m_pChanListData,'\n');
		line.stripWhiteSpace();
		n--;
		if(*(line.ptr()) == 'L'){
			if(kvi_strEqualCSN("LISTFROM:",line.ptr(),9))
			{
				line.cutLeft(9);
				line.stripWhiteSpace();
				m_szLastServer = line.ptr();
				continue;
			} else if(kvi_strEqualCSN("LISTDATE:",line.ptr(),9))
			{
				line.cutLeft(9);
				line.stripWhiteSpace();
				m_szLastDate = line.ptr();
				continue;
			}
		}
		if(line.hasData())processListEntry(line.ptr());
	}

	m_pListView->setUpdatesEnabled(true);
	m_pListView->update();

	if(!(*m_pChanListData)){
		m_pChanListLoadTimer->stop();
		delete m_pChanListLoadTimer;
		m_pChanListLoadTimer = 0;
		m_pChanListData = 0;

		m_pRequestList->setEnabled(true);
		m_pApplyFilter->setEnabled(true);

		m_szChanListData = ""; // free some KB of mem

		if(m_szLastServer.isEmpty())m_szLastServer = "???";
		if(m_szLastDate.isEmpty())m_szLastDate = "???";

		m_bInRequest = false;

		output(KVI_OUT_INTERNAL,__tr("Cached channel list from %s, received at %s"),m_szLastServer.ptr(),m_szLastDate.ptr());
		output(KVI_OUT_INTERNAL,__tr("Total: %u channels"),m_pChanList->count());

		updateDisplayedItems();
	}
}

void KviListWindow::saveChanList()
{
	if(m_pChanList->count() == 0)return; //empty channel list...do not save

	KviStr szName;
	g_pApp->getLocalKVircDirectory(szName,KviApp::Config,"kvi.chanlist.conf");

	QFile f(szName.ptr());
	if(!f.open(IO_WriteOnly|IO_Truncate)){
		output(KVI_OUT_ERROR,__tr("Unable to save the channel list to %s"),szName.ptr());
		return;
	}

	KviStr descr(KviStr::Format,"LISTFROM:%s\nLISTDATE:%s",m_szLastServer.ptr(),m_szLastDate.ptr());

	if(!kvi_writeLine(&f,descr.ptr()))
	{
		f.close();
		output(KVI_OUT_ERROR,__tr("Unable to save the channel list to %s"),szName.ptr());
		return;
	}

	for(KviChanListEntry * e=m_pChanList->first();e;e=m_pChanList->next())
	{
		KviStr tmp(KviStr::Format,"%s %d :%s",e->channel.ptr(),e->nUsers,e->topic.ptr());
		if(!kvi_writeLine(&f,tmp.ptr()))
		{
			f.close();
			output(KVI_OUT_ERROR,__tr("Unable to save the channel list to %s"),szName.ptr());
			return;
		}
	}

	f.close();
}

void KviListWindow::notifyDisconnected()
{
/* FIXME:
         "1. ADD A "JOINCHANNEL" FEATURE ON DBLCLK"
         "2. ADD A "IN REQUEST" INDICATOR SOMEWHERE"
*/
	if(m_bInRequest)
	{
		m_bInRequest = false;
		output(KVI_OUT_ERROR,__tr("Disconnected while receiving channel list"));
	}
//	m_pRequestList->setEnabled(false);
}

//void KviListWindow::notifyConnected()
//{
//	m_pRequestList->setEnabled(true);
//}

void KviListWindow::saveProperties()
{
	KviWindowProperty p;
	p.rect = externalGeometry();
	p.isDocked = isAttached();
	QValueList<int> l(m_pSplitter->sizes());
	if(l.count() >= 1)p.splitWidth1 = *(l.at(0));
	if(l.count() >= 2)p.splitWidth2 = *(l.at(1));
	p.timestamp = m_pView->timestamp();
	p.imagesVisible = m_pView->imagesVisible();
	p.isMaximized = isAttached() && isMaximized();
	QValueList<int> l2(m_pVerSplitter->sizes());
	if(l2.count() >= 1)p.topSplitWidth1 = *(l2.at(0));
	if(l2.count() >= 2)p.topSplitWidth2 = *(l2.at(1));
	p.topSplitWidth3 = 0;
	g_pOptions->m_pWinPropertiesList->setProperty(caption(),&p);
}

void KviListWindow::setProperties(KviWindowProperty *p)
{
	QValueList<int> l;
	l.append(p->splitWidth1);
	l.append(p->splitWidth2);
	m_pSplitter->setSizes(l);
	QValueList<int> l2;
	l2.append(p->topSplitWidth1);
	l2.append(p->topSplitWidth2); 
	m_pVerSplitter->setSizes(l);
	m_pView->setTimestamp(p->timestamp);
	m_pView->setShowImages(p->imagesVisible);
}

QPixmap * KviListWindow::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_LIST];
}

void KviListWindow::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages,false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);
}

void KviListWindow::resizeEvent(QResizeEvent *)
{
	m_pVerSplitter->setGeometry(0,0,width(),height());
}

void KviListWindow::updateDisplayedItems()
{
	KviStr tmp(KviStr::Format,__tr("Displayed %d of %u"),m_pListView->childCount(),m_pChanList->count());
	m_pDisplayedItems->setText(tmp.ptr());
}

void KviListWindow::startOfList()
{
	outputNoFmt(KVI_OUT_INTERNAL,__tr("Received beginning of list : Resetting"));
	clearChanList();
	m_pListView->clear();
	m_pChannelRegexp->setText("*");
	m_pTopicRegexp->setText("*");
	m_pMinUsers->setText("0");
	m_pMaxUsers->setText("-1");
	updateDisplayedItems();
	m_szLastServer = m_pFrm->m_global.szCurrentServerHost;
	m_szLastDate = QDateTime::currentDateTime().toString();
	m_bInRequest = true;
}

void KviListWindow::applyFilter()
{
	KviStr chan  = m_pChannelRegexp->text();
	KviStr topic = m_pTopicRegexp->text();
	KviStr min   = m_pMinUsers->text();
	KviStr max   = m_pMaxUsers->text();

	bool bOk;
	int iMin = min.toInt(&bOk);
	if(!bOk)iMin = 0;
	int iMax = max.toInt(&bOk);
	if(!bOk)iMax = -1;

#define KVI_MATCHBITS_MINUSERSMATCHED 1
#define KVI_MATCHBITS_MAXUSERSMATCHED 2
#define KVI_MATCHBITS_TOPICMATCHED 4
#define KVI_MATCHBITS_CHANNELMATCHED 8
#define KVI_MATCHBITS_TOTALMATCH 15

	int defmatchbits = 0;
	if(iMin <= 0)defmatchbits =  KVI_MATCHBITS_MINUSERSMATCHED;
	if(iMax < 0)defmatchbits |= KVI_MATCHBITS_MAXUSERSMATCHED;

	chan.stripWhiteSpace();
	topic.stripWhiteSpace();
	if(kvi_strEqualCS(chan.ptr(),"*") || chan.isEmpty())defmatchbits |= KVI_MATCHBITS_CHANNELMATCHED;
	if(kvi_strEqualCS(topic.ptr(),"*") || topic.isEmpty())defmatchbits |= KVI_MATCHBITS_TOPICMATCHED;

	QRegExp expChan(_CHAR_2_QSTRING(chan.ptr()),false,true);
	QRegExp expTopic(_CHAR_2_QSTRING(topic.ptr()),false,true);

	output(KVI_OUT_INTERNAL,__tr("Applying filter [%s,%s,%d,%d]"),chan.ptr(),topic.ptr(),iMin,iMax);

	m_pListView->clear();
	m_pListView->setUpdatesEnabled(false);

	for(KviChanListEntry * e=m_pChanList->first();e;e=m_pChanList->next()){
		int matchbits = defmatchbits;
		if(!(matchbits & KVI_MATCHBITS_MINUSERSMATCHED))
		{
			// Have to check the minimum users
			if(e->nUsers >= iMin)matchbits |= KVI_MATCHBITS_MINUSERSMATCHED;
			else continue;
		}
		if(!(matchbits & KVI_MATCHBITS_MAXUSERSMATCHED))
		{
			if(e->nUsers <= iMax)matchbits |= KVI_MATCHBITS_MAXUSERSMATCHED;
			else continue;
		}
		if(!(matchbits & KVI_MATCHBITS_TOPICMATCHED))
		{
			if(expTopic.match(_CHAR_2_QSTRING(e->topic.ptr())) > -1)matchbits |= KVI_MATCHBITS_TOPICMATCHED;
			else continue;
		}
		if(!(matchbits & KVI_MATCHBITS_CHANNELMATCHED))
		{
			if(expChan.match(_CHAR_2_QSTRING(e->channel.ptr())) > -1)matchbits |= KVI_MATCHBITS_CHANNELMATCHED;
			else continue;
		}
		if(matchbits == KVI_MATCHBITS_TOTALMATCH)
		{
			QString nU;
			nU.setNum(e->nUsers);
			QListViewItem *it = new QListViewItem(m_pListView,_CHAR_2_QSTRING(e->channel.ptr()),
			nU,_CHAR_2_QSTRING(e->topic.ptr()));
			(void)it;
		}
	}

	m_pListView->setUpdatesEnabled(true);
	m_pListView->repaint();

	updateDisplayedItems();
}

void KviListWindow::clearChanList()
{
	while(m_pChanList->first())m_pChanList->removeFirst();
}

void KviListWindow::endOfList()
{
	outputNoFmt(KVI_OUT_INTERNAL,__tr("Received end of list"));
	output(KVI_OUT_INTERNAL,__tr("Total: %u channels"),m_pChanList->count());
	updateDisplayedItems();
	m_bInRequest = false;
}

void KviListWindow::processListEntry(const char *msg)
{
	if(!m_bInRequest)startOfList();

	KviChanListEntry * e = new KviChanListEntry;
	msg = kvi_extractToken(e->channel,msg,' ');
	if(e->channel.isEmpty()){
		delete e;
		brokenListEntry(msg);
		return;
	}
	KviStr tmp;
	msg = kvi_extractToken(tmp,msg,' ');
	bool bOk = false;
	e->nUsers = tmp.toInt(&bOk);
	if(!bOk){
		delete e;
		brokenListEntry(msg);
		return;
	}
	while((*msg) && (isspace(*msg)) || (*msg == ':'))msg++;
	e->topic = msg;
	m_pChanList->append(e);
	QListViewItem *it = new QListViewItem(m_pListView,_CHAR_2_QSTRING(e->channel.ptr()),_CHAR_2_QSTRING(tmp.ptr()),_CHAR_2_QSTRING(e->topic.ptr()));
	updateDisplayedItems();
	(void)it; //avoid warnings
}

void KviListWindow::brokenListEntry(const char *msg)
{
	output(KVI_OUT_ERROR,__tr("Received broken list entry: [%s]"),msg ? msg : __tr("EMPTY MESSAGE"));
}

void KviListWindow::requestList()
{
//	if(m_bInRequest)outputNoFmt(KVI_OUT_ERROR,__tr("WARNING: A /LIST request has been already sent and it is being processed by the server."));
	KviStr tmp = m_pListParams->text();
	tmp.stripWhiteSpace();
	bool bSent = false;
	if(tmp.isEmpty())bSent = m_pFrm->m_pSocket->sendData("LIST",4);
	else bSent = m_pFrm->m_pSocket->sendFmtData("LIST %s",tmp.ptr());
	if(!bSent)outputNoFmt(KVI_OUT_ERROR,__tr("Not connected to server : Cannot request channel list"));
	else outputNoFmt(KVI_OUT_INTERNAL,__tr("Channel list requested : Waiting for reply"));
}



#include "m_kvi_list.moc"
