//
//   SPEAK plugin for KVirc 1.1 Phoenix.
//
//   This plugin enables you to use IBM's ViaVoiceOutloud to speak
//   messages to you.  Messages that are directed to you, over the
//   Relay Chat, can be spoken outloud or sentences spoken by specific
//   people can also heard outloud.
//
//   To use the plugin, one needs to have a working IBM ViaVoice
//   Outloud, which can be retrieved from IBM's site.  For information
//   about IBM ViaVoice http://www-4.ibm.com/software/speech/dev/
//
//   Copyright (C) rn E. Hansen (oe.hansen@gamma.telenordia.se)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "trees.h"

Node::Node()
{
  left  = 0;
  right = 0;
}

Node::~Node()
{
  if (left) delete left;;
  if (right) delete right;
}

Node *Node::Left(Node *p_np)
{
  Node *sav = left;

  left = p_np;
  return sav;
}

Node *Node::Right(Node *p_np)
{
  Node *sav = right;

  right = p_np;
  return sav;
}

BasicTree::BasicTree()
{
  head = 0;
}

BasicTree::~BasicTree()
{
  if (head)
    delete head;
}

//
// Length
//
// return the number of Nodes found in the tree
//

int BasicTree::Length()
{
  return Length(head);
}

//
// Length of node
//
// return the total number of nodes in this branch
//

int BasicTree::Length(Node *ref)
{
  if (ref == NULL)
    return 0;
  return 1 + Length(ref->Left()) + Length(ref->Right());
}

//
// Operator ()
//
// Return the numbered node in the tree.
//

Node *BasicTree::operator () (int n)
{
  Node *ref = head;
  int l;

  l = Length(head);
  if (n < 1 || n > l)
    return NULL;
  while(ref) {
    l = Length(ref->Left()) + 1;
    if (n == l)
      return ref;
    else if (n < l)
      ref = ref->Left();
    else {
      n = n - l;
      ref = ref->Right();
    }
  }
  return NULL;
}

//
// Insert
//
// Insert the node by pushing it down the header Node
//

void BasicTree::Insert(Node *ref)
{
  head = Down(head, ref);
}

//
// Unlink
//
// We unlink the node, by relinking the parent node to
// the children.  We avoid deleting a node that doesn't
// belong to this tree.
//

void BasicTree::Unlink(Node *ref)
{
  Node *par = Parent(ref);

  if (par == NULL && ref == head) {
      head = Delete(ref);
  } else if (par != NULL) {
    if (ref == par->Left())
      par->Left(Delete(ref));
    else
      par->Right(Delete(ref));
  }
}

//
// Delete
//
//  (x)         ->     (x)     =>  (o)  -  X
//  / \                  \         /
// o   o                  o       o
//                       /
//                      o
//

Node *BasicTree::Delete(Node *ref)
{
  Node *np;

  if (ref == 0)
    return 0;
  np = ref->Left();
  ref->Left(NULL);
  if (ref->Right())
    np = Down(np, ref->Right());
  ref->Left(NULL);
  if (head == ref)
    head = np;
  delete ref;
  return np;
}

//
// Down
//
// A method to push a node down into the tree structure,
// and try to maintain a resonably basic balance.
//
//  X   = New Node
// ( )  = Base Node
//  O   = Any Node
//
// X + (NULL)   -> (X)
//
// X + (O)      -> X<0   => (O)  else  (O)
//                          /            \                    dummy
//                         X              X
//
// X + (O1)     -> X>=O1 => (O1) else O2 + (X)
//     /                    /  \             \                dummy
//    O2                   O2   X             O1
//
// X + (O1)     ->  X<O1 => (O1) else O2 + (X)
//        \                 /  \           /
//         O2              X    O2        O1
//
// X + (O1)     ->  x<O1 => (X) + O1 else    O1 + (X)
//     /  \                 /       \       /       \         dummy
//    O2   O3              O2        O3     O2      O3
// 
//
// 

Node *BasicTree::Down(Node *base, Node *ref)
{
  Node *sav;

  // Check for instance X + (NULL)
  if (base == NULL)
    return ref;

  // Check for instance X + (o)
  if (base->Left() == 0 && base->Right() == 0) {
    if (ref->Compare(base) >= 0) {
      base->Right(ref);
    } else
      base->Left(ref);
    return base;
  }

  // Instances 3 and 4, one side of base is empty
  if (base->Right() == NULL) {
    if (ref->Compare(base) >= 0) {
      base->Right(ref);
      return base;
    }
    sav = base->Left();
    base->Left(NULL);
    ref->Right(Down(ref->Right(), base));
    return Down(ref, sav);
  } else if (base->Left() == NULL) {
    if (ref->Compare(base) < 0) {
      base->Left(ref);
      return base;
    }
    sav = base->Right();
    base->Right(NULL);
    ref->Left(Down(ref->Left(), base));
    return Down(ref, sav);
  }

  // Finally, split the base and make two #2 nodes to splice.
  if (ref->Compare(base) < 0) {
    sav = base->Left(0);
    ref = Down(sav,ref);
  } else {
    sav = base->Right(0);
    ref = Down(ref,sav);
  }

  return Down(base,ref);
}

//
// Lookup
//
// This is a binary search on the tree, and does not
// support multiple instances of an item.  It will only
// retract the first instance of a given item.
//
// The Compare(void *) routine is a basic component, that
// must be filled in by the derived class.
//

Node *BasicTree::Lookup(void *key)
{
  Node *np = head;
  int n;

  while (np) {
    n = np->Compare(key);
    if (n == 0)
      return np;
    if (n < 0)
      np = np->Right();
    else
      np = np->Left();
  }
  return 0;
}

//
// Parent
//
// This does a lookup on the tree for a specific node,
// and if found, returns the parent node.
//

Node *BasicTree::Parent(Node *ref)
{
  Node *np = head;
  Node *pa = 0;
  int n;

  while (np) {
    n = np->Compare(ref);
    if (n == 0)
      return pa;
    pa = np;
    if (n < 0)
      np = np->Right();
    else
      np = np->Left();
  }
  return 0;
}
