// =============================================================================
//
//      --- kvi_imagelibrary_kde.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Robin Verduijn <robin@debian.org>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviImageLibrary"

#include <qbitmap.h>

#include "kvi_debug.h"
#include "kvi_fileutils.h"
#include "kvi_imagelibrary_kde.h"
#include "kvi_locale.h"

KviImageLibrary::KviImageLibrary(const char *path1, const char *path2, int imageWidth, int imageHeight)
{
	m_count = -1;
	__range_valid(path1);
	__range_valid(path2);
	if( kvi_fileExists(path1) ) {
		loadLibrary(path1);
		if( !libraryLoaded() )
			loadLibrary(path2);
	} else loadLibrary(path2);
	__range_valid(imageWidth  > 0);
	__range_valid(imageHeight > 0);
	setItemSize(QSize(imageWidth, imageHeight));
}

KviImageLibrary::~KviImageLibrary()
{
	m_count = 0;
}

bool KviImageLibrary::loadLibrary(const char *path)
{
	m_count = -1;
	setPixmap(QPixmap(path));
	if( !libraryLoaded() ) {
		m_count = 0;
		debug(_i18n_("WARNING: cannot load image library %s"), path);
	}
	return (!pixmap().isNull());
}

int KviImageLibrary::imageCount()
{
	if( !libraryLoaded() ) return 0;
	if( m_count != -1 )
		return m_count;

	QSize size = itemSize();
	if( (size.width() < 1) || (size.height() < 1) )
		return 0;
	int nRows = pixmap().width() / size.width();
	return (nRows * (pixmap().height() / size.height()));
}

bool KviImageLibrary::libraryLoaded()
{
	return (!pixmap().isNull());
}

QPixmap KviImageLibrary::getImage(int zeroBasedIndex)
{
	__range_valid(zeroBasedIndex >= 0);
	__range_valid(zeroBasedIndex < imageCount());

	QPixmap image(itemSize(), pixmap().depth());
	if( (zeroBasedIndex >= imageCount() ) || (zeroBasedIndex < 0) ) {
		image.fill();
		return image;
	}
	QRect rect = coordinates(zeroBasedIndex);
	// Copy the image data
	if( !rect.isEmpty() )
		bitBlt(&image, QPoint(0, 0), &pixmap(), rect, Qt::CopyROP);
	if( pixmap().mask() ) {
		// Copy the mask too
		QBitmap bmp(itemSize());
		bitBlt(&bmp, QPoint(0, 0), pixmap().mask(), rect, Qt::CopyROP);
		image.setMask(bmp);
	}
	return image;
}
