// =============================================================================
//
//      --- kvi_alias.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviAlias"

#include <qfile.h>

#include "kvi_alias.h"
#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_locale.h"

KviAliasManager::KviAliasManager()
{
	m_pAliasList = new QPtrList<KviAlias>;
	m_pAliasList->setAutoDelete(true);
}

KviAliasManager::~KviAliasManager()
{
	if( m_pAliasList ) {
		delete m_pAliasList;
		m_pAliasList = 0;
	}
}

KviAlias *KviAliasManager::findAlias(const char *name)
{
	int r;
	for( KviAlias *a = m_pAliasList->first(); a; a = m_pAliasList->next() ) {
		r = kvi_strcmpCI(a->szName.ptr(), name);
		if( r == 0 ) return a; // Equal
		if( r <  0 ) return 0; // Found a greater one
	}
	return 0;
}

bool KviAliasManager::addAlias(KviAlias *s)
{
	__range_valid(s);
	s->bLocked = false;

	KviAlias *a = findAlias(s->szName.ptr());
	if( a ) {
		if( a->bLocked ) {
			debug("Attempt to change a locked alias : %s", a->szName.ptr());
			return false; // It is locked!
		}
		m_pAliasList->removeRef(a);
	}
	int idx = 0;
	for( a = m_pAliasList->first(); a; a = m_pAliasList->next() ) {
		if( kvi_strcmpCI(a->szName.ptr(), s->szName.ptr()) < 0 ) {
			// Found a greater one
			m_pAliasList->insert(idx, s);
			return true;
		}
		idx++;
	}
	// Found no greater aliases
	m_pAliasList->append(s);
	return true;
}

void KviAliasManager::clearAll()
{
	if( m_pAliasList ) delete m_pAliasList;
	m_pAliasList = new QPtrList<KviAlias>;
	m_pAliasList->setAutoDelete(true);
}

int KviAliasManager::count()
{
	return m_pAliasList->count();
}

void KviAliasManager::removeAlias(KviAlias *s)
{
	__range_valid(s);
	m_pAliasList->removeRef(s);
}

bool KviAliasManager::removeAlias(const char *name)
{
	KviAlias *a = findAlias(name);
	if( a ) {
		if( a->bLocked )
			debug("Attempt to remove a locked alias : %s", a->szName.ptr());
		else {
			m_pAliasList->removeRef(a);
			return true;
		}
	}
	return false;
}

bool KviAliasManager::load(const char *filename, KviStr &error)
{
	KviStr buffer;
	if( !kvi_loadFile(filename, buffer) ) {
		error = _i18n_("Unable to load file");
		return false;
	}
	if( !kvi_strEqualCSN(KVI_MAGIC_STRING, buffer.ptr(), KVI_MAGIC_STRING_LENGTH) ) {
		error = _i18n_("Invalid magic");
		return false; // Not a KVIrc file
	}
	buffer.cutLeft(KVI_MAGIC_STRING_LENGTH);
	buffer.stripLeftWhiteSpace(); // Be flexible... allow some spaces... for manual editing by experts :)
	if( buffer.isEmpty() )
		return true; // Done

	while( buffer.hasData() ) {
		// [ALIAS <aliasname> bytes <numofbytes>]\n
		// ...
		// ...
		// ...
		// \n[ENDALIAS]\n
		if( !kvi_strEqualCIN("[ALIAS ", buffer.ptr(), 7) ) {
			error = _i18n_("Unrecognized [ALIAS] label.");
			return false;
		}
		buffer.cutLeft(7);

		buffer.stripLeftWhiteSpace(); // Be flexible... allow some spaces... for manual editing by experts :)

		KviAlias *a = new KviAlias;

		if( !buffer.getToken(a->szName, ' ') ) {
			// There must be at least another token
			error = _i18n_("Syntax error in [ALIAS] label");
			if( a ) delete a;
			return false;
		}
		if( a->szName.isEmpty() ) {
			// Syntax error somewhere
			error = _i18n_("Empty alias name?");
			if( a ) delete a;
			return false;
		}

		buffer.stripLeftWhiteSpace(); // Be flexible... allow some spaces... for manual editing by experts :)

		if( !kvi_strEqualCIN("bytes ", buffer.ptr(), 6) ) {
			error = _i18n_("Syntax error in [ALIAS] label (alias length)");
			if( a ) delete a;
			return false;
		}
		buffer.cutLeft(6);
		buffer.stripLeftWhiteSpace(); // Be flexible... allow some spaces... for manual editing by experts :)

		if( !isdigit(*(buffer.ptr())) ) {
			error = _i18n_("Syntax error in [ALIAS] label (digit expected)");
			if( a ) delete a;
			return false;
		}

		KviStr numBytes;
		if( !buffer.getToken(numBytes, ']') ) {
			error = _i18n_("Syntax error in [ALIAS] label (cannot extract alias length)");
			if( a ) delete a;
			return false;
		}

		bool bOk = false;
		int bytes = numBytes.toInt(&bOk);
		if( !bOk ) {
			// Syntax error again... someone messed with the file
			error = _i18n_("Unrecogized token ");
			error += numBytes;
			if( a ) delete a;
			return false;
		}
		buffer.cutLeft(1); // Cut the \n
		if( buffer.len() < bytes ) {
			error = _i18n_("Alias buffer smaller than declared length (");
			error += numBytes;
			error += ")";
			if( a ) delete a;
			return false;
		}
		a->szBuffer.setStr(buffer.ptr(), bytes);
		buffer.cutLeft(bytes);

		if( !kvi_strEqualCIN("\n[ENDALIAS]", buffer.ptr(), 11) ) {
			error = _i18n_("Unrecognized [ENDALIAS] label");
			if( a ) delete a;
			return false;
		}
		buffer.cutLeft(11);

		addAlias(a);

		buffer.stripLeftWhiteSpace(); // Be flexible... Allow some spaces... For manual editing by experts :)
	}
	return true;
}

bool KviAliasManager::save(const char *filename)
{
	QFile f(filename);
	if( !f.open(IO_WriteOnly | IO_Truncate) )
		return false;

	f.writeBlock(KVI_MAGIC_STRING, KVI_MAGIC_STRING_LENGTH);

	for( KviAlias *a = m_pAliasList->first(); a; a = m_pAliasList->next() ) {
		f.writeBlock("[ALIAS ", 7);
		f.writeBlock(a->szName.ptr(), a->szName.len());
		f.writeBlock(" bytes ", 7);
		KviStr szNBytes(KviStr::Format, "%d", a->szBuffer.len());
		f.writeBlock(szNBytes.ptr(), szNBytes.len());
		f.writeBlock("]\n", 2);
		f.writeBlock(a->szBuffer.ptr(), a->szBuffer.len());
		f.writeBlock("\n[ENDALIAS]\n", 12);
	}

	f.close();
	return true;
}

QPtrList<KviAlias> *KviAliasManager::aliasList()
{
	return m_pAliasList;
}
