// =============================================================================
//
//      --- kvi_options_ctcp_protection.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviOptionsCtcpProtection"

#include <qlayout.h>

#include "kvi_boolselector.h"
#include "kvi_integerselector.h"
#include "kvi_locale.h"
#include "kvi_options_ctcp_protection.h"
#include "kvi_options.h"

/*
	@quickhelp: KviOptionsCtcpProtection
	@widget: CTCP protection options
		KVIrc has a primitive built-in protection against the CTCP FLOOD attack.<br>
		When protection is enabled KVIrc will reply only to a limited
		number of CTCP messages.<br>
		For example, if you allow one CTCP request in 5 seconds,
		KVIrc will happily reply to the first CTCP and then ignore all
		the other requests until 5 seconds have been passed.<br>
		The unreplied requests are treated as "CTCP flood" attacks,
		will trigger an <a href="onctcpflood.kvihelp">OnCtcpFlood</a> event and eventually will be signaled with a warning.<br>
		The protection mechanism works only for the CTCP requests that require a reply.<br>
		A CTCP ACTION (for example) does not require a reply, so it cannot be used
		in a "CTCP flood" attack.<br>
		The flood warnings will be signaled in the current window by default, but may
		be redirected to the console.<br>
		Additionally, you can choose to ignore specific CTCP requests.<br>
		Even if you ignore a specific CTCP, the <a href="onctcp.kvihelp">OnCtcp</a> event
		will be still triggered.<br>
*/
KviOptionsCtcpProtection::KviOptionsCtcpProtection(QWidget *parent)
	: KviOptionsWidget(parent, KVI_GENERAL_OPTIONS_WIDGET_ID_CTCPPROTECTION)
{
	QGridLayout *g = new QGridLayout(
		this, 12, 2, KVI_GENERAL_OPTIONS_GRIDLAYOUT_BORDER, KVI_GENERAL_OPTIONS_GRIDLAYOUT_SPACE
	);

	KviBoolSelector *b1 = new KviBoolSelector(this,
		_i18n_("Enable CTCP flood protection"), &(g_pOptions->m_bUseAntiCtcpFlood)
	);
	g->addMultiCellWidget(b1, 0, 0, 0, 1);

	KviIntegerSelector *i = new KviIntegerSelector(this,
		_i18n_("Allow maximum of:"),
		&(g_pOptions->m_iMaxCtcpCount), KviIntegerSelector::Int, 1, 65535, 6,
		g_pOptions->m_bUseAntiCtcpFlood, 100, 0, _i18n_(" CTCP(s)")
	);
	connect(b1, SIGNAL(toggled(bool)), i, SLOT(setEnabled(bool)));
	g->addWidget(i, 1, 0);

	i = new KviIntegerSelector(this,
		_i18n_(" in:"),
		&(g_pOptions->m_iMaxCtcpTime), KviIntegerSelector::Int, 1, 65535, 10,
		g_pOptions->m_bUseAntiCtcpFlood, 100, 0, _i18n_(" second(s)")
	);
	connect(b1, SIGNAL(toggled(bool)), i, SLOT(setEnabled(bool)));
	g->addWidget(i, 1, 1);

	KviBoolSelector *b = new KviBoolSelector(this,
		_i18n_("CTCP flood warnings to console"),
		&(g_pOptions->m_bCtcpFloodWarningsToConsole), g_pOptions->m_bUseAntiCtcpFlood
	);
	connect(b1, SIGNAL(toggled(bool)), b, SLOT(setEnabled(bool)));
	g->addMultiCellWidget(b, 2, 2, 0, 1);

	QFrame *f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f, 3, 3, 0, 1);

	b = new KviBoolSelector(this, _i18n_("Ignore CTCP PING"), &(g_pOptions->m_bIgnoreCtcpPingRequests));
	g->addMultiCellWidget(b, 4, 4, 0, 1);
	b = new KviBoolSelector(this, _i18n_("Ignore CTCP VERSION"), &(g_pOptions->m_bIgnoreCtcpVersionRequests));
	g->addMultiCellWidget(b, 5, 5, 0, 1);
	b = new KviBoolSelector(this, _i18n_("Ignore CTCP TIME"), &(g_pOptions->m_bIgnoreCtcpTimeRequests));
	g->addMultiCellWidget(b, 6, 6, 0, 1);
	b = new KviBoolSelector(this, _i18n_("Ignore CTCP USERINFO"), &(g_pOptions->m_bIgnoreCtcpUserinfoRequests));
	g->addMultiCellWidget(b, 7, 7, 0, 1);
	b = new KviBoolSelector(this, _i18n_("Ignore CTCP CLIENTINFO"), &(g_pOptions->m_bIgnoreCtcpClientinfoRequests));
	g->addMultiCellWidget(b, 8, 8, 0, 1);
	b = new KviBoolSelector(this, _i18n_("Ignore CTCP SOURCE"), &(g_pOptions->m_bIgnoreCtcpSourceRequests));
	g->addMultiCellWidget(b, 9, 9, 0, 1);
	b = new KviBoolSelector(this, _i18n_("Ignore CTCP FINGER"), &(g_pOptions->m_bIgnoreCtcpFingerRequests));
	g->addMultiCellWidget(b, 10, 10, 0, 1);

	// Trick
	f = new QFrame(this);
	g->addMultiCellWidget(f, 11, 11, 0, 1);
	g->setRowStretch(11, 1);
}

KviOptionsCtcpProtection::~KviOptionsCtcpProtection()
{
	// Nothing here
}

void KviOptionsCtcpProtection::commitChanges()
{
	KviBoolSelector::commitAll(this);
	KviIntegerSelector::commitAll(this);
}

#include "m_kvi_options_ctcp_protection.moc"
