// =============================================================================
//
//      --- kvi_eventeditor.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviEventEditor"

#include <qcheckbox.h>
#include <qsplitter.h>
#include <qvbox.h>

#include "kvi_app.h"
#include "kvi_event.h"
#include "kvi_eventeditor.h"
#include "kvi_frame.h"
#include "kvi_label.h"
#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_messagebox.h"
#include "kvi_pushbutton.h"
#include "kvi_script_datalistview.h"
#include "kvi_script_editor.h"

// TODO: Totally missing quick help!

// Declared in kvi_app.cpp:
extern KviEventManager *g_pEventManager;

KviStr g_szLastSelectedEvent = "";

/*
	@quickhelp: KviEventEditor
	@widget: Event editor
		Here you edit the <a href="syntax_events.kvihelp">event handlers</a>.<br>
		To edit an event handler, select its name in the left list box.<br>
		When you are done editing the event, enable it by checking the "Enable" checkbox.<br>
*/

/**
 *
 * EVENT EDITOR
 *
 * This widget allows editing of the events
 *
 */
KviEventEditor::KviEventEditor(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	m_pCurItem  = 0;
	m_pSplitter = new QSplitter(this);
	m_pSplitter->setOpaqueResize();

	QVBox *left_box = new QVBox(m_pSplitter);
	m_pListView = new KviListView(left_box);
	m_pListView->addColumn(_i18n_("Event"));
	m_pListView->addColumn(_i18n_("-"));
	m_pListView->addColumn(_i18n_("!"));
	m_pListView->setSelectionMode(QListView::Single);
	m_pListView->setAutoResize(false);
	m_pClear = new KviPushButton(_i18n_("&Clear Handlers"), left_box);

	QVBox *right_box = new QVBox(m_pSplitter);
	m_pEnabled = new QCheckBox(_i18n_("Enable event"), right_box);
	m_pEnabled->setEnabled(false);
	m_pParamsLabel = new KviLabel(right_box);
	m_pParamsLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	m_pParamsLabel->setEnabled(false);

	m_pScriptEditor = new KviScriptEditor(right_box);
	m_pScriptEditor->setEnabled(false);

	fillEventListView();
	connect(m_pListView,
		SIGNAL(selectionChanged(KviListViewItem *)), this, SLOT(listViewSelectionChanged(KviListViewItem *))
	);
	connect(m_pEnabled, SIGNAL(toggled(bool)), this, SLOT(enableCheckToggled(bool)));
	connect(m_pClear,   SIGNAL(clicked()),     this, SLOT(clearAllHandlers()));

	KviScriptDataListViewItem *it = findEventItem(g_szLastSelectedEvent.ptr());
	if( it )
		m_pListView->setSelected(it, true);
}

KviEventEditor::~KviEventEditor()
{
	// Nothing here
}

KviScriptDataListViewItem *KviEventEditor::findEventItem(const char *item_key)
{
	for( KviScriptDataListViewItem *it = (KviScriptDataListViewItem *) m_pListView->firstChild()
	   ; it
	   ; it = (KviScriptDataListViewItem *) it->nextSibling()
	) {
		KviStr tmp;
		it->makeKey(tmp);
		if( kvi_strEqualCI(tmp.ptr(), item_key) )
			return it;
	}
	return 0;
}

void KviEventEditor::fillEventListView()
{
	KviScriptDataListViewItem *it = 0;
	for( int i = 0; i < KVI_NUM_EVENTS; i++ ) {
		it = new KviScriptDataListViewItem(m_pListView,
			g_pEventManager->m_szEventBuffer[i].ptr(),
			g_pEventManager->m_eventDescriptor[i].eventName,
			g_pEventManager->m_szEventBuffer[i].hasData() ? "X" : "",
			g_pEventManager->m_bEventEnabled[i] ? "X" : "",
			g_pEventManager->m_eventDescriptor[i].eventParams
		);
	}
}

void KviEventEditor::clearAllHandlers()
{
	if( KviMessageBox::warningYesNo(
		_i18n_("Are you sure you want to clear all event handlers?"),
		_i18n_("Clear Handlers"),
		this) == KviMessageBox::Yes
	) {
		for( KviScriptDataListViewItem *it = (KviScriptDataListViewItem *) m_pListView->firstChild()
		   ; it
		   ; it = (KviScriptDataListViewItem *) it->nextSibling()
		) {
			it->m_szBuffer = "";
			it->setText(1, "");
			it->setText(2, "");
		}
		m_pScriptEditor->setText("");
		m_pEnabled->setChecked(false);
	}
}

void KviEventEditor::saveCurrentItem()
{
	if( !m_pCurItem ) return;

	m_pCurItem->m_szBuffer = m_pScriptEditor->text();
	if( m_pCurItem->m_szBuffer.hasNonWhiteSpaceData() ) {
		m_pCurItem->setText(1, "X");
		m_pCurItem->setText(2, m_pEnabled->isChecked() ? "X" : "");
	} else {
		// Empty implementation
		m_pCurItem->m_szBuffer = "";
		m_pCurItem->setText(1, "");
		m_pCurItem->setText(2, "");
	}
}

void KviEventEditor::enableCheckToggled(bool bEnabled)
{
	if( !m_pCurItem ) return;

	m_pScriptEditor->setEnabled(bEnabled);
	KviStr tmp = m_pScriptEditor->text();
	m_pCurItem->setText(1, tmp.hasNonWhiteSpaceData() ? "X" : "");
	m_pCurItem->setText(2, bEnabled ? "X" : "");
}

void KviEventEditor::listViewSelectionChanged(KviListViewItem *it)
{
	saveCurrentItem();
	m_pCurItem = 0;
	if( it ) {
		m_pScriptEditor->setText(((KviScriptDataListViewItem *) it)->m_szBuffer.ptr());
		KviStr tmp = ((KviListViewItem *) it)->text(2);
		m_pEnabled->setChecked(tmp.hasData());
		m_pScriptEditor->setFocus();
		m_pParamsLabel->setText(((KviListViewItem *) it)->text(3));
	} else {
		m_pScriptEditor->setText("");
		m_pParamsLabel->setText("");
	}
	m_pCurItem = (KviScriptDataListViewItem *) it;
	m_pParamsLabel->setEnabled(it);
	m_pEnabled->setEnabled(it);
}

void KviEventEditor::commit()
{
	if( m_pCurItem )
		m_pCurItem->makeKey(g_szLastSelectedEvent);

	saveCurrentItem();
	g_pEventManager->cleanup();
	for( KviScriptDataListViewItem *it = (KviScriptDataListViewItem *) m_pListView->firstChild()
	   ; it
	   ; it = (KviScriptDataListViewItem *) it->nextSibling()
	) {
		KviStr tmp = it->text(1);
		if( tmp.hasData() ) {
			tmp = it->text(2);
			bool bEnabled = tmp.hasData();
			g_pEventManager->setHandler(it->text(0).ascii(), it->m_szBuffer.ptr(), bEnabled);
		}
	}
	g_pEventManager->disableEmptyHandlers();

	for( KviFrame *f = g_pApp->m_pFrameList->first(); f; f = g_pApp->m_pFrameList->next() )
		f->restartIdleTimer();
}

void KviEventEditor::resizeEvent(QResizeEvent *)
{
	m_pSplitter->setGeometry(0,0, width(), height());
}

#include "m_kvi_eventeditor.moc"
