// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptLabel"

#include <qpixmap.h>

#include "kvi_app.h"
#include "kvi_error.h"
#include "kvi_label.h"
#include "kvi_script_label.h"
#include "kvi_script_objectclassdefinition.h"

// TODO: Alignment in the label?
// TODO: An example with tags inside?

/*
	@class: label
	@short:
		A simple label widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:

		!fn: $text()
		Returns the text displayed on the label.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be displayed on the label.<br>

		!fn: $setFrameShape(&lt;style&gt;)
		none, box, panel, winpanel, hline, vline

		!fn: $setFrameShadow(&lt;type&gt;)
		plain, sunken, raised

		!fn: $setImage(\[image path\])
		Sets the picture to be displayed on the label.<br>
		The \[image path\] must be an absolute path to the image file.<br>
		If \[image path\] is an empty string, the current picture is unset.<br>


	@description:
		This class implements a simple label widget.<br>
		The label can display static text that can be set and retrieved by using
		the [classfnc:label]$setText[/classfnc]() and the [classfnc:label]$text[/classfnc]() function.<br>
		The label has different kind of frames available:<br>
		The frame appearance can be controlled by using the [classfnc:label]$setFrameShape[/classfnc]()
		and [classfnc:label]$setFrameShadow[/classfnc]() functions.<br>
		The constructor for this widget accepts an optional additional parameter:
		the text to be displayed.<br>
		The label text can handle a limited set of html tags to change the text
		appearance.<br>
		The supported tags include : &lt;font&gt; (face and color attributes), <br>
		&lt;h1&gt; (and the other headings), &lt;center&gt;, &lt;br&gt;, &lt;hr&gt;,
		&lt;li&gt;, &lt;pre&gt;, &lt;tt&gt;, &lt;table&gt;...<br>
		More tags may be supported, depending on the underlying Qt library.<br>
	@examples:
		OK. A complex example this time.<br>
		A simple minesweeper game done with labels.<br>
		You can improve this game in many ways. Have fun :)<br>
		<example>
		&nbsp;[cmd]if[/cmd]([fnc]$classDefined[/fnc](minebutton))[cmd]clearobjects[/cmd] minebutton
		&nbsp;[cmd]if[/cmd]([fnc]$classDefined[/fnc](minesweeper))[cmd]clearobjects[/cmd] minesweeper
		&nbsp;[cmd]class[/cmd](minebutton, [class]label[/class])
		&nbsp;{
		&nbsp;	[cmd:class]function[/cmd](constructor)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->[classfnc:label]$setFrameShape[/classfnc](winpanel)
		&nbsp;		[fnc]$this[/fnc]->[classfnc:label]$setFrameShadow[/classfnc](raised)
		&nbsp;		[fnc]$this[/fnc]->[classfnc:widget]$setMinimumWidth[/classfnc](20)
		&nbsp;		[fnc]$this[/fnc]->[classfnc:widget]$setMinimumHeight[/classfnc](20)
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](press)
		&nbsp;	{
		&nbsp;		[cmd]if[/cmd](("[fnc]$this[/fnc]->%pressed" == "") <a href="s_calc.kvihelp">&&</a> [fnc]$this[/fnc]->[classfnc:widget]$isEnabled[/classfnc]())
		&nbsp;		{
		&nbsp;			[fnc]$this[/fnc]->[classfnc:label]$setFrameShape[/classfnc](panel)
		&nbsp;			[fnc]$this[/fnc]->[classfnc:label]$setFrameShadow[/classfnc](sunken)
		&nbsp;			[fnc]$this[/fnc]->%pressed = 1
		&nbsp;			[cmd]if[/cmd]([fnc]$this[/fnc]->%nbombs == -1){
		&nbsp;				[fnc]$this[/fnc]->[classfnc:label]$setText[/classfnc]("@")
		&nbsp;				[fnc]$this[/fnc]->[classfnc:object]$parent[/classfnc]()->$bombPressed()
		&nbsp;			} [cmd:if]else[/cmd] {
		&nbsp;				[cmd]if[/cmd]([fnc]$this[/fnc]->%nbombs &gt; 0)[fnc]$this[/fnc]->[classfnc:label]$setText[/classfnc]([fnc]$this[/fnc]->%nbombs)
		&nbsp;				[cmd:if]else[/cmd] [fnc]$this[/fnc]->[classfnc:object]$parent[/classfnc]()->$emptyButtonPressed([fnc]$this[/fnc]->%x, [fnc]$this[/fnc]->%y)
		&nbsp;				[fnc]$this[/fnc]->[classfnc:object]$parent[/classfnc]()->$nonBombPressed()
		&nbsp;			}
		&nbsp;		}
		&nbsp;	}
		&nbsp;	[cmd:class]event[/cmd]([event]OnMouseButtonPress[/event])
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->$press()
		&nbsp;	}
		&nbsp;}
		&nbsp;[cmd]class[/cmd](minesweeper, [class]widget[/class])
		&nbsp;{
		&nbsp;	[cmd:class]function[/cmd](constructor)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->%layout = [fnc]$new[/fnc]([class]layout[/class], [fnc]$this[/fnc], my_layout)
		&nbsp;		[cmd]foreach[/cmd](%x, [fnc]$range[/fnc](0, 19))
		&nbsp;		{
		&nbsp;			[fnc]$this[/fnc]->%bombs[[fnc]$rand[/fnc](15)_[fnc]$rand[/fnc](15)] = bomb
		&nbsp;		}
		&nbsp;		[fnc]$this[/fnc]->%totalbombs = 0;
		&nbsp;		[cmd]foreach[/cmd](%x, [fnc]$range[/fnc](0, 14))
		&nbsp;		{
		&nbsp;			[cmd]foreach[/cmd](%y, [fnc]$range[/fnc](0, 14))
		&nbsp;			{
		&nbsp;				%l = [fnc]$new[/fnc](minebutton, [fnc]$this[/fnc], label%x\_%y)
		&nbsp;				[fnc]$this[/fnc]->%layout->[classfnc:layout]$add[/classfnc](%l, %x, %y)
		&nbsp;				[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[%x\_%y]" == "bomb"){
		&nbsp;					%l->%nbombs = -1
		&nbsp;					[fnc]$this[/fnc]->%totalbombs++
		&nbsp;				} [cmd:if]else[/cmd] {
		&nbsp;					%nbombs = 0;
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x - 1)_[fnc]$calc[/fnc](%y - 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x - 1)_%y]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x - 1)_[fnc]$calc[/fnc](%y + 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x + 1)_[fnc]$calc[/fnc](%y - 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x + 1)_%y]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x + 1)_[fnc]$calc[/fnc](%y + 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[%x\_[fnc]$calc[/fnc](%y - 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[%x\_[fnc]$calc[/fnc](%y + 1)]" == "bomb")%nbombs++
		&nbsp;					%l->%nbombs = %nbombs
		&nbsp;				}
		&nbsp;				[fnc]$this[/fnc]->%labels[%x\_%y] = %l
		&nbsp;				%l->%x = %x
		&nbsp;				%l->%y = %y
		&nbsp;			}
		&nbsp;		}
		&nbsp;		[fnc]$this[/fnc]->%status = [fnc]$new[/fnc]([class]label[/class], [fnc]$this[/fnc], status_label)
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setFrameShape[/classfnc](panel)
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setFrameShadow[/classfnc](sunken)
		&nbsp;		[fnc]$this[/fnc]->%layout->[classfnc:layout]$addMultiCell[/classfnc]([fnc]$this[/fnc]->%status, 15, 15, 0, 14)
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setText[/classfnc]("New game : total bombs : [fnc]$this[/fnc]->%totalbombs")
		&nbsp;		[fnc]$this[/fnc]->%buttonsLeft = [fnc]$calc[/fnc]((15 * 15) - [fnc]$this[/fnc]->%totalbombs)
		&nbsp;		[fnc]$this[/fnc]->[classfnc:widget]$show[/classfnc](1)
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](pressButton)
		&nbsp;	{
		&nbsp;		[cmd]if[/cmd]("$1" != "")$1->$press();
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](disableChildren)
		&nbsp;	{
		&nbsp;		[cmd]foreach[/cmd](%x, [fnc]$range[/fnc](0, 14))
		&nbsp;		{
		&nbsp;			[cmd]foreach[/cmd](%y, [fnc]$range[/fnc](0, 14))[fnc]$this[/fnc]->%labels[%x\_%y]->[classfnc:widget]$setEnabled[/classfnc](0)
		&nbsp;		}
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](bombPressed)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setText[/classfnc]("Boooooom");
		&nbsp;		[fnc]$this[/fnc]->$disableChildren()
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](nonBombPressed)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->%buttonsLeft--
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setText[/classfnc]("Total bombs : [fnc]$this[/fnc]->%totalbombs")
		&nbsp;		[cmd]if[/cmd]([fnc]$this[/fnc]->%buttonsLeft == 0){
		&nbsp;			[fnc]$this[/fnc]->%status->[classfnc:label]$setText[/classfnc]("Great job! All bombs discovered.");
		&nbsp;			[fnc]$this[/fnc]->$disableChildren()
		&nbsp;		}
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](emptyButtonPressed)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 - 1)_[fnc]$calc[/fnc]($2 - 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 - 1)_$2])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 - 1)_[fnc]$calc[/fnc]($2 + 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 + 1)_[fnc]$calc[/fnc]($2 - 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 + 1)_$2])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 + 1)_[fnc]$calc[/fnc]($2 + 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[$1_[fnc]$calc[/fnc]($2 - 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[$1_[fnc]$calc[/fnc]($2 + 1)])
		&nbsp;	}
		&nbsp;}
		&nbsp;%x = [fnc]$new[/fnc](minesweeper, [fnc]$root[/fnc], minesweeper)
		&nbsp;%x->[classfnc:widget]$show[/classfnc](1)
		</example>
	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * LABEL class
 */
void KviScriptLabel::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",        (scriptObjectFunction) &KviScriptLabel::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",           (scriptObjectFunction) &KviScriptLabel::builtinFunction_TEXT);
	d->addBuiltinFunction("setFrameShape",  (scriptObjectFunction) &KviScriptLabel::builtinFunction_SETFRAMESHAPE);
	d->addBuiltinFunction("setFrameShadow", (scriptObjectFunction) &KviScriptLabel::builtinFunction_SETFRAMESHADOW);
	d->addBuiltinFunction("setImage",       (scriptObjectFunction) &KviScriptLabel::builtinFunction_SETIMAGE);
}

KviScriptLabel::KviScriptLabel(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptLabel::~KviScriptLabel()
{
	// Nothing here
}

bool KviScriptLabel::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new KviLabel(((KviScriptWidget *) parent())->m_pWidget, name());
		}
	}
	if( !m_pWidget )
		m_pWidget = new KviLabel(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			((KviLabel *) m_pWidget)->setText(pS->ptr());
	}
	return true;
}

int KviScriptLabel::builtinFunction_TEXT(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((KviLabel *) m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptLabel::builtinFunction_SETTEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((KviLabel *) m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((KviLabel *) m_pWidget)->setText("");
	return KVI_ERROR_Success;
}

int KviScriptLabel::builtinFunction_SETIMAGE(QPtrList<KviStr> *params, KviStr &)
{
	if( params ) {
		KviStr *pS1 = params->first();
		if( pS1 ) {
			KviStr path;
			if( g_pApp->findImage(path, pS1->ptr()) ) {
				((KviLabel *) m_pWidget)->setAutoResize(true);
				((KviLabel *) m_pWidget)->setPixmap(QPixmap(path.ptr()));
			}
			return KVI_ERROR_Success;
		}
	}
	((KviLabel *) m_pWidget)->setAutoResize(false);
	((KviLabel *) m_pWidget)->setPixmap(QPixmap()); // Null pixmap
	return KVI_ERROR_Success;
}

int KviScriptLabel::builtinFunction_SETFRAMESHAPE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			if( kvi_strEqualCI(pS->ptr(), "none") )          ((KviLabel *) m_pWidget)->setFrameShape(QFrame::NoFrame);
			else if( kvi_strEqualCI(pS->ptr(), "box") )      ((KviLabel *) m_pWidget)->setFrameShape(QFrame::Box);
			else if( kvi_strEqualCI(pS->ptr(), "panel") )    ((KviLabel *) m_pWidget)->setFrameShape(QFrame::Panel);
			else if( kvi_strEqualCI(pS->ptr(), "winpanel") ) ((KviLabel *) m_pWidget)->setFrameShape(QFrame::WinPanel);
			else if( kvi_strEqualCI(pS->ptr(), "hline") )    ((KviLabel *) m_pWidget)->setFrameShape(QFrame::HLine);
			else if( kvi_strEqualCI(pS->ptr(), "vline") )    ((KviLabel *) m_pWidget)->setFrameShape(QFrame::VLine);
			else return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptLabel::builtinFunction_SETFRAMESHADOW(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			if( kvi_strEqualCI(pS->ptr(), "plain") )       ((KviLabel *) m_pWidget)->setFrameShadow(QFrame::Plain);
			else if( kvi_strEqualCI(pS->ptr(), "sunken") ) ((KviLabel *) m_pWidget)->setFrameShadow(QFrame::Sunken);
			else if( kvi_strEqualCI(pS->ptr(), "raised") ) ((KviLabel *) m_pWidget)->setFrameShadow(QFrame::Raised);
			else return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

#include "m_kvi_script_label.moc"
