// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptToolBar"

#include "kvi_error.h"
#include "kvi_frame.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_script_toolbar.h"
#include "kvi_toolbar.h"

/*
	@class: toolbar
	@short:
		A toolbar class
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $dock(&lt;docksite&gt;)
		Docks the toolbar to one of the sides of the main window.<br>
		Returns 1 if the operation was sucessfull or 0 if the toolbar is not
		a children or a KVIrc main window and thus cannot be docked successfully.<br>
		The &lt;docksite&gt; can be one of :<br>
		top, left, right, bottom, minimized.<br>

		!fn: $setDockEnabled(&lt;docksite&gt;, &lt;bEnabled&gt;)
		Enables or disables a specific docksite for this toolbar.<br>
		The &lt;docksite&gt; can be one of :<br>
		top, left, right, bottom, minimized.<br>
		&lt;bEnabled&gt; can be '1' to enable the docksite and '0' to disable it.<br>
		Returns 1 if the operation was sucessfull or 0 if the toolbar is not
		a children or a KVIrc main window and thus cannot be docked successfully.<br>

	@description:
		A toolbar widget.<br>
		It is used mainly with the [class]toolbutton[/class] objects.<br>
		If the parent of the toolbar is a non-widget object ([fnc]$root[/fnc] for example),
		the toolbar will be owned by the main KVIrc window.<br>
		In that case, it will be possible to dock it to one of the four sides
		of the window and drag it around.<br>

	@examples:

	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		class [class]toolbutton[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * TOOLBAR class
 */
void KviScriptToolBar::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("dock",           (scriptObjectFunction) &KviScriptToolBar::builtinFunction_DOCK);
	d->addBuiltinFunction("setDockEnabled", (scriptObjectFunction) &KviScriptToolBar::builtinFunction_SETDOCKENABLED);
}

KviScriptToolBar::KviScriptToolBar(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptToolBar::~KviScriptToolBar()
{
	// Nothing here
}

bool KviScriptToolBar::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new KviToolBar(
				name(), controller()->mainFrame(), ((KviScriptWidget *) parent())->m_pWidget, Qt::DockTop, false, name()
			);
		}
	}
	if( !m_pWidget )
		m_pWidget = new KviToolBar(name(), controller()->mainFrame(), controller()->mainFrame(), Qt::DockTop, false, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	return true;
}

Qt::Dock KviScriptToolBar::getDockId(const char *str)
{
	if( !str ) return DockMinimized;

	if( kvi_strEqualCI(str, "top") )    return DockTop;
	if( kvi_strEqualCI(str, "bottom") ) return DockBottom;
	if( kvi_strEqualCI(str, "left") )   return DockLeft;
	if( kvi_strEqualCI(str, "right") )  return DockRight;
	return DockMinimized;
}

int KviScriptToolBar::builtinFunction_DOCK(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( parent()->inherits("KviScriptWidget") ) {
		// Cannot dock
		buffer.append('0');
		return KVI_ERROR_Success;
	}
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			KviMainWindow *w = (KviMainWindow *) ((KviToolBar *) m_pWidget)->mainWindow();
			if( w ) {
				w->moveToolBar((KviToolBar *) m_pWidget, getDockId(pS->ptr()));
				buffer.append('1');
			} else {
				// Cannot dock
				buffer.append('0');
			}
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptToolBar::builtinFunction_SETDOCKENABLED(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( parent()->inherits("KviScriptWidget") ) {
		// Cannot dock
		buffer.append('0');
		return KVI_ERROR_Success;
	}
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			KviStr *pS2 = params->next();
			bool bEnable = true;
			if( pS2 ) {
				if( kvi_strEqualCI(pS2->ptr(), "0") )
					bEnable = false;
			}
			KviMainWindow *w = (KviMainWindow *) ((KviToolBar *) m_pWidget)->mainWindow();
			if( w ) {
				w->setDockEnabled((KviToolBar *) m_pWidget, getDockId(pS->ptr()), bEnable);
				buffer.append('1');
			} else {
				// Cannot dock
				buffer.append('0');
			}
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

#include "m_kvi_script_toolbar.moc"
