// =============================================================================
//
//      --- kvi_syntaxhighlighter.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Robin Verduijn <robin@debian.org>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviSyntaxHighlighter"

#define _KVI_SCRIPTEDITORWIDGET_CPP_

#include "kvi_config.h"
#include "kvi_options.h"
#include "kvi_syntaxhighlighter.h"

#define KVI_SYNTAX_STATE_NEWCOMMAND 0
#define KVI_SYNTAX_STATE_INCOMMAND  1
#define KVI_SYNTAX_STATE_INSTRING   2
#define KVI_SYNTAX_STATE_INCOMMENT  3

KviSyntaxHighlighter::KviSyntaxHighlighter(QTextEdit *textEdit)
	: QSyntaxHighlighter(textEdit)
{
	// Nothing here
}

KviSyntaxHighlighter::~KviSyntaxHighlighter()
{
	// Nothing here
}

int KviSyntaxHighlighter::highlightParagraph(const QString &text, int /* ignored */)
{
	const char *pC = text.ascii();
	QColor *clr = 0;

	m_textState = KVI_SYNTAX_STATE_NEWCOMMAND;
	int current = 0;
	int len     = 0;

	while( (*pC) && (*pC != '\n') ) {
		len = extractNextWord(pC);
		clr = getColor(pC, len);
		pC += len;
		if( clr )
			setFormat(current, len, *clr);
		current += len;
	}

	return KVI_SYNTAX_STATE_NEWCOMMAND;
}

int KviSyntaxHighlighter::extractNextWord(const char *s)
{
	int len = 0;
	// Spaces and tabs
	if( (*s == ' ') || (*s == '\t') ) {
		len++;
		s++;
		while( ( *s == ' ') || (*s == '\t') ) {
			len++;
			s++;
		}
		return len;
	}

	if( (*s == '#') && m_textState == KVI_SYNTAX_STATE_NEWCOMMAND ) {
		len++;
		s++;
		while( (*s) && (*s != ' ') && (*s != '\t') && (*s != ';') && (*s != '\n') ) {
			len++;
			s++;
		}
		return len;
	}

	if( m_textState == KVI_SYNTAX_STATE_INCOMMENT ) {
		while( (*s) && (*s != ' ') && (*s != '\t') && (*s != ';') && (*s != '\n') ) {
			len++;
			s++;
		}
		return len;
	}

	if( *s == '\\' ) {
		// Escape!
		len++;
		s++;
		if( *s && (*s != '\n') && (*s != ' ') && (*s != '\t') )
			len++;
		return len;
	}

	if( *s == '"' )
		return 1;

	if( isalnum(*s) || (*s == '_') || (*s == '.') ) {
		// Command or normal text
		s++;
		len++;
		while( *s && (isalnum(*s) || (*s == '_') || (*s == '.')) ) {
			s++;
			len++;
		}
		return len;
	}

	if( (*s == '%') || (*s == '$') ) {
		// Variable or identifier
		s++;
		len++;
		while( *s && (isalnum(*s) || (*s == '_') || (*s == '.') || (*s == ':')) ) {
			s++;
			len++;
		}
		return len;
	}

	if( (*s == '{') || (*s == '}') )
		return 1;

	if( (*s == '(') || (*s == ')') || (*s == '[') ||
		(*s == ']') || (*s == ',') || (*s == ':') ||
		(*s == ';') || (*s == '=') || (*s == '-') ||
		(*s == '<') || (*s == '>') || (*s == '&') ||
		(*s == '|') || (*s == '+') || (*s == '*') ||
		(*s == '/') || (*s == '!') || (*s == '^')
	) {
		len++;
		s++;
		while(
			(*s == '(') || (*s == ')') || (*s == '[') ||
			(*s == ']') || (*s == ',') || (*s == ':') ||
			(*s == ';') || (*s == '=') || (*s == '-') ||
			(*s == '<') || (*s == '>') || (*s == '&') ||
			(*s == '|') || (*s == '+') || (*s == '*') ||
			(*s == '/') || (*s == '!') || (*s == '^')
		) {
			len++;
			s++;
		}
		return len;
	}

	// Senseless string
	s++;
	len++;
	while( (*s) && (isalnum(*s)) ) {
		len++;
		s++;
	}
	return len;
}

QColor *KviSyntaxHighlighter::getColor(const char *s, int len)
{
	if( (*s == ' ') || (*s == '\t') )
		return 0;

	if( m_textState == KVI_SYNTAX_STATE_NEWCOMMAND ) {
		if( *s == '#' ) {
			m_textState = KVI_SYNTAX_STATE_INCOMMENT;
			return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_COMMENT];
		} else {
			m_textState = KVI_SYNTAX_STATE_INCOMMAND;
			if( isalnum(*s) || (*s == '_') || (*s == '.') )
				return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_COMMAND];
		}
	} else if( m_textState == KVI_SYNTAX_STATE_INCOMMENT ) {
		if( (*s != ';') && (*s != '\n') )
			return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_COMMENT];
	}

	if( *s == '"' ) {
		if( m_textState != KVI_SYNTAX_STATE_INSTRING )
			m_textState = KVI_SYNTAX_STATE_INSTRING;
		else
			m_textState = KVI_SYNTAX_STATE_INCOMMAND;
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_STRING];
	}

	if( m_textState == KVI_SYNTAX_STATE_INSTRING )
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_STRING];

	if( *s == ';' )
		m_textState = KVI_SYNTAX_STATE_NEWCOMMAND;

	if( *s == '\\' )
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_ESCAPE];

	if( (*s == '{') || (*s == '}') )
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_BRACE];
	if( *s == '%' ) {
		s++;
		if( (!*s) && (!(isalnum(*s))) && (*s != '_') && (*s != '.') )
			return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_OPERATOR];
		if( islower(*s) )
			return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_LOCVAR]; // Local variable
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_GLOBVAR];
	}
	if( *s == '$' ) {
		s++;
		len--;
		if( len && (*s == 't') || (*s == 'T') ) {
			s++;
			len--;
			if( len && (*s == 'h') || (*s == 'H') ) {
				s++;
				len--;
				if( kvi_strEqualCIN(s, "is", len) )
					return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_THIS];
			}
		}
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_IDENT];
	}

	if( (*s == '(') || (*s == ')') || (*s == '[') ||
		(*s == ']') || (*s == ',') || (*s == ':') ||
		(*s == ';') || (*s == '=') || (*s == '-') ||
		(*s == '<') || (*s == '>') || (*s == '&') ||
		(*s == '|') || (*s == '+') || (*s == '*') ||
		(*s == '/') || (*s == '!') || (*s == '^')
	) {
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_OPERATOR];
	}
	return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_TEXT];
}
