// =============================================================================
//
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviEditorWindow"

#define __KVIRC_PLUGIN__

#include "kvi_accel.h"
#include "kvi_editor_icon.h"
#include "kvi_editor_window.h"
#include "kvi_filedialog.h"
#include "kvi_locale.h"
#include "kvi_menubar.h"
#include "kvi_popupmenu.h"
#include "kvi_simple_editor.h"
#include "kvirc_plugin.h"

static QPixmap *g_pIcon;
void *g_handle = 0;

KviEditorWindow::KviEditorWindow(KviFrame *lpFrm)
	: KviWindow("Editor: Untitled", KVI_WND_TYPE_PLUGIN, lpFrm)
{
	m_pMenuBar = new KviMenuBar(this);
	m_pEditor  = new KviSimpleEditor(this);

	connect(m_pEditor,
		SIGNAL(fileNameChanged(KviSimpleEditor *, const QString &)), this,
		  SLOT(fileNameChanged(KviSimpleEditor *, const QString &))
	);

	KviPopupMenu *file = new KviPopupMenu(m_pMenuBar);
	file->insertItem("&New",  this, SLOT(newFile()),  QAccel::stringToKey(_i18n_("Ctrl+N")));
	file->insertItem("&Open", this, SLOT(openFile()), QAccel::stringToKey(_i18n_("Ctrl+O")));
	file->insertSeparator();
	file->insertItem("&Save",       m_pEditor, SLOT(saveFile()),   QAccel::stringToKey(_i18n_("Ctrl+S")));
	file->insertItem("Save &As...", m_pEditor, SLOT(saveFileAs()), QAccel::stringToKey(_i18n_("Ctrl+A")));
	file->insertSeparator();
	file->insertItem("&Close", this, SLOT(close()), QAccel::stringToKey(_i18n_("Ctrl+W")));

	KviPopupMenu *edit = new KviPopupMenu(m_pMenuBar);
	edit->insertItem("Toggle &Find Widget", m_pEditor, SLOT(toggleFindWidget()), QAccel::stringToKey(_i18n_("Ctrl+F")));
	edit->insertItem("&Switch Mode", m_pEditor, SLOT(switchMode()));

	m_pMenuBar->insertItem("&File", file);
	m_pMenuBar->insertItem("&Edit", edit);
	m_pMenuBar->setBackgroundMode(KviMenuBar::PaletteBackground);

	setFocusHandler(m_pEditor);
	m_pEditor->setFocus();
}

KviEditorWindow::~KviEditorWindow()
{
	// Nothing here
}

QPixmap *KviEditorWindow::myIconPtr()
{
	return g_pIcon;
}

void KviEditorWindow::fileNameChanged(KviSimpleEditor *, const QString &filename)
{
	KviStr tmp(KviStr::Format, "Editor: %s", filename.ascii());
	setWindowCaption(tmp.ptr());
}

void KviEditorWindow::resizeEvent(QResizeEvent *)
{
	int hght = m_pMenuBar->heightForWidth(width());
	m_pMenuBar->setGeometry(0, 0, width(), hght);
	m_pEditor->setGeometry(0, hght, width(), height() - hght);
}

void KviEditorWindow::closeEvent(QCloseEvent *e)
{
	e->ignore();
	if( m_pEditor->isModified() ) {
		if( !m_pEditor->closeFile() )
			return;
	}
	KviWindow::closeEvent(e);
}

void KviEditorWindow::openFile(const QString &filename)
{
	m_pEditor->loadFile(filename);
}

void KviEditorWindow::newFile()
{
	KviEditorWindow *wnd = new KviEditorWindow(m_pFrm);
	kvirc_plugin_add_window(g_handle, m_pFrm, wnd);
}

void KviEditorWindow::openFile()
{
	QString tmp = KviFileDialog::getOpenFileName();
	if( !tmp.isEmpty() ) {
		if( (m_pEditor->fileName() == "") && !m_pEditor->isModified() ) {
			// Open in this window
			openFile(tmp);
		} else {
			// Open in another window
			KviEditorWindow *wnd = new KviEditorWindow(m_pFrm);
			kvirc_plugin_add_window(g_handle, m_pFrm, wnd);
			wnd->openFile(tmp);
			wnd->setFocus();
		}
	}
}

bool editor_plugin_command_edit(KviPluginCommandStruct *cmd)
{
	KviEditorWindow *wnd = new KviEditorWindow(cmd->frame);
	kvirc_plugin_add_window(cmd->handle, cmd->frame, wnd);
	QString tmp = kvirc_plugin_param(cmd, 1);
	if( !tmp.isEmpty() )
		wnd->openFile(tmp.utf8());
	return true;
}

bool editor_plugin_init(KviPluginCommandStruct *cmd)
{
	g_pIcon = new QPixmap(editor_icon_xpm);
	kvirc_plugin_register_command(cmd->handle, "EDIT", editor_plugin_command_edit);
	g_handle = cmd->handle; // Need it outside commands
	return true;
}

void editor_plugin_cleanup()
{
	if( g_pIcon ) {
		delete g_pIcon;
		g_pIcon = 0;
	}
}

/**
	@document: doc_plugin_editor.kvihelp
	@title: The text editor plugin
		Adds a simple text editor to KVIrc.<br>
		You can open files by using the command<br>
		<b>/edit [filename]</b><br>
		Filename must be an absolute file path.<br>
		If no filename is specified, a new file will be created.<br>
		The editor has three working modes:<br>
		Normal, Cpp (for C source code, with syntax highlighting) and HTML
		(with light syntax highlighting).<br>
		The plugin window has a lot of keyboard shortcuts:<br>
		Here is the list:<br>
		CTRL+C: copy the selected text to clipboard.<br>
		CTRL+V: paste the clipboard contents to the current position in the file.<br>
		CTRL+X: copy & then remove the selection.<br>
		CTRL+M: switch mode.<br>
		CTRL+I: indent one level (only in Cpp mode).<br>
		CTRL+U: unindent one level (only in Cpp mode).<br>
		CTRL+O: comment out (only in Cpp mode).<br>
		CTRL+SHIFT+O: comment out (alternate) (only in Cpp mode).<br>
		CTRL+P: remove comment (only in Cpp mode).<br>
		CTRL+R: record/stop recording keystrokes.<br>
		CTRL+T: replay keystrokes.<br>
		CTRL+S: save file.<br>
		CTRL+A: save file as...<br>
		CTRL+F: show/hide the find widget.<br>
		CTRL+N: find next.<br>
		CTRL+H: find previous.<br>
		CTRL+B: find next, treating the string to find as a regular expression.<br>
		CTRL+G: find previous (regular expression).<br>
		CTRL+J: replace the current selection.<br>
		CTRL+K: replace all the occurrences in the file.<br>
		CTRL+L: replace all the occurrences in the file (regular expression).<br>
		CTRL+Q: replace and find next.<br>
		CTRL+W: replace and find next (regular expression).<br>
		CTRL+E: replace all in selection.<br>
		CTRL+D: replace all in selection (regular expression).<br>
		<br>
		You can use the editor as the default "player"
		for text files (see <a href="play.kvihelp">PLAY</a>).<br>
		In order to do so, put "EDIT <a href="s_execfilename.kvihelp">$execFileName</a>" in the
		"Run script" field (misc options dialog, mimetypes tab)
		for the mimetypes that you want to play with the editor.<br>
		Note:<br>
		If you unload the plugin, the /EDITOR command will no longer work.<br>
		Here is a simple workaround "commandline field" script for this problem:<br>
		Version 1: Use the plugin if it is loaded.<br>
		<b><a href="if.kvihelp">if</a>(<a href="s_pluginLoaded.kvihelp">$pluginLoaded</a>(editor))edit <a href="s_execfilename.kvihelp">$ExecFileName</a>; else <a href="exec.kvihelp">exec</a> -q myviewer <a href="s_execfilename.kvihelp">$ExecFileName</a></b><br>
		Workaround 2: Use the plugin always; if it is not loaded, load it.<br>
		<b><a href="if.kvihelp">if</a>(!<a href="s_pluginloaded.kvihelp">$pluginLoaded</a>(editor))<a href="plugin.kvihelp">PLUGIN</a> load libkvieditor.so; edit <a href="s_execfilename.kvihelp">$ExecFilename</a></b><br>
		Note:<br>
		I am currently using this widget (inserted into a separate application) in Cpp mode to develop
		KVIrc. :)
*/

void editor_plugin_help()
{
	kvirc_plugin_show_help_page("doc_plugin_editor.kvihelp");
}

KviPlugin kvirc_plugin =
{
	"Editor",
	"A simple text editor",
	"0.1",
	"Szymon Stefanek <stefanek@tin.it>",
	"<html>"
		"Exports the <b>/EDIT</b> command."
		"</html>",
	editor_plugin_init,
	editor_plugin_cleanup,
	0,
	editor_plugin_help
};

#include "m_kvi_editor_window.moc"
