#ifndef _KVI_IRC_USERCHANLIST_H_INCLUDED_
#define _KVI_IRC_USERCHANLIST_H_INCLUDED_

// =============================================================================
//
//      --- kvi_irc_userchanlist.h ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include "kvi_irc_userlist.h"

/**
 * Client of the global list.<br>
 * This keeps only pointers to data and flags.<br>
 * NOTE: I am sure that one day I will completely forget how it works :)
 */
typedef struct _KviIrcUserChanData
{
	// THIS IS (should be) A READ-ONLY STRUCTURE!
	// NEVER CHANGE DIRECTLY
	// (except for extFlag that is used in the KviUserListBox to keep track of selections)
	_KviIrcUserChanData *prev;
	KviIrcUserListNode  *pNode; // Pointer to the global data structure
	char qFlag;		            // UnrealIRCd owner -Trisk
	char oFlag;                 // Is user +o on this channel?
	char hFlag;
	char vFlag;                 // Is user +v on this channel?
	char uFlag;		            // Userop... -Trisk
	char extFlag;               // The only R/W flag in this structure
	_KviIrcUserChanData *next;
} KviIrcUserChanData;

class KviIrcUserChanList
{
public:
	KviIrcUserChanList(KviIrcUserList *pList);
	/**
	 * Makes everyone part from this channel
	 */
	~KviIrcUserChanList();
private:
	KviIrcUserList *m_pGlobalList;
	KviIrcUserChanData *m_pHead;
	KviIrcUserChanData *m_pTail;
	KviIrcUserChanData *m_pCur; // Same comments as for the global class
	int m_iCount;
	int m_iOwnerCount;
	int m_iOpCount;
	int m_iHalfOpCount;
	int m_iVoiceCount;
	int m_iUserOpCount;
public:
	// Unsafe iterators
	// CAUTION!!
	// firstNode(), lastNode(), first() or next() MUST be called before
	// any iteration attempt with nextNode(), next(), prevNode() and prev().
	KviIrcUser *firstUser();
	KviIrcUser *lastUser();
	KviIrcUser *nextUser();
	KviIrcUser *prevUser();

	KviIrcUserChanData *firstData();
	KviIrcUserChanData *lastData();
	KviIrcUserChanData *nextData();
	KviIrcUserChanData *prevData();

	void clearList();
	/**
	 * A user joins... adds an entry here and in the global list.
	 */
	KviIrcUserChanData *join(
		const KviIrcUser &user, char bOp = 0, char bVoice = 0, char bHalfOp = 0, char bUserOp = 0, char bOwner = 0
	);

	/**
	 * A user parts; removes the entry here and in the global list.
	 */
	bool part(const char *nick);

	/**
	 * A user parts; removes the entry here and in the global list.<br>
	 * Returns false if the user was not here.
	 */
	bool part(const KviIrcUser &user);

	bool select(const char *nick);
	bool select(const KviIrcUser &user);

	bool deselect(const char *nick);
	bool deselect(const KviIrcUser &user);

	// Careful with that AXE Eugene!
	// Never change the KviIrcUserChanData structure directly
	// (I mean from outside this class).
	// Or all this stuff will coredump sooner or later.
	// There are surely safer ways to do the same thing,
	// though they are also SURELY slower (and a bit more complex inside?)
	KviIrcUserChanData *findData(const KviIrcUser &user);
	KviIrcUserChanData *findData(const char *nick);

	/**
	 * Finds a user in this list.
	 */
	KviIrcUser *findUser(const KviIrcUser &user);
	KviIrcUser *findUser(const char *nick);
	int findUserPosition(const char *nick);

	/**
	 * Ops the user. This is the right way to do it.<br>
	 * Returns false if the user was not in the list.
	 */
	bool owner(const char *nick, char bOwner);
	bool owner(const KviIrcUser &user, char bOwner);

	bool op(const char *nick, char bOp);
	bool op(const KviIrcUser &user, char bOp);

	bool halfop(const char *nick, char bHalfOp);
	bool halfop(const KviIrcUser &user, char bHalfOp);

	bool voice(const char *nick, char bVoice);
	bool voice(const KviIrcUser &user, char bVoice);

	bool userop(const char *nick, char bUserOp);
	bool userop(const KviIrcUser &user, char bUserOp);

	bool isOwner(const char *nick);
	bool isOwner(const KviIrcUser &user);

	bool isOp(const char *nick);
	bool isOp(const KviIrcUser &user);

	bool isHalfOp(const char *nick);
	bool isHalfOp(const KviIrcUser &user);

	bool isVoice(const char *nick);
	bool isVoice(const KviIrcUser &user);

	bool isUserOp(const char *nick);
	bool isUserOp(const KviIrcUser &user);

	bool nickChange(const KviIrcUser &nicker, const char *newNick);

	// Internal user count
	int count();
	int ownerCount();
	int opCount();
	int halfopCount();
	int voiceCount();
	int useropCount();
protected:
	/**
	 * Insert data keeping the op-voice-alphabetic order.
	 */
	void insertData(KviIrcUserChanData *data);

	/**
	 * Insert an owner.
	 */
	void insertOwnerData(KviIrcUserChanData *data);

	/**
	 * Skips owners and inserts an op user.
	 */
	void insertOpData(KviIrcUserChanData *data);

	/**
	 * Skips ops and inserts a halfop user.
	 */
	void insertHalfOpData(KviIrcUserChanData *data);

	/**
	 * Skips ops and halfops and inserts a voiced user.
	 */
	void insertVoiceData(KviIrcUserChanData *data);

	/**
	 * Skips ops, halfops, and voiced users, and inserts a userop user.
	 */
	void insertUserOpData(KviIrcUserChanData *data);

	/**
	 * Skips ops, halfops, userops, and voiced users, and inserts a normal user.
	 */
	void insertNormalData(KviIrcUserChanData *data);

	/**
	 * Does not remove the user list entry from the global list
	 * (it must be done before)!<br>
	 * (deletes the data structure)
	 */
	void removeData(KviIrcUserChanData *data);

private:
	/**
	 * Used for internal reorganisation (op-deop list order changes)
	 */
	void removeDataNoDelete(KviIrcUserChanData *data);
};

#endif // _KVI_IRC_USERCHANLIST_H_INCLUDED_
