// =============================================================================
//
//      --- kvi_dirbrowser_thread.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDirBrowserThread"

#include <sys/stat.h>
#include <unistd.h>

#include "kvi_application.h"
#include "kvi_dirbrowser_thread.h"
#include "kvi_mimemanager.h"
#include "kvi_options.h"

KviDirectoryThread::KviDirectoryThread(KviDirBrowserJob *job)
{
	m_pJob = job;
	m_pDir = 0;
}

KviDirectoryThread::~KviDirectoryThread()
{
	if( m_pDir ) {
		closedir(m_pDir);
		m_pDir = 0;
	}
	if( m_pJob ) {
		if( m_pJob->entries ) {
			delete m_pJob->entries;
			m_pJob->entries = 0;
		}
		delete m_pJob;
		m_pJob = 0;
	}
}

void KviDirectoryThread::run()
{
	m_pDir = opendir(m_pJob->szDirPath.ptr());
	if( !m_pDir )
		return;
	m_pJob->entries = new QPtrList<KviDirBrowserEntry>;
	m_pJob->entries->setAutoDelete(true);

	dirent *file;
	while( (file = readdir(m_pDir)) && !m_bStopRequested ) {
		KviDirBrowserEntry *e = new KviDirBrowserEntry();
		e->szAbsFilePath = m_pJob->szDirPath.ptr();
		e->szAbsFilePath.ensureLastCharIs('/');
		e->szAbsFilePath.append(file->d_name);
		struct stat st;
		if( lstat(e->szAbsFilePath.ptr(), &st) != 0 ) {
			// Cannot stat the file
			delete e;
			continue;
		}
		e->szFileName = file->d_name;
		e->szDirPath  = m_pJob->szDirPath.ptr();
		if( kvi_strEqualCS(e->szFileName.ptr(), ".") && S_ISDIR(st.st_mode) ) {
			// Current dir inode
			delete e;
			continue;
		}
		if( !(m_pJob->bShowHidden) ) {
			if( e->szFileName.firstCharIs('.') ) {
				// Hidden file - do not show
				delete e;
				continue;
			}
		}
		e->uSize         = st.st_size;
		e->bIsDirectory  = S_ISDIR(st.st_mode);
		e->bIsReadable   = (
			getuid() == 0) || (S_IROTH & st.st_mode) ||
			((S_IRGRP & st.st_mode) && (getgid() == st.st_gid)) ||
			((S_IRUSR & st.st_mode) && (getuid() == st.st_uid)
		);
		e->bIsSymLink    = S_ISLNK(st.st_mode);
		e->bIsFile       = S_ISREG(st.st_mode);
		e->bIsSocket     = S_ISFIFO(st.st_mode) || S_ISSOCK(st.st_mode);
		e->bIsDevice     = S_ISBLK(st.st_mode)  || S_ISCHR(st.st_mode);
		if( e->bIsFile ) {
			KviMimeType *m   = g_pOptions->m_pMimeManager->findMatch(e->szAbsFilePath.ptr(), true);
			e->szMimeName    = m->mimeName;
			e->szCommandline = m->commandline;
			e->szIconPath    = m->iconPath;
		} else if( e->bIsSymLink ) {
			// Read the link
			char buffer[256];
			memset(buffer, '\0', 256);
			int len = readlink(e->szAbsFilePath.ptr(), buffer, 256);
			if( len > -1 ) {
				e->szMimeName.setStr(buffer, len);
				if( stat(e->szAbsFilePath.ptr(), &st) == 0 ) {
					e->bIsDirectory = S_ISDIR(st.st_mode);
				}
			}
		}
		// Sort
		if( m_pJob->bGroupByExtension ) {
			if( m_pJob->bSortBySize ) {
				// Group by extension then sort by size in each group.
				// Directories first, then files without extension, then extensions in alphabetic order
				// Inside each group of extensions sort by size
				int idx = e->szFileName.findLastIdx('.');
				KviStr tmp(KviStr::Format, "%u", e->uSize);
				if( idx >= 1 ) { // No hidden files
					// Has extension
					if( e->bIsDirectory )
						e->szKey.sprintf("1 %s %s", e->szFileName.ptr() + idx, file->d_name);
					else
						e->szKey.sprintf("3 %s %d %u", e->szFileName.ptr() + idx, tmp.len(), e->uSize);
				} else {
					// No extension
					if( e->bIsDirectory )
						e->szKey.sprintf("1 _ %s", file->d_name);
					else
						e->szKey.sprintf("2 _ %d %u", tmp.len(), e->uSize);
				}
			} else {
				int idx = e->szFileName.findLastIdx('.');
				if( idx >= 1 ) { // No hidden files
					e->szKey.sprintf("%s %s %s", e->bIsDirectory ? "1" : "3", e->szFileName.ptr() + idx, file->d_name);
				} else {
					e->szKey.sprintf("%s _ %s", e->bIsDirectory ? "1" : "2", file->d_name);
				}
			}
		} else {
			if( m_pJob->bSortBySize ) {
				// Sort by size: directories first, then by size string length then by size string
				KviStr tmp(KviStr::Format, "%u", e->uSize);
				if( e->bIsDirectory )
					e->szKey.sprintf("1 %s", file->d_name);
				else
					e->szKey.sprintf("2 %d %u", tmp.len(), e->uSize);
			} else {
				// Sort by name: directories first, then by name
				e->szKey.sprintf("%s %s", e->bIsDirectory ? "1" : "2", file->d_name);
			}
		}
		if( m_pJob->bCaseInsensitive )
			e->szKey.toLower();
		m_pJob->entries->append(e);
	}
	QCustomEvent *ev = new QCustomEvent(QEvent::User, m_pJob);
	KviApplication::postEvent(m_pJob->parent, ev);
}
