// =============================================================================
//
//      --- kvi_event.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviEvent"

#include <qfile.h>

#include "kvi_defines.h"
#include "kvi_event.h"
#include "kvi_locale.h"
#include "kvi_settings.h"

// TODO: Protect events against self-modification
// TODO: OnDCCSendFailure
// TODO: Get rid of OnMePart, OnMeBan etc...
// TODO: OnShutdown seems not to be executed
// TODO: OnWindowCreated
// TODO: OnWindowDestroyed
// TODO: OnQueryCreated

KviEventDescriptor KviEventManager::m_eventDescriptor[KVI_NUM_EVENTS] =
{
	{
		/*
			@event: OnStartup
			@short:
				New KVIrc server window opened
			@parameters:
				$1 = bApplicationStartup<br>
				$2- = KVIrc version string
			@window:
				Console
			@description:
				This event is 'fired' when a new KVIrc server window is opened.<br>
				You can use it to setup your script, set internal variables or simply
				echo a 'script logo' to the console.<br>
				If you call <a href="halt.kvihelp">halt</a> in this event you will prevent KVIrc from printing
				the version string on the console.
				The first parameter contains "1" if this is the only server
				window present (so it is the application startup at all), "0" otherwise.<br>
				You can use this event to load your script configuration.<br>
		*/
		"OnStartup", "$1 version"
	},
	{
		/*
			@event: OnShutdown
			@short:
				KVIrc server window closed
			@parameters:
				None
			@window:
				Console
			@description:
				This event is 'fired' when a KVIrc server window is closed.<br>
				Useful to save your script configuration.<br>
				Calling <a href="halt.kvihelp">halt</a> in this event has no particular meaning.
		*/
		"OnShutdown", "none"
	},
	{
		/*
			@event: OnServerPing
			@short:
				Received a PING from server
			@parameters:
				$1 = server name<br>
				$2 = ping parameter
			@window:
				Console
			@description:
				Event executed when a PING from server has been received.<br>
				KVIrc first replies to the ping (so you cannot stop the reply),
				and then executes the event.<br>
				By calling <a href="halt.kvihelp">halt</a> in the event handler you will stop
				the PING notification output in the console (assuming that it has been enabled).
		*/
		"OnServerPing", "$1 server, $2 parameter"
	},
	{
		/*
			@event: OnConnect
			@short:
				Connected to an IRC server
			@parameters:
				$1 = server hostname<br>
				$2 = server ip<br>
				$3 = port<br>
				$4 = network name
			@window:
				Console
			@description:
				This event is 'fired' when an irc connection has been estabilished.<br>
				No data has been sent to the server yet, so all the login operations
				are still to be done (even your nickname is still undefined!).<br>
				For this reason you should not attempt to send anything to the server
				unless you really know what you are doing.<br>
				Calling <a href="halp.kvihelp">halt</a> in this event will stop the 'Connected to ...' output in the console.
		*/
		"OnConnect", "$1 server host, $2 server ip, $3 port, $4 network"
	},
	{
		/*
			@event: OnDisconnect
			@short:
				Disconnected from an IRC server
			@parameters:
				$1 = server hostname<br>
				$2 = server ip<br>
				$3 = port<br>
				$4 = network name
			@window:
				Console
			@description:
				This event is 'fired' when an irc connection has been terminated.<br>
				All the channel windows have been closed, and KVIrc is in the middle of the "general system reset".<br>
				Obviously you should not attempt to send anything to the server : you are not connected.<br>
				You should use this event to reset your script state and stop any connection related asynchronous work
				that might be still running.<br>
				Calling <a href="halt.kvihelp">halt</a> in this event will avoid the 'Connection terminated ...' output in the console.
		*/
		"OnDisconnect", "$1 server host, $2 server ip, $3 port, $4 network"
	},
	{
		/*
			@event: OnIrc
			@short:
				Login operations completed
			@parameters:
				$1 = nickname
			@window:
				Console
			@description:
				This event is 'fired' when all the login operations have been completed.<br>
				The nickname in $1 is the one that the server accepted (that may be also the secondary one or even auto-modified by KVIrc).
				From this moment you are "really" on Irc.<br>
				Calling <a href="halt.kvihelp">halt</a> in this event will stop the 'Login operations complete : happy IRCing...' output in the console.<br>
				(In fact I wanted to call this event Event OnHappyIrcing).
		*/
		"OnIrc", "$1 nickname"
	},
	{
		/*
			@event: OnJoin
			@short:
				A user joined a channel
			@parameters:
				$1 = channel name<br>
				$2 = joiner nickname<br>
				$3 = joiner username<br>
				$4 = joiner host
			@window:
				Joined channel
			@description:
				Event executed when a user (that is NOT you) joins a channel.<br>
				The special case in that YOU join a channel is handled by the event <a href="onMeJoin.kvihelp">OnMeJoin</a>.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you will avoid the '&lt;nick&gt; has joined &lt;channel&gt;...' output.
			@examples:
				Customize the join output:<br>
				<example>
					<a href="echo.kvihelp">echo</a> -i <a href="s_icon.kvihelp">$icon(join)</a> Signon : $2!$3@$4
					<a href="halt.kvihelp">halt</a>
				</example>
				Auto-op a specific user (by leumaS &lt;samuel@blomqvist.nu&gt;):<br>
				<example>
					<a href="if.kvihelp">if</a>("username, host.domain.top, #channel" == "$3, $4, $1") <a href="op.kvihelp">op</a> $2
				</example>
				Auto-op all a specific user that has a dialup account (by leumaS &lt;samuel@blomqvist.nu&gt;):<br>
				<example>
					<a href="if.kvihelp">if</a>("user, *!*@*.domani.top, #channel" == "$3, <a href="s_mask.kvihelp">$Mask</a>(10, $2), $1") <a href="op.kvihelp">op</a> $2
				</example>
		*/
		"OnJoin", "$1 channel name, $2 joiner nick, $3 joiner username, $4 joiner host"
	},
	{
		/*
			@event: OnChannelSync
			@short:
				Channel synchronisation completed
			@parameters:
				$1 = channel name<br>
				$2 = sync time (secs.msecs)
			@window:
				Channel
			@description:
				Event executed when a channel has been completely synchronized after a join.<br>
				When you join a channel KVIrc has to request some information about it:<br>
				-Channel mode<br>
				-Topic<br>
				-List of users (and optionally the masks of all users)<br>
				-Ban list<br>
				This event fires when all of these information have been received.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you will avoid the "Channel synchronized in ... secs" output.<br>
				<docsubtitle>Limitations</docsubtitle>
				The exception list state is still "undefined".<br>
				It is because we cannot know if the server supports it.<br>
				So you can try to check it, but be aware that it may be not in sync.<br>
				<docsubtitle>More details</docsubtitle>
				When a RPL_ENDOFNAMES is received, and there is only one user on the channel
				the channel is assumed to be "new".<br>
				On "new" channels the topic is assumed to be "not set": KVIrc will NOT wait for a RPL_TOPIC from server.<br>
				There is an exception for networks where a Channel Service is available
				and the channel is Registered.<br>
				In this case the topic may be unset in this event and set later by ChanServ.
		*/
		"OnChannelSync", "$1 channel name, $2 synctime"
	},
	{
		/*
			@event: OnMeJoin
			@short:
				Joined a new channel
			@parameters:
				$1 = channel name
			@window:
				Joined channel
			@description:
				Event executed when YOU join a channel.<br>
				This is a special case for the event <a href="onJoin.kvihelp">OnJoin</a>.<br>
				Please note that this event is called just after the channel window has been created (in response to the
				server JOIN message).<br>
				If you try to check other users in the channel, you will strangely discover yourself ALONE!.<br>
				This is because the channel has not been synchronized yet.<br>
				For the same reason the topic and the channel mode is still undefined.<br>
				There is a special event that signals you channel synchronisation : <a href="onChannelSync.kvihelp">OnChannelSync</a>.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you will avoid the '&lt;nick&gt; has joined &lt;channel&gt;...' output.
		*/
		"OnMeJoin", "$1 channel name"
	},
	{
		/*
			@event: OnPart
			@short:
				A user has left a channel
			@parameters:
				$1 = channel name<br>
				$2 = parter nickname<br>
				$3 = parter username<br>
				$4 = parter host<br>
				$5-= part message
			@window:
				Channel
			@description:
				Event executed when a user (that is NOT you) leaves a channel.<br>
				The special case in that YOU leave a channel is handled by the event <a href="onMePart.kvihelp">OnMePart</a>.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you will avoid the '&lt;nick&gt; has left &lt;channel&gt;...' output.
				This event is called when the user is STILL on the channel, and will leave
				just after the event handler has returned.<br>
			@examples:
				Customize the part output:<br>
				<example>
					<a href="echo.kvihelp">echo</a> -i <a href="s_icon.kvihelp">$icon(part)</a> Signoff : $2!$3@$4 [$5-]
					<a href="halt.kvihelp">halt</a>
				</example>
		*/
		"OnPart", "$1 channel name, $2 nick, $3 username, $4 host, $5- part message"
	},
	{
		/*
			@event: OnMePart
			@short:
				Left a channel
			@parameters:
				$1 = channel name<br>
				$2-= part message
			@window:
				Channel
			@description:
				Event executed when YOU leave a channel.<br>
				This is a special case for the event <a href="onPart.kvihelp">OnPart</a>.<br>
				Called immediately before the channel window is being destroyed.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you will avoid the 'You have left &lt;channel&gt;...' output in the console.
		*/
		"OnMePart", "$1 channel name, $2- part message"
	},
	{
		/*
			@event: OnKick
			@short:
				A user was kicked from a channel
			@parameters:
				$1 = channel name<br>
				$2 = kicked nick<br>
				$3 = kicker nick<br>
				$4 = kicker username<br>
				$5 = kicker host<br>
				$6-= kick reason
			@window:
				Channel
			@description:
				Event executed when a user (that is NOT you) has been kicked from a channel.<br>
				The special case in that YOU have been kicked is handled by the event <a href="onMeKick.kvihelp">OnMeKick</a>.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you will avoid the '&lt;nick&gt; has been kicked from &lt;channel&gt;...' output.
		*/
		"OnKick", "$1 channel name, $2 kicked nick, $3 kicker nick, $4 kicker username, $5 kicker host, $6- kick reason"
	},
	{
		/*
			@event: OnMeKick
			@short:
				Kicked from a channel
			@parameters:
				$1 = channel name<br>
				$2 = kicker nick<br>
				$3 = kicker username<br>
				$4 = kicker host<br>
				$5-= kick reason
			@window:
				Console
			@description:
				Event executed when YOU have been kicked from a channel.<br>
				This is a special case for the event <a href="onKick.kvihelp">OnKick</a>.<br>
				Called just after the channel window has been closed (in response to the
				server KICK message).<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you will avoid the 'You have been kicked from &lt;channel&gt; by...' output in the console.
		*/
		"OnMeKick", "$1 channel name, $2 kicker nick, $3 kicker username, $4 kicker host, $5- kick reason"
	},
	{
		/*
			@event: OnTopic
			@short:
				The topic of a channel was changed
			@parameters:
				$1 = channel name<br>
				$2 = source nickname<br>
				$3 = source mask<br>
				$4-= new topic
			@window:
				Channel
			@description:
				Event executed when a user (or server) sets a new topic for a channel.<br>
				By calling <a href="halp.kvihelp">halt</a> in this event you will avoid the '&lt;nick&gt; changes topic to...' output.
		*/
		"OnTopic", "$1 channel name, $2 source nick, $3 source mask, $4- new topic"
	},
	{
		/*
			@event: OnInvite
			@short:
				Received an INVITE message
			@parameters:
				$1 = inviter nick<br>
				$2 = inviter mask<br>
				$3 = channel<br>
				$4 = autojoin_is_enabled?
			@window:
				Console
			@description:
				Event executed when a user invites you to join a channel.<br>
				If you call <a href="halp.kvihelp">halt</a> in this event, you will stop the "user foo invites you to join channel bar" output
				and the autojoin-in-invite feature.<br>
				$4 tells you if the feature is enabled by the user in the options dialog, so you can implement it
				by yourself.
			@examples:
			<example>
				<a href="echo.kvihelp">echo</a> -i <a href="s_icon.kvihelp">$icon(invite)</a> User $1 invites you to join channel $3
				<a href="if.kvihelp">if</a>($4)<a href="join.kvihelp">join</a> $3
				else echo The autojoin feature is disabled
				<a href="halt.kvihelp">halt</a>
			</example>
		*/
		"OnInvite", "$1 inviter nick, $2 inviter mask, $3 channel, $4 autojoin_enabled?"
	},
	{ /* KviEvent_OnCTCP : halt stops the "CTCP ... REQUEST ..." output and the reply, bound as output */
		"OnCTCP", "$1 source nick, $2 source mask, $3 target, $4 ctcp type, $5 reply_disabled?, $6- data"
	},
	{ /* KviEvent_OnCTCPReply : halt stops the "CTCP REPLY ..." output, bound as output*/
		"OnCTCPReply", "$1 source nick, $2 source mask, $3 target, $4 ctcp type, $5- data"
	},
	{ /* KviEvent_OnCTCPFlood : halt stops the "CTCP FLOOD ..." output, bound to ActiveWindow or console, depending on options */
		"OnCTCPFlood", "$1 source nick, $2 source mask, $3 target, $4 ctcp type, $5- data"
	},
	{ /* KviEvent_OnAction : halt stops the "user ..." output, bound as output */
		"OnAction", "$1 source nick, $2 source mask, $3 target, $4 open_query_on_privmsg, $5- data"
	},

	{ /* KviEvent_OnOp : halt stops the "user sets mode +o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnOp", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnDeop : halt stops the "user sets mode -o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnDeop", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnMeOp : halt stops the "user sets mode +o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeOp", "$1 channel name, $2 source nick, $3 source mask"
	},
	{ /* KviEvent_OnMeDeop : halt stops the "user sets mode -o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeDeop", "$1 channel name, $2 source nick, $3 source mask"
	},

	{ /* KviEvent_OnVoice : halt stops the "user sets mode +v user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnVoice", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnDevoice : halt stops the "user sets mode -v user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnDevoice", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnMeVoice : halt stops the "user sets mode +v user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeVoice", "$1 channel name, $2 source nick, $3 source mask"
	},
	{ /* KviEvent_OnMeDevoice : halt stops the "user sets mode -v user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeDevoice", "$1 channel name, $2 source nick, $3 source mask"
	},

	{ /* KviEvent_OnBan : halt stops the "user sets mode +b mask" output, bound to chan or console, dep. on options */
		"OnBan", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask[, $5 set_by, $6 set_at]"
	},
	{ /* KviEvent_OnUnban : halt stops the "user sets mode -b mask" output, bound to chan or console, dep. on options */
		"OnUnban", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask"
	},
	{ /* KviEvent_OnMeBan : halt stops the "user sets mode +b mask" output, bound to chan or console, dep. on options */
		"OnMeBan", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask[, $5 set_by, $6 set_at]"
	},
	{ /* KviEvent_OnMeUnban : halt stops the "user sets mode -b mask" output, bound to chan or console, dep. on options */
		"OnMeUnban", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask"
	},

	{ /* KviEvent_OnBanException : halt stops the "user sets mode +e mask" output, bound to chan or console, dep. on options */
		"OnBanException", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask[, $5 set_by, $6 set_at]"
	},
	{ /* KviEvent_OnBanExceptionRemove : halt stops the "user sets mode -e mask" output, bound to chan or console, dep. on options */
		"OnBanExceptionRemove", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask"
	},
	{ /* KviEvent_OnMeBanException : halt stops the "user sets mode +e mask" output, bound to chan or console, dep. on options */
		"OnMeBanException", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask[, $5 set_by, $6 set_at]"
	},
	{ /* KviEvent_OnMeBanExceptionRemove : halt stops the "user sets mode -e mask" output, bound to chan or console, dep. on options */
		"OnMeBanExceptionRemove", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask"
	},

	{
		/*
			@event: OnChannelMode
			@short:
				The channel mode changed
			@parameters:
				$1 = channel name<br>
				$2 = source nick/host<br>
				$3 = source mask (user@host)<br>
				$4 = mode flag(s)<br>
				$5 = parameter
			@window:
				Channel
			@description:
				Executed when a channel mode is changed by a user or a server.<br>
				In that case $4 contains a single mode flag (like +s or -t) and $5
				contains the (optional) parameter.<br>
				The event is firead also when you join a new channel and the
				server sends you the current channel mode.<br>
				In that case $2 and $3 contain the same string (the server name),
				and $4 contains the entire mode string (like +smtn) or an empty string
				if the channel is modeless.<br>
				If you call <a href="halt.kvihelp">halt</a> in this event, you will
				stop the "nick sets mode..." output.<br>
		*/
		"OnChannelMode", "$1 channel name, $2 source nick, $3 source mask, $4 mode, $5 params"
	},

	{ /* KviEvent_OnChannelMessage : halt stops the Message output, bound to chan */
		/*
			@event: OnChannelMessage
			@short:
				A channel private message was received
			@parameters:
				$1 = source nickname<br>
				$2 = source mask (user@host)<br>
				$3- = notice text
			@window:
				Channel
			@description:
				Event fired when a channel private message arrives.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you
				will stop the message output.<br>
		*/
		"OnChannelMessage", "$1 source nick, $2 source mask, $3- message"
	},
	{
		/*
			@event: OnMePrivateMessage
			@short:
				A private message was received
			@parameters:
				$1 = source nickname<br>
				$2 = source mask (user@host)<br>
				$3- = notice text
			@window:
				Query if exists, otherwise Console
			@description:
				Event fired when a private message directed to the user arrives.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you
				will stop the message output.<br>
				If no query window exists, halt will alsto stop the query creation.<br>
		*/
		"OnMePrivateMessage", "$1 source nick, $2 source mask, $3- message"
	},

	{
		/*
			@event: OnServerNotice
			@short:
				A server notice was received
			@parameters:
				$1 = source server/service<br>
				$2 = target (nickname/channel)<br>
				$3- = notice text
			@window:
				Console
			@description:
				Event fired when a server/service notice arrives.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you
				will stop the message output.<br>
				The target of the output is depending on <a href="s_option.kvihelp">$option()</a>.
		*/
		"OnServerNotice", "$1 source, $2 target, $3- notice"
	},
	{
		/*
			@event: OnChannelNotice
			@short:
				A channel notice was received
			@parameters:
				$1 = source nickname<br>
				$2 = source mask (user@host)<br>
				$3- = notice text
			@window:
				Channel
			@description:
				Event fired when a channel notice arrives.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you
				will stop the message output.<br>
		*/
		"OnChannelNotice", "$1 source nick, $2 source mask, $3- notice text"
	},
	{
		/*
			@event: OnMePrivateNotice
			@short:
				A private notice was received
			@parameters:
				$1 = source nickname<br>
				$2 = source mask (user@host)<br>
				$3- = notice text
			@window:
				Query if exists, otherwise Console
			@description:
				Event fired when a private notice arrives.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you
				will stop the message output.<br>
				If no query window exists, halt will stop also the query creation.<br>
		*/
		"OnMePrivateNotice", "$1 source nick, $2 source mask, $3- notice text"
	},
	{
		/*
			@event: OnNotify
			@short:
				A "notified" user went online/offline
			@parameters:
				$1 nick
				$2 isOnline? (boolean)
			@window:
				Console
			@description:
				Executed when a "notified" user goes online or offline.<br>
		*/
		"OnNotify", "$1 nick, $2 isonline?"
	},
	{ /* KviEvent_OnInviteException : halt stops the "user sets mode +I mask" output, bound to chan or console, dep. on options */
		"OnInviteException", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask[, $5 set_by, $6 set_at]"
	},
	{ /* KviEvent_OnBanExceptionRemove : halt stops the "user sets mode -I mask" output, bound to chan or console, dep. on options */
		"OnInviteExceptionRemove", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask"
	},
	{ /* KviEvent_OnMeInviteException : halt stops the "user sets mode +I mask" output, bound to chan or console, dep. on options */
		"OnMeInviteException", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask[, $5 set_by, $6 set_at]"
	},
	{ /* KviEvent_OnMeInviteExceptionRemove : halt stops the "user sets mode -I mask" output, bound to chan or console, dep. on options */
		"OnMeInviteExceptionRemove", "$1 channel name, $2 source nick, $3 source mask, $4 dest mask"
	},
	{ /* KviEvent_OnConsoleInput (46) : halt stops the "RAW to server..." output and the send, bound to console */
		"OnConsoleInput", "$1 text"
	},
	{ /* KviEvent_OnChannelInput (47) : halt stops the <nick>... output and the send, bound to channel */
		"OnChannelInput", "$1 window, $2 text"
	},
	{ /* KviEvent_OnQueryInput (48) : halt stops the <nick>... output and the send, bound to query */
		"OnQueryInput", "$1 window, $2 text"
	},
	{ /* KviEvent_OnDCCInput (49) : halt stops the <nick>... output and the send, bound to DCC window */
		"OnDCCInput", "$1 window, $2 text"
	},
	{
		/*
			@event: OnUnhandledLiteral
			@short:
				Message that KVIrc ignores
			@parameters:
				$1 source mask<br>
				$2 literal command<br>
				$3- parameters
			@window:
				Console
			@description:
				Executed when KVIrc receives a literal message
				that is not understood.<br>
				This may happen for literal commands not handled yet by KVIrc.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you
				will stop the message output to console or active window
				(the target window is depending on an <a href="s_option.kvihelp">$option()</a>).<br>
		*/
		"OnUnhandledLiteral", "$1 source mask, $2 literal command, $3- parameters"
	},
	{
		/*
			@event: OnUnhandledNumeric
			@short:
				Message that KVIrc ignores
			@parameters:
				$1 source mask<br>
				$2 numeric reply<br>
				$3- parameters
			@window:
				Console
			@description:
				Executed when KVIrc receives a numeric message
				that is not understood.<br>
				This may happen for numeric replies not handled yet by KVIrc.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you
				will stop the message output to console or active window
				(the target window is depending on an <a href="s_option.kvihelp">$option()</a>).<br>
		*/
		"OnUnhandledNumeric", "$1 source mask, $2 numeric reply, $3- parameters"
	},
	{
		/*
			@event: OnKeyShortcut
			@short:
				A shortcut key was pressed.
			@parameters:
				$1 = Key code.<br>
			@window:
				The one in which the key is pressed
			@description:
				Executed when a shortcut key is pressed.<br>
				The code of the key is in the $1 parameter.<br>
				It may be one of the following strings:<br>
				F1, F2, ..., F12, Ctrl+0, Ctrl+1, ..., Ctrl+9, Shift+F1, Shift+F2..., Shift+F12.<br>
				A simple and funny example of its use could be a code like this:<br>
				<example>
					<a href="if.kvihelp">if</a>("<a href="s_commandline.kvihelp">$commandline</a>(<a href="s_window.kvihelp">$window</a>)" != "")<a href="window.kvihelp">window</a> <a href="s_window.kvihelp">$window</a> commandline <a href="s_b.kvihelp">$b</a><a href="s_u.kvihelp">$u</a><a href="s_commandline.kvihelp"><a href="s_upcase.kvihelp">$upcase</a>($commandline</a>(<a href="s_window.kvihelp">$window</a>))
				</example>
				The handler for this event will usually be a [cmd]switch[/cmd] command:<br>
				<example>
				&nbsp;[cmd]switch[/cmd]($1)
				&nbsp;{
				&nbsp;	case(F1)[cmd]echo[/cmd] F1 pressed!
				&nbsp;	case(F2)[cmd]echo[/cmd] F2 pressed!
				&nbsp;	case(F3)[cmd]echo[/cmd] F3 pressed!
				&nbsp;	case(F4)[cmd]echo[/cmd] F4 pressed!
				&nbsp;	case(Ctrl+1)[cmd]echo[/cmd] Ctrl+1 pressed!
				&nbsp;}
				</example>
				If your [cmd]switch[/cmd] has to be huge (i.e. you are handling a lot of shortcut keys),
				it is better to split the handling in more aliases (each one containing a [cmd]switch[/cmd]):<br>
				<example>
				&nbsp;[cmd]if[/cmd]("[fnc]$strLeft[/fnc](1, $1)" == "F")handleSimpleSortcuts $1
				&nbsp;[cmd:if]else[/cmd] [cmd]if[/cmd]("[fnc]$strLeft[/fnc](1, $1)" == "C")handleCtrlShortcuts $1
				&nbsp;[cmd:if]else[/cmd] handleShiftShortcuts $1
				</example>
		*/
		"OnKeyShortcut", "$1 = KeyCode"
	},
	{
		/*
			@event: OnNickDoubleClicked
			@short:
				A nickname was doubleclicked in a channel (or console) listbox
			@parameters:
				$1 = nickname<br>
			@window:
				Channel or console in which the nick is clicked.<br>
			@description:
				Executed when a nickname in a listbox is double clicked.<br>
				By calling halt in this event you will stop the default action
				that is to call /QUERY nickname.<br>
		*/
		"OnNickDoubleClicked", "nickname"
	},
	{ /* KviEvent_OnUserWindowInput (75) : halt stops the output and the send, bound to window */
		/*
			@event: OnUserWindowInput
			@short:
				Something was entered in a user window
			@parameters:
				$1 = window<br>
				$2- = text<br>
			@window:
				The window in which the text was entered.<br>
			@description:
				Executed when text was entered in a user window.<br>
				By calling halt in this event you will stop the text output.<br>
		*/
		"OnUserWindowInput", "$1 window, $2- text"
	},
	{ /* KviEvent_OnUserWindowFileDrop (76) bound to window */
		/*
			@event: OnUserWindowFileDrop
			@short:
				A file was dropped (DnD) in a user window.
			@parameters:
				$1 = window<br>
				$2 = filename<br>
			@window:
				The userwindow in which the file was dropped.<br>
			@description:
				Executed when a file is droppend in a user window (view zone).<br>
		*/
		"OnUserWindowFileDrop", "$1 window, $2 filename"
	},
	{ /* KviEvent_OnUserWindowSelected (77) bound to window */
		/*
			@event: OnUserWindowSelected
			@short:
				A list item was selected in a user window listbox
			@parameters:
				$1 = window<br>
				$2 = double-clicked?<br>
				$3 = index<br>
				$4- = item name<br>
			@window:
				The userwindow in which the item is selected.<br>
			@description:
				Executed when the selection in the listbox changes ($2 = 0).<br>
				Or when an item in the listbox is double-clicked ($2 = 1).<br>
		*/
		"OnUserWindowSelected", "$1 window, $2 double-clicked, $3 index, $4- item name"
	},
	{
		/*
			@event: OnWhoisReply
			@short:
				Whois reply from server
			@parameters:
				$1 = reply name<br>
				$2 = nickname<br>
				$3- = reply dependant params<br>
			@window:
				Console<br>
			@description:
				Executed when KVIrc recieves a whois reply from the server.<br>
				The parameters starting from $3 are reply dependant:<br>
				if $1 is USER:<br>
				$3 is the username, $4 is the hostname and $5- is the realname of the user.<br>
				if $1 is SERVER:<br>
				$3- contains the server that the user is connected to.<br>
				if $1 is OPER:<br>
				There are no reply dependand parameters : the user is an IRCOp.<br>
				if $1 is WAS:<br>
				$3- contains the server history entry for the user.<br>
				if $1 is STATUS:<br>
				$3- constains other information about the user status : AWAY, HelpOp etc...<br>
				if $1 is IDLE:<br>
				$3 contains the total user idle time in seconds, $4 contains the hours, $5 the minutes in the last hour, and $6 the seconds in the last minute.<br>
				if $1 is UIDLE:<br>
				$3- contains the server specific idle time format string.<br>
				if $1 is SIGNON:<br>
				$3- contains the user signon time string.<br>
				if $1 is ENDWHOWAS or ENDWHOIS:<br>
				$3 contains the name of the server that generated the reply<br>
				if $1 is CHANS:<br>
				$3- contains the list of the channels that the user is on.<br>
				Halt will stop the whois-output in all these replies.<br>
		*/
		"OnWhoisReply", "$1 reply name, $2 nick, $3- reply dependant params"
	},
	{
		/*
			@event: OnDCCChatMessage
			@short:
				DCC chat remote message
			@parameters:
				$1 = nickname<br>
				$2 = username or "*" if not known<br>
				$3 = hostname or "*" if not known<br>
				$4 = ip_address<br>
				$5- = text<br>
			@window:
				DCC chat window<br>
			@description:
				Executed when a user connected via DCC chat
				writes a line of text.<br>
				By calling 'halt' in this event you will avoid the output
				to the DCC window.<br>
		*/
		"OnDCCChatMessage", "$1 nick, $2 username or '*', $3 hostname or '*', $4 ip_address, $5- text"
	},
	{
		/*
			@event: OnDCCChatConnected
			@short:
				DCC chat connection estabilished
			@parameters:
				$1 = nickname<br>
				$2 = username or "*" if not known<br>
				$3 = hostname or "*" if not known<br>
				$4 = ip_address<br>
			@window:
				DCC chat window<br>
			@description:
				Executed when a DCC chat connection is estabilished
		*/
		"OnDCCChatConnected", "$1 nick, $2 username or '*', $3 hostname or '*', $4 ip_address"
	},
	{
		/*
			@event: OnDCCChatTerminated
			@short:
				DCC chat connection terminated
			@parameters:
				$1 = nickname<br>
				$2 = username or "*" if not known<br>
				$3 = hostname or "*" if not known<br>
				$4 = ip_address<br>
				$5- = error description (if any)<br>
			@window:
				DCC chat window<br>
			@description:
				Executed when a DCC chat connection is terminated
				either by a remote close, error or local close.<br>
				This event does NOT fire if the connection was not estabilished yet.<br>
				By calling halt in this event you will disable the
				output "Connection terminated (host:port)"<br>
		*/
		"OnDCCChatTerminated", "$1 nick, $2 username or '*', $3 hostname or '*', $4 ip_address, $5- error message"
	},
	{
		/*
			@event: OnDCCGetTransferComplete
			@short:
				Ad DCC get transfer has been successfully completed
			@parameters:
				$1 = remote nickname<br>
				$2 = remote username or "*" if not known<br>
				$3 = remote hostname or "*" if not known<br>
				$4 = remote ip_address<br>
				$5 = filename with path<br>
				$6 = length of the file in bytes
			@window:
				DCC send window<br>
			@description:
				Executed when a DCC get transfer has been SUCCESSFULLY completed.<br>
		*/
		"OnDCCGetTransferComplete", "$1 nick, $2 username or '*', $3 hostname or '*', $4 ip_address, $5 filename, $6 length in bytes"
	},
	{
		/*
			@event: OnDCCSendTransferComplete
			@short:
				Ad DCC send transfer has been successfully completed
			@parameters:
				$1 = remote nickname<br>
				$2 = remote username or "*" if not known<br>
				$3 = remote hostname or "*" if not known<br>
				$4 = remote ip_address<br>
				$5 = filename with path<br>
				$6 = length of the file in bytes
			@window:
				DCC send window<br>
			@description:
				Executed when a DCC send transfer has been SUCCESSFULLY completed.<br>
		*/
		"OnDCCSendTransferComplete", "$1 nick, $2 username or '*', $3 hostname or '*', $4 ip_address, $5 filename, $6 length in bytes"
	},
	{
		/*
			@event: OnCTCPMultimedia
			@short:
				A user sent a CTCP Multimedia
			@parameters:
				$1 = source nickname<br>
				$2 = source mask<br>
				$3 = target (you or a channel name)<br>
				$4 = 1 if a query window should be created<br>
				$5 = filename
			@window:
				Query or channel<br>
			@description:
				Executed when a remote user sends a CTCP MULTIMEDIA (SOUND) to you or a common channel.<br>
				Halt stops the output and the automatic query creation.<br>
		*/
		"OnCTCPMultimedia", "$1 source nick, $2 source mask, $3 target, $4 open_query_on_privmsg, $5 filename"
	},
	{
		/*
			@event: OnWallops
			@short:
				Wallops message received
			@parameters:
				$1 = source mask<br>
				$2- = message text<br>
			@window:
				Console<br>
			@description:
				Called when a WALLOPS message is received from the server.<br>
				The source parameter may be either a full user mask or a server hostname.<br>
				By calling halt in this event you will stop the default message output.<br>
		*/
		"OnWallops", "$1 source mask, $2- message text"
	},
	{
		/*
			@event: OnChannelOpNotice
			@short:
				A wallchops notice was received
			@parameters:
				$1 = source nickname<br>
				$2 = source mask (user@host)<br>
				$3- = notice text
			@window:
				Channel
			@description:
				Event fired when a channel op notice arrives.<br>
				By calling <a href="halt.kvihelp">halt</a> in this event you
				will stop the message output.<br>
		*/
		"OnChannelOpNotice", "$1 source nick, $2 source mask, $3- notice text"
	},
	{
		/*
			@event: OnUrl
			@short:
				Triggered when a URL is about to be shown in an output window
			@parameters:
				$1 = url<br>
			@window:
				Window in which the url is going to appear
			@description:
				Triggered when a URL is about to be shown in an output window.<br>
				<b>WARNING</b><br>
				If you <a href="echo.kvihelp">echo</a> the URL again in the window
				you will get infinite recursion of this event!.<br>
				So never call:<br>
				<a href="echo.kvihelp">echo</a> $1<br>
		*/
		"OnUrl", "$1 url"
	},
	{
		/*
			@event: OnNick
			@short:
				Triggered when a nick change occurs.
			@parameters:
				$1  = current nick<br>
				$2  = user name<br>
				$3  = user host<br>
				$4  = new nick<br>
				$5- = list of windows affected by the change<br>
			@window:
				Console.
			@description:
				Triggered when a nick change has occurred, and the change is about
				to appear in a window.<br>
		*/
		"OnNick", "$1 source nick, $2 source mask, $3 target nick, $4- window list"
	},
	{
		/*
			@event: OnMeNick
			@short:
				Triggered when own nick changes.
			@parameters:
				$1  = current nick<br>
				$2  = user name<br>
				$3  = user host<br>
				$4  = new nick<br>
				$5- = list of windows affected by the change<br>
			@window:
				Console.
			@description:
				Triggered when own nick changes, just before the change is
				displayed in the window.<br>
		*/
		"OnMeNick", "$1 source nick, $2 source mask, $3 target nick, $4- window list"
	},
	{
		/*
			@event: OnError
			@short:
				Triggered on server ERROR messages
			@parameters:
				$1  = server name<br>
				$2- = error description string<br>
			@window:
				Console.
			@description:
				Triggered when a server ERROR message is received.
		*/
		"OnError", "$1 server name, $2- error description string"
	},
	{
		/*
			@event: OnEventError
			@short:
				Triggered when an error occurs inside an event handler
			@parameters:
				$1  = server name<br>
				$2- = error description string<br>
			@window:
				Console.
			@description:
				Triggered when an error occurs inside an event handler (that is not OnEventError obviously).<br>
		*/
		"OnEventError", "$1- error description string"
	},
	{
		/*
			@event: OnQuit
			@short:
				Triggered when a user quits IRC
			@parameters:
				$1 = nick<br>
				$2 = user<br>
				$3 = host<br>
				$4 = bWasSplit?<br>
				$5- = quit message<br>
			@window:
				Console
			@description:
				Triggered when a user quits IRC.<br>
				The event is triggered when the user is still on all the channels that he was on.<br>
				halt will stop all the KVIrc output about the quitting user:<br>
				this includes all the channel window messages and the terminated queries warnings.<br>
		*/
		"OnQuit", "$1 nick, $2 user, $3 host, $4 bWasSplit, $5- quit message"
	},
	{
		/*
			@event: OnIdleStart
			@short:
				Triggered when no user activity is detected
			@parameters:
				$1 = time of inactivity<br>
			@window:
				All
			@description:
				Triggered when there is no user activity, ie:
				No keys are pressed or no commands are being
				executed by the user. The time of inactivity
				after which user is considered idle is
				configurable.<br>
		*/
		"OnIdleStart", "$1 idleTime"
	},
	{
		/*
			@event: OnIdleStop
			@short:
				Triggered when user stops being idle
			@parameters:
				none<br>
			@window:
				All
			@description:
				Triggered when user inputs a text or
				(optionally) executes a command while being
				idle. This event is *not* triggered when you
				browse the commands history etc.<br>
		*/
		"OnIdleStop", "none"
	},
	{
		"OnLogin", "none"
	},

	{ /* KviEvent_OnHalfop : halt stops the "user sets mode +o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnHalfop", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnDehalfop : halt stops the "user sets mode -o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnDehalfop", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnMeHalfop : halt stops the "user sets mode +o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeHalfop", "$1 channel name, $2 source nick, $3 source mask"
	},
	{ /* KviEvent_OnMeDehalfop : halt stops the "user sets mode -o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeDehalfop", "$1 channel name, $2 source nick, $3 source mask"
	},

	{ /* KviEvent_OnUserop : halt stops the "user sets mode +o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnUserop", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnDeuserop : halt stops the "user sets mode -o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnDeuserop", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnMeUserop : halt stops the "user sets mode +o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeUserop", "$1 channel name, $2 source nick, $3 source mask"
	},
	{ /* KviEvent_OnMeDeuserop : halt stops the "user sets mode -o user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeDeuserop", "$1 channel name, $2 source nick, $3 source mask"
	},

	{ /* KviEvent_OnOwner : halt stops the "user sets mode +q user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnOwner", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnDeowner : halt stops the "user sets mode -q user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnDeowner", "$1 channel name, $2 source nick, $3 source mask, $4 dest nick"
	},
	{ /* KviEvent_OnMeOwner : halt stops the "user sets mode +q user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeOwner", "$1 channel name, $2 source nick, $3 source mask"
	},
	{ /* KviEvent_OnMeDeowner : halt stops the "user sets mode -q user" output and the statusbat text, bound to chanel or console, dep. on options */
		"OnMeDeowner", "$1 channel name, $2 source nick, $3 source mask"
	}
};

KviEventManager::KviEventManager()
{
#ifdef COMPILE_PLUGIN_SUPPORT
	for( int i = 0; i < KVI_NUM_EVENTS; i++ ) {
		m_pEventHandlerList[i] = 0;
	}
#endif
	cleanup();
}

void KviEventManager::cleanup()
{
	for( int i = 0; i < KVI_NUM_EVENTS; i++ ) {
		m_bEventEnabled[i] = false;
		m_szEventBuffer[i] = "";
	}
}

KviEventManager::~KviEventManager()
{
#ifdef COMPILE_PLUGIN_SUPPORT
	for( int i = 0; i < KVI_NUM_EVENTS; i++ ) {
		if( m_pEventHandlerList[i] ) {
			debug("WARNING: there are %u registered event handlers for event %d", m_pEventHandlerList[i]->count(), i);
			delete m_pEventHandlerList[i];
			m_pEventHandlerList[i] = 0;
		}
	}
#endif
}

void KviEventManager::setHandler(const char *evName, const char *buffer, bool bEnabled)
{
	for( int i = 0; i < KVI_NUM_EVENTS; i++ ) {
		if( kvi_strEqualCI(evName, m_eventDescriptor[i].eventName) ) {
			m_szEventBuffer[i] = buffer;
			m_bEventEnabled[i] = bEnabled;
			return;
		}
	}
	debug("Unknown handler %s", evName);
}

void KviEventManager::disableEmptyHandlers()
{
	for( int i = 0; i < KVI_NUM_EVENTS; i++ ) {
		m_szEventBuffer[i].stripWhiteSpace();
		if( m_szEventBuffer[i].isEmpty() )
			m_bEventEnabled[i] = false;
	}
}

int KviEventManager::indexOfEvent(const char *eventName)
{
	unsigned int len = strlen(eventName);
	for( int i = 0; i < KVI_NUM_EVENTS; i++ ) {
		if( kvi_strEqualCIN(m_eventDescriptor[i].eventName, eventName, len) ) {
			if( len == strlen(m_eventDescriptor[i].eventName) )
				return i;
		}
	}
	return -1;
}

bool KviEventManager::load(const char *filename, KviStr &error)
{
	KviStr buffer;
	if( !kvi_loadFile(filename, buffer) ) {
		error = _i18n_("Unable to load file");
		return false;
	}
	if( !kvi_strEqualCSN(KVI_MAGIC_STRING, buffer.ptr(), KVI_MAGIC_STRING_LENGTH) ) {
		error = _i18n_("Invalid magic");
		return false; // Not a KVIrc file
	}
	buffer.cutLeft(KVI_MAGIC_STRING_LENGTH);
	buffer.stripLeftWhiteSpace(); // Be flexible... allow some spaces... for manual editing by experts :)
	if( buffer.isEmpty() )
		return true; // Done

	while( buffer.hasData() ) {
		// [EVENT <eventname> <ENABLED|DISABLED> bytes <numofbytes>]\n
		// ...
		// ...
		// ...
		// \n[ENDEVENT]\n

		// Event label
		if( !kvi_strEqualCIN("[EVENT ", buffer.ptr(), 7) ) {
			error = _i18n_("Unrecognized [EVENT] label.");
			return false;
		}

		buffer.cutLeft(7);
		buffer.stripLeftWhiteSpace();

		// Event name
		KviStr eventName;
		if( !buffer.getToken(eventName, ' ') ) {
			error = _i18n_("Syntax error in [EVENT] label");
			return false;
		} // There must be at least one other token

		// Find the event
		int index = indexOfEvent(eventName.ptr());
		KviStr *ptrToEventBuffer;
		bool dummy;
		bool *ptrToEventEnableFlag;

		if( (index < 0) || (index >= KVI_NUM_EVENTS) ) {
			debug("Unrecognized event handler %s, ignoring.", eventName.ptr());
			ptrToEventBuffer     = &eventName; // <-- Put it here and ignore...
			ptrToEventEnableFlag = &dummy;
		} else {
			ptrToEventBuffer     = &(m_szEventBuffer[index]);
			ptrToEventEnableFlag = &(m_bEventEnabled[index]);
		}

		buffer.stripLeftWhiteSpace();

		if( kvi_strEqualCIN("ENABLED ", buffer.ptr(), 8) ) {
			buffer.cutLeft(8);
			*ptrToEventEnableFlag = true;
		} else if( kvi_strEqualCIN("DISABLED ", buffer.ptr(), 9) ) {
			buffer.cutLeft(9);
			*ptrToEventEnableFlag = false;
		} else {
			error = _i18n_("Syntax error in [EVENT] label (event state)");
			return false;
		}

		buffer.stripLeftWhiteSpace();

		if( !kvi_strEqualCIN("bytes ", buffer.ptr(), 6) ) {
			error = _i18n_("Syntax error in [EVENT] label (event length)");
			return false;
		}
		buffer.cutLeft(6);

		buffer.stripLeftWhiteSpace();

		if( !isdigit(*(buffer.ptr())) ) {
			error = _i18n_("Syntax error in [EVENT] label (digit expected)");
			return false;
		}

		KviStr numBytes;
		if( !buffer.getToken(numBytes, ']') ) {
			error = _i18n_("Syntax error in [EVENT] label (cannot extract alias length)");
			return false;
		}

		bool bOk = false;
		int bytes = numBytes.toInt(&bOk);
		if( !bOk ) {
			error  = _i18n_("Unrecogized token ");
			error += numBytes;
			return false;
		} // Syntax error again... someone messed with the file
		buffer.cutLeft(1); // Cut the \n
		if( buffer.len() < bytes ) {
			error  = _i18n_("Alias buffer smaller than declared length (");
			error += numBytes;
			error += ")";
			return false;
		}

		ptrToEventBuffer->setStr(buffer.ptr(), bytes);
		buffer.cutLeft(bytes);

		if( !kvi_strEqualCIN("\n[ENDEVENT]", buffer.ptr(), 11) ) {
			error = _i18n_("Unrecognized [ENDEVENT] label");
			return false;
		}
		buffer.cutLeft(11);

		buffer.stripLeftWhiteSpace();
	}
	disableEmptyHandlers();
	return true;
}

bool KviEventManager::save(const char *filename)
{
	QFile f(filename);
	if( !f.open(IO_WriteOnly | IO_Truncate) )
		return false;

	f.writeBlock(KVI_MAGIC_STRING, KVI_MAGIC_STRING_LENGTH);

	for( int i = 0; i < KVI_NUM_EVENTS; i++ ) {
		f.writeBlock("[EVENT ", 7);
		f.writeBlock(m_eventDescriptor[i].eventName, strlen(m_eventDescriptor[i].eventName));
		if( m_bEventEnabled[i] )
			f.writeBlock(" ENABLED", 8);
		else
			f.writeBlock(" DISABLED", 9);
		f.writeBlock(" bytes ", 7);
		KviStr szNBytes(KviStr::Format, "%d", m_szEventBuffer[i].len());
		f.writeBlock(szNBytes.ptr(), szNBytes.len());
		f.writeBlock("]\n", 2);
		f.writeBlock(m_szEventBuffer[i].ptr(), m_szEventBuffer[i].len());
		f.writeBlock("\n[ENDEVENT]\n", 12);
	}

	f.close();
	return true;
}
