/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef EDITVIEW_H
#define EDITVIEW_H

#include <qscrollview.h>
#include <qpoint.h>
#include <qpointarray.h>
#include <qimage.h>
#include <kaction.h>
#include <qtimer.h>
#include "stitch.h"
#include "kxstitchdoc.h"

class KXStitchDoc;
class KPrinter;
class QRect;
class QPainter;
class QMimeSource;

/** Editor window for modifying patterns
  */
class EditView : public QScrollView
{
  Q_OBJECT
public:
  enum ToolMode {Paint, Draw, Erase, Rectangle, FillRectangle, Ellipse, FillEllipse, FillPolyline, SelectRectangle, Backstitch, ColorPicker, Paste};
  enum ViewMode {Regular, BWSymbols, ColorSymbols, ColorBlocks, ColorBlocksSymbols, BWLines, ColorLines};

  EditView(KXStitchDoc* doc,QWidget *parent);

  void resizeGrid();
/** configure the editor to the users preferences
  */
  void configure(bool init);
/** get the current selection area
  */
  QRect selectionArea();
/** clear the selection area
  */
  void clearSelection();
/** print the view
    @return The number of pages printed so far
    @param printer Pointer to the KPrinter object
    @param painter Pointer to the painter object
    @param width Width of the printable page
    @param height Height of the printable page
    @param pages The number of pages printed so far
  */
  int print(KPrinter *printer, QPainter *painter, int width, int height, int pages);

/** set the current @ref ToolMode tool in use
    @param m The tool to be selected
  */
  void setMode(ToolMode m);
/** set the background image to a pixmap
    @param pixmap A pointer to the pixmap to be used
  */
  void setBackgroundImage(QImage image);
/** clear the background image
  */
  void clearBackgroundImage();

public slots:
/** set the center of the view
    @param pos The cell that should be at the center of the view
  */
  void slotSetCenter(QPoint pos);

signals:
/** emitted when the coordinated of visible cells are changed, when zooming or scrolling
    @param rect The top left cell and the number of cells wide and high that are visible
  */
  void changedVisibleCells(QRect rect);
/** emitted when a selection area is made or cleared
    @param made true if a selection is made, false if cleared
  */
  void selectionMade(bool made);
/** emitted to indicate points or sizes
  */
  void locationOrSize(QPoint point);
/** emitted when the format of the scales has changed
    @param a string indicating the current format
  */
  void scaleFormat(QString format);
/** emitted when the status of the stitch mask has changed
    @param enable boolean value indicating if the mask is enabled or not
  */
  void stitchMask(bool enable);
/** emitted when the status of the color mask has changed
    @param enable boolean value indicating if the mask is enabled or not
  */
  void colorMask(bool enable);

protected:
/** draw the visible parts of the scrollview that are not part of the contents,
    e.g. the scales.
    @param event QPaintEvent defining the area to be repainted.
  */
  virtual void paintEvent(QPaintEvent *event);
/** draw the contents of the editor view clipping to a rectangle
    @param p Pointer to the painter object
    @param clipx ContentsX position of the clip rectangle
    @param clipy ContentsY position of the clip rectangle
    @param clipw Contents width of the clip rectangle
    @param cliph Contetns height of the clip rectangle
  */
  virtual void drawContents(QPainter *p, int clipx, int clipy, int clipw, int cliph);
/** print the contents of the editor view to a printer
    @param printer Pointer to the KPrinter object
    @param painter Pointer to the painter object
    @param x X position of the top left of this page on the contents
    @param y Y position of the top left of this page on the contents
    @param printable Rectangle defining the printable area of the paper
    @param pageCellsWide The number of cells that can be printed on the width of paper
    @param pageCellsTall The number of cells that can be printed on the height of paper
    @param printCellSize The size of the cells to be printed
  */
  void printContents(KPrinter *printer, QPainter *painter, int x, int y, QRect printable, int pageCellsWide, int pageCellsTall, int printCellSize);
/** implemented to handle context menu events
  */
  virtual void contextMenuEvent(QContextMenuEvent* e);
/** handle the initial mouse press interaction with the editor window
    @param e Pointer to the mouse event
  */
  virtual void contentsMousePressEvent(QMouseEvent *e);
/** handle and mouse moves in the editor window while a button is pressed
    @param e Pointer to the mouse event
  */
  virtual void contentsMouseMoveEvent(QMouseEvent *e);
/** handle the mouse release in the editor window, terminates some of the drawing functions
    @param e Pointer to the mouse event
  */
  virtual void contentsMouseReleaseEvent(QMouseEvent *e);

/** Get the coordinates of the cell at the center of the editor view
    @return QPoint defining the center cell
  */
  QPoint visibleCenterCell();
/** Get the coordinate of the top left cell in the editor view, and the number of cells
    horizontally and vertically that are visible in the view
    @return QRect defining the visible cells
  */
  QRect visibleCells();
/** convert a contents location into a cell reference that contains the contents location
    @return A QPoint defining the cell that contains the contents location
    @param contents A contents point to be translated
  */
  QPoint contentsToCell(QPoint contents);
/** Convert a contents location into a snap point, there are 9 snap points for each cell
    @return A QPoint defining the snap point of the contents location
    @param contents A contents point to be translated
  */
  QPoint contentsToSnap(QPoint contents);
/** convert a snap point to a viewport coordinate
    @return A QPoint defining the snap location translated to the viewport
    @param snap The snap location to be translated
  */
  QPoint snapToViewport(QPoint snap);
/** convert a snap point to a contents coordinate
    @return A QPoint defining the snap location translated to the contents
    @param snap The snap location to be translated
 */
  QPoint snapToContents(QPoint snap);
/** convert a cell coordinate to a contents coordinate rectangle encompassing the cell
    @return a QRect defining the contents coordinates of the cell
    @param cell The cell coordinates to be translated
  */
  QRect cellToRect(QPoint cell);
/** convert a QRect of cells to a contents QRect
    @return a QRect defining the contents coordinates of the cells
    @param cells A QRect defining the bounding rectangle of the cells to be converted
  */
  QRect cellsToRect(QRect cells);
/** convert contents QRect to viewport QRect
    @return a QRect defining the viewport coordinates of the cells
    @param cells A QRect defining the bounding rectangle of the cells to be converted
  */
  QRect rectToViewport(QRect rect);
/** convert a pixmap to a QPointArray
    @return an array of points defining cells highlighted on pixmap
    @param pix Pixmap to be checked
  */
  QPointArray processPixmap(QPixmap& pix);

private slots:
/** zoom in on the editor window
  */
  void slotZoomIn();
/** zoom out on the editor window
  */
  void slotZoomOut();
/** put the marked text/object into the clipboard and remove
    it from the document
  */
  void slotEditCut();
/** put the marked text/object into the clipboard
  */
  void slotEditCopy();
/** paste the clipboard into the document
  */
  void slotEditPaste();
/** create a library pattern
  */
  void slotLibraryCreateLibraryPattern();
/** insert a library pattern
  */
  void slotLibraryPattern();
/** insert a text string
  */
  void slotTextTool();
/** set current stitch type
  */
  void slotStitchFull();
  void slotStitchTBHalf();
  void slotStitchBTHalf();
  void slotStitchBLQtr();
  void slotStitchTLQtr();
  void slotStitchTRQtr();
  void slotStitchBRQtr();
  void slotStitchBL3Qtr();
  void slotStitchTL3Qtr();
  void slotStitchTR3Qtr();
  void slotStitchBR3Qtr();
  void slotStitchFRKnot();
/** set current tool type
  */
  void slotToolPaint();
  void slotToolDraw();
  void slotToolBackStitch();
  void slotToolErase();
  void slotRectangle();
  void slotFillRectangle();
  void slotEllipse();
  void slotFillEllipse();
  void slotFillPolyline();
  void slotSelectRectangle();
  void slotColorPicker();
/** set current view mode
  */
  void slotRegular();
  void slotBWSymbols();
  void slotColorSymbols();
  void slotColorBlocks();
  void slotColorBlocksSymbols();
/** check if clipboard contains data that can be used
  */
  void slotClipboardDataChanged();
/** extend the pattern
  */
  void slotPatternExtend();
/** center the pattern
  */
  void slotPatternCenter();
/** crop the canvas to the pattern
  */
  void slotCropCanvasToPattern();
/** crop the canvas to the selection rectangle
  */
  void slotCropCanvasToSelection();
/** fit the background image to the selection
  */
  void slotFitBackgroundToSelection();
/** toggle the stitchMask flag
  */
  void slotStitchMask();
/** toggle the colorMask flag
  */
  void slotColorMask();
/** display the full width of the document
  */
  void slotFitWidth();
/** display the full height of the document
  */
  void slotFitHeight();
/** display the full document
  */
  void slotFitPage();
/** set scales to Stitches
  */
  void slotScaleFormatStitches();
/** set scales to CM
  */
  void slotScaleFormatCM();
/** set scales to Inches
  */
  void slotScaleFormatInches();
/** toggle the show / hide stitches
  */
  void slotShowStitches();
/** toggle the show / hide back stitches
  */
  void slotShowBackStitches();
/** toggle the show / hide grid lines
  */
  void slotShowGrid();
/** toggle the show / hide rule scales
  */
  void slotShowScales();
/** Connected to the contentsMoving signal, used to scroll the scales
    @param dx X offset
    @param dy Y offset
  */
  void slotScrolling(int dx, int dy);
/** Connected to the selectedArea timer, to produce a marching ant effect
  */
  void slotSelectionTimedOut();
/** toggle viewing the background image
  */
  void slotShowBackground();

private:
/** set the current ViewMode in use
    @param m The view mode to be used
  */
  void setViewMode(ViewMode m);
/** set the viewmode of backstitches
    @param m The viewmode to be used
  */
  void setBSViewMode(ViewMode m);
/** draw a bounding line
    @param s Start point
    @param e End point
  */
  void drawBoundingLine(QPoint s, QPoint e);
/** draw a bounding rectangle
    @param s TopLeft corner
    @param e BottomRight corner
  */
  void drawBoundingRect(QPoint s, QPoint e);
/** draw a bounding ellipse
    @param s TopLeft corner of bounding rectangle
    @param e BottomRight corner of bounding rectangle
  */
  void drawBoundingEllipse(QPoint s, QPoint e);
/** draw a selection rectangle
  */
  void drawSelectionRectangle();
  void hideSelectionRect();
  void showSelectionRect();
/** draw the contents of a cell
    @param p Pointer to the painter object
    @param rect QRect of the cells contents coordinates
    @param pStitches Pointer to a Queue of Stitch
    @param viewMode viewMode to be used to render the Queue
  */
  void renderCell(QPainter *p, QRect rect, Stitch::Queue *pStitches, ViewMode viewMode);

  KAction*        m_zoomIn;
  KAction*        m_zoomOut;
  KAction*        m_editCut;
  KAction*        m_editCopy;
  KAction*        m_editPaste;
  KAction*        m_createLibraryPattern;
  KAction*        m_cropToSelection;
  KAction*        m_backgroundToSelection;
  KToggleAction*  m_showBackground;
  KToggleAction*  m_showStitches;
  KToggleAction*  m_showBackstitches;
  KToggleAction*  m_showGrid;
  KToggleAction*  m_showScales;

  KXStitchDoc*    m_doc;
  QMimeSource*    m_pasteData;
  bool            m_pasteDataIsLibrary;
  bool            m_backgroundImage;
  bool            m_stitches;     // Editor_ShowStitches
  bool            m_backstitches; // Editor_ShowBackstitches
  bool            m_grid;         // Editor_ShowGrid
  int             m_cellSize;     // Editor_CellSize
  int             m_cellGrouping; // Editor_CellGrouping
  QColor          m_gridMajor;    // Editor_GridMajorColor
  QColor          m_gridMinor;    // Editor_GridMinorColor
  bool            m_scales;       // Editor_ShowScales
  int             m_scaleSize;    // Editor_ScaleSize
  enum SCALEFORMAT
  {
    Stitches,
    CM,
    Inches
  } m_scaleFormat;                // Editor_ScaleFormat
  ViewMode        m_vmode;        // Editor_StitchType
  ViewMode        m_bsvmode;      // Editor_BackStitch::Type

  Stitch::Type    m_currentStitchType;
  bool            m_stitchMask;
  bool            m_colorMask;
  ToolMode        m_oldMode;
  ToolMode        m_mode;
  bool            m_updatingDisplay;
  QTimer          m_selectionTimer;
  QRect           m_selectedArea;
  QPoint          m_selectedAreaOffset;
  bool            m_selectedRectangleVisible;
  QPointArray     m_pointArray;
  int             m_points;
  QPixmap*        m_pasteImage;
  QImage          m_background;
  QRect           m_requestedSize;
  QPixmap         m_backgroundScaled;
  /** start is the cell where the mouse was pressed,
      tracking is the current cell during a mouse move
      end is the cell where the mouse was released
    */
  QPoint          m_start;
  QPoint          m_tracking;
  QPoint          m_end;
};

#endif
