/***************************************************************************
 *
 *  $Id: kzenexplorer.cpp,v 1.56 2005/07/03 18:27:51 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>
#include <fstream>

#include <libnjb.h>

#include <qstring.h>
#include <qregexp.h>
#include <qfile.h>
#include <qapplication.h>
#include <qdir.h>
#include <qsignalmapper.h>

#include <kmainwindow.h>
#include <kstdaction.h>
#include <ktoolbar.h>
#include <klocale.h>
#include <kinputdialog.h>
#include <kmessagebox.h>
#include <kcompletion.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <kcmdlineargs.h>
#include <krun.h>
#include <kprocess.h>

#include <fileref.h>
#include <tag.h>

#include "kzenexplorerview.h"
#include "kzenitem.h"
#include "kzenartist.h"
#include "kzenalbum.h"
#include "kzenplaylist.h"
#include "kzenmediafiles.h"
#include "kzenprogressdialog.h"
#include "kzentageditor.h"
#include "kzensmartplaylistdialog.h"
#include "kzensearch.h"
#include "kzensearchdialog.h"
#include "kzenconfigdialog.h"
#include "kzenconfig.h"
#include "jukebox.h"
#include "kzensynchronize.h"

#include "kzenexplorer.h"

static njb_t njbs[NJB_MAX_DEVICES];
static njb_t *fNJB = NULL;
static bool njb_connected = FALSE;

int progressReadDB(u_int64_t sent, u_int64_t total, const char* , unsigned , void *)
{
  long Sent = sent;
  long Total = total;
  int Percent = (Sent*100)/Total;
  
  kdDebug() << "read track database: " << Percent << "\%" << endl;

  if (gExplorer->splashScreen()) {
    QString m;
    m.sprintf(i18n("Reading track Database from jukebox: %d%% done"),Percent);
    gExplorer->splashScreen()->message(m);
  }

  return 0;
}

int progressWriteDB(u_int64_t sent, u_int64_t total, const char* , unsigned , void *)
{
  long Sent = sent;
  long Total = total;
  int Percent = (Sent*100)/Total;
  
  kdDebug() << "Write track database: " << Percent << "\%" << endl;
  
  return 0;
}

KZenExplorer * gExplorer = 0;

KZenExplorer::KZenExplorer(KZenSplashScreen * splash)
  :KMainWindow(0,"KZenExplorer")
{
  gExplorer = this;

  fDatabaseID = 0;

  fSplash = splash;
  fSplash->message(i18n("Constructing main window"));
  
  fView = new KZenExplorerView(this,"View");
  setCentralWidget(fView);

  connect(this,SIGNAL(removeTrackItem(KZenTrackItem *)),
	  fView,SLOT(removeTrackItem(KZenTrackItem *)));
  connect(this,SIGNAL(removeNaviItem(KZenItem *)),
	  fView,SLOT(removeNaviItem(KZenItem *)));
  connect(this,SIGNAL(selectNaviItem(KZenItem *)),
	  fView,SLOT(selectNaviItem(KZenItem *)));
  connect(this,SIGNAL(removeArtist(const char *)),
	  fView,SLOT(removeArtist(const char *)));
  connect(this,SIGNAL(removeGenre(const char *)),
	  fView,SLOT(removeGenre(const char *)));
  connect(this,SIGNAL(removeAlbum(const char *, const char *)),
	  fView,SLOT(removeAlbum(const char *, const char *)));
  connect(fView,SIGNAL(editTracks(QPtrList<KZenTrackItem> &)),
	  this,SLOT(editTracks(QPtrList<KZenTrackItem> &)));
  connect(fView,SIGNAL(deleteTracks(QPtrList<KZenTrackItem> &, KZenPlayList *)),
	  this,SLOT(deleteTracks(QPtrList<KZenTrackItem> &, KZenPlayList *)));
  connect(fView,SIGNAL(destroyTracks(QPtrList<KZenTrackItem> &)),
	  this,SLOT(destroyTracks(QPtrList<KZenTrackItem> &)));
  
  fTrackList = new QPtrList<KZenTrack>();
  fTrackList->setAutoDelete(true);

  fSearchTrackList = new QPtrList<KZenTrack>();
  fSearchTrackList->setAutoDelete(false);

  fArtistList = new QPtrList<KZenArtist>();
  fArtistList->setAutoDelete(true);

  fGenreList = new QPtrList<KZenGenre>();
  fGenreList->setAutoDelete(true);

  fPlayLists = new QPtrList<KZenPlayList>();
  fPlayLists->setAutoDelete(true);

  fSmartPlayLists = new QPtrList<KZenSmartPlayList>();
  fSmartPlayLists->setAutoDelete(true);

  fEditor = new KZenTagEditor(this, "editor");

  fSmartPlayListDialog = new KZenSmartPlayListDialog(this, "smart");

  fProgressDialog = new KZenProgressDialog(this, "progress");

  connect(fProgressDialog,SIGNAL(cancelRequest(bool)),
	  this,SLOT(cancelRequest(bool)));

  fCommandThread = new KZenCommandThread(this);
  fCommandThread->start();
  
  fSearchDialog = 0;
  
  fConfigDialog = 0;

  fPlayTimer = new QTimer(this);
  connect(fPlayTimer,SIGNAL(timeout()),this,SLOT(playTimeOut()));

  fBusy = false;
}

KZenExplorer::~KZenExplorer()
{
  fPlayLists->clear();
  delete fPlayLists;

  fSmartPlayLists->clear();
  delete fSmartPlayLists;

  fArtistList->clear();
  delete fArtistList;

  fGenreList->clear();
  delete fGenreList;

  fSearchTrackList->clear();
  delete fSearchTrackList;
  
  fTrackList->clear();
  delete fTrackList;

  fCommandThread->stop();
  fCommandThread->wait();
  delete fCommandThread;
}
 
void KZenExplorer::saveGlobalProperties(KConfig * cfg)
{
  kdDebug() << "saveGlobalProperties" << endl;

  cfg->setGroup("Geometry");
  cfg->writeEntry("Size",size());
  cfg->writeEntry("Navi",fView->getNaviWidth());

  cfg->setGroup("ToolBar");
  cfg->writeEntry("IconSize",toolBar()->iconSize());
  cfg->writeEntry("IconText",toolBar()->iconText());

  KZenConfig::get()->saveConfig(cfg);
}

void KZenExplorer::readGlobalProperties(KConfig * cfg)
{
  kdDebug() << "readGlobalProperties" << endl;

  if (cfg->hasGroup("Geometry")) {
    cfg->setGroup("Geometry");
    resize(cfg->readPropertyEntry("Size",QSize(800,600)).toSize());
    fView->setNaviWidth(cfg->readPropertyEntry("Navi",200).toInt());
  }

  if (cfg->hasGroup("ToolBar")) {
    cfg->setGroup("ToolBar");
    toolBar()->setIconText((KToolBar::IconText)cfg->readPropertyEntry("IconText",
								      KToolBar::IconOnly).toInt(),
			   true);
    toolBar()->setIconSize(cfg->readPropertyEntry("IconSize",16).toInt(),
			   true);
  }

  KZenConfig::get()->readConfig(cfg);
}

bool KZenExplorer::readTrackDBFile(const QString & fn)
{
  kdDebug() << "readTrackDBFile" << endl;

  fSmartPlayLists->clear();

  if (!KStandardDirs::exists(fn)) {
    kdDebug() << "~readTrackDBFile --> no database" << endl;
    
    fDatabaseID = 0;
    return true;
  }
  
  uint count;
  uint gversion;
  uint version;
  KZenTrack * track;
  QFile file(fn);
  if (file.open(IO_ReadOnly)) {
    QDataStream stream(&file);

    stream >> gversion;

    if (gversion==1) {

      stream >> fCategories;

      stream >> version;

      // KZenTrack version 1
      if (version==1) {

	stream >> count;
	
	uint id,rating;
	QBitArray categories;
	for (uint t=0;t<count;t++) {
	  stream >> id;
	  stream >> rating;
	  stream >> categories;

	  for (uint i=0;i<fTrackList->count();i++) {
	    track = fTrackList->at(i);
	    if (track->getTrackID()==id) {
	      track->setRating(rating);
	      track->setCategories(categories);
	    }
	  }
	}
      }

      // KZenTrack version 2
      if (version==2) {

	stream >> count;
	
	uint id,rating,compilation;
	QBitArray categories;
	for (uint t=0;t<count;t++) {
	  stream >> id;
	  stream >> rating;
	  stream >> categories;
	  stream >> compilation;

	  for (uint i=0;i<fTrackList->count();i++) {
	    track = fTrackList->at(i);
	    if (track->getTrackID()==id) {
	      track->setRating(rating);
	      track->setCategories(categories);
	      track->setIsCompilation((bool)compilation);
	    }
	  }
	}
      }
      
      stream >> version;

      // KZenSmartPlayList version 1
      if (version==1) {

	stream >> count;

	KZenSmartPlayList * spl;
	QString name;
	for (uint p=0;p<count;p++) {
	  stream >> name;
	  spl = new KZenSmartPlayList(name);
	  fSmartPlayLists->append(spl);
	  stream >> *spl;
	}
      }
    }
    
    file.close();
  }

  kdDebug() << "~readTrackDBFile --> OK" << endl;
  
  return true;
}

bool KZenExplorer::readTrackDB()
{
  kdDebug() << "readTrackDB" << endl;

  QString FileName;
  
  KCmdLineArgs * args = KCmdLineArgs::parsedArgs();
  FileName = args->getOption("db");

  if (FileName.isNull()) {
    FileName.sprintf("%skzenexplorer.db",
		     (const char*)KGlobal::dirs()->saveLocation("tmp"));

    njb_datafile_t * df;
    u_int64_t size = 0;
    NJB_Reset_Get_Datafile_Tag(fNJB);
    while ((df=NJB_Get_Datafile_Tag(fNJB))) {
      QString name = df->filename;
      if (name=="kzenexplorer.db" ) {
	fDatabaseID = df->dfid;
	size = df->filesize;
      }
      NJB_Datafile_Destroy(df);
    }
    NJB_Error_Dump(fNJB, stderr);
    
    if (size) {
      if (NJB_Get_File(fNJB, fDatabaseID, (u_int32_t) size, FileName, progressReadDB, NULL)==-1) {
	NJB_Error_Dump(fNJB, stderr);
	return true;
      }
    } else {
      return true;
    }
  } else {
    njb_datafile_t * df;
    u_int64_t size = 0;
    NJB_Reset_Get_Datafile_Tag(fNJB);
    while ((df=NJB_Get_Datafile_Tag(fNJB))) {
      QString name = df->filename;
      if (name=="kzenexplorer.db" ) {
	fDatabaseID = df->dfid;
	size = df->filesize;
	if (NJB_Delete_Datafile(fNJB, fDatabaseID)==-1) {
	  NJB_Error_Dump(fNJB, stderr);
	}
	NJB_Datafile_Destroy(df);
      }
    }
    NJB_Error_Dump(fNJB, stderr);

    fDatabaseID = 0;
  }

  readTrackDBFile(FileName);

  kdDebug() << "~readTrackDB --> OK" << endl;
  
  return true;
}

void KZenExplorer::writeTrackDBFile(const QString & fn)
{
  kdDebug() << "writeTrackDBFile" << endl;

  uint count = 0;
  KZenTrack * track;
  for (uint i=0;i<fTrackList->count();i++) {
    track = fTrackList->at(i);
    if (track->needsWrite()) {
      count++;
    }
  }

  kdDebug() << "file=" << fn << endl;

  QFile file(fn);
  if (file.open(IO_WriteOnly)) {
    QDataStream stream(&file);

    stream << version();

    stream << fCategories;

    stream << KZenTrack::version();
    stream << count;
    
    for (uint i=0;i<fTrackList->count();i++) {
      track = fTrackList->at(i);
      if (track->needsWrite()) {

	track->compress();

	stream << track->getTrackID();
	stream << track->getRating();
	stream << track->getCategories();
	stream << (uint)track->getIsCompilation();
      }
    }
    
    stream << KZenSmartPlayList::version();
    stream << fSmartPlayLists->count();
    
    KZenSmartPlayList * spl;
    for (uint i=0;i<fSmartPlayLists->count();i++) {
      spl = fSmartPlayLists->at(i);
      stream << spl->getName();
      stream << *spl;
    }
    
    file.close();
  }
  
  kdDebug() << "~writeTrackDBFile" << endl;
}

void KZenExplorer::writeTrackDB()
{
  kdDebug() << "writeTrackDB" << endl;

  QString FileName;
  FileName.sprintf("%skzenexplorer.db",
		   (const char*)KGlobal::dirs()->saveLocation("tmp"));
  
  writeTrackDBFile(FileName);

  if (fDatabaseID!=0) {
    if (NJB_Delete_Datafile(fNJB, fDatabaseID)==-1) {
      NJB_Error_Dump(fNJB, stderr);
    }
  }
  if (NJB_Send_File(fNJB, FileName, NULL, NULL, progressWriteDB, NULL, &fDatabaseID)==-1) {
    NJB_Error_Dump(fNJB,stderr);
  }
  
  kdDebug() << "~writeTrackDB" << endl;
}

void KZenExplorer::setupActions()
{
  kdDebug() << "setupActions" << endl;

  fPlayTracks = new KAction(i18n("Play Tracks"), "play_track.png", KShortcut("CTRL+p"),
			    fView, SLOT(playTracks()),
			    actionCollection(), "play_tracks");
  fQueueTracks = new KAction(i18n("Queue Tracks"), "queue_track.png", KShortcut("CTRL+q"),
			     fView, SLOT(queueTracks()),
			     actionCollection(), "queue_tracks");
  fStopPlay = new KAction(i18n("Stop Playback"), "stop_play.png", KShortcut("CTRL+s"),
			  this, SLOT(stopPlay()),
			  actionCollection(), "stop_play");
  fEditTracks = new KAction(i18n("Edit Tracks"), "edit_track.png", KShortcut("CTRL+e"),
			    fView, SLOT(editTracks()),
			    actionCollection(), "edit_tracks");
  fSaveTracks = new KAction(i18n("Save Tracks"), "save_track.png", KShortcut("CTRL+ALT+s"),
			    fView, SLOT(saveTracks()),
			    actionCollection(), "save_tracks");
  fDeleteTracks = new KAction(i18n("Delete Tracks"), "delete_track.png", KShortcut("CTRL+ALT+d"),
			      fView, SLOT(deleteTracks()),
			      actionCollection(), "delete_tracks");
  fDestroyTracks = new KAction(i18n("Destroy Tracks"), "destroy.png", KShortcut("CTRL+ALT+x"),
			       fView, SLOT(destroyTracks()),
			       actionCollection(), "destroy_tracks");

  fNewPlayList = new KAction(i18n("New Playlist"), "new_playlist.png", KShortcut("CTRL+n"), 
			     this, SLOT(newPlayList()),
			     actionCollection(), "new_playlist");
  fNewSmartPlayList = new KAction(i18n("New Smart Playlist"), "new_smartplaylist.png", KShortcut("CTRL+SHIFT+n"), 
				  this, SLOT(newSmartPlayList()),
				  actionCollection(), "new_smart_playlist");
  fClearSearch = new KAction(i18n("Clear Search"), "clear_search.png", KShortcut(), 
			     this, SLOT(clearSearch()),
			     actionCollection(), "clear_search");
  fAdvancedSearch = new KAction(i18n("Advanced Search"), "advanced_search.png", KShortcut(), 
				this, SLOT(advancedSearch()),
				actionCollection(), "advanced_search");

  QSignalMapper * duplicatesMapper = new QSignalMapper(this);
  connect(duplicatesMapper, SIGNAL(mapped(int)), this, SLOT(findDuplicates(int)));
  
  fFindDuplicatesTitle = new KAction(i18n("Title"), "duplicate.png", KShortcut(), 
				     duplicatesMapper, SLOT(map()),
				     actionCollection(), "find_duplicates_title");
  duplicatesMapper->setMapping(fFindDuplicatesTitle, 1);

  fFindDuplicatesTitleArtist = new KAction(i18n("Title && Artist"), "duplicate.png", KShortcut(), 
					   duplicatesMapper, SLOT(map()),
					   actionCollection(), "find_duplicates_title_artist");
  duplicatesMapper->setMapping(fFindDuplicatesTitleArtist, 2);

  fFindDuplicatesTitleArtistAlbum = new KAction(i18n("Title, Artist && Album"), "duplicate.png", KShortcut(), 
						duplicatesMapper, SLOT(map()),
						actionCollection(), "find_duplicates_title_artist_album");
  duplicatesMapper->setMapping(fFindDuplicatesTitleArtistAlbum, 3);

  fRenamePlayList = new KAction(i18n("Rename"), "rename.png", KShortcut(), 
				fView, SLOT(renamePlayList()),
				actionCollection(), "rename_playlist");
  fEditSmartPlayList = new KAction(i18n("Edit"), "edit_smartplaylist.png", KShortcut(), 
				   fView, SLOT(editSmartPlayList()),
				   actionCollection(), "edit_smart_playlist");
  fShufflePlayList = new KAction(i18n("Shuffle"), "shuffle.png", KShortcut(), 
				 fView, SLOT(shufflePlayList()),
				 actionCollection(), "shuffle_playlist");
  fUpdatePlayList = new KAction(i18n("Update Playlist"), "update.png", KShortcut(),
				fView, SLOT(updatePlayList()),
				actionCollection(), "update_playlist");
  fUpdateAllPlayLists = new KAction(i18n("Update all Playlists"), "updateall.png", KShortcut(), 
				    this, SLOT(updateAllPlayLists()),
				    actionCollection(), "update_all_playlists");
  fDeletePlayList = new KAction(i18n("Delete"), "delete_playlist.png", KShortcut(), 
				fView, SLOT(deleteNaviItem()),
				actionCollection(), "delete_playlist");
  fDestroy = new KAction(i18n("Destroy"), "destroy.png", KShortcut("CTRL+SHIFT+x"), 
			 fView, SLOT(destroyNaviItem()),
			 actionCollection(), "destroy");
  fIsCompilation = new KAction(i18n("Mark as Compilation"), "compilation.png", KShortcut(), 
			       fView, SLOT(isCompilationClicked()),
			       actionCollection(), "iscompilation");
  fShowDetails = new KAction(i18n("Show Details"), "details.png", KShortcut(), 
			       fView, SLOT(showDetails()),
			       actionCollection(), "showdetails");
  fSelectAll = new KAction(i18n("Select All"), "select_all.png", KShortcut(), 
			   fView, SLOT(selectAll()),
			   actionCollection(), "select_all");
  fPlayListFromSelection = new KAction(i18n("New Playlist From Selection"), 
				       "new_playlist_selection.png", 
				       KShortcut(),
				       this, SLOT(newPlayListSelection()),
				       actionCollection(), "new_playlist_selection");
  fExportDB = new KAction(i18n("Export Track DB"), "exportdb.png", KShortcut(), 
			  this, SLOT(exportDB()),
			  actionCollection(), "exportdb");
  fBrowseLocalLib = new KAction(i18n("Browse Library"), "local_lib.png", KShortcut("CTRL+b"), 
				this, SLOT(browseLocalLib()),
				actionCollection(), "local_lib");
  fConfigure = new KAction(i18n("Configure"), "configure", KShortcut(),
			   this, SLOT(showConfigDialog()),
			   actionCollection(), "kzenexplorer_configure");
  fQuit = new KAction(i18n("Quit"), "quit.png", KShortcut(), 
		      kapp, SLOT(quit()),
		      actionCollection(), "quit");
  
  fSynchronize = new KAction(i18n("Synchronize"), "synchronize.png", KShortcut(), 
			     this, SLOT(synchronize()),
			     actionCollection(), "synchronize");
  
  fView->setupActions();

  setUpdateEnabled(false);
  setUpdateAllEnabled(false);
  setTrackOpsEnabled(false);
  setPlayListOpsEnabled(false);
  
  createGUI();
}

bool KZenExplorer::queryExit()
{
  kdDebug() << "queryExit" << endl;

  setBusy(true);

  updateAllPlayLists();
  
  writeTrackDB();
  
  disconnectNJB();

  saveGlobalProperties(KGlobal::config());

  return true;
}

bool KZenExplorer::connectNJB()
{
  if (njb_connected) return true;

  kdDebug() << "connectNJB" << endl;

  if (fSplash) fSplash->message(i18n("Looking for jukeboxes ..."));

  int n;
  if (NJB_Discover(njbs, 0, &n) == -1 || n==0) {
    if (fSplash) {
      fSplash->message(i18n("Could not find any jukeboxes"));
      fSplash->wait();
    }
    return false;
  }

  fNJB = &njbs[0];

  if (fSplash) fSplash->message(i18n("Trying to open jukebox"));
  
  if (NJB_Open(fNJB) == -1) {
    if (fSplash) {
      fSplash->message(i18n("Could not open "));
      fSplash->wait();
    }
    NJB_Error_Dump(fNJB,stderr);
    return false;
  }

  if (fSplash) fSplash->message(i18n("Trying to capture jukebox"));

  if (NJB_Capture(fNJB) == -1) {
    if (fSplash) {
      fSplash->message(i18n("Could not capture jukebox"));
      fSplash->wait();
    }
    NJB_Error_Dump(fNJB,stderr);
    NJB_Close(fNJB);
    return false;
  }
  
  Jukebox::get()->setNJB(fNJB);

  if (fSplash) fSplash->message(i18n("Getting track list from jukebox"));
  
  if (!getTrackList()) {
    fSplash->message(i18n("Error getting track list"));
    fSplash->wait();
    return false;
  }
  
  if (fSplash) fSplash->message(i18n("Reading track database from jukebox"));
  
  if (!readTrackDB()) {
    fSplash->message(i18n("Error reading track database"));
    fSplash->wait();
    return false;
  }
  
  fArtistList->clear();

  fVariousArtists = new KZenArtist(i18n("Various Artists"));
  fArtistList->append(fVariousArtists);
 
  KZenTrack * track;
  KZenArtist * temp;
  KZenGenre * gtemp;
  QString artist;
  QString genre;
  for (uint i=0;i<fTrackList->count();i++) {
    track = fTrackList->at(i);
    artist = track->getArtist();

    if (track->getIsCompilation()) {
      temp = fVariousArtists;
    } else {
      temp = 0;
      for (uint a=0;a<fArtistList->count();a++) {
	temp = fArtistList->at(a);
	if (artist==temp->getName()) break;
	temp = 0;
      }
    }
    
    if (!temp) {
      temp = new KZenArtist(artist);
      fArtistList->append(temp);
    }
    track->setZenArtist(temp);
    temp->addTrack(track);
    
    genre = track->getGenre();
    if (genre.isEmpty()) genre = i18n("Unknown");
    
    gtemp = 0;
    for (uint a=0;a<fGenreList->count();a++) {
      gtemp = fGenreList->at(a);
      if (genre==gtemp->getName()) break;
      gtemp = 0;
    }
    if (!gtemp) {
      gtemp = new KZenGenre(genre);
      fGenreList->append(gtemp);
    }
    track->setZenGenre(gtemp);
    gtemp->addTrack(track);
  }
  
  if (fSplash) fSplash->message(i18n("Getting playlists from jukebox"));

  if (!getPlayLists()) {
    fSplash->message(i18n("Error reading playlists"));
    fSplash->wait();
    return false;
  }

  fView->displayPlayLists(fPlayLists, fSmartPlayLists);
  fView->displayArtistList(fArtistList);
  fView->displayGenreList(fGenreList);

  fView->naviTabClicked(KZen::TabAll);

  fEditor->setup(fTrackList);

  refreshSmartPlayLists();

  fCommandThread->setNJB(fNJB);

  fCommandThread->updateUsage();

  fCommandThread->getOwnerString();

  fCommandThread->getClock();

  njb_connected = true;
  return true;
}

void KZenExplorer::disconnectNJB()
{
  if (!njb_connected) return;

  kdDebug() << "disconnectNJB" << endl;

  NJB_Stop_Play(fNJB);
  
  NJB_Release(fNJB);
  NJB_Close(fNJB);
  
  njb_connected = false;
  fNJB = 0;
}

KZenConfigDialog * KZenExplorer::configDialog()
{
  if (!fConfigDialog) fConfigDialog = new KZenConfigDialog(this);
  return fConfigDialog;
}

bool KZenExplorer::getTrackList()
{
  NJB_Reset_Get_Track_Tag(fNJB);
  
  int n = 0;

  fTrackList->clear();
  njb_songid_t *songtag;
  while ((songtag = NJB_Get_Track_Tag(fNJB))) {
    KZenTrack * track = new KZenTrack(songtag);
    fTrackList->append(track);
    NJB_Songid_Destroy(songtag);

    if (fSplash) {
      QString m = i18n("Getting track list from jukebox");
      m += " ";
      for (int i=0;i<n;i++) m += ".";
      fSplash->message(m);
      n++;
      if (n==6) n = 0;
    }
  }

  if (NJB_Error_Pending(fNJB)) {
    NJB_Error_Dump(fNJB,stderr);
    return false;
  }
  return true;
}

bool KZenExplorer::getPlayLists()
{
  NJB_Reset_Get_Playlist(fNJB);
  
  int n = 0;

  fPlayLists->clear();
  njb_playlist_t *playlist;
  KZenSmartPlayList * spl = 0;
  while ((playlist = NJB_Get_Playlist(fNJB))) {
    for (uint p=0;p<fSmartPlayLists->count();p++) {
      spl = fSmartPlayLists->at(p);
      if (spl->getName()==playlist->name) break;
      spl = 0;
    }

    if (spl) {
      spl->setModified(false);
      spl->init(playlist,fTrackList);
    } else {
      KZenPlayList * pl = new KZenPlayList(playlist,fTrackList);
      pl->setModified(false);
      fPlayLists->append(pl);
    }

    if (fSplash) {
      QString m = i18n("Getting playlists from jukebox");
      m += " ";
      for (int i=0;i<n;i++) m += ".";
      fSplash->message(m);
      n++;
      if (n==6) n = 0;
    }
  }
  
  if (NJB_Error_Pending(fNJB)) {
    NJB_Error_Dump(fNJB,stderr);
    return false;
  }
  return true;
}

void KZenExplorer::selectionChangedNavi(QListViewItem * i)
{
  KZenItem * item = (KZenItem *)i;
  
  switch (item->getItemType()) {
  case KZenItem::ItemTypeArtist:
    {
      KZenArtistItem * artistitem = (KZenArtistItem *)item;
      KZenArtist * artist = artistitem->getArtist();
      if (artist->getName()!=i18n("Various Artists")) {
	fView->displayTrackList(artist->getTrackList(), ViewModeArtist);
      } else {
	fView->displayTrackList(artist->getTrackList(), ViewModeAll);
      }

      setDestroyEnabled(true);
      setUpdateEnabled(false);
      setTrackOpsEnabled(false);
      setPlayListOpsEnabled(false);
      setSmartPlayListOpsEnabled(false);
    }
    break;
  case KZenItem::ItemTypeAlbum:
    {
      KZenAlbumItem * albumitem = (KZenAlbumItem *)item;
      KZenAlbum * album = albumitem->getAlbum();
      if (albumitem->getIsCompilation()) {
	  fView->displayTrackList(album->getTrackList(), ViewModeAll);
      } else {
	fView->displayTrackList(album->getTrackList(), ViewModeAlbum);
      }

      setDestroyEnabled(true);
      setUpdateEnabled(false);
      setTrackOpsEnabled(false);
      setPlayListOpsEnabled(false);
      setSmartPlayListOpsEnabled(false);
    }
    break;
  case KZenItem::ItemTypeAlbumTL:
    {
      KZenAlbumItemTL * albumitem = (KZenAlbumItemTL *)item;
      if (albumitem->getIsCompilation()) {
	fView->displayTrackList(albumitem->getTrackList(), ViewModeAll);
      } else {
	fView->displayTrackList(albumitem->getTrackList(), ViewModeAlbum);
      }

      setDestroyEnabled(true);
      setUpdateEnabled(false);
      setTrackOpsEnabled(false);
      setPlayListOpsEnabled(false);
      setSmartPlayListOpsEnabled(false);
    }
    break;
  case KZenItem::ItemTypeGenre:
    {
      KZenGenreItem * genreitem = (KZenGenreItem *)item;
      KZenGenre * genre = genreitem->getGenre();
      fView->displayTrackList(genre->getTrackList(), ViewModeAll);

      setDestroyEnabled(true);
      setUpdateEnabled(false);
      setTrackOpsEnabled(false);
      setPlayListOpsEnabled(false);
      setSmartPlayListOpsEnabled(false);
    }
    break;
  case KZenItem::ItemTypePlayList:
    {
      KZenPlayListItem * playlistitem = (KZenPlayListItem *)item;
      KZenPlayList * playlist = playlistitem->getPlayList();
      if (playlist->isModified()) {
	setUpdateEnabled(true);
      } else {
	setUpdateEnabled(false);
      }
      fView->displayTrackListPL(playlist);

      setDestroyEnabled(true);
      setTrackOpsEnabled(false);
      setPlayListOpsEnabled(true);
      setSmartPlayListOpsEnabled(false);
    }
    break;
  case KZenItem::ItemTypeSmartPlayList:
    {
      KZenSmartPlayListItem * playlistitem = (KZenSmartPlayListItem *)item;
      KZenSmartPlayList * playlist = (KZenSmartPlayList*)playlistitem->getPlayList();
      if (playlist->isModified()) {
	setUpdateEnabled(true);
      } else {
	setUpdateEnabled(false);
      }
      playlist->refresh(fTrackList);
      fView->displayTrackListPL(playlist);

      setDestroyEnabled(true);
      setTrackOpsEnabled(false);
      setPlayListOpsEnabled(true);
      setSmartPlayListOpsEnabled(true);
    }
    break;
  default:
    {
      fView->displayTrackList(fTrackList, ViewModeAll);

      setDestroyEnabled(true);
      setUpdateEnabled(false);
      setTrackOpsEnabled(false);
      setPlayListOpsEnabled(false);
      setSmartPlayListOpsEnabled(false);
    }
    break;
  }
  
  fView->setSearchText("");
}

KZenTrack * KZenExplorer::findTrack(const KURL & url)
{
  TagLib::FileRef f(url.path());
  if (f.isNull() || !f.tag()) return 0;
  
  TagLib::Tag * tag = f.tag();
  QString artist = TStringToQString(tag->artist()).stripWhiteSpace();
  QString album = TStringToQString(tag->album()).stripWhiteSpace();
  QString title = TStringToQString(tag->title()).stripWhiteSpace();
  
  KZenTrack * track = 0;
  for (uint i=0;i<fTrackList->count();i++) {
    track = fTrackList->at(i);
    if (track->getArtist().stripWhiteSpace()==artist &&
	track->getAlbum().stripWhiteSpace()==album &&
	track->getTitle().stripWhiteSpace()==title ) {
      break;
    }
    track = 0;
  }
  
  return track;
}

void KZenExplorer::filesDropped(const QStringList & filenames, KZenPlayList * PL)
{
  if (!connectNJB()) return;

  fCommandThread->lock();

  for (uint i=0;i<filenames.count();i++) {
    KURL url(filenames[i]);

    kdDebug() << "dropped: " << url << endl;

    if (KZenMediaFiles::isMediaFile(url)) {
      KZenTrack * newTrack = 0;
      if (KZenMediaFiles::isMP3(url)) {
	if (KZenConfig::get()->smartImport()) newTrack = findTrack(url);
	if (!newTrack) {
	  newTrack = KZenMediaFiles::createTrackMP3(url);
	  if (newTrack) {
	    fProgressDialog->addTrack(newTrack, true);
	    fCommandThread->addTrack(newTrack,url,PL);
	  }
	}
      }
    } else if (KZenMediaFiles::isDirectory(url)) {
      importDirectory(url,PL);
    } else if (KZenMediaFiles::isPlaylistFile(url)) {
      importPlayList(url);
    }
  }

  fCommandThread->updateUsage();

  fCommandThread->unlock();
}

void KZenExplorer::importDirectory(const KURL & url, KZenPlayList * PL)
{
  kdDebug() << "import directory: " << url.path() << endl;

  QDir d(url.path());

  const QFileInfoList *list = d.entryInfoList();
  QFileInfoListIterator it( *list );
  QFileInfo *fi;

  while ((fi = it.current())) {
    if (fi->isDir() && fi->isExecutable() && fi->isReadable()) {
      if (fi->fileName()!="." && fi->fileName()!="..") {
	KURL newurl = url;
	newurl.addPath(fi->fileName().latin1());
	importDirectory(newurl, PL);
      }
    } else if(fi->isReadable()){
      KURL newurl;
      newurl.setPath(fi->absFilePath());

      kdDebug() << "import: " << newurl.path() << endl;

      KZenTrack * newTrack = 0;
      if (KZenMediaFiles::isMP3(newurl)) {
	if (KZenConfig::get()->smartImport()) newTrack = findTrack(newurl);
	if (!newTrack) {
	  newTrack = KZenMediaFiles::createTrackMP3(newurl);
	  if (newTrack) {
	    fProgressDialog->addTrack(newTrack, true);
	    fCommandThread->addTrack(newTrack,newurl,PL);
	  }
	}
      }
    }
    ++it;
  }
}

void KZenExplorer::importPlayList(const KURL & url)
{
  kdDebug() << "import playlist: " << url.path() << endl;

  QStringList files;
  KURL u;

  QFile file(url.path());
  if (file.open(IO_ReadOnly)) {
    QTextStream stream(&file);
    QString line;
    int i = 1;
    while (!stream.atEnd()) {
      line = stream.readLine().stripWhiteSpace();
      if (!line.startsWith("#EXT")) {
	printf( "%3d: %s\n", i++, line.latin1() );
	u.setPath(line);
	if (KZenMediaFiles::isMP3(u)) {
	  files += line;
	}
      }
    }
    file.close();
  }

  if (files.size()==0) return;

  QString name = url.filename();
  if (name.endsWith(".m3u",false)) name.remove(name.findRev(".m3u",-1,false),4);

  bool ok;
  name = KInputDialog::getText(i18n("Import Playlist"),
			       i18n("Please enter the name of the playlist"),
			       name,&ok);
  if (ok) {
    name.stripWhiteSpace();

    KZenPlayList * tpl = 0;
    for (uint i=0;i<fPlayLists->count();i++) {
      tpl = fPlayLists->at(i);
      if (name==tpl->getName()) {
	return;
      }
    }

    KZenSmartPlayList * spl = 0;
    for (uint i=0;i<fSmartPlayLists->count();i++) {
      spl = fSmartPlayLists->at(i);
      if (name==spl->getName()) {
	return;
      }
    }

    KZenPlayList * pl = new KZenPlayList(name);    
    fView->naviTabClicked(KZen::TabPlayLists);

    fCommandThread->newPlayList(pl);

    for (QStringList::Iterator it = files.begin(); it != files.end(); ++it) {
      u.setPath(*it);
      
      KZenTrack * newTrack = 0;
      if (KZenMediaFiles::isMP3(u)) {
	if (KZenConfig::get()->smartImport()) newTrack = findTrack(u);
	if (!newTrack) {
	  newTrack = KZenMediaFiles::createTrackMP3(u);
	  if (newTrack) {
	    fProgressDialog->addTrack(newTrack, true);
	    fCommandThread->addTrack(newTrack,u,pl);
	  }
	} else {
	  pl->addTrack(newTrack);
	}
      }
    }
    
    addPlayList(pl);
    getView()->addPlayList(pl);
    emitSelectNaviItem(pl->getItem());
  }
}

void KZenExplorer::customEvent(QCustomEvent * e)
{
  if (e->type() == 12345) {
    KZenProgressEvent * p = (KZenProgressEvent*) e;
    fProgressDialog->refresh(p->getProgress());
  }
  if (e->type() == 12346) {
    KZenItemEvent * p = (KZenItemEvent*) e;
    fView->addTrack(p->getTrack());
  }
}

void KZenExplorer::cancelRequest(bool t)
{
  fCommandThread->setCancelRequest(t);
}

void KZenExplorer::tracksDropped(const QStringList & list, KZenPlayList * PL)
{
  KZenTrack * track;
  
  for (uint i=0;i<list.count();i++) {
    uint ID = list[i].toInt();
    
    for (uint i=0;i<fTrackList->count();i++) {
      track = fTrackList->at(i);
      if (track->getTrackID()==ID) {
	PL->addTrack(track);
      }
    }
  }
}

void KZenExplorer::playTracks(QPtrList<KZenTrackItem> & tracks)
{
  KZenTrack * track;
  for (uint i=0;i<tracks.count();i++) {
    track = tracks.at(i)->getTrack();

    if (i==0) {
      fCommandThread->playTrack(track);
    } else {
      fCommandThread->queueTrack(track);
    }
  }

//   fPlayTimer->start(1000);
}

void KZenExplorer::queueTracks(QPtrList<KZenTrackItem> & tracks)
{
  KZenTrack * track;
  for (uint i=0;i<tracks.count();i++) {
    track = tracks.at(i)->getTrack();

    if (i==0 && !fCommandThread->isPlaying()) {
      fCommandThread->playTrack(track);
    } else {
      fCommandThread->queueTrack(track);
    }
  }
}

void KZenExplorer::stopPlay()
{
  fCommandThread->stopPlay();
}

void KZenExplorer::editTracks(QPtrList<KZenTrackItem> & tracks)
{
  fEditor->setTrackList(tracks);
  
  if (fEditor->exec()==QDialog::Accepted) {

    KZenTrack * track;
    for (uint i=0;i<tracks.count();i++) {
      track = tracks.at(i)->getTrack();
	
      fCommandThread->updateTrack(track);
    }
  }
}

void KZenExplorer::saveTracks(QPtrList<KZenTrackItem> & tracks)
{
  if (tracks.count()==0) return;

  QString dir = KFileDialog::getExistingDirectory(KZenConfig::get()->localLibrary().path());
  
  if (!dir.isEmpty()) {
    KURL url;
    url.setPath(dir);
    
    KZenTrack * track;
    for (uint i=0;i<tracks.count();i++) {
      track = tracks.at(i)->getTrack();
      
      fProgressDialog->addTrack(track, false);
      fCommandThread->saveTrack(track,url);
    }
  }
}

void KZenExplorer::deleteTracks(QPtrList<KZenTrackItem> & tracks, KZenPlayList * pl)
{
  KZenTrack * track;
  for (uint i=0;i<tracks.count();i++) {
    track = tracks.at(i)->getTrack();
    if (pl->getTrackList()->remove(track)) pl->setModified(true);

    emit removeTrackItem(tracks.at(i));
  }
}

void KZenExplorer::emitRemoveTrackItem(KZenTrackItem * i)
{
  emit removeTrackItem(i);
}

void KZenExplorer::emitRemoveArtist(const char * artist)
{
  emit removeArtist(artist);
}

void KZenExplorer::emitRemoveGenre(const char * genre)
{
  emit removeGenre(genre);
}

void KZenExplorer::emitRemoveAlbum(const char * artist, const char *album)
{
  emit removeAlbum(artist,album);
}

void KZenExplorer::emitRemoveNaviItem(KZenItem * item)
{
  emit removeNaviItem(item);
}

void KZenExplorer::emitSelectNaviItem(KZenItem * item)
{
  emit selectNaviItem(item);
}

void KZenExplorer::destroyTracks(QPtrList<KZenTrackItem> & tracks)
{ 
  int ret = KMessageBox::questionYesNo(this, 
				       i18n("Do you really want to destroy the selected tracks"),
				       i18n("Destroy Tracks - KZenExplorer"));
  if (ret!=KMessageBox::Yes) return;

  fCommandThread->lock();
  
  KZenTrack * track;
  for (uint i=0;i<tracks.count();i++) {
    track = tracks.at(i)->getTrack();
    fCommandThread->destroyTrack(track);
  }

  fCommandThread->updateUsage();

  fCommandThread->unlock();
}

void KZenExplorer::updatePlayList(KZenPlayList * PL, QStringList & list)
{
  KZenTrack * Ztrack;
  uint pos = 0;

  for (QStringList::Iterator it = list.begin();it!=list.end();++it) {
    QString trackID = *it;
    uint ID = trackID.toInt();
    
    for (uint i=0;i<PL->getTrackList()->count();i++) {
      Ztrack = PL->getTrackList()->at(i);
      if (Ztrack->getTrackID()==ID) {
	Ztrack = PL->getTrackList()->take(i);
	PL->getTrackList()->insert(pos, Ztrack);

	if (pos!=i) PL->setModified(true);
      }
    }
    pos++;
  }
}

void KZenExplorer::addPlayList(KZenPlayList * pl)
{
  fPlayLists->append(pl);
}

void KZenExplorer::addSmartPlayList(KZenSmartPlayList * pl)
{
  fSmartPlayLists->append(pl);
}

void KZenExplorer::newPlayList()
{
  if (!connectNJB()) return;

  bool ok;
  QString name = KInputDialog::getText(i18n("New Playlist"),
				       i18n("Please enter the name of the new playlist"),
				       i18n("noname"),&ok);
  if (ok) {
    name.stripWhiteSpace();

    KZenPlayList * pl = 0;
    for (uint i=0;i<fPlayLists->count();i++) {
      pl = fPlayLists->at(i);
      if (name==pl->getName()) {
	return;
      }
    }

    KZenSmartPlayList * spl = 0;
    for (uint i=0;i<fSmartPlayLists->count();i++) {
      spl = fSmartPlayLists->at(i);
      if (name==spl->getName()) {
	return;
      }
    }

    fView->naviTabClicked(KZen::TabPlayLists);

    fCommandThread->newPlayList(name);
  }
}

void KZenExplorer::newSmartPlayList()
{
  if (!connectNJB()) return;

  KZenSmartPlayList * spl = new KZenSmartPlayList(i18n("New Smart Playlist"));
  fSmartPlayListDialog->setPlayList(spl);
  if (fSmartPlayListDialog->exec()==QDialog::Accepted) {
    bool dupName = false;
    QString name = spl->getName().stripWhiteSpace();
    spl->setName(name);

    KZenPlayList * pl = 0;
    for (uint i=0;i<fPlayLists->count();i++) {
      pl = fPlayLists->at(i);
      if (spl->getName()==pl->getName()) {
	dupName = true;
      } 
    }
    KZenSmartPlayList * tspl = 0;
    for (uint i=0;i<fSmartPlayLists->count();i++) {
      tspl = fSmartPlayLists->at(i);
      if (spl->getName()==tspl->getName()) {
	dupName = true;
      }
    }

    while (dupName || name.isEmpty()) {
      bool ok;
      QString name = KInputDialog::getText(i18n("New Smart Playlist - KZenExplorer"),
					   i18n("Please enter a different name for the new smart playlist"),
					   spl->getName(),&ok);
      if (ok) {
	name.stripWhiteSpace();
	dupName = false;
	for (uint i=0;i<fPlayLists->count();i++) {
	  pl = fPlayLists->at(i);
	  if (spl->getName()==pl->getName()) dupName = true; 
	}
	for (uint i=0;i<fSmartPlayLists->count();i++) {
	  tspl = fSmartPlayLists->at(i);
	  if (spl->getName()==tspl->getName()) dupName = true; 
	}
	if (!dupName) spl->setName(name);
      } else {
	delete spl;
	return;
      }
    }

    fView->naviTabClicked(KZen::TabPlayLists);

    fCommandThread->newSmartPlayList(spl);
  } else {
    delete spl;
  }
}

void KZenExplorer::renamePlayList(KZenPlayListItem * item)
{
  if (!connectNJB()) return;

  KZenPlayList * pl = item->getPlayList();

  bool ok;
  QString name = KInputDialog::getText(i18n("Rename Playlist - KZenExplorer"),
				       i18n("Please enter the new name of the playlist"),
				       pl->getName(),&ok);
  if (ok) {
    name.stripWhiteSpace();

    KZenPlayList * tpl = 0;
    for (uint i=0;i<fPlayLists->count();i++) {
      tpl = fPlayLists->at(i);
      if (name==tpl->getName()) {
	return;
      }
    }

    KZenSmartPlayList * spl = 0;
    for (uint i=0;i<fSmartPlayLists->count();i++) {
      spl = fSmartPlayLists->at(i);
      if (name==spl->getName()) {
	return;
      }
    }

    fCommandThread->renamePlayList(pl, name);
  }
}

void KZenExplorer::editSmartPlayList(KZenSmartPlayListItem * item)
{
  KZenSmartPlayList * ospl = (KZenSmartPlayList*)item->getPlayList();
  if (!ospl) return;

  KZenSmartPlayList * spl = new KZenSmartPlayList(ospl);

  fSmartPlayListDialog->setPlayList(spl);
  if (fSmartPlayListDialog->exec()==QDialog::Accepted) {
    bool dupName = false;
    QString name = spl->getName().stripWhiteSpace();
    spl->setName(name);
    
    KZenPlayList * pl = 0;
    KZenSmartPlayList * tspl = 0;
    if (ospl->getName()!=spl->getName()) {
      for (uint i=0;i<fPlayLists->count();i++) {
	pl = fPlayLists->at(i);
	if (spl->getName()==pl->getName()) dupName = true; 
      }
      for (uint i=0;i<fSmartPlayLists->count();i++) {
	tspl = fSmartPlayLists->at(i);
	if (spl->getName()==tspl->getName()) dupName = true; 
      }
    }
    
    while (dupName || name.isEmpty()) {
      bool ok;
      QString name = KInputDialog::getText(i18n("Edit Smart Playlist - KZenExplorer"),
					   i18n("Please enter a different name for this smart playlist"),
					   spl->getName(),&ok);
      if (ok) {
	name.stripWhiteSpace();
	dupName = false;
	for (uint i=0;i<fPlayLists->count();i++) {
	  pl = fPlayLists->at(i);
	  if (spl->getName()==pl->getName()) dupName = true; 
	}
	for (uint i=0;i<fSmartPlayLists->count();i++) {
	  tspl = fSmartPlayLists->at(i);
	  if (spl->getName()==tspl->getName()) dupName = true; 
	}
	if (!dupName) spl->setName(name);
      } else {
	spl->setName(ospl->getName());
	break;
      }
    }

    ospl->assign(spl);
  }

  ospl->refresh(fTrackList);
  item->listView()->setSelected(item,false); 
  item->listView()->setSelected(item,true); 
  
  delete spl;
}

void KZenExplorer::refreshSmartPlayLists(KZenTrack * track)
{
  kdDebug() << "refreshSmartPlayLists(KZenTrack * track)" << endl;

  KZenSmartPlayList * spl;
  for (uint i=0;i<fSmartPlayLists->count();i++) {
    spl = fSmartPlayLists->at(i);
    spl->refresh(track);
  }
}

void KZenExplorer::refreshSmartPlayLists()
{
  kdDebug() << "refreshSmartPlayLists()" << endl;

  KZenSmartPlayList * spl;
  for (uint i=0;i<fSmartPlayLists->count();i++) {
    spl = fSmartPlayLists->at(i);
    spl->refresh(fTrackList);
  }
}

void KZenExplorer::updateAllPlayLists()
{
  KZenPlayList * pl;
  for (uint i=0;i<fPlayLists->count();i++) {
    pl = fPlayLists->at(i);
    updatePlayList(pl);
  }

  KZenSmartPlayList * spl;
  for (uint i=0;i<fSmartPlayLists->count();i++) {
    spl = fSmartPlayLists->at(i);
    updatePlayList(spl);
  }
}

void KZenExplorer::updatePlayList(KZenPlayList * PL)
{
  if (!connectNJB()) return;

  if (!PL->isModified()) return;

  fCommandThread->updatePlayList(PL);
}

void KZenExplorer::setDestroyEnabled(bool e)
{
  fDestroy->setEnabled(e);
}

void KZenExplorer::setUpdateEnabled(bool e)
{
  fUpdatePlayList->setEnabled(e);
}

void KZenExplorer::setUpdateAllEnabled(bool e)
{
  fUpdateAllPlayLists->setEnabled(e);
}

void KZenExplorer::setTrackOpsEnabled(bool e)
{
  fPlayTracks->setEnabled(e);
  fQueueTracks->setEnabled(e);
  fEditTracks->setEnabled(e);
  fSaveTracks->setEnabled(e);
  fDeleteTracks->setEnabled(e);
  fDestroyTracks->setEnabled(e);
}

void KZenExplorer::setPlayListOpsEnabled(bool e)
{
  fRenamePlayList->setEnabled(e);
  fEditSmartPlayList->setEnabled(e);
  fShufflePlayList->setEnabled(e);
  fDeletePlayList->setEnabled(e);
}

void KZenExplorer::setSmartPlayListOpsEnabled(bool e)
{
  fEditSmartPlayList->setEnabled(e);
}

void KZenExplorer::deleteNaviItem(KZenItem * item)
{
  if (!connectNJB()) return;

  if (item->getItemType()==KZenItem::ItemTypePlayList ||
      item->getItemType()==KZenItem::ItemTypeSmartPlayList) {

    int ret = KMessageBox::questionYesNo(this, 
					 i18n("Do you really want to delete this playlist"),
					 i18n("Delete Playlist - KZenExplorer"));
    if (ret==KMessageBox::Yes) {
      KZenPlayList * pl = ((KZenPlayListItem*)item)->getPlayList();
      
      fCommandThread->deletePlayList(pl);
      
      fView->displayTrackList(fTrackList, ViewModeAll);
    }
  }
}

void KZenExplorer::destroyNaviItem(KZenItem * item)
{
  if (!connectNJB()) return;

  if (item->getItemType()==KZenItem::ItemTypePlayList) {
    
    int ret = KMessageBox::questionYesNo(this, 
					 i18n("Do you really want to destroy this playlist"),
					 i18n("Destroy Playlist - KZenExplorer"));
    if (ret==KMessageBox::Yes) {
      fCommandThread->lock();
      
      KZenPlayList * pl = ((KZenPlayListItem*)item)->getPlayList();
      
      fCommandThread->deletePlayList(pl);
      
      KZenTrack * track;
      for (uint i=0;i<pl->getTrackList()->count();i++) {
	track = pl->getTrackList()->at(i);
	fCommandThread->destroyTrack(track);
      }

      fCommandThread->updateUsage();

      fCommandThread->unlock();
    }
  } else if (item->getItemType()==KZenItem::ItemTypeArtist) {
    
    KZenArtist * artist = ((KZenArtistItem*)item)->getArtist();
    int ret = KMessageBox::questionYesNo(this, 
					 i18n("Do you really want to destroy all tracks of this artist"),
					 i18n("Destroy Artist - KZenExplorer"));
    
    if (ret==KMessageBox::Yes) {
      
      fCommandThread->lock();
      
      KZenTrack * track;
      for (uint i=0;i<artist->getTrackList()->count();i++) {
	track = artist->getTrackList()->at(i);
	fCommandThread->destroyTrack(track);
      }
      
      fCommandThread->updateUsage();
      
      fCommandThread->unlock();
    }
  } else if (item->getItemType()==KZenItem::ItemTypeAlbum) {
    
    int ret = KMessageBox::questionYesNo(this,
					 i18n("Do you really want to destroy all tracks of this album"),
					 i18n("Destroy Album - KZenExplorer"));
    
    if (ret==KMessageBox::Yes) {
      
      fCommandThread->lock();

      KZenAlbum * album = ((KZenAlbumItem*)item)->getAlbum();
      KZenTrack * track;
      for (uint i=0;i<album->getTrackList()->count();i++) {
	track = album->getTrackList()->at(i);
	fCommandThread->destroyTrack(track);
      }
      
      fCommandThread->updateUsage();
      
      fCommandThread->unlock();
    }
  } else if (item->getItemType()==KZenItem::ItemTypeAlbumTL) {
    
    int ret = KMessageBox::questionYesNo(this,
					 i18n("Do you really want to destroy all tracks of this album"),
					 i18n("Destroy Album - KZenExplorer"));
    
    if (ret==KMessageBox::Yes) {
      
      fCommandThread->lock();

      KZenAlbumItemTL * albumItem = (KZenAlbumItemTL*)item;
      KZenTrack * track;
      for (uint i=0;i<albumItem->getTrackList()->count();i++) {
	track = albumItem->getTrackList()->at(i);
	fCommandThread->destroyTrack(track);
      }
      
      fCommandThread->updateUsage();
      
      fCommandThread->unlock();
    }
  } else if (item->getItemType()==KZenItem::ItemTypeGenre) {
    
    KZenGenre * Genre = ((KZenGenreItem*)item)->getGenre();
    int ret = KMessageBox::questionYesNo(this,
					 i18n("Do you really want to destroy all tracks of this genre"),
					 i18n("Destroy Genre - KZenExplorer"));
    
    if (ret==KMessageBox::Yes) {
      
      fCommandThread->lock();
      
      KZenTrack * track;
      for (uint i=0;i<Genre->getTrackList()->count();i++) {
	track = Genre->getTrackList()->at(i);
	fCommandThread->destroyTrack(track);
      }
      
      fCommandThread->updateUsage();
      
      fCommandThread->unlock();
    }
  }      
}

void KZenExplorer::clearSearch()
{
  fView->setSearchText("");
  fSearchTrackList->clear();
  fView->displayTrackList(fTrackList, ViewModeAll);
}

void KZenExplorer::searchChanged(const QString & search)
{
  if (!search.isEmpty()) {
    fSearchTrackList->clear();
    KZenTrack * track;
    bool cs = KZenConfig::get()->caseSensitiveSearch();
    for (uint i=0;i<fTrackList->count();i++) {
      track = fTrackList->at(i);
      if (track->getArtist().contains(search, cs) ||
	  track->getAlbum().contains(search, cs) ||
	  track->getTitle().contains(search, cs) ||
	  track->getGenre().contains(search, cs)) {
	fSearchTrackList->append(track);
      }
    }
    fView->displayTrackList(fSearchTrackList, ViewModeSearch);
  } else {
    fSearchTrackList->clear();
    fView->displayTrackList(fTrackList, ViewModeAll);
  }
}

void KZenExplorer::findDuplicates(int mode)
{
  fSearchTrackList->clear();

  QPtrList<KZenTrack> * TrackList = fView->currentTrackList();

  if (TrackList->count()<2) return;
  
  KProgressDialog progress(gExplorer, "progress",
			   i18n("Duplicates"),
			   i18n("Finding duplicated tracks"),
			   true);
  progress.progressBar()->setTotalSteps(TrackList->count()-1);
  progress.progressBar()->setProgress(0);
  progress.setAutoClose(true);
  progress.show();
  qApp->processEvents();

  bool cancelled = false;
  bool match;
  bool cs = KZenConfig::get()->caseSensitiveSearch();
  bool exact = KZenConfig::get()->exactSearch();

  KZenTrack * track;
  KZenTrack * strack;
  for (uint i=0;i<TrackList->count()-1;i++) {
    track = TrackList->at(i);
    
    progress.progressBar()->advance(1);
    qApp->processEvents();
    cancelled = progress.wasCancelled();
    if (cancelled) break;

    for (uint s=i+1;s<TrackList->count();s++) {
      strack = TrackList->at(s);
      
      match = true;
      if (exact) {
	if (track->getTitle()!=strack->getTitle()) match = false;
      } else {
	if (!track->getTitle().startsWith(strack->getTitle(), cs) &&
	    !strack->getTitle().startsWith(track->getTitle(), cs)) match = false;

      }

      if (match && mode>1) {
	if (exact) {
	  if (track->getArtist()!=strack->getArtist()) match = false;
	} else {
	  if (!track->getArtist().startsWith(strack->getArtist(), cs) &&
	      !strack->getArtist().startsWith(track->getArtist(), cs)) match = false;
	}
      }
      
      if (match && mode>2) {
	if (exact) {
	  if (track->getAlbum()!=strack->getAlbum()) match = false;
	} else {
	  if (!track->getAlbum().startsWith(strack->getAlbum(), cs) &&
	      !strack->getAlbum().startsWith(track->getAlbum(), cs)) match = false;
	}
      }

      if (match) {
	if (fSearchTrackList->find(track)==-1) fSearchTrackList->append(track);
	if (fSearchTrackList->find(strack)==-1) fSearchTrackList->append(strack);
      }
    }
  }
  
  if (cancelled) {
    fSearchTrackList->clear();
  } else {
    fView->displayTrackList(fSearchTrackList, ViewModeSearch);
    fView->setSortColumn(KZen::FieldTitle);
  }
}

void KZenExplorer::advancedSearch()
{
  if (!fSearchDialog) fSearchDialog = new KZenSearchDialog(this);
  
  fSearchTrackList->clear();

  if (fSearchDialog->exec()==QDialog::Accepted) {
    fSearchTrackList->clear();
    KZenTrack * track;
    for (uint i=0;i<fTrackList->count();i++) {
      track = fTrackList->at(i);
      if (fSearchDialog->getSearch()->checkTrack(track)) {
	fSearchTrackList->append(track);
      }
    }
    fView->displayTrackList(fSearchTrackList, ViewModeSearch);
  }
}

void KZenExplorer::trackEdited(QListViewItem * item, const QString & text, int col)
{
  KZenTrackItem * i = (KZenTrackItem*)item;
  KZenTrack * track = i->getTrack();

  QString newValue = text.stripWhiteSpace();
  bool modified = false;
  switch (KZen::getTagField(col)) {
  case KZen::FieldArtist:
    if (track->getArtist()!=newValue) modified = true;
    track->setArtist(newValue);
    break;
  case KZen::FieldAlbum:
    if (track->getAlbum()!=newValue) modified = true;
    track->setAlbum(newValue);
    break;
  case KZen::FieldTitle:
    if (track->getTitle()!=newValue) modified = true;
    track->setTitle(newValue);
    break;
  case KZen::FieldTrack:
    if (track->getTrackNumber()!=newValue.toUInt()) modified = true;
    track->setTrackNumber(newValue.toUInt());
    break;
  case KZen::FieldGenre:
    if (track->getGenre()!=newValue) modified = true;
    track->setGenre(newValue);
    break;
  case KZen::FieldYear:
    if (track->getYear()!=newValue.toUInt()) modified = true;
    track->setYear(newValue.toUInt());
    break;
  default:
    break;
  }

  if (modified) {
    kdDebug() << "track modified --> update" << endl;
    fCommandThread->updateTrackInline(track);    
  }
}

void KZenExplorer::newPlayListSelection()
{
  kdDebug() << "newPlayListSelection" << endl;

  if (!connectNJB()) return;

  bool ok;
  QString name = KInputDialog::getText(i18n("New Playlist From Selection"),
				       i18n("Please enter the name of the new playlist"),
				       i18n("New Playlist"),&ok);

  if (ok) {
    name.stripWhiteSpace();

    KZenPlayList * tpl = 0;
    for (uint i=0;i<fPlayLists->count();i++) {
      tpl = fPlayLists->at(i);
      if (name==tpl->getName()) {
	return;
      }
    }

    KZenSmartPlayList * spl = 0;
    for (uint i=0;i<fSmartPlayLists->count();i++) {
      spl = fSmartPlayLists->at(i);
      if (name==spl->getName()) {
	return;
      }
    }

    KZenPlayList * pl = new KZenPlayList(name);
    
    KZenTrackItem * tracki;
    for (uint i=0;i<fView->selectedTracks().count();i++) {
      tracki = fView->selectedTracks().at(i);
      pl->addTrack(tracki->getTrack());
    }
    
    fView->naviTabClicked(KZen::TabPlayLists);

    fCommandThread->newPlayList(pl);
  }
}

void KZenExplorer::setBusy(bool busy)
{
  fBusy = busy;

//   if (fBusy) {
//     fEditTracks->setEnabled(false);
//     fSaveTracks->setEnabled(false);
//     fDeleteTracks->setEnabled(false);
//     fDestroyTracks->setEnabled(false);
    
//     fNewPlayList->setEnabled(false);
//     fNewSmartPlayList->setEnabled(false);
//     fAdvancedSearch->setEnabled(false);
//     fRenamePlayList->setEnabled(false);
//     fEditSmartPlayList->setEnabled(false);
//     fShufflePlayList->setEnabled(false);
//     fUpdatePlayList->setEnabled(false);
//     fUpdateAllPlayLists->setEnabled(false);
//     fDeletePlayList->setEnabled(false);
//     fDestroy->setEnabled(false);
//   } else {
//     fEditTracks->setEnabled(true);
//     fSaveTracks->setEnabled(true);
//     fDeleteTracks->setEnabled(true);
//     fDestroyTracks->setEnabled(true);
    
//     fNewPlayList->setEnabled(true);
//     fNewSmartPlayList->setEnabled(true);
//     fAdvancedSearch->setEnabled(true);
//     fRenamePlayList->setEnabled(true);
//     fEditSmartPlayList->setEnabled(true);
//     fShufflePlayList->setEnabled(true);
//     fUpdatePlayList->setEnabled(false);
//     fUpdateAllPlayLists->setEnabled(true);
//     fDeletePlayList->setEnabled(true);
//     fDestroy->setEnabled(true);
//   }

  fView->setBusy(fBusy);
}

void KZenExplorer::exportDB()
{  
  QString fn = KFileDialog::getSaveFileName(KZenConfig::get()->localLibrary().path(),
					    "*.zdb",
					    this,
					    i18n("Export Track DB"));
  if (KStandardDirs::exists(fn)) {
    int ret = KMessageBox::questionYesNo(this, 
					 i18n("Do you really want to overwrite the file?"),
					 i18n("File Exists - KZenExplorer"));
    if (ret!=KMessageBox::Yes) return;
  }

  if (!fn.endsWith(".zdb")) fn += ".zdb";
  
  writeTrackDBFile(fn);
}

void KZenExplorer::showConfigDialog()
{  
  configDialog()->setup();

  if (configDialog()->exec()==QDialog::Accepted) {
    configDialog()->acceptData();
    
    if (configDialog()->ownerStringModified()) {
      kdDebug() << "set owner string" << endl;
      fCommandThread->setOwnerString(configDialog()->getOwnerString());
    }

    if (configDialog()->clockModified()) {
      kdDebug() << "set clock" << endl;
      fCommandThread->setClock(configDialog()->getClock());
    }
    
    configDialog()->setModified(false);
  }
}

void KZenExplorer::playTimeOut()
{  
//   fCommandThread->updatePlayProgress();

//   if (!fCommandThread->isPlaying()) fPlayTimer->stop();
}

void KZenExplorer::browseLocalLib()
{
  KURL url = KZenConfig::get()->localLibrary();

  fCommandThread->lock();

  kdDebug() << "starting file manager @ " << url.url() << endl;
  new KRun(url);

  fCommandThread->unlock();
}

void KZenExplorer::synchronize()
{
  kdDebug() << "synchronize" << endl;

  KZenSynchronize sync;
  sync.synchronize();
}

#include "kzenexplorer.moc"
