/***************************************************************************
 *
 *  $Id: kzentrack.cpp,v 1.17 2005/06/27 16:45:53 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>
#include <string>

#include <qpainter.h>
#include <qvariant.h>
#include <qbitmap.h>

#include <kdebug.h>

#include <libnjb.h>

#include "kzencategories.h"
#include "kzentracklistview.h"

#include "kzentrack.h"

QStringList KZenTrack::fTitles;

KZenTrack::KZenTrack()
{
  fTrackID = 0;
  fArtist = "";
  fAlbum = "";
  fTitle = "";
  fTrackNumber = 0;
  fLength = 0;
  fGenre = "";
  fYear = 0;
  fFileName = "";
  fCodec = CodecUNKNOWN;
  fRating = 0;
  fIsCompilation = false;

  fZenArtist = 0;
  fZenAlbum = 0;
  fZenGenre = 0;  
}

KZenTrack::KZenTrack(njb_songid_t * song)
{
  fTrackID = 0;
  fArtist = "";
  fAlbum = "";
  fTitle = "";
  fTrackNumber = 0;
  fLength = 0;
  fGenre = "";
  fYear = 0;
  fFileName = "";
  fCodec = CodecUNKNOWN;
  fRating = 0;
  fIsCompilation = false;
  
  njb_songid_frame_t * frame;
  NJB_Songid_Reset_Getframe(song);
  setTrackID(song->trid);
  while ((frame = NJB_Songid_Getframe(song))) {
    if (strcmp(frame->label,FR_ARTIST)==0 && frame->type == NJB_TYPE_STRING) {
      setArtist(frame->data.strval);
    }
    if (strcmp(frame->label,FR_ALBUM)==0 && frame->type == NJB_TYPE_STRING) {
      setAlbum(frame->data.strval);
    }
    if (strcmp(frame->label,FR_TITLE)==0 && frame->type == NJB_TYPE_STRING) {
      setTitle(frame->data.strval);
    }
    if (strcmp(frame->label,FR_TRACK)==0 && frame->type == NJB_TYPE_UINT16) {
      setTrackNumber(frame->data.u_int16_val);
    }
    if (strcmp(frame->label,FR_LENGTH)==0 && frame->type == NJB_TYPE_UINT16) {
      setLength(frame->data.u_int16_val);
    }
    if (strcmp(frame->label,FR_GENRE)==0 && frame->type == NJB_TYPE_STRING) {
      setGenre(frame->data.strval);
    }
    if (strcmp(frame->label,FR_YEAR)==0 && frame->type == NJB_TYPE_UINT16) {
      setYear(frame->data.u_int16_val);
    }
    if (strcmp(frame->label,FR_FNAME)==0 && frame->type == NJB_TYPE_STRING) {
      setFileName(frame->data.strval);
    }
    if (strcmp(frame->label,FR_SIZE)==0 && frame->type == NJB_TYPE_UINT32) {
      setFileSize(frame->data.u_int32_val);
    }
    if (strcmp(frame->label,FR_CODEC)==0 && frame->type == NJB_TYPE_STRING) {
      if (strcmp(frame->data.strval,NJB_CODEC_MP3)==0) {
	setCodec(CodecMP3);
      } else if (strcmp(frame->data.strval,NJB_CODEC_WMA)==0) {
	setCodec(CodecWMA);
      } else if (strcmp(frame->data.strval,NJB_CODEC_WAV)==0) {
	setCodec(CodecWAV);
      } else {
	setCodec(CodecUNKNOWN);
      }
    }
  }

  fZenArtist = 0;
  fZenAlbum = 0;
  fZenGenre = 0;  
}

KZenTrack::~KZenTrack()
{

}

bool KZenTrack::needsWrite()
{
  if (getRating()!=0) return true;
  for (uint i=0;i<fCategories.size();i++) {
    if (fCategories.testBit(i)) return true;
  }
  if (fIsCompilation) return true;

  return false;
}

void KZenTrack::compress()
{
  uint last = 0;
  for (uint i=0;i<fCategories.size();i++) {
    if (fCategories[i]) last = i;
  }

  fCategories.resize(last+1);
}

const char * KZenTrack::getCodecName()
{
  if (fCodec==CodecMP3)
    return NJB_CODEC_MP3;
  else if (fCodec==CodecWMA)
    return NJB_CODEC_WMA;
  else if (fCodec==CodecWAV)
    return NJB_CODEC_WAV;

  return 0;
}

void KZenTrack::setTrackID(uint TrackID)
{
  fTrackID = TrackID;
}

void KZenTrack::setFileSize(uint size)
{
  fFileSize = size;
}

void KZenTrack::setArtist(const char * Artist)
{
  fArtist = Artist;
}

void KZenTrack::setAlbum(const char * Album)
{
  fAlbum = Album;
}

void KZenTrack::setTitle(const char * Title)
{
  fTitle = Title;

  if (!KZenTrack::getTitles().contains(Title)) {
    KZenTrack::getTitles().append(Title);
    KZenTrack::getTitles().sort();
  }
}

void KZenTrack::setTrackNumber(uint TrackNumber)
{
  fTrackNumber = TrackNumber;
}

void KZenTrack::setLength(uint Length)
{
  fLength = Length;
}

void KZenTrack::setGenre(const char * Genre)
{
  fGenre = Genre;
}

void KZenTrack::setYear(uint Year)
{
  fYear = Year;
}

void KZenTrack::setFileName(const char * FileName)
{
  fFileName = FileName;
}

void KZenTrack::setCodec(CodecType codec) 
{ 
  fCodec = codec; 
}

void KZenTrack::setRating(uint rating) 
{ 
  fRating = rating;
}

void KZenTrack::setIsCompilation(bool c) 
{ 
  fIsCompilation = c;
}

void KZenTrack::setCategory(uint idx, bool v)
{
  if (idx>=fCategories.size()) {
    fCategories.resize(idx+1);
  }

  fCategories.setBit(idx,v);
}

void KZenTrack::setCategories(const QBitArray & ba)
{
  fCategories.resize(ba.size());
  for (uint i=0;i<ba.size();i++) {
    fCategories[i] = ba[i];
  }
}

void KZenTrack::removeCategory(uint idx)
{
  if (fCategories.size()<=idx) return;

  for (uint i=idx;i<fCategories.size()-1;i++) {
    fCategories[i] = fCategories[i+1];
  }
  fCategories[fCategories.size()-1] = false;
}

void KZenTrack::refresh()
{
  if (fItem) fItem->init();
}

QStringList & KZenTrack::getTitles()
{
  return fTitles;
}

bool operator==(const QBitArray &ba1, const QBitArray &ba2)
{
  if (ba1.size()!=ba2.size()) return false;
  for (uint i=0;i<ba1.size();i++) {
    if (ba1.testBit(i)!=ba2.testBit(i)) return false;
  }
  return true;
}

KZenTrackItem::KZenTrackItem(QListView * parent, KZenTrack * Track)
  :KListViewItem(parent)
{
  fTrack = Track;

  init();
}

KZenTrackItem::KZenTrackItem(QListView * parent, QListViewItem * after, KZenTrack * Track)
  :KListViewItem(parent,after)
{
  fTrack = Track;
  
  init();
}

KZenTrackItem::~KZenTrackItem()
{
  if (fTrack) fTrack->setItem(0);
}

void KZenTrackItem::init()
{
  QString t;

  setText(KZen::getColumn(KZen::FieldTitle), fTrack->getTitle());
  setText(KZen::getColumn(KZen::FieldArtist), fTrack->getArtist());
  setText(KZen::getColumn(KZen::FieldAlbum), fTrack->getAlbum());
    
  if (fTrack->getTrackNumber()!=0) {
    t.setNum(fTrack->getTrackNumber());
    setText(KZen::getColumn(KZen::FieldTrack), t);
  }
  
  t = "";
  for (uint i=0;i<fTrack->getCategories().size();i++) {
    if (fTrack->getCategories().testBit(i)) {
      if (t.length()>0) t += ", ";
      t += gCategories->category(i);
    }
  }
  setText(KZen::getColumn(KZen::FieldCategory),t);
  
  if (fTrack->getYear()!=0) {
    t.setNum(fTrack->getYear());
    setText(KZen::getColumn(KZen::FieldYear), t);
  }

  int length = fTrack->getLength();
  int hours,minutes,seconds;
  
  seconds = length%60;
  length -= seconds;
  length /= 60;
  minutes = length%60;
  length -= minutes;
  hours = length/60;
  
  if (hours>0) {
    t.sprintf("%02d:%02d:%02d",hours,minutes,seconds);
  } else {
    t.sprintf("%02d:%02d",minutes,seconds);
  }
  setText(KZen::getColumn(KZen::FieldLength),t);
  
  setText(KZen::getColumn(KZen::FieldGenre),fTrack->getGenre());
  
  switch (fTrack->getCodec()) {
  case KZenTrack::CodecMP3:
    setText(KZen::getColumn(KZen::FieldCodec), "mp3");
    break;
  case KZenTrack::CodecWMA:
    setText(KZen::getColumn(KZen::FieldCodec),"wma");
    break;
  case KZenTrack::CodecWAV:
    setText(KZen::getColumn(KZen::FieldCodec),"wav");
    break;
  default:
    break;
  }
}

int KZenTrackItem::compare(QListViewItem * i, int col, bool ascending) const
{
  KZenTrackItem * item = (KZenTrackItem*) i;
  
  KZenTrack * track = item->getTrack();
  KZenTrack * mytrack = getTrack();

  TagField field = getTagField(col);
  TagFieldType coltype = getTagFieldType(col);

  switch (field)
    {
    case FieldArtist:
      {
	int comp = QString::localeAwareCompare(key(col,ascending),
					       item->key(col,ascending));
	if (comp==0) {
	  comp = QString::localeAwareCompare(key(getColumn(FieldAlbum),ascending),
					     item->key(getColumn(FieldAlbum),ascending));
	  if (comp==0) {
	    QVariant Value(item->text(getColumn(FieldTrack)));
	    QVariant MyValue(text(getColumn(FieldTrack)));
	    
	    if (MyValue.toInt() > Value.toInt()) {
	      return 1;
	    } else if (MyValue.toInt() < Value.toInt()) {
	      return -1;
	    } else {
	      return 0;
	    }
	  }
	}
	return comp;
      }
      break;
    case FieldAlbum:
      {
	int comp = QString::localeAwareCompare(key(col,ascending),
					       item->key(col,ascending));
	if (comp==0) {
	  QVariant Value(item->text(getColumn(FieldTrack)));
	  QVariant MyValue(text(getColumn(FieldTrack)));
	  
	  if (MyValue.toInt() > Value.toInt()) {
	    return 1;
	  } else if (MyValue.toInt() < Value.toInt()) {
	    return -1;
	  } else {
	    return 0;
	  } 
	}
	return comp;
      }
      break;
    default:
      break;
    }
  
  switch (coltype)
    {
    case FieldTypeText:
      {
	return QString::localeAwareCompare(key(col,ascending),
					   item->key(col,ascending));
      }
      break;
    case FieldTypeInt:
      {  
	QVariant Value(item->text(col));
	QVariant MyValue(text(col));
	
	if (MyValue.toInt() > Value.toInt()) {
	  return 1;
	} else if (MyValue.toInt() < Value.toInt()) {
	  return -1;
	} else {
	  return 0;
	}
      }
      break;
    case FieldTypeTime:
      {
	if (mytrack->getLength() > track->getLength()) {
	  return 1;
	} else if (mytrack->getLength() < track->getLength()) {
	  return -1;
	} else {
	  return 0;
	}
      }
      break;
    case FieldTypeRating:
      {
	if (mytrack->getRating() > track->getRating()) {
	  return 1;
	} else if (mytrack->getRating() < track->getRating()) {
	  return -1;
	} else {
	  return 0;
	}
      }
      break;
    default:
      return 0;
    }
  
  return 0;
}

void KZenTrackItem::paintCell(QPainter * p, const QColorGroup & cg,
			      int column, int width, int align)
{
  if (KZen::getTagFieldType(column)!=FieldTypeRating) {
    KListViewItem::paintCell(p, cg, column, width, align);
  } else {
    if (!p) return;

    KZenTrackListView * lv = (KZenTrackListView*) listView();
    if (!lv) return;

    static QPixmap buffer;
    buffer.resize(width, height());


    if(buffer.isNull()) {
      KListViewItem::paintCell(p, cg, column, width, align);
      return;
    }

    QPainter pBuf(&buffer, true);
    if (isSelected()) {
      pBuf.fillRect(buffer.rect(),cg.highlight());
    } else {
      pBuf.fillRect(buffer.rect(),KZen::backgroundColor(this, column));
    }

    int rating = fTrack->getRating();
    for (int i=0;i<5;i++) {
      if (i<rating) {
	pBuf.setPen(cg.color(QColorGroup::Dark));
 	pBuf.setBrush(cg.brush(QColorGroup::Dark));
	pBuf.drawPie(15 + 15 * i - 4, height()/2 - 4, 9, 9, 0*16, 360*16);
      } else {
	pBuf.setPen(cg.color(QColorGroup::Background));
 	pBuf.setBrush(cg.brush(QColorGroup::Background));
	pBuf.drawPie(15 + 15 * i - 4, height()/2 - 4, 9, 9, 0*16, 360*16);
	pBuf.setPen(cg.color(QColorGroup::Dark));
	pBuf.drawEllipse(15 + 15 * i - 4, height()/2 - 4, 9, 9);
      }
    }
  
    pBuf.end();
    p->drawPixmap(0, 0, buffer);
  }  
}

KZenTrackPLItem::KZenTrackPLItem(QListView * parent, KZenTrack * Track, 
				 KZenPlayList * PL)
  :KZenTrackItem(parent,Track)
{
  fPlayList = PL;
}

KZenTrackPLItem::KZenTrackPLItem(QListView * parent, QListViewItem * after,
				 KZenTrack * Track, KZenPlayList * PL)
  :KZenTrackItem(parent,after,Track)
{
  fPlayList = PL;
}

KZenTrackPLItem::~KZenTrackPLItem()
{
  
}
