//LabPlot : ConvolutionListDialog.cc

#include <math.h>
#include "ConvolutionListDialog.h"

#ifdef HAVE_GSL
#include <gsl/gsl_fft_real.h>
#include <gsl/gsl_fft_halfcomplex.h>
#endif

ConvolutionListDialog::ConvolutionListDialog(MainWin *mw, const char *name)
	: ListDialog(mw, name)
{
	setCaption(i18n("Convolution Dialog"));
	KConfig *config = mw->Config();
	config->setGroup( "Convolution" );

	QTabWidget *tw = new QTabWidget(vbox);
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);

	type = new KComboBox(hb);
	QStringList tlist;
	tlist << i18n("convolute") << i18n("deconvolute");
	type->insertStringList(tlist);
	type->setCurrentItem(config->readNumEntry("Type",0));

	hb = new QHBox(tab1);
	new QLabel(i18n("set "),hb);
	set1ni = new KIntNumInput(config->readNumEntry("Set1",1),hb);
	set1ni->setRange(1,1000,1,false);
	new QLabel(i18n("with set "),hb);
	set2ni = new KIntNumInput(config->readNumEntry("Set2",2),hb);
	set2ni->setRange(1,1000,1,false);
	
	// TODO : select gsl or fftw

	hb = new QHBox(tab1);
	new QLabel(i18n("x-values : "),hb);
	xvalue = new KComboBox(hb);
	QStringList vlist;
	vlist << i18n("index") << i18n("signal");
	xvalue->insertStringList(vlist);
	xvalue->setCurrentItem(config->readNumEntry("Xvalue",0));
	
	Style *style=0;
	Symbol *symbol=0;
	QVBox *styletab;
	if(p->getPlot(p->API())->Type() == PSURFACE)
		styletab = surfaceStyle(tw,true);
	else
		styletab = simpleStyle(tw, style, symbol);

	tw->addTab(tab1,i18n("Parameter"));
	tw->addTab(styletab,i18n("Style"));
	
	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
        QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));

	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void ConvolutionListDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Convolution" );

	config->writeEntry("Type", type->currentItem());
	config->writeEntry("Set1", set1ni->value());
	config->writeEntry("Set2", set2ni->value());
	config->writeEntry("Xvalue", xvalue->currentItem());
}

int ConvolutionListDialog::apply_clicked() {
	GraphList *gl = p->getPlot(p->API())->getGraphList();
	if(gl->Number()==0) {
		KMessageBox::error(this,i18n("No graph found!"));
		return -2;
	}

	unsigned int item1 = set1ni->value()-1;
	unsigned int item2 = set2ni->value()-1;

	if (item1 > gl->Number() || item2 > gl->Number()) {
		KMessageBox::error(this, i18n("Sorry. Can not find selected data sets!"));
		return -1;
	}
	
	GRAPHType s1 = gl->getStruct(item1);
	GRAPHType s2 = gl->getStruct(item2);

	Style *style=0;
	Symbol *symbol=0;
	
	if(s1 != GRAPHM && s2 != GRAPHM) {
		style = new Style(cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
			width->value(),pencb->currentItem(),brushcb->currentItem());
		style->setBoxWidth(boxwidth->value());
		style->setAutoBoxWidth(autobox->isChecked());
		style->setPointsSorting(sortpointscb->isChecked());
		symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->value(),
			(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	}

	if ((s1 == GRAPH2D || s1 == GRAPH3D || s1 == GRAPH4D) && 
		(s2 == GRAPH2D || s2 == GRAPH3D || s2 == GRAPH4D)) {
		Graph2D *g1=0, *g2=0;
		Graph3D *g31=0, *g32=0;
		Graph4D *g41=0, *g42=0;
		int nr1=0, nr2=0;
		QString label1, label2;
		switch(s1) {
		case GRAPH2D: g1 = gl->getGraph2D(item1); nr1 = g1->Number(); label1 = g1->getLabel()->simpleTitle(); break;
		case GRAPH3D: g31 = gl->getGraph3D(item1); nr1 = g31->Number(); label1 = g31->getLabel()->simpleTitle(); break;
		case GRAPH4D: g41 = gl->getGraph4D(item1); nr1 = g41->Number(); label1 = g41->getLabel()->simpleTitle(); break;
		default : break;
		}
		switch(s2) {
		case GRAPH2D: g2 = gl->getGraph2D(item2); nr2 = g2->Number(); label2 = g2->getLabel()->simpleTitle(); break;
		case GRAPH3D: g32 = gl->getGraph3D(item2); nr2 = g32->Number(); label2 = g32->getLabel()->simpleTitle(); break;
		case GRAPH4D: g42 = gl->getGraph4D(item2); nr2 = g42->Number(); label2 = g42->getLabel()->simpleTitle(); break;
		default : break;
		}

		int signr, resnr;
		Point *sigdata=0, *resdata=0;		// data
		Point3D *sig3data=0, *res3data=0;	// data
		Point4D *sig4data=0, *res4data=0;	// data
		if (nr1>nr2) {
			signr = nr1;
			resnr = nr2;
			switch(s1) {
			case GRAPH2D : sigdata = g1->Data(); break;
			case GRAPH3D : sig3data = g31->Data(); break;
			case GRAPH4D : sig4data = g41->Data(); break;
			default : break;
			}
			switch(s2) {
			case GRAPH2D : resdata = g2->Data(); break;
			case GRAPH3D : res3data = g32->Data(); break;
			case GRAPH4D : res4data = g42->Data(); break;
			default : break;
			}
		}
		else {
			signr = nr2;
			resnr = nr1;
			switch(s2) {
			case GRAPH2D : sigdata = g2->Data(); break;
			case GRAPH3D : sig3data = g32->Data(); break;
			case GRAPH4D : sig4data = g42->Data(); break;
			default : break;
			}
			switch(s1) {
			case GRAPH2D : resdata = g1->Data(); break;
			case GRAPH3D : res3data = g31->Data(); break;
			case GRAPH4D : res4data = g41->Data(); break;
			default : break;
			}
		}

		double *dsig = new double[signr], *dres = new double[resnr];		// y data values
		for (int i=0;i<signr;i++) {
			if(sigdata)
				dsig[i]=sigdata[i].Y();
			else if(sig3data)
				dsig[i]=sig3data[i].Y();
			else if(sig4data)
				dsig[i]=sig4data[i].Y();
		}
		for (int i=0;i<resnr;i++) {
			if(resdata)
				dres[i]=resdata[i].Y();
			else if(res3data)
				dres[i]=res3data[i].Y();
			else if(res4data)
				dres[i]=res4data[i].Y();
		}

		int nr = signr+resnr/2;		// tmp number of points

		// round to nearest factor of two (not needed when using mixed radix fft's !)
		int nrnew = 64;
		while(nrnew<nr && nrnew>0)
			nrnew *= 2;
		nr=nrnew;
		
		// TODO : convolute here using fftw too (see fit list dialog)
#ifdef HAVE_GSL
		double *sig = new double[nr], *res = new double[nr];
		
		memset(res,0,nr*sizeof(double));
		for (int i=0;i<resnr/2;i++) {
			res[i] = dres[resnr/2+i];
			res[nr-resnr/2+i] = dres[i];
		}
		
		if(resnr/2%2==1)
			res[resnr/2]=dres[resnr];
			
		memset(sig,0,nr*sizeof(double));
		memcpy(sig,dsig,signr*sizeof(double));
	
		// calculate ffts
		gsl_fft_real_radix2_transform(res,1,nr);
		gsl_fft_real_radix2_transform(sig,1,nr);
		
		double re, im, size;
		int t = type->currentItem();	// 0: convolve, 1: deconvolve
		// multiply/divide both ffts
		for (int i=0;i<nr/2;i++) {
			if(i==0 || i==nr/2-1) {
				if(t == 0)
					res[i] = res[i]*sig[i];
				else 
					res[i] = sig[i]/res[i];
			}
			else {
				if(t == 0) {
					re = res[i]*sig[i]-res[nr-i]*sig[nr-i];
					im = res[i]*sig[nr-i]+res[nr-i]*sig[i];
				}
				else {
					size = res[i]*res[i]+res[nr-i]*res[nr-i];
					re = res[i]*sig[i]+res[nr-i]*sig[nr-i];
					im = res[i]*sig[nr-i]-res[nr-i]*sig[i];
					re /= size;
					im /= size;
				}
				
				res[i] = re;
				res[nr-i] = im;
			}
		}
		// inverse fft
		gsl_fft_halfcomplex_radix2_inverse(res,1,nr);
		
		double *conv = new double[signr];
		memcpy(conv,res,signr*sizeof(double));
		
		free(sig);free(res);
		
		// make new data set from conv[signr];
		Point *ptr = new Point[signr];
		double xmin=0, xmax=1, ymin=0, ymax=1;
		for (int i = 0;i<signr;i++) {
			double x=0;
			switch(xvalue->currentItem()) {
			case 0:	x=i; break;
			case 1:	x=sigdata[i].X(); break;
			}
			double y = conv[i];

			ptr[i].setPoint(x,y);
		}

		mw->calculateRanges2D(ptr,signr,&xmin,&xmax,&ymin,&ymax);

		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);

		QString fun;
		if (t==0)
			fun = QString(i18n("convolution of ")+label1+i18n(" with ")+label2);
		else
			fun = QString(i18n("deconvolution of ")+label1+i18n(" with ")+label2);

		Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr,signr);
		mw->addGraph2D(ng,sheetcb->currentItem());
#else
		KMessageBox::error(this, i18n("Sorry. Your installation doesn't support the GSL!"));
#endif
	}

	updateList();

	return 0;
}
