/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB/NJN
 *
 *	$Id: waitsome.c,v 6.3 1999/05/25 22:08:28 kmeyer1 Exp $
 *
 *	Function:	- wait for at least one request to finish
 *	Accepts:	- # of requests
 *			- array of requests
 *			- # completed requests (out)
 *			- array of indices (out)
 *			- array of status (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <string.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <rpisys.h>


int
MPI_Waitsome(count, reqs, ndone, indices, stats)

int			count;
MPI_Request		*reqs;
int			*ndone;
int			*indices;
MPI_Status		*stats;

{
	int		i;
	int		err;			/* error code */
	int		done;			/* test return flag */
	int		first;			/* first request ready */
	int		fl_err;			/* error ocurred? */
	int		fl_trace;		/* do tracing? */
	MPI_Status	firststat;		/* status of first request */
	MPI_Status	*p;

	lam_initerr_m();
	lam_setfunc_m(BLKMPIWAITSOME);

	if (count < 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITSOME,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((count > 0) && ((reqs == 0) || (indices == 0))) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITSOME,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if (ndone == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITSOME,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Generate a start wrapper trace.
 */
	fl_trace = ((_kio.ki_rtf & RTF_TRON) == RTF_TRON);

	if (fl_trace) {
		lam_tr_wrapstart(BLKMPIWAITSOME);
	}
/*
 * Loop initializing all status members.
 */
	if (stats != MPI_STATUSES_IGNORE) {
		for (i = 0, p = stats; i < count; ++i, ++p) {
			lam_emptystat(p);
		}
	}
/*
 * Wait till the first request is done.
 */
	err = MPI_Waitany(count, reqs, &first, &firststat);

	if (err != MPI_SUCCESS) {

		if (first == MPI_UNDEFINED) {
			return(lam_errfunc(MPI_COMM_WORLD,
					BLKMPIWAITSOME, err));
		} else {
			if (stats != MPI_STATUSES_IGNORE) {
				stats[first].MPI_ERROR = err;
			}

			if (fl_trace) {
				lam_tr_wrapend(BLKMPIWAITSOME);
			}

			return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITSOME,
					lam_mkerr(MPI_ERR_IN_STATUS, 0)));
		}
	}

	if (first == MPI_UNDEFINED) {
		*ndone = first;

		if (fl_trace) {
			lam_tr_wrapend(BLKMPIWAITSOME);
		}

		lam_resetfunc_m(BLKMPIWAITSOME);
		return(MPI_SUCCESS);
	}
/*
 * Loop testing if any other request is done and skip inactive requests.
 * I am going to be nice and maintain the sequential order
 * in the result arrays, just as if MPI_Testsome() was used.
 */
	*ndone = 0;
	fl_err = 0;

	for (i = 0; i < count; ++i, ++reqs) {

		if (i == first) {
			++(*ndone);
			*indices++ = i;
			if (stats != MPI_STATUSES_IGNORE) {
				*stats = firststat;
				stats->MPI_ERROR = MPI_SUCCESS;
				++stats;
			}
		}
		else if ((*reqs == MPI_REQUEST_NULL) ||
				((*reqs)->rq_state == LAM_RQSINIT)) {
			continue;
		}
		else {
			err = MPI_Test(reqs, &done, stats);

			if ((err != MPI_SUCCESS) || done) {
				++(*ndone);
				*indices++ = i;

				if (err != MPI_SUCCESS) {
					if (stats != MPI_STATUSES_IGNORE) {
						stats->MPI_ERROR = err;
					}

					fl_err = 1;
				}

				if (stats != MPI_STATUSES_IGNORE) {
					++stats;
				}
			}
		}
	}
/*
 * Generate an end wrapper trace.
 */
	if (fl_trace) {
		lam_tr_wrapend(BLKMPIWAITSOME);
	}

	if (fl_err) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITSOME,
				lam_mkerr(MPI_ERR_IN_STATUS, 0)));
	}

	lam_resetfunc_m(BLKMPIWAITSOME);
	return(MPI_SUCCESS);
}
