/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB
 *
 *	$Id: lamclean.c,v 6.6 2000/09/05 17:11:49 jsquyres Exp $
 *
 *	Function:	- command to clean out several daemons on all nodes
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

#include <args.h>
#include <bfreq.h>
#include <debug.h>
#include <freq.h>
#include <ksignal.h>
#include <laminternal.h>
#include <net.h>
#include <preq.h>
#include <priority.h>
#include <terror.h>
#include <trreq.h>
#include <typical.h>

/*
 * local functions
 */
static void		help();

int
main(argc, argv)

int			argc;
char			*argv[];

{
	int4		*all;		/* array of all node IDs */
	int4		i;
	int4		nall;		/* count of all nodes */
	int		fl_verbose;	/* verbosity flag */

	/* Ensure that we are not root */

	if (getuid() == 0 || geteuid() == 0) {
	  show_help(NULL, "deny-root", NULL);
	  exit(EACCES);
	}

/*
 * Parse command-line arguments.
 */
	validopts("hv");

	if (do_args(&argc, argv) || (argc != 1)) {
		fprintf(stderr, "lamclean [-hv]\n");
		exit(LAM_EEXIT);
	}

	if (opt_taken('h')) {
		help();
		exit(0);
	}

	fl_verbose = opt_taken('v');
/*
 * Initialize and attach to Trollius kernel.
 */
	if (kinit(PRCMD)) {
	  show_help(NULL, "no-lamd", "lamclean", NULL);
	  exit(LAM_EEXIT);
	}
/*
 * Get an array of all node IDs.
 */
	nall = getnall();
	all = (int4 *) malloc((unsigned) (nall * sizeof(int4)));

	if (all == 0) lamfail("lamclean (malloc)");

	if (getall(all, nall)) lamfail("lamclean (getall)");
/*
 * Doom all appl processes.
 */
	if (fl_verbose) nodespin_init("killing processes,");

	for (i = 0; i < nall; ++i) {

		if (fl_verbose) nodespin_next(all[i]);

		if (rpdoom(all[i], SELECT_APPL, 0, SIGUDIE)) {
			lamfail("lamclean (rpdoom)");
		}
	}
/*
 * Reset file daemons.
 */
	if (fl_verbose) {
		nodespin_end();
		nodespin_init("closing files,");
	}

	for (i = 0; i < nall; ++i) {

		if (fl_verbose) nodespin_next(all[i]);

		if (lam_rfrmfd(all[i], NOTFD)) {
			lamfail("lamclean (rfrmfd)");
		}
	}
/*
 * Reset trace deamons.
 */
	if (fl_verbose) {
		nodespin_end();
		nodespin_init("sweeping traces,");
	}

	for (i = 0; i < nall; ++i) {

		if (fl_verbose) nodespin_next(all[i]);

		if (lam_rtrsweep(all[i], TRANYLIST)) {
			lamfail("lamclean (lam_rtrsweep)");
		}
	}
/*
 * Cleanup registered objects.
 */
	if (fl_verbose) {
		nodespin_end();
		nodespin_init("cleaning up registered objects,");
	}

	for (i = 0; i < nall; ++i) {

		if (fl_verbose) nodespin_next(all[i]);

		if (lam_rtrcleanobjs(all[i])) {
			lamfail("lamclean (lam_rtrcleanobjs)");
		}
	}
/*
 * Reset buffers.
 */
	sleep(1);
	if (fl_verbose) {
		nodespin_end();
		nodespin_init("sweeping messages,");
	}

	for (i = 0; i < nall; ++i) {

		if (fl_verbose) nodespin_next(all[i]);

		if (rbfwipe(all[i])) {
			lamfail("lamclean (rbfwipe)");
		}
	}

	if (fl_verbose) nodespin_end();

	kexit(0);
	return(0);
}

/*
 *	help
 *
 *	Function:	- prints helpful information on this command
 */
static void
help()

{
	printf("\nSynopsis:	lamclean [options]\n");
	printf("\nDescription:	Clean up resources on all nodes.\n");
	printf("\nOptions:	-h\t\tPrint this help message.\n");
	printf("\t\t-v\t\tBe verbose.\n");
}
