/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1995 The Ohio State University
 *	RBD/GDB
 *
 *	$ID$
 * 
 *	Function:	- boot schema parser
 *			- host file syntax version
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <lamnet.h>
#include <net.h>
#include <typical.h>
#include <args.h>

/*
 * constants
 */
#define MAXLINE		256		/* input line size */
#define LNDBLOCK	10		/* array allocation block */

/*
 * static variables
 */
static char		linebuf[MAXLINE];

/*
 * static functions
 */
static void parseline(char **phost, int *pncpus, char **puser, int lineno, 
		      char *linedata);
#if 0
/* After much discussion, we decided to not have a "-o" option to
   lamboot/recon/wipe that affects the old parsing behavior.  Instead,
   we'll just recognize the first word after the hostname with no = in
   it to be the username.  This preserves backward compatibility.
   Ick.  This is only because Raja insisted.  :-) See the thread
   starting at http://www.mpi.nd.edu/Internal/llamas/msg01496.php3. */
static void old_parseline(char **phost, int *pncpus, char **puser);
#endif

/*
 *	parse_host
 *
 *	Function:	- parse a host file
 *			- fill the network description
 *			- node IDs are [0, N-1]
 *	Accepts:	- input file stream
 *			- node description array (value returned)
 *			- array size (value returned)
 */
int
bhostparse(FILE *fp, struct lamnode **plamnet, int *pnlamnet)
{
	char		*host;		/* host machine name */
	char		*user;		/* user name */
	char		*linecopy;	/* copy of current line */
	int		ncpus;		/* number of CPUs */
	int		nlndalloc;	/* current allocation size */
	int		node;		/* current node ID */
	int		i;		/* favorite counter */
	int		found;		/* if found duplicate */
	int		lineno;		/* line number in host file */
	struct lamnode	*lamnet;	/* current node desc. array */

/*
 * Loop till EOF reading and parsing line-by-line.
 */
	lamnet = 0;
	nlndalloc = 0;
	node = 0;
	lineno = 0;

	while (! feof(fp)) {

		if (fgets(linebuf, MAXLINE, fp) == 0) break;
/*
 * Parse the line for hostname, num CPUs, and username.
 */
		lineno++;
		linecopy = strdup(linebuf);
		if (linecopy == NULL)
		  return LAMERROR;
		i = strlen(linecopy);
		if (linecopy[i - 1] == '\n')
		  linecopy[i - 1] = '\0';

#if 0
		/* See note above */
		if (opt_taken('o'))
		  old_parseline(&host, &ncpus, &user);
		else
#endif
		  parseline(&host, &ncpus, &user, lineno, linecopy);

		free(linecopy);

/*
 * If a machine name is found, add the node to the network.
 */
		if (!host) {
			continue;
		}
/*
 * If the hostname is already used, accumulate the new CPU count into its CPU
 * count
 */
		for (found = i = 0; i < node; i++) {
		  if (strcmp(host, lamnet[i].lnd_hname) == 0 &&
		      ((user == NULL && lamnet[i].lnd_uname == NULL) ||
		       (user != NULL && lamnet[i].lnd_uname != NULL &&
			strcmp(user, lamnet[i].lnd_uname) == 0))) {
		    lamnet[i].lnd_ncpus += ncpus;
		    found = 1;
		    break;
		  }
		}
		if (found)
		  continue;
/*
 * Allocate initial node description array.
 */
		if (lamnet == 0) {
			lamnet = (struct lamnode *) malloc((unsigned)
					LNDBLOCK * sizeof(struct lamnode));
			if (lamnet == 0) return(LAMERROR);
			nlndalloc = LNDBLOCK;
		}
/*
 * Extend node description array.
 */
		if (nlndalloc <= node) {
			nlndalloc += LNDBLOCK;
			lamnet = (struct lamnode *) realloc((char *) lamnet,
					(unsigned) nlndalloc *
					sizeof(struct lamnode));
			if (lamnet == 0) return(LAMERROR);
		}

		lamnet[node].lnd_nodeid = node;
		lamnet[node].lnd_type = NT_BOOT;
		lamnet[node].lnd_ncpus = ncpus;
/*
 * Set host name.
 */
		lamnet[node].lnd_hname = malloc((unsigned) strlen(host) + 1);
		if (lamnet[node].lnd_hname == 0) return(LAMERROR);
		strcpy(lamnet[node].lnd_hname, host);
/*
 * Set optional user name.
 */
		lamnet[node].lnd_uname = 0;

		if (user) {
			lamnet[node].lnd_uname = malloc((unsigned)
					strlen(user) + 1);
			if (lamnet[node].lnd_uname == 0) return(LAMERROR);
			strcpy(lamnet[node].lnd_uname, user);
		}

		node++;
	}

	*pnlamnet = node;
	*plamnet = lamnet;
	return(0);
}

/*
 *	parseline
 *
 *	Function:	- parses the line buffer for hostname and username
 *			- Format: hostname [key=value]*
 *			- key is one of:
 *			-   cpu: number of CPUs
 *			-   user: user name
 *			- skip comments: everything after a # character
 *			- fills hostname/username or NULL
 *	Accepts:	- host name (returned value)
 *			- number of CPUs (returned value)
 *			- user name (returned value)
 *			- line number (for error messages)
 *			- line contents (for error messages)
 */
static void
parseline(char **phost, int *pncpus, char **puser, int lineno, 
	  char *linedata)
{
  char		*p;		/* favourite pointer */
  char		*q;		/* end of current token */
  char		*equals;	/* equals sign in key=value */
  char		*value;		/* value in key=value */
  char		lineno_str[20];	/* string for lineno */
  int		ncpus;		/* numerical value of n in cpu=n */

  *phost = 0;
  *pncpus = 0;
  *puser = 0;
  linebuf[MAXLINE - 1] = '\0';

  snprintf(lineno_str, 19, "%d", lineno);
  lineno_str[19]='\0';

  /*
   * Skip comments by replacing the first '#' with a NULL.
   */
  for (p = linebuf; *p; ++p) {

    if (*p == '#') {
      *p = '\0';
      break;
    }
  }

  /*
   * For each token in the file, dispatch:
   *   If it does not contain an '=' character, it is a hostname (error if
   *   there already was one).
   *   If it contains an '=' character, choose based on the left side what
   *   parameter to set.
   */
  for (p = linebuf; *p; p = q) {
    /* If *p is whitespace, just continue. */
    if (isspace(*p)) {
      q = p + 1;
      continue;
    }

    /* Find end of token */
    for (q = p; *q && !isspace(*q); q++)
      continue;
    if (*q != '\0') {
      *q = '\0';
      ++q;
    }

    /* Test for '=' character */
    equals = strchr(p, '=');
    if (equals == NULL) {
      /* This is a hostname */
      if (*phost == NULL) {
	*phost = p;
      } 
      /* Nope, not the hostname -- must be a username */
      else if (*puser == NULL) {
	*puser = p;
      }
      /* None of the above -- must be garbage */
      else {
	show_help("bhost-parse", "garbage-after-hostname", lineno_str, 
		  linedata, NULL);
	*phost = NULL;
	return;
      }
    } else {
      /* This is a key=value pair */
      if (*phost == NULL) {
	show_help("bhost-parse", "hostname-must-be-first", lineno_str, 
		  linedata, NULL);
	*phost = NULL;
	return;
      }

      *equals = '\0';
      value = equals + 1;
      if (strcasecmp(p, "cpu") == 0) {
	if (*pncpus == 0) {
	  ncpus = atoi(value);
	  if (ncpus >= 1) {
	    *pncpus = ncpus;
	  } else {
	    show_help("bhost-parse", "invalid-cpu-count", lineno_str, 
		      linedata, NULL);
	    *phost = NULL;
	    return;
	  }
	} else {
	  show_help("bhost-parse", "multiple-cpu-counts", lineno_str, 
		    linedata, NULL);
	  *phost = NULL;
	  return;
	}
      } else if (strcasecmp(p, "user") == 0) {
	if (*puser == NULL) {
	  if (*value == '\0') {
	    show_help("bhost-parse", "empty-username", lineno_str, 
		      linedata, NULL);
	    *phost = NULL;
	    return;
	  } else {
	    *puser = value;
	  }
	} else {
	  show_help("bhost-parse", "multiple-usernames", lineno_str, 
		    linedata, NULL);
	  *phost = NULL;
	  return;
	}
      } else {
	show_help("bhost-parse", "invalid-key", lineno_str, linedata, NULL);
	*phost = NULL;
	return;
      }
    }
  }

  /* Fix up number of CPUs if user did not set it */

  if (*pncpus == 0)
    *pncpus = 1;
}


#if 0
/* See note above */
/*
 *	old_parseline
 *
 *	Function:	- parses the line buffer for hostname and username
 *			- Format: hostname[:ncpus] [username]
 *			- skip comments: everything after a # character
 *			- fills hostname/username or NULL
 *                      - this function is only kept around to
 *                        preserve backward compatability with the "old
 *                        way" of parsing boot schema files.
 *	Accepts:	- host name (returned value)
 *			- number of CPUs (returned value)
 *                      - user name (returned value) 
*/
static void
old_parseline(char **phost, int *pncpus, char **puser)
{
	char		*p;		/* favourite pointer */

	*phost = 0;
	*pncpus = 1;
	*puser = 0;
	linebuf[MAXLINE - 1] = '\0';
/*
 * Skip comments by replacing the first '#' with a NULL.
 */
	for (p = linebuf; *p; ++p) {

		if (*p == '#') {
			*p = '\0';
			break;
		}
	}
/*
 * Find beginning and end of hostname and null-terminate it.
 */
	for (p = linebuf; *p; ++p) {

		if ( ! isspace((int) *p)) {
			*phost = p;
			break;
		}
	}

/*
 * Find the end of the hostname.  If we find :xxx, get the number of
 * CPUs.
 */
	for ( ; *p; ++p) {

		if (isspace((int) *p) || (*p) == ':') {
		  if (*p == ':') {
		    *p++ = '\0';
		    *pncpus = atoi(p);
		    /* Prevent stupid users */
		    if (*pncpus <= 0)
		      *pncpus = 1;
		    for (; *p && isdigit(*p); p++)
		      continue;
		  } else 
		    *p++ = '\0';
		  break;
		}
	}
/*
 * Find beginning and end of username and null-terminate it.
 */
	for ( ; *p; ++p) {

		if ( ! isspace((int) *p)) {
			*puser = p;
			break;
		}
	}

	for ( ; *p; ++p) {

		if (isspace((int) *p)) {
			*p++ = '\0';
			break;
		}
	}
}
#endif
