/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	$Id: ibcast.c,v 1.2 2000/10/19 16:48:47 jsquyres Exp $
 *
 *	Function:	- IMPI broadcast synchronization
 *	Accepts:	- buffer
 *			- count
 *			- datatype
 *			- root
 *			- communicator
 *	Returns:	- MPI_SUCCESS or an MPI error code
 */

#include <mpi.h>
#include <mpisys.h>
#include <impi.h>
#include <impi-defs.h>
#include <blktype.h>


/*
 * Local functions
 */
static int master_bcast(void *buff, int count, MPI_Datatype datatype, 
			int root, MPI_Comm comm);


/*
 * Almost identical to the pseudocode in section 4.11 of the IMPI
 * standard
 *
 * This function should only be called from the main MPI_Bcast
 * function.
 */
int
IMPI_Bcast(void *buff, int count, MPI_Datatype datatype, 
	   int root, MPI_Comm comm)
{
  int ret;
  int local_root;
  int myrank = comm->c_group->g_myrank;

  lam_setfunc(BLKIMPIBCAST);

  /* Switch to use the IMPI collective context ID */

  lam_mkimpicoll(comm);

  /* Global phase */

  if (myrank == root ||
      (IMPI_Is_master(myrank, comm) && !IMPI_Are_local(myrank, root, comm)))
    master_bcast(buff, count, datatype, root, comm);

  /* Switch back to the pt2pt context ID */

  lam_mkpt(comm);

  /* Local phase */

  if (IMPI_Are_local(myrank, root, comm)) {
    local_root = IMPI_Rank_impi2shadow(root, comm);
    ret = MPI_Bcast(buff, count, datatype, local_root, comm->c_shadow);
  }
  else {
    local_root = IMPI_Local_master_rank(myrank, comm);
    local_root = IMPI_Rank_impi2shadow(local_root, comm);
    ret = MPI_Bcast(buff, count, datatype, local_root, comm->c_shadow);
  }

  lam_resetfunc(BLKIMPIBCAST);
  return ret;
}


/*
 * Almost identical to the pseudocode in section 4.11 of the IMPI
 * standard
 */
static int 
master_bcast(void *buff, int count, MPI_Datatype datatype, 
	     int root, MPI_Comm comm)
{
  int myrank, nmasters, mynum, rootnum, vnum, dim, hibit;
  int i, peer, mask;

  myrank = comm->c_group->g_myrank;
  nmasters = IMPI_Num_masters(comm);

  if (nmasters <= IMPI_MAX_LINEAR_BCAST) {
    /* Linear broadcast between masters */

    if (myrank == root) {
      for (i = 0; i < nmasters; i++) {
	if (i == IMPI_Local_master_num(root, comm))
	  continue;

	MPI_Send(buff, count, datatype, IMPI_Master_rank(i, comm),
		 IMPI_BCAST_TAG, comm);
      }
}
    else {
      MPI_Recv(buff, count, datatype, root, IMPI_BCAST_TAG, comm, 
	       MPI_STATUS_IGNORE);
    }
  } else {
    /* Tree broadcast between masters */

    mynum = IMPI_Master_num(myrank, comm);
    rootnum = IMPI_Master_num(root, comm);
    vnum = (mynum + nmasters - rootnum) % nmasters;
    dim = lam_cubedim(nmasters);
    hibit = lam_hibit(vnum, dim);
    --dim;

    /* Receive data from parent in the tree */

    if (vnum > 0) {
      peer = ((vnum & ~(1 << hibit)) + rootnum) % nmasters;
      peer = IMPI_Master_rank(peer, comm);
      if (IMPI_Are_local(peer, root, comm))
	peer = root;

      MPI_Recv(buff, count, datatype, peer, IMPI_BCAST_TAG, comm, 
	       MPI_STATUS_IGNORE);
    }

    /* Send data to children */

    for (i = hibit + 1, mask = 1 << i; i <= dim; ++i, mask <<= 1) {
      peer = vnum | mask;
      if (peer < nmasters) {
	peer = IMPI_Master_rank((peer + rootnum) % nmasters, comm);
	MPI_Send(buff, count, datatype, peer, IMPI_BCAST_TAG, comm);
      }
    }
  }
  
  return MPI_SUCCESS;
}

