/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1995 The Ohio State University
 *	GDB
 *
 *	$Id: krecv.c,v 6.3 1999/09/10 05:20:46 jsquyres Exp $
 * 
 *	Function:	- kernel message receive
 */

#include <errno.h>

#include <kio.h>
#include <kreq.h>
#include <portable.h>
#include <terror.h>
#include <typical.h>

/*
 * external variables
 */
extern struct kio_t	_kio;			/* kernel I/O block */

/*
 * external functions
 */
extern int		_cipc_krecvback();	/* ITB/OTB msg recv */
extern int		_cipc_krecvfront();	/* ITB/OTB msg recv */
extern void		_ksig_follow();		/* check signals */

/*
 *	krecv
 *
 *	Function:	- receives a message from another process
 *			- retries kernel request after signal interruption
 *			  or transfer timeout
 *	Accepts:	- kernel message descriptor ptr
 *	Returns:	- 0 or ERROR
 */
int
krecv(pkmsg)

struct kmsg		*pkmsg;			/* message desc. ptr */

{
	int4		r;
/*
 * Loop through possible signal interruptions.
 */
	do {
		if (krecvfront(pkmsg) < 0) {
			return(LAMERROR);
		}

		r = krecvback(pkmsg);

	} while ((r > 0) && ((r & _kio.ki_sigretry) == r));

	if (r > 0) {
		errno = EINTR;
		return(LAMERROR);
	} else if (r < 0) {
		return(LAMERROR);
	}

	return(0);
}

/*
 *	krecvfront
 *
 *	Function:	- request portion of kernel message receive
 *	Accepts:	- kernel message descriptor ptr
 *	Returns:	- kernel-client socket descriptor or ERROR
 */
int
krecvfront(pkmsg)

struct kmsg		*pkmsg;		/* message desc. ptr */

{
	struct kreq	req;		/* kernel request */
/*
 * Formulate the KQRECV request.
 */
	LAM_ZERO_ME(req);
	req.kq_req = KQRECV;
	req.kq_index = _kio.ki_index;
	req.kq_msg.k_event = pkmsg->k_event;
	req.kq_msg.k_type = pkmsg->k_type;
	req.kq_msg.k_flags = pkmsg->k_flags;
	req.kq_msg.k_length = pkmsg->k_length;
	req.kq_msg.k_msg = pkmsg->k_msg;

	return(_cipc_krecvfront(&req));
}

/*
 *	krecvback
 *
 *	Function:	- reply portion of kernel message receive
 *	Accepts:	- kernel message descriptor ptr
 *	Returns:	- signal if request was interrupted
 *			- 0 or ERROR or signal
 */
int4
krecvback(pkmsg)

struct kmsg		*pkmsg;

{
	struct kreq	req;		/* kernel request */
	struct kreply	reply;		/* kernel reply */

	req.kq_msg.k_flags = pkmsg->k_flags;
	req.kq_msg.k_msg = pkmsg->k_msg;

	if (_cipc_krecvback(&req, &reply)) {
		return(LAMERROR);
	}

	if (pkmsg->k_flags & KTRACE) {
		_kio.ki_blktime += reply.kr_blktime;
	}

	if (reply.kr_signal) {
		_kio.ki_signal |= reply.kr_signal;
		_ksig_follow();

		if (reply.kr_reply == EINTR) {
			errno = EINTR;
			return(reply.kr_signal);
		}
	}

	if (reply.kr_reply) {
		errno = reply.kr_reply;
		return(LAMERROR);
	}

	pkmsg->k_type = reply.kr_type;
	pkmsg->k_length = reply.kr_length;
	return(0);
}
