/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	JRV/RBD
 *
 *	$Id: allreduce.c,v 6.5 2000/10/13 17:57:32 jsquyres Exp $
 *
 *	Function:	- reduce and return data to all processes
 *	Accepts:	- send buffer
 *			- receive buffer
 *			- count of elements
 *			- type of elements
 *			- operation to perform
 *			- communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <lam_config.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Allreduce - Combines values from all processes and distribute the result
                back to all processes

Input Parameters:
+ sbuf - starting address of send buffer (choice) 
. count - number of elements in send buffer (integer) 
. dtype - data type of elements of send buffer (handle) 
. op - operation (handle) 
- comm - communicator (handle) 

Output Parameter:
. rbuf - starting address of receive buffer (choice) 

.N IMPI_YES

.N fortran

.N collops

.N Errors
.N MPI_ERR_COMM
.N MPI_ERR_INTERCOMM
.N MPI_ERR_IMPI
.N MPI_ERR_BUFFER
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_OP

.N ACK
@*/
int MPI_Allreduce(void *sbuf, void *rbuf, int count, 
		  MPI_Datatype dtype, MPI_Op op, MPI_Comm comm)
{
	int		err;			/* error code */

	lam_initerr();
	lam_setfunc(BLKMPIALLREDUCE);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return (lam_errfunc(comm, BLKMPIALLREDUCE,
				lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (dtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(comm, BLKMPIALLREDUCE,
				lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	LAM_TRACE(lam_tr_cffstart(BLKMPIALLREDUCE));
/*
 * Reduce to 0 and broadcast.
 */
	err = MPI_Reduce(sbuf, rbuf, count, dtype, op, 0, comm);
	if (err != MPI_SUCCESS) {
		return (lam_errfunc(comm, BLKMPIALLREDUCE, err));
	}

	err = MPI_Bcast(rbuf, count, dtype, 0, comm);
	if (err != MPI_SUCCESS) {
		return (lam_errfunc(comm, BLKMPIALLREDUCE, err));
	}

	LAM_TRACE(lam_tr_cffend(BLKMPIALLREDUCE, -1, comm, dtype, count));

	lam_resetfunc(BLKMPIALLREDUCE);
	return(MPI_SUCCESS);
}
