/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: cartrank.c,v 6.3 1999/12/31 21:27:18 jsquyres Exp $
 *
 *	Function:	- translate coordinates to rank
 *	Accepts:	- communicator
 *			- coordinates array
 *			- ptr rank (returned value)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Cart_rank - Determines process rank in communicator given Cartesian
                location

Input Parameters:
+ comm - communicator with cartesian structure (handle) 
- coords - integer array (of size  'ndims') specifying the cartesian 
  coordinates of a process 

Output Parameter:
. prank - rank of specified process (integer) 

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_TOPOLOGY
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Cart_rank(MPI_Comm comm, int *coords, int *prank)
{
	int		rank;
	int		dim;
	int		ord;
	int		factor;
	int		i;
	int		*d;
	int		*c;

	lam_initerr();
	lam_setfunc(BLKMPICARTRANK);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPICARTRANK, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm,
			BLKMPICARTRANK, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (!LAM_IS_CART(comm)) {
		return(lam_errfunc(comm, BLKMPICARTRANK,
				lam_mkerr(MPI_ERR_TOPOLOGY, 0)));
	}

	if ((coords == 0) || (prank == 0)) {
		return(lam_errfunc(comm,
			BLKMPICARTRANK, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Loop over coordinates computing the rank.
 */
	factor = 1;
	rank = 0;
	i = comm->c_topo_ndims - 1;
	d = comm->c_topo_dims + i;
	c = coords + i;

	for (; i >= 0; --i, --c, --d) {
		dim = (*d > 0) ? *d : -(*d);
		ord = *c;
		if ((ord < 0) || (ord >= dim)) {
			if (*d > 0) {
				return(lam_errfunc(comm, BLKMPICARTRANK,
					lam_mkerr(MPI_ERR_ARG, 0)));
			}

			ord %= dim;
			if (ord < 0) {
				ord += dim;
			}
		}

		rank += factor * ord;
		factor *= dim;
	}

	*prank = rank;

	lam_resetfunc(BLKMPICARTRANK);
	return(MPI_SUCCESS);
}
