/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/NJN
 *
 *	$Id: finalize.c,v 6.15.2.1 2001/11/09 20:57:50 brbarret Exp $
 *
 *	Function:	- terminate the MPI session
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <lam_config.h>

#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <MPISYSF.h>
#include <mpisys.h>
#include <rpisys.h>
#include <terror.h>
#include <trreq.h>
#include <typical.h>
#include <t_types.h>
#include <lamthreads.h>
#if LAM_WANT_IMPI
#include <impi.h>
#endif

/*
 * external functions
 */
extern void		_cio_cleanup();
extern void		lam_kexit();

/*
 * local functions
 */
static int		waitall();
static void		free_comm();
static void		free_dtype();
static void		free_env();
static void		free_errhdl();
static void		free_f77();
static void		free_op();
static void		free_rdtype();

/*@
   MPI_Finalize - Terminates MPI execution environment

Notes:

All processes must call this routine before exiting.  The number of
processes running `after` this routine is called is undefined; it is
best not to perform much more than a 'return rc' after calling
'MPI_Finalize'.

MPI mandates that the same thread invoke 'MPI_Init' (or
'MPI_Init_thread') and 'MPI_Finalize'; if a different thread invokes
'MPI_Finalize', 'MPI_ERR_OTHER' will be returned, and MPI will not be
finalized.

.N fortran

.N Errors
.N MPI_ERR_INTERN
. MPI_ERR_OTHER - Returned if a different thread attempts to finalize
MPI than the thread that initialized MPI.

.seealso MPI_Init, MPI_Init_thread

.N ACK
@*/
int MPI_Finalize(void)
{
	int		err;			/* error code */
	
	lam_initerr();
	lam_setfunc(BLKMPIFINAL);
/*
 * Ensure that if we started with MPI_Init_thread that this is the
 * same thread that invoked MPI_Init_thread.
 */
	if (lam_thread_level > MPI_THREAD_SINGLE && 
	    !lam_thread_compare(lam_thread_self(), lam_main_thread)) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIFINAL, 
				   lam_mkerr(MPI_ERR_ARG, 0)));
	}

	LAM_TRACE(lam_tr_cffstart(BLKMPIFINAL));
/*
 * Block until all pending requests are done.
 */
	err = waitall();
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIFINAL, err));
	}
/*
 * IMPI finalization
 */
#if LAM_WANT_IMPI
	if (_kio.ki_rtf & RTF_IMPI) {
	  if (IMPI_Finalize() != 0)
	    return(lam_errfunc(MPI_COMM_WORLD, BLKMPIFINAL, err));
	}
#endif
/*
 * RPI specific finalization.
 */
	if (RPI_SPLIT(_rpi_lamd_finalize, _rpi_c2c_finalize, (0))) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIFINAL,
					lam_mkerr(MPI_ERR_INTERN, errno)));
	}
/*
 * Cleanup.
 */
	lam_nukepublished();
	lam_ports_free();
	lam_free_onesided();

	free_env();
	free_rdtype();
	free_comm();
	free_op();
	free_dtype();
	free_errhdl();
	free_f77();

	LAM_TRACE(lam_tr_cffend(BLKMPIFINAL, -1, 0, 0, 0));

	lam_tr_off();
	lam_nuketrace();
 	lam_nukeprocs();

	if (_kio.ki_rtf & RTF_MPISIGS) {
		lam_mpi_reset_sighandlers();
	}

	lam_flfinal = 1;
	lam_kexit(0);

	lam_resetfunc(BLKMPIFINAL);
	lam_nukefunc();

	if (lam_thread_level > MPI_THREAD_SINGLE) {
	  lam_thread_finalize();
	}

	return(MPI_SUCCESS);
}


/*
 *	waitall
 *
 *	Function:	- wait till all pending requests are done
 *			- window handler requests are ingored
 *	Returns:	- MPI_SUCCESS or error code
 */
static int
waitall()

{
	int		err;			/* error code */
	int		nreqs;			/* # pending requests */
	MPI_Request	*reqs;			/* request array */
	int		i;

	nreqs = _mpi_req_getn();
	if (nreqs == 0) return(MPI_SUCCESS);

	reqs = (MPI_Request *) malloc((unsigned) nreqs * sizeof(MPI_Request));
	if (reqs == 0) {
		return(lam_mkerr(MPI_ERR_OTHER, errno));
	}

	_mpi_req_get(nreqs, reqs);
/*
 * Ignore window handler requests.
 */
	for (i = 0; i < nreqs; i++) {
	    if (reqs[i]->rq_hdlr == lam_osd_handler) {
		reqs[i] = MPI_REQUEST_NULL;
	    }
	}

	/* Must use LAM_MPI_C_STATUSES_IGNORE as MPISYSF #undefs
	   MPI_STATUSES_IGNORE due to name clash */
	err = MPI_Waitall(nreqs, reqs, LAM_MPI_C_STATUSES_IGNORE);

	free((char *) reqs);
	return(err);
}

/*
 *	free_errhdl
 *
 *	Function:	- deallocate pre-defined error handles
 */
static void
free_errhdl()

{
/*
 * Nothing to free.
 */
}

/*
 *	free_comm
 *
 *	Function:	- deallocate pre-defined communicators
 */
static void
free_comm()

{
        MPI_Errhandler err;
/*
 * Free the "parent" communicator if any.
 */
	if (lam_comm_parent != MPI_COMM_NULL) {
		lam_comm_free(lam_comm_parent);
	}
/*
 * Free the "world" communicator.
 * Free the error handler, if a user error handler was defined on it
 * (otherwise, we have a memory leak)
 */
	err = MPI_COMM_WORLD->c_errhdl;
	if (err != 0 && err != MPI_ERRHANDLER_NULL && 
	    err != MPI_ERRORS_ARE_FATAL && err != MPI_ERRORS_RETURN)
	  MPI_Errhandler_free(&MPI_COMM_WORLD->c_errhdl);
	lam_rmcid(MPI_COMM_WORLD->c_contextid);
	free((char *) MPI_COMM_WORLD->c_group);
	ah_free(MPI_COMM_WORLD->c_keys);

/*
 * Free the "self" communicator.
 * Free the error handler, if a user error handler was defined on it
 * (otherwise, we have a memory leak)
 */
	err = MPI_COMM_SELF->c_errhdl;
	if (err != 0 && err != MPI_ERRHANDLER_NULL && 
	    err != MPI_ERRORS_ARE_FATAL && err != MPI_ERRORS_RETURN)
	  MPI_Errhandler_free(&MPI_COMM_WORLD->c_errhdl);
	lam_rmcid(MPI_COMM_SELF->c_contextid);
	free((char *) MPI_COMM_SELF->c_group);
	if (MPI_COMM_SELF->c_keys) {
		ah_free(MPI_COMM_SELF->c_keys);
	}

	al_free(lam_comms);

	lam_nukecids();
}

/*
 *	free_dtype
 *
 *	Function:	- deallocate basic (intrinsic) datatypes
 */
static void
free_dtype()

{
/*
 * Nothing to free.
 */
}

/*
 *	free_rdtype
 *
 *	Function:	- deallocate reduction datatypes
 */
static void
free_rdtype()

{
	lam_type_free(MPI_2INT);
	lam_type_free(MPI_FLOAT_INT);
	lam_type_free(MPI_DOUBLE_INT);
	lam_type_free(MPI_LONG_DOUBLE_INT);
	lam_type_free(MPI_LONG_INT);
	lam_type_free(MPI_SHORT_INT);
	lam_type_free(MPI_F_2INTEGER);
	lam_type_free(MPI_F_2REAL);
	lam_type_free(MPI_F_2DOUBLE_PRECISION);
}

/*
 *	free_op
 *
 *	Function:	- deallocate intrinsic reduction operations
 */
static void
free_op()

{
/*
 * Nothing to free.
 */
}

/*
 *	free_env
 *
 *	Function:	- deallocate environment attributes
 */
static void
free_env()

{
	MPI_Attr_delete(MPI_COMM_WORLD, MPI_TAG_UB);
	MPI_Attr_delete(MPI_COMM_WORLD, MPI_HOST);
	MPI_Attr_delete(MPI_COMM_WORLD, MPI_IO);
	MPI_Attr_delete(MPI_COMM_WORLD, MPI_WTIME_IS_GLOBAL);
	MPI_Attr_delete(MPI_COMM_WORLD, MPI_UNIVERSE_SIZE);
	if (lam_appnum >= 0) {
	    MPI_Attr_delete(MPI_COMM_WORLD, MPI_APPNUM);
	}

	lam_nukekeys();
}

/*
 *	free_f77
 *
 *	Function:	- deallocate f77 support structures
 */
static void
free_f77()

{
	free((char *) lam_F_handles);
}
