/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/JRV
 *
 *	$Id: gather.c,v 6.9 2000/11/04 22:54:36 arodrig6 Exp $
 *
 *	Function:	- gather buffers at root in process rank order
 *	Accepts:	- send buffer
 *			- send count
 *			- send datatype
 *			- recv buffer
 *			- recv count
 *			- recv datatype
 *			- root
 *			- communicator
 *	Returns:	- MPI_SUCCESS or an MPI error code
 */

#include <lam_config.h>

#include <app_mgmt.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>
#if LAM_WANT_IMPI
#include <impi.h>
#endif


/*
 * Local functions
 */

static int gather(void *sbuf, int scount, MPI_Datatype sdtype, 
		  void *rbuf, int rcount, MPI_Datatype rdtype, 
		  int root, MPI_Comm comm);

/*@

MPI_Gather - Gathers together values from a group of processes
 
Input Parameters:
+ sbuf - starting address of send buffer (choice) 
. scount - number of elements in send buffer (integer) 
. sdtype - data type of send buffer elements (handle) 
. rcount - number of elements for any single receive (integer, 
significant only at root) 
. rdtype - data type of recv buffer elements 
(significant only at root) (handle) 
. root - rank of receiving process (integer) 
- comm - communicator (handle) 

Output Parameter:
. rbuf - address of receive buffer (choice, significant only at 'root') 

.N IMPI

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_INTERCOMM
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_BUFFER
.N MPI_ERR_ROOT

.N ACK
@*/
int MPI_Gather(void *sbuf, int scount, MPI_Datatype sdtype, 
	       void *rbuf, int rcount, MPI_Datatype rdtype, 
	       int root, MPI_Comm comm)
{
	int		rank;			/* my rank */
	int		size;			/* group size */

	lam_initerr();
	lam_setfunc(BLKMPIGATHER);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm, BLKMPIGATHER,
					lam_mkerr(MPI_ERR_COMM, 0)));
	}

	MPI_Comm_rank(comm, &rank);
	MPI_Comm_size(comm, &size);

	if ((root >= size) || (root < 0)) {
		return(lam_errfunc(comm, BLKMPIGATHER,
					lam_mkerr(MPI_ERR_ROOT, 0)));
	}

	if ((sdtype == MPI_DATATYPE_NULL)
			|| (rank == root && rdtype == MPI_DATATYPE_NULL)) {
		return(lam_errfunc(comm, BLKMPIGATHER,
					lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if ((scount < 0) || (rank == root && rcount < 0)) {
		return(lam_errfunc(comm, BLKMPIGATHER,

					lam_mkerr(MPI_ERR_COUNT, 0)));
	}

#if LAM_WANT_IMPI
	/* Remove this when IMPI collectives are implemented */
 
        if (LAM_IS_IMPI(comm)) {
	  return lam_err_comm(comm, MPI_ERR_COMM, 0, 
			      "Collectives not yet implemented on IMPI communicators");
	}
#endif

	LAM_TRACE(lam_tr_cffstart(BLKMPIGATHER));

#if LAM_WANT_IMPI && 0
        if (LAM_IS_IMPI(comm)) {
	  IMPI_Gather(sbuf, scount, sdtype, rbuf, rcount, rdtype, 
			    root, comm);
	} else {
	  gather(sbuf, scount, sdtype, rbuf, rcount, rdtype, root, comm);
	}
#else
	gather(sbuf, scount, sdtype, rbuf, rcount, rdtype, root, comm);
#endif
	LAM_TRACE(lam_tr_cffend(BLKMPIGATHER, root, comm, sdtype, scount));

	lam_resetfunc(BLKMPIGATHER);
	return(MPI_SUCCESS);
}


static int 
gather(void *sbuf, int scount, MPI_Datatype sdtype, 
       void *rbuf, int rcount, MPI_Datatype rdtype, 
       int root, MPI_Comm comm)
{
  int		i;			/* favourite index */
  int		err;			/* error code */
  int		rank;			/* my rank */
  int		size;			/* group size */
  char		*ptmp;			/* temporary buffer */
  MPI_Aint	incr;			/* increment size */
  MPI_Aint	extent;			/* datatype extent */
  struct _gps	*p;			/* favourite pointer */

  MPI_Comm_rank(comm, &rank);
  MPI_Comm_size(comm, &size);

/*
 * Remember required parameters.
 */
  p = &(comm->c_group->g_procs[root]->p_gps);
  
  lam_setparam(BLKMPIGATHER, root | (p->gps_grank << 16),
	       (p->gps_node << 16) | p->gps_idx);
/*
 * Switch to collective communicator.
 */
  lam_mkcoll(comm);
/*
 * Everyone but root sends data and returns.
 */
  if (rank != root) {

    err = MPI_Send(sbuf, scount, sdtype, root, BLKMPIGATHER, comm);
    lam_mkpt(comm);
    
    if (err != MPI_SUCCESS)
      return(lam_errfunc(comm, BLKMPIGATHER, err));
    
    LAM_TRACE(lam_tr_cffend(BLKMPIGATHER,
			    root, comm, sdtype, scount));
    
    lam_resetfunc(BLKMPIGATHER);
    return(MPI_SUCCESS);
  }
/*
 * I am the root, loop receiving the data.
 */
  MPI_Type_extent(rdtype, &extent);
  incr = extent * rcount;

  for (i = 0, ptmp = (char *) rbuf; i < size; ++i, ptmp += incr) {
/*
 * simple optimization
 */
    if (i == rank) {
      err = lam_dtsndrcv(sbuf, scount, sdtype, ptmp,
			 rcount, rdtype, BLKMPIGATHER, comm);
    } else {
      err = MPI_Recv(ptmp, rcount, rdtype, i,
		     BLKMPIGATHER, comm, MPI_STATUS_IGNORE);
    }
    
    if (err != MPI_SUCCESS) {
      lam_mkpt(comm);
      return(lam_errfunc(comm, BLKMPIGATHER, err));
    }
  }
  
  lam_mkpt(comm);
  
  return MPI_SUCCESS;
}

