/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: wwait.c,v 1.2 2001/02/03 20:51:50 arodrig6 Exp $
 *
 *	Function:	- one sided window wait
 *	Accepts:	- window
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>

/*@
MPI_Win_wait - Completes an RMA exposure epoch

Input Parameter:
+ win -  window object (handle) 

Notes:

Completes an RMA exposure epoch started by a call to MPI_WIN_POST on
win. This call matches calls to MPI_WIN_COMPLETE(win) issued by each
of the origin processes that were granted access to the window during
this epoch. The call to MPI_WIN_WAIT will block until all matching
calls to MPI_WIN_COMPLETE have occurred. This guarantees that all
these origin processes have completed their RMA accesses to the local
window. When the call returns, all these RMA accesses will have
completed at the target window.

.N fortran
.N IMPI_MPI2

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_WIN
.N MPI_ERR_EPOCH

.seealso MPI_Put(3), MPI_Get(3), MPI_Win_create(3), MPI_Win_start(3), MPI_Win_complete(3), MPI_Win_fence(3), MPI_Win_free(3), MPI_Win_get_group(3), MPI_Win_get_group(3), MPI_Accumulate(3)

.N WEB
  @*/


int MPI_Win_wait(MPI_Win win)
{
	MPI_Comm	comm;
	int		err;			/* error code */

	lam_initerr_m();
	lam_setfunc_m(BLKMPIWINWAIT);
/*
 * Check arguments.
 */
	if (win == MPI_WIN_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIWINWAIT, lam_mkerr(MPI_ERR_WIN, 0)));
	}

	comm = win->w_comm;

	if (!(win->w_flags & LAM_WFPOSTED)) {
		return(lam_errfunc(comm,
			BLKMPIWINWAIT, lam_mkerr(MPI_ERR_EPOCH, 0)));
	}
/*
 * Loop until everybody has completed.
 */
	_mpi_req_blkclr_m();
	_mpi_req_blkset_m(win->w_req);

	while (win->w_ncomplete > 0) {

		err = lam_osd_blkadv(win);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIWINWAIT, err));
		}
	}
/*
 * Complete all remote one-sided requests in the window.
 */
	err = lam_osd_complete(win, LAM_RQFOSTARG);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINWAIT, err));
	}

	win->w_flags &= ~LAM_WFPOSTED;

        lam_resetfunc_m(BLKMPIWINWAIT);
	return(MPI_SUCCESS);
}
