/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB/RBD/NJN
 *
 *	$Id: hcc.c,v 6.47.2.2 2001/10/07 19:18:09 jsquyres Exp $
 *
 *	Function:	- wrapper for C program compilation
 */

#include <lam_config.h>

#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>

#include <args.h>
#include <boot.h>
#include <portable.h>

#ifndef DEFTCC
#define DEFTCC LAM_CC
#endif


/*
 * external functions
 */
extern int		cnfexec();


/*
 * local functions
 */
static void strip_white(char* str);
static int check_file(char *dir, char *file);


/*
 * local variables
 */
static char		ph[4][LAM_PATH_MAX];
static char		plib[LAM_PATH_MAX];
static char		extra[LAM_PATH_MAX];


int
main(int argc, char *argv[])
{
	int		ac;		/* cc # args */
	int		status;		/* cc return code */
	int		i, j;		/* favourite index */
	int		fl_libs;	/* add libs to command line */
	int		fl_show;	/* show what would be done */
	int             fl_building;    /* are we building LAM right now? */
	int             fl_profile;     /* want profile lib */
	int             fl_cpp;         /* preprocessor, or compile/link? */
	int             fl_compiling_cpp; /* compiling C++ or not */
	int             num_incs;       /* num of -I's */
	int             num_libs;       /* num of -L's */
	char		**tempav;	/* temp cc arg vector */
	char		**av;		/* cc arg vector */
	char		**sys;		/* system dependent arg vector */
	char		*inchome;	/* include base dir */
	char		*libhome;	/* library base dir */
	char		*tcc;		/* cc tool */
	char		*argv0;		/* basename of argv0 */
	char            *syslibs = strdup(LAM_SYSLIBS);
	char            *extra_cflags = strdup(EXTRA_MPICC_FLAGS);
	char            *extra_cxxflags = strdup(EXTRA_MPICPP_FLAGS);
	char            *extra_flags;
#if LAM_WANT_ROMIO || LAM_WANT_MPI2CPP
	int             found;
#if LAM_WANT_MPI2CPP
	int             fl_want_mpi2cpp = 0;
#endif
#if LAM_WANT_ROMIO
	int             fl_want_romio = 0;
#endif
#endif
#if LAM_WANT_PROFILE
	int             added_lpmpi = 0;
#endif
#if LAM_HAVE_THREADS
	char            *threadcflags = LAM_SYSLIBS;
#endif

/*
 * LAM_TOP_BUILDDIR and LAM_TOP_SRCDIR should only be used internally
 * -- it is *not* a user-level flag (and as such, is not documented
 * :-)
 */
	libhome = getenv("LAM_TOP_BUILDDIR");
	if (libhome != 0) {
	  inchome = getenv("LAM_TOP_SRCDIR");
	  if (inchome == 0) {
	    show_help("compile", "internal_build", NULL);
	    exit(1);
	  }
	} else {
	  libhome = getenv("LAMHOME");
	  if (libhome == 0) libhome = getenv("TROLLIUSHOME");
	  if (libhome == 0) libhome = LAM_PREFIX;

	  inchome = libhome;
	}

	/* Clean up inchome and libhome -- trim any extra whitespace
           or any trailing /'s so that we can do some easy strcmp's
           below.  This also catches the corner case where $prefix is
           simply "/". */

	for (i = strlen(inchome) - 1; i >= 0; ++i) {
	  if (isspace(inchome[i]) || inchome[i] == '/')
	    inchome[i] = '\0';
	  else
	    break;
	}
	for (i = strlen(libhome) - 1; i >= 0; ++i) {
	  if (isspace(libhome[i]) || libhome[i] == '/')
	    libhome[i] = '\0';
	  else
	    break;
	}

	fl_compiling_cpp = 0;
	if (getenv("LAM_IS_COMPILING_CPP") != 0) {
	  fl_compiling_cpp = 1;
	  extra_flags = extra_cxxflags;
	} else
	  extra_flags = extra_cflags;
/*
 * Add system dependent defines.
 */
	fl_show = 0;
	fl_libs = 1;
	fl_building = 0;
	fl_profile = 0;
	fl_cpp = 0;

	for (i = 1; i < argc; ++i) {
		if (strcmp(argv[i], "-showme") == 0) {
			fl_show = 1;
		} else if (strcmp(argv[i], "-c") == 0) {
			fl_libs = 0;
		} else if (strcmp(argv[i], "-E") == 0 ||
			   strcmp(argv[i], "-M") == 0) {
			fl_libs = 0;
			fl_cpp = 1;
		} else if (strcmp(argv[i], "-S") == 0) {
			fl_libs = 0;
		} else if (strcmp(argv[i], "-lam-building") == 0) {
			fl_building = 1;
		} else if (strcmp(argv[i], "-lpmpi") == 0) {
			fl_profile = 1;
		}
	}
#if !LAM_WANT_PROFILE
/*
 * Sanity check
 */
	if (fl_profile == 1) {
	  show_help("compile", "no-profiling-support", argv[0], NULL);
	  fl_profile = 0;
	}
#endif
/*
 * Form include directory pathname.
 */
	if ((strlen(inchome) + 32) >= LAM_PATH_MAX ||
	    (strlen(libhome) + 32) >= LAM_PATH_MAX) {
	  char buffer[32];

	  snprintf(buffer, 32, "%d", LAM_PATH_MAX - 32);
	  if (strlen(inchome) + 32 >= LAM_PATH_MAX)
	    show_help("compile", "phome-too-long", inchome, buffer, "hcc", 
		      NULL);
	  else
	    show_help("compile", "phome-too-long", libhome, buffer, "hcc", 
		      NULL);
	  exit(1);
	}
/*
 * Form include directory pathname.
 * If we're building LAM:
 *
 * - if it's a VPATH build, lam_config.h will be in
 * LAM_TOP_BUILDDIR/share/include, and the rest will be in
 * LAM_TOP_SRCDIR/share/include.  But we also need the mpi2c++ header
 * files (ugh!), which are in LAM_TOP_SRCDIR/mpi2c++/src/mpi2c++, and
 * are referenced through LAM_TOPSRCDIR/mpi2c++/src.
 *
 * - if it's not a VPATH build, everything will be in
 * LAM_TOP_SRCDIR/share/include.  So it doesn't hurt to list both
 * (from above); it simplifies the logic.
 *
 * If we're not building LAM, everything is in $prefix/include 
 *
 * Special case: if $prefix is /usr and we're not lam_building, don't
 * -I it -- this may disrupt the compiler's normal sequence of include
 * directories.  This causes a problem for some versions of RedHat
 * (and probably other distros as well), for example.
 */
	num_incs = 0;
	if (fl_building) {
	  strcpy(ph[0], "-I");
	  strcat(ph[0], inchome);
	  strcat(ph[0], STRSDIR);
	  strcat(ph[0], "share");
	  strcat(ph[0], STRSDIR);
	  strcat(ph[0], "include");

	  strcpy(ph[1], "-I");
	  strcat(ph[1], inchome);
	  strcat(ph[1], STRSDIR);
	  strcat(ph[1], "mpi2c++");
	  strcat(ph[1], STRSDIR);
	  strcat(ph[1], "src");

	  strcpy(ph[2], "-I");
	  strcat(ph[2], inchome);
	  strcat(ph[2], STRSDIR);
	  strcat(ph[2], "mpi2c++");
	  strcat(ph[2], STRSDIR);
	  strcat(ph[2], "src");
	  strcat(ph[2], STRSDIR);
	  strcat(ph[2], "mpi2c++");

	  strcpy(ph[3], "-I");
	  strcat(ph[3], libhome);
	  strcat(ph[3], STRSDIR);
	  strcat(ph[3], "share");
	  strcat(ph[3], STRSDIR);
	  strcat(ph[3], "include");
	  num_incs = 4;
	} else {

	  /* Ensure that we don't -I/usr/include, for the reasons
             listed above */

	  if (strcmp(inchome, "/usr") != 0) {
	    strcpy(ph[0], "-I");
	    strcat(ph[0], inchome);
	    strcat(ph[0], STRSDIR);
	    strcat(ph[0], "include");
	    num_incs = 1;
	  }
	}
/*
 * Form library directory pathname.
 *
 * If we're building LAM, use "-LLAM_TOP_BUILDDIR/share
 * -LLAM_TOP_BUILDDIR/share/pmpi -LLAM_TOP_BUILDDIR/share/mpi".
 *
 * Otherwise, use "-L$prefix/lib".
 *
 * Apply similar logic here as we did with -I -- if libhome is /usr,
 * there's no need to explicitly add it, because the compiler will
 * already be looking there.
 */
	num_libs = 0;
	strcpy(plib, "-L");
	strcat(plib, libhome);
	strcat(plib, STRSDIR);
	strcat(plib, "lib");
	if (!fl_building && strcmp(libhome, "/usr") != 0) {
	  num_libs = 1;
	}

	tcc = getenv("LAMHCC");
	if (tcc == 0) tcc = getenv("TROLLIUSHCC");
	if (tcc == 0) tcc = DEFTCC;

	if ((tempav = sfh_argv_break(tcc, ' ')) == 0) {
	  show_help(NULL, "lib-call-fail", "sfh_argv_break", NULL);
	  exit(errno);
	}
	ac = sfh_argv_count(tempav);
	argv0 = strrchr(tempav[0], '/');
/*
 * Ugh.  If we're acting as the preprocessor, ditch any libtool
 * arguments.
 */
	if (fl_cpp &&
	    (strcmp(tempav[0], "libtool" ) == 0 ||
	     (argv0 != NULL && strcmp(argv0, "/libtool") == 0))) {
	  i = 1;
	  while (strncmp(tempav[i], "--", 2) == 0)
	    i++;

	  j = 0;
	  av = 0;
	  for (; i < ac; i++)
	    sfh_argv_add(&j, &av, tempav[i]);
	  
	  sfh_argv_free(tempav);
	  tempav = 0;
	  ac = j;
	}
/*
 * Ugh.  If the first argument is libtool and we're compiling a C++
 * file, we have to insert a "--mode" command because libtool doesn't
 * know how to infer the mode from C++ compilers [yet].
 */
	else if (fl_building && fl_compiling_cpp &&
	    (strcmp(tempav[0], "libtool" ) == 0||
	     (argv0 != NULL && strcmp(argv0, "/libtool") == 0))) {

	  j = 0;
	  av = NULL;
	  sfh_argv_add(&j, &av, tempav[0]);
	  if (fl_libs)
	    strcpy(extra, "--mode=link");
	  else
	    strcpy(extra, "--mode=compile");
	  sfh_argv_add(&j, &av, extra);

	  for (i = 1; i < ac; i++)
	    sfh_argv_add(&j, &av, tempav[i]);

	  sfh_argv_free(tempav);
	  tempav = 0;
	  ac = j;
	} else
	  av = tempav;
/*
 * If we're building LAM right now, #define LAM_BUILDING to be 1 so
 * that we don't try to include mpi++.h and mpio.h in mpi.h.  Also set
 * libtool to compile or link if we're compiling C++ (since it doesn't
 * [yet] recognize C++ compiler options).
 */
	if (fl_building) {
	  strcpy(extra, "-DLAM_BUILDING=1");
	  sfh_argv_add(&ac, &av, extra);
	} 
/* 
 * If we're not building LAM, check to see if we want ROMIO and the 
 * MPI 2 C++ bindings.  Optimize a bit -- if we're only doing -c, don't
 * check for ROMIO because it doesn't need any -I arguments
 */
#if LAM_WANT_ROMIO || LAM_WANT_MPI2CPP
	else {
#if LAM_WANT_ROMIO
	  if (fl_libs) {
	    found = check_file(plib + 2, "liblammpio.a");
	    if (found)
	      fl_want_romio = 1;
	    else {
	      found = check_file(plib + 2, "liblammpi.so");
	      if (found) 
		fl_want_romio = 1;
	      else {
		fl_want_romio = 0;
		if (!fl_building) {
		  fprintf(stderr, 
			  "WARNING: %s expected to find liblammpio.* in %s\n",
			  argv[0], plib + 2);
		  fprintf(stderr, 
			  "WARNING: MPI-2 IO support will be disabled\n");
		}
	      }
	    }
	  }
#endif

#if LAM_WANT_MPI2CPP
	  found = check_file(plib + 2, "liblammpi++.a");
	  if (found)
	    fl_want_mpi2cpp = 1;
	  else {
	    found = check_file(plib + 2, "liblammpi++.so");
	    if (found)
	      fl_want_mpi2cpp = 1;
	    else {
	      fl_want_mpi2cpp = 0;
	      if (!fl_building) {
		fprintf(stderr, 
			"WARNING: %s expected to find liblammpi++.* in %s\n",
			argv[0], plib + 2);
		fprintf(stderr, 
			"WARNING: MPI C++ API support will be disabled\n");
	      }
	    }
	  }
#endif
	}
#endif

	for (i = 0; i < num_incs; i++)
	  sfh_argv_add(&ac, &av, ph[i]);
	if (fl_libs) {
		if (num_libs == 1)
		  sfh_argv_add(&ac, &av, plib);

/*
 * Add system dependent link arguments.
 */
		if (strlen(LAM_HBIND) > 0) {
			if ((sys = sfh_argv_break(LAM_HBIND, ' ')) == 0) {
			  show_help(NULL, "lib-call-fail", "sfh_argv_break", 
				    NULL);
			  exit(errno);
			}

			for ( ; *sys; sys++) {
				sfh_argv_add(&ac, &av, *sys);
			}
		}
	}

/*
 * Don't copy -showme or -lam-building, and -lpmpi was already insertted
 */
	for (i = 1; i < argc; ++i) {
		if (strcmp(argv[i], "-showme") &&
		    strcmp(argv[i], "-lam-building") &&
		    strcmp(argv[i], "-lpmpi")) {
			sfh_argv_add(&ac, &av, argv[i]);
		}
	}

	if (fl_libs) {
/*
 * Do we want MPI 2 C++ bindings support?
 * Tricky -- we have to test for presence of libmpi++.a first, because
 * hcc is used to build libmpi++.a.  Chicken and egg problem.
 */
#if LAM_WANT_MPI2CPP
		if (fl_want_mpi2cpp == 1) {
		    sfh_argv_add(&ac, &av, "-llammpi++");
		}
#endif
/*
 * Do we want ROMIO support?
 * Tricky -- we have to test for presence of libmpio.a first, because
 * hcc is used to build libmpio.a.  Chicken and egg problem.
 */
#if LAM_WANT_ROMIO
		if (fl_want_romio == 1) {
		  sfh_argv_add(&ac, &av, "-llammpio");
		  /* Don't have to check for LAM_WANT_PROFILE here; we
		     can't have LAM_WANT_ROMIO without it */
		  sfh_argv_add(&ac, &av, "-lpmpi");
		  added_lpmpi = 1;
#if HAVE_LIBAIO
		  sfh_argv_add(&ac, &av, "-laio");
#endif
		}
#endif
/*
 * Add standard LAM libraries.  If we're building LAM, add the libtool
 * .la objects.
 */
		if (fl_building) {
		  strcpy(plib, libhome);
		  strcat(plib, STRSDIR);
		  strcat(plib, "share");
		  strcat(plib, STRSDIR);
		  strcat(plib, "libmpi.la");
		  sfh_argv_add(&ac, &av, plib);
	
#if LAM_WANT_PROFILE	  
		  strcpy(plib, libhome);
		  strcat(plib, STRSDIR);
		  strcat(plib, "share");
		  strcat(plib, STRSDIR);
		  strcat(plib, "pmpi");
		  strcat(plib, STRSDIR);
		  strcat(plib, "libpmpi.la");
		  sfh_argv_add(&ac, &av, plib);
		  added_lpmpi = 1;
#endif

		  strcpy(plib, libhome);
		  strcat(plib, STRSDIR);
		  strcat(plib, "share");
		  strcat(plib, STRSDIR);
		  strcat(plib, "liblam.la");
		  sfh_argv_add(&ac, &av, plib);
		} else {
		
#if LAM_WANT_PROFILE  
		  if (fl_profile && !added_lpmpi)
		    sfh_argv_add(&ac, &av, "-lpmpi");
#endif
#if LAM_WANT_FORTRAN
		  sfh_argv_add(&ac, &av, "-llamf77mpi");
#endif
		  sfh_argv_add(&ac, &av, "-lmpi");
		  sfh_argv_add(&ac, &av, "-llam");
		}
/*
 * Add system dependent libraries.
 */
		strip_white(syslibs);
		if (strlen(syslibs) > 0) {
			if ((sys = sfh_argv_break(syslibs, ' ')) == 0) {
			  show_help(NULL, "lib-call-fail", "sfh_argv_break", 
				    NULL);
			  exit(errno);
			}

			j = sfh_argv_count(sys);
			for (i = 0; i < j; i++)
			  sfh_argv_add(&ac, &av, sys[i]);
			sfh_argv_free(sys);
		}

/*
 * Add any extra flags
 */
		strip_white(extra_flags);
		if (strlen(extra_flags) > 0) {
			if ((sys = sfh_argv_break(extra_flags, ' ')) == 0) {
			  show_help(NULL, "lib-call-fail", "sfh_argv_break", 
				    NULL);
			  exit(errno);
			}
			
			j = sfh_argv_count(sys);
			for (i = 0; i < j; i++)
			  sfh_argv_add(&ac, &av, sys[i]);
			sfh_argv_free(sys);
		}
#if LAM_RPI_MYRI
		tempav = sfh_argv_break(LAM_RPI_REAL_LDFLAGS, ' ');
		j = sfh_argv_count(tempav);
		for (i = 0; i < j; i++)
		  sfh_argv_add(&ac, &av, tempav[i]);
		sfh_argv_free(tempav);
#endif
	}

	if (fl_show) {
		for (i = 0; i < ac; i++) {
			printf("%s ", av[i]);
		}
		printf("\n");
	} else {
		status = cnfexec(av);

		if (status != 0) {
		  if (errno != 0)
		    perror("hcc");
		  exit(status);
		}
	}

	/* Do some cleanup 'cause we're good little programmers who
           like to have bcheck-clean programs... */
	free(syslibs);
	free(extra_cflags);
	free(extra_cxxflags);
	sfh_argv_free(av);

	return(0);
}


/*
 * Remove leading and trailing white space from a given string.
 * Must be sent a null-terminated string.
 */
static void 
strip_white(char* str)
{
  char *src = str;

  /* Remove leading whitespace */

  for (; src != '\0'; src++) {
    if (!isspace(*src)) {
      strcpy(str, src);
      break;
    }
  }

  /* Remove trailing whitespace */

  for (src = str + strlen(str); src != str; src--) {
    if (isspace(*src)) 
      *src = '\0';
    else
      break;
  }
}


static int 
check_file(char *dir, char *file)
{
  int ret;
  struct stat buf;
  char *name = malloc(strlen(dir) + strlen(file) + 8);

  sprintf(name, "%s/%s", dir, file);
  ret = stat(name, &buf);
  free(name);

  return (ret == 0);
}
