/***************************************************************************
 *   Copyright (C) 2005 - 2006 by Christian Muehlhaeuser, Last.fm Ltd.     *
 *   chris@last.fm                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#ifdef WIN32
#include <windows.h>
#endif

#include <QtGui>
#include <QTimer>

#include "settings.h"
#include "tagdialog.h"
#include "webserviceconnector.h"


TagDialog::TagDialog( QWidget *parent )
    : QDialog( parent, Qt::Dialog )
    , m_collapsed( true )
    , m_song( new Song() )
{
    ui.setupUi( this );

    connect( WebserviceConnector::instance(), SIGNAL( tagsForArtistResult( QStringList ) ), this, SLOT( tagsForArtistResult( QStringList ) ) );
    connect( WebserviceConnector::instance(), SIGNAL( tagsForUserResult( QStringList ) ), this, SLOT( tagsForUserResult( QStringList ) ) );
    connect( WebserviceConnector::instance(), SIGNAL( tagsForUserArtistResult( QStringList ) ), this, SLOT( tagsForUserArtistResult( QStringList ) ) );
    connect( WebserviceConnector::instance(), SIGNAL( tagsForUserTrackResult( QStringList ) ), this, SLOT( tagsForUserTrackResult( QStringList ) ) );
    connect( WebserviceConnector::instance(), SIGNAL( tagsForUserAlbumResult( QStringList ) ), this, SLOT( tagsForUserAlbumResult( QStringList ) ) );

    connect( ui.tagTypeBox, SIGNAL( activated( int ) ), this, SLOT( tagTypeChanged( int ) ) );
    connect( ui.okButton, SIGNAL( clicked() ), this, SLOT( okPressed() ) );
    connect( ui.collapseButton, SIGNAL( clicked() ), this, SLOT( toggleCollapse() ) );
    connect( ui.personalTagsList, SIGNAL( itemDoubleClicked( QListWidgetItem* ) ), this, SLOT( addTag( QListWidgetItem* ) ) );
    connect( ui.publicTagsList, SIGNAL( itemDoubleClicked( QListWidgetItem* ) ), this, SLOT( addTag( QListWidgetItem* ) ) );

    WebserviceConnector::instance()->tagsForUser( Settings::instance()->username() );
}


void
TagDialog::setSong( Song *song )
{
    m_song->fetch( *song );
    WebserviceConnector::instance()->tagsForArtist( m_song->artist() );
    tagTypeChanged( ui.tagTypeBox->currentIndex() );

    ui.tagTypeBox->setItemText( 0, QString( "Artist (%1)" ).arg( m_song->artist() ) );
    ui.tagTypeBox->setItemText( 1, QString( "Album (%1 - %2)" ).arg( m_song->artist() ).arg( m_song->album() ) );
    ui.tagTypeBox->setItemText( 2, QString( "Song (%1 - %2)" ).arg( m_song->artist() ).arg( m_song->track() ) );
}


void
TagDialog::okPressed()
{
    WebserviceConnector::instance()->setTag( m_song, ui.tagTypeBox->currentIndex(), ui.tagEdit->text() );
}


void
TagDialog::toggleCollapse()
{
    if ( m_collapsed )
    {
        m_collapsed = false;
        layout()->setEnabled( true );

        QIcon image( QApplication::applicationDirPath() + "/data/collapse.png" );
        ui.collapseButton->setIcon( image );
        ui.collapseLabel->setText( "Hide favourites" );

        resize( 393, 373 );
        setMinimumSize( 393, 373 );
        setMaximumSize( 393 * 4, 373 * 4 );

        setSizeGripEnabled( true );
    }
    else
    {
        m_collapsed = true;
        layout()->setEnabled( false );

        QIcon image( QApplication::applicationDirPath() + "/data/expand.png" );
        ui.collapseButton->setIcon( image );
        ui.collapseLabel->setText( "Show favourites" );

        resize( 393, ui.extraLabel1->y() - 4 );
        setMinimumSize( 393, ui.extraLabel1->y() - 4 );
        setMaximumSize( 393, ui.extraLabel1->y() - 4 );

        setSizeGripEnabled( false );
    }
}


void
TagDialog::tagTypeChanged( int type )
{
    ui.tagTypeBox->setEnabled( false );
    ui.tagEdit->setText( "Loading..." );

    switch ( type )
    {
        case 0:
            WebserviceConnector::instance()->tagsForUserArtist( Settings::instance()->username(), m_song->artist() );
            break;

        case 1:
            WebserviceConnector::instance()->tagsForUserAlbum( Settings::instance()->username(), m_song->artist(), m_song->album() );
            break;

        case 2:
            WebserviceConnector::instance()->tagsForUserTrack( Settings::instance()->username(), m_song->artist(), m_song->track() );
            break;
    }
}


void
TagDialog::tagsForArtistResult( const QStringList &result )
{
    ui.publicTagsList->clear();

    for ( int i = 0; i < result.count(); i++ )
    {
        ui.publicTagsList->addItem( result[i] );
    }
}


void
TagDialog::tagsForUserResult( const QStringList &result )
{
    ui.personalTagsList->clear();

    for ( int i = 0; i < result.count(); i++ )
    {
        ui.personalTagsList->addItem( result[i] );
    }
}


void
TagDialog::tagsForUserArtistResult( const QStringList &result )
{
    QString tags = "";

    for ( int i = 0; i < result.count(); i++ )
    {
        tags += ( i > 0 ? ", " : "" ) + result[i];
    }

    ui.tagTypeBox->setEnabled( true );
    ui.tagEdit->setEnabled( true );
    ui.tagEdit->setText( tags );
    ui.tagEdit->setFocus();
}


void
TagDialog::tagsForUserTrackResult( const QStringList &result )
{
    QString tags = "";

    for ( int i = 0; i < result.count(); i++ )
    {
        tags += ( i > 0 ? ", " : "" ) + result[i];
    }

    ui.tagTypeBox->setEnabled( true );
    ui.tagEdit->setEnabled( true );
    ui.tagEdit->setText( tags );
    ui.tagEdit->setFocus();
}


void
TagDialog::tagsForUserAlbumResult( const QStringList &result )
{
    QString tags = "";

    for ( int i = 0; i < result.count(); i++ )
    {
        tags += ( i > 0 ? ", " : "" ) + result[i];
    }

    ui.tagTypeBox->setEnabled( true );
    ui.tagEdit->setEnabled( true );
    ui.tagEdit->setText( tags );
    ui.tagEdit->setFocus();
}


void
TagDialog::addTag( QListWidgetItem *item )
{
    if ( !ui.tagEdit->text().trimmed().isEmpty() )
        ui.tagEdit->setText( ui.tagEdit->text() + ", " + item->text() );
    else
        ui.tagEdit->setText( item->text() );
}

