/*
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright (C) 2003 NEC Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

int defcpu_ia64 = -1;
extern void *G_dump_header_asm;

/*
 * defcpu_ia64_cmd() -- Run the 'defcpu' command.
 */
int
defcpu_ia64_cmd(command_t *cmd)
{
	int cpu, num_cpus;
	void *ptr;
	kaddr_t stack, task;

	if (CORE_IS_KMEM) {
		fprintf(cmd->efp, "Can't use this command on live system\n");
		return (1);
	}
	if (cmd->nargs == 0) {
		if (defcpu_ia64 == -1) {
                	defcpu_ia64 = KL_INT(G_dump_header_asm,
                       		dha_typename, "dha_dumping_cpu");
		}
		fprintf(cmd->ofp, "Available cpus are :");
		num_cpus = KL_INT(G_dump_header_asm, 
			dha_typename, "dha_num_cpus");
		for (cpu = 0; cpu < num_cpus; cpu++) {
			fprintf(cmd->ofp, " %d", cpu);
			if (cpu == defcpu_ia64) {
				fprintf(cmd->ofp, "*");
			}
        		ptr = G_dump_header_asm + 
				kl_member_offset(dha_typename, 
				"dha_smp_current_task") + (cpu * KL_NBPW);
			if (!KL_GET_UINT64(ptr)) {
				fprintf(cmd->ofp, "!");
			}
		}
		fprintf(cmd->ofp, "\n");
		return(0);
	} 

	cpu = strtol(cmd->args[0], NULL, 10);

	if (cpu >= num_cpus) {
		fprintf(cmd->efp, "Error setting defcpu to %s\n", cmd->args[0]);
		return (1);
	}
	defcpu_ia64 = cpu;
	fprintf(cmd->ofp, "Default cpu is %d\n", defcpu_ia64);

        ptr = G_dump_header_asm + 
		kl_member_offset(dha_typename, "dha_stack") + (cpu * KL_NBPW);
	stack = KL_GET_UINT64(ptr);
	if (stack) {
        	ptr = G_dump_header_asm + kl_member_offset(dha_typename,
			"dha_smp_current_task") + (cpu * KL_NBPW);
		task = KL_GET_UINT64(ptr);
		deftask = task;
		fprintf(cmd->ofp, "Default task is 0x%016"FMT64"x\n", deftask);
	}
	return (0);
}

#define _DEFCPU_USAGE	"[-w outfile] [cpu]"

/*
 * defcpu_ia64_usage() -- Print the usage string for the 'defcpu' command.
 */
void
defcpu_ia64_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _DEFCPU_USAGE);
}

/*
 * defcpu_ia64_help() -- Print the help information for the 'defcpu' command.
 */
void
defcpu_ia64_help(command_t *cmd)
{
	CMD_HELP(cmd, _DEFCPU_USAGE,
	"Set the default cpu if one is indicated. Otherwise print the "
	"value of default cpu." 
        "When 'lcrash' is run on a live system, defcpu has no "
        "meaning.\n\n"
	"This command also sets the default task to the task running "
	"on the default cpu at the time the dump is taken. "
	"The rd command will display the registers on the default cpu "
	"at the time the dump is taken. "
        "The trace command will display a trace wrt the task "
        "running on the default cpu at the time the dump is taken. ");
}

/*
 * defcpu_ia64_parse() -- Parse the command line arguments for 'defcpu'.
 */
int
defcpu_ia64_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_WRITE), 0)) {
		return(1);
	}
	return(0);
}
