/*
 * $Id: cmd_mmap.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

/*
 * mmap_cmd() -- Run the 'mmap' command.
 */
int
mmap_cmd(command_t *cmd)
{
	int i, mmap_cnt = 0, mode, first_time = TRUE;
	void *mmp;
	kaddr_t mmap;
	uint64_t value;

	mmp = kl_alloc_block(MM_STRUCT_SZ, K_TEMP);
	if (KL_ERROR) {
		return(1);
	}
	for (i = 0; i < cmd->nargs; i++) {
		kl_get_value(cmd->args[i], &mode, 0, &value);
		if (KL_ERROR) {
			continue;
		}
		mmap = (kaddr_t)value;
		if (first_time || (cmd->flags & (C_FULL|C_NEXT))) {
			mm_struct_banner(cmd->ofp, BANNER|SMAJOR);
			first_time = FALSE;
		}

		if (mode == 2) {
			GET_BLOCK(mmap, MM_STRUCT_SZ, mmp);
			if (KL_ERROR) {
				continue;
			}
			print_mm_struct(mmap, mmp, cmd->flags, cmd->ofp);
			mmap_cnt++;
		} else {
			fprintf(KL_ERRORFP, "Invalid mm_struct : "
				"0x%"FMTPTR"x\n", mmap);
		}
	}
	kl_free_block(mmp);
	if (mmap_cnt) {
		mm_struct_banner(cmd->ofp, SMAJOR);
	}
	PLURAL("active mm_struct struct", mmap_cnt, cmd->ofp);
	return(0);
}

#define _MMAP_USAGE "[-f] [-n] [-w outfile] mmap_list"

/*
 * mmap_usage() -- Print the usage string for the 'mmap' command.
 */
void
mmap_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _MMAP_USAGE);
}

/*
 * mmap_help() -- Print the help information for the 'mmap' command.
 */
void
mmap_help(command_t *cmd)
{
	CMD_HELP(cmd, _MMAP_USAGE,
        "Display relevant information for each entry in mmap_list. ");
}

/*
 * mmap_parse() -- Parse the command line arguments for 'mmap'.
 */
int
mmap_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_TRUE|C_FULL|C_NEXT|C_WRITE), 0)) {
		return(1);
	}
	return(0);
}

/*
 * mmap_complete() -- Complete arguments of 'mmap' command.
 */
char *
mmap_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	mmap_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
