/*
 * $Id: cmd_savedump.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#include <lcrash.h>

#define DUMP_DEFAULT_DIR "."

static int dump_level = KL_DUMP_LEVEL_ALL;
static char dump_target_dir[1024] = DUMP_DEFAULT_DIR;

void vmdump(int, int, char* );

/*
 * savedump_cmd() -- Run the 'savedump' command.
 */
int
savedump_cmd(command_t *cmd)
{
	if(!(cmd->flags & C_LDUMP_COMPR_NONE) && !(cmd->flags & C_LDUMP_COMPR_RLE))
		cmd->flags |= C_LDUMP_COMPR_GZIP; /* default */
	vmdump(dump_level, cmd->flags,dump_target_dir);
	return(0);
}

#define _DUMP_USAGE_2_4 \
"[-c none|rle|gzip][-x [u][b][p][f]][-t <dir>][-i][-v]"

#define _DUMP_USAGE_2_6 \
"[-c none|rle|gzip][-x [u][p][f]][-t <dir>][-i][-v]"

/*
 * savedump_usage() -- Print the usage string for the 'savedump' command.
 */
void
savedump_usage(command_t *cmd)
{
	if(KL_LINUX_RELEASE >= LINUX_2_6_0)
		CMD_USAGE(cmd, _DUMP_USAGE_2_6)
	else
		CMD_USAGE(cmd, _DUMP_USAGE_2_4)
}

#define _DUMP_HELP_2_4 \
"This command can be used for two purposes:"\
"\n"\
"\n1. Create a live dump: When /dev/mem is specified at startup of lcrash,"\
"\nthis command can be used to create a live dump."\
"\n"\
"\n2. Reduce dump size: When a system dump is specified at startup of"\
"\nlcrash, this command can be used to reduce the size of the system dump"\
"\nusing compression and the '-x' option for discarding special memory"\
"\npages like buffer cache, page cache, free and user-space pages."\
"\nThis capability is especially useful, when huge dumps should be"\
"\ntransfered e.g. over a network."\
"\n"\
"\nOPTIONS:"\
"\n-c: compression type:"\
"\n       - none: No compression:"\
"\n       - rle:  Runlength encoding"\
"\n       - gzip: gzip compression (default)"\
"\n-x: exclude memory regions to reduce dump size:"\
"\n       - u: Exclude <U>ser pages"\
"\n       - b: Exclude <B>uffer cache pages"\
"\n       - p: Exclude <P>age cache pages"\
"\n       - f: Exclude <F>ree (unused) pages"\
"\n-t: target directory where dump is written (default is current directory)"\
"\n-i: write info file which describes dump content" \
"\n-v: verbose output"


#define _DUMP_HELP_2_6 \
"This command can be used for two purposes:"\
"\n"\
"\n1. Create a live dump: When /dev/mem is specified at startup of lcrash,"\
"\nthis command can be used to create a live dump."\
"\n"\
"\n2. Reduce dump size: When a system dump is specified at startup of"\
"\nlcrash, this command can be used to reduce the size of the system dump"\
"\nusing compression and the '-x' option for discarding special memory"\
"\npages like buffer cache, page cache, free and user-space pages."\
"\nThis capability is especially useful, when huge dumps should be"\
"\ntransfered e.g. over a network."\
"\n"\
"\nOPTIONS:"\
"\n-c: compression type:"\
"\n       - none: No compression:"\
"\n       - rle:  Runlength encoding"\
"\n       - gzip: gzip compression (default)"\
"\n-x: exclude memory regions to reduce dump size:"\
"\n       - u: Exclude <U>ser pages"\
"\n       - p: Exclude <P>age cache pages"\
"\n       - f: Exclude <F>ree (unused) pages"\
"\n-t: target directory where dump is written (default is current directory)"\
"\n-i: write info file which describes dump content" \
"\n-v: verbose output"

/*
 * savedump_help() -- Print the help information for the 'savedump' command.
 */
void
savedump_help(command_t *cmd)
{
	if(KL_LINUX_RELEASE >= LINUX_2_6_0)
		CMD_HELP(cmd, _DUMP_USAGE_2_6, _DUMP_HELP_2_6);
	else
		CMD_HELP(cmd, _DUMP_USAGE_2_4, _DUMP_HELP_2_4);
}

/*
 * savedump_parse() -- Parse the command line arguments for 'savedump'.
 */
int
savedump_parse(command_t *cmd)
{
	option_t *op;
	char* ptr;

	if (set_cmd_flags(cmd, 0, "c:x:vt:ib")) {
		return(1);
	}
	strcpy(dump_target_dir,DUMP_DEFAULT_DIR);
	op = cmd->options;
	while (op) {
		switch(op->op_char) {
			case 'c':
				if(strcmp(op->op_arg,"gzip") == 0){
					cmd->flags |= C_LDUMP_COMPR_GZIP;
				} else if(strcmp(op->op_arg,"rle") == 0){
					cmd->flags |= C_LDUMP_COMPR_RLE;
				} else if(strcmp(op->op_arg,"none") == 0){
					cmd->flags |= C_LDUMP_COMPR_NONE;
				} else {
					fprintf(KL_ERRORFP, "compression type '%s' not supported!\n",op->op_arg);
                                        return(1);
				}
				break;
			case 'x':
				ptr = op->op_arg;
				while(*ptr){
					if(KL_LINUX_RELEASE >= LINUX_2_6_0) {
						switch(*ptr){
							case 'u':
								cmd->flags |= C_LDUMP_X_USER_PAGES;
								break;
							case 'p':
								cmd->flags |= C_LDUMP_X_PCACHE;
								break;
							case 'f':
								cmd->flags |= C_LDUMP_X_FREE_PAGES;
								break;
							default:
								fprintf(KL_ERRORFP, "Invalid exclude memory type '%c'!\n",*ptr);
                                        			return(1);
						}
					} else {
						switch(*ptr){
							case 'u':
								cmd->flags |= C_LDUMP_X_USER_PAGES;
								break;
							case 'b':
								cmd->flags |= C_LDUMP_X_BCACHE;
								break;
							case 'p':
								cmd->flags |= C_LDUMP_X_PCACHE;
								break;
							case 'f':
								cmd->flags |= C_LDUMP_X_FREE_PAGES;
								break;
							default:
								fprintf(KL_ERRORFP, "Invalid exclude memory type '%c'!\n",*ptr);
                                        			return(1);
						}
					}
					ptr++;
				}
				break;
			case 'i':
				cmd->flags |= C_LDUMP_INFO_FILE; 
				break;
			case 't':
				strcpy(dump_target_dir,op->op_arg);
				break;
			case 'v':
				cmd->flags |= C_LDUMP_VERBOSE;
				break;
		}
		op = op->op_next;
	}
	return(0);
}

/*
 * savedump_complete() -- Complete arguments of 'savedump' command.
 */
char *
savedump_complete(command_t *cmd)
{
	char *ret;

	if(cmd->nargs >=2){
		if(strcmp(cmd->args[cmd->nargs -2],"-t") == 0){
			return (complete_file_name(cmd->args[cmd->nargs -1], 100));
		}
	}

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	savedump_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
