/*
 * $Id: dump.c,v 1.2 2005/02/23 19:31:14 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 *
 * Module for generating compressed system crash dumps from live 
 * system memory or from system dumps.
 *
 */

#include <lcrash.h>
#include <klib.h>
#include <zlib.h>

/* Defines
 */

/* #define VMDUMP */

/* strings for info file */

#define USER_PAGE    "U"
#define BCACHE_PAGE  "B"
#define PCACHE_PAGE  "P"
#define FREE_PAGE    "F"
#define MISSING_PAGE "M"

/* All existing exclude flags */

#define EXCL_FLAGS (C_LDUMP_X_FREE_PAGES |\
                    C_LDUMP_X_USER_PAGES |\
                    C_LDUMP_X_BCACHE |\
                    C_LDUMP_X_PCACHE)

/* junk size for "userpage checking" memory alloction */

#define TASKS_MMS_JUNKSZ 10

/* bit in struct page flags which indicates that page has blocks associated */
/* on disk. See include/linux/page-flags.h! */

#define PG_MAPPEDTODISK 0x20000 /* Bit 17. XXX check if this works also on   */
                                /* other architectures besides of s390/s390x */

// extern int get_dump_header_asm(dump_header_asm_t *);
// extern int get_dump_header(dump_header_t *);

extern kaddr_t get_pg_struct_addr(uint64_t, void *, int, unsigned long *);

/* static global variables 
 */

static kl_dump_header_t     dump_header;
static kl_dump_header_asm_t dump_header_asm;
static char *dpcpage = NULL;
static void *dump_page_buf = NULL;
static char *page_map;       /* page map with pages to dump */
static int  verbose = 0;     /* verbose flag */
static FILE* dump_info_file; /* for page info */
static uint64_t memory_size;

static kaddr_t* tasks_mm_structs = NULL; /* list of already checked mm structs */
static int tasks_mm_structs_last = 0;    /* index of last checked task in tasks_mm_structs */
static int tasks_mm_structs_last_alloc = 0; /* last allocated entry */

/* for statistics */

static int pcache_count;
static int pcache_count_sum;
static int buffer_count;
static int buffer_count_sum;
static int user_count;
static int user_count_sum;
static int free_count;
static int free_count_sum;

/* Compression routine
 */

kl_compress_fn_t vmdump_compress;

/* Forward declarations
 */

static int init_vmdump_header(int);

/*
 * init_vmdump()
 */
static int
init_vmdump(int flags)
{
	pcache_count = pcache_count_sum = 0;
	buffer_count = buffer_count_sum = 0;
	user_count   = user_count_sum   = 0;
	free_count   = free_count_sum   = 0;

	/* findout memory size */

        memory_size = NUM_PHYSPAGES * KL_PAGE_SIZE;

	fprintf(KL_ERRORFP,"Compression type\t\t: ");
	if(flags & C_LDUMP_COMPR_NONE){
		vmdump_compress = kl_compress_none;
		fprintf(KL_ERRORFP,"uncompressed\n");
	} else if(flags & C_LDUMP_COMPR_RLE){
		vmdump_compress = kl_compress_rle;
		fprintf(KL_ERRORFP,"RLE\n");
	} else if(flags & C_LDUMP_COMPR_GZIP){
		vmdump_compress = kl_compress_gzip;
		fprintf(KL_ERRORFP,"GZIP\n");
	} else {
		fprintf(KL_ERRORFP, "ERROR: compression type not supported!\n");
		return(1);
	}

	dpcpage = (char*)kl_alloc_block(KL_DUMP_PAGE_SIZE, K_TEMP);

	/* set up one-time dump header values 
	 */
	if (init_vmdump_header(flags) != 0) {
		fprintf(KL_ERRORFP, "init_vmdump_header() failed!\n");
		return(1);
	}

	if(flags & EXCL_FLAGS){
		fprintf(KL_ERRORFP,"Pages which will NOT be dumped\t: ");
	}
	if(flags & C_LDUMP_X_BCACHE){
		fprintf(KL_ERRORFP,"'buffer cache' ");
		/* XXX dump_header.dump_level |= DUMP_LEVEL_EXCL_BCACHE; */
	}
	if(flags & C_LDUMP_X_PCACHE){
		fprintf(KL_ERRORFP,"'page cache' ");
		/* XXX dump_header.dump_level |= DUMP_LEVEL_EXCL_PCACHE; */
	}
	if(flags & C_LDUMP_X_USER_PAGES){
		fprintf(KL_ERRORFP,"'user pages' ");
		tasks_mm_structs_last = 0;
		/* XXX dump_header.dump_level |= DUMP_LEVEL_EXCL_UPAGES; */
	}
	if(flags & C_LDUMP_X_FREE_PAGES){
		fprintf(KL_ERRORFP,"'free pages' ");
		/* XXX dump_header.dump_level |= DUMP_LEVEL_EXCL_UNUSED; */
	}
	fprintf(KL_ERRORFP,"\n");

	return(0);
}

/*
 * Name: vmdump_close()
 * Func: Close the vmdump file.
 */
static void
vmdump_close(FILE *dump_file)
{
	if (dump_file) {
		fclose(dump_file);
	}
	if (dump_info_file) {
		fclose(dump_info_file);
	}
}

#ifdef VMDUMP

/*
 * Name: vmdump_page_valid()
 * Func: Make sure the page address passed in is valid in the physical
 *       address space.  Right now, we validate all pages.
 */
static int
vmdump_page_valid(uint64_t mem_offset)
{
	if (kl_valid_physmem(mem_offset)) {
		return(1);
	} else {
		return(0);
	}
}

static uint64_t
vmdump_next_valid_page(uint64_t mem_offset)
{
	uint64_t vaddr;

	vaddr = kl_next_valid_physaddr(mem_offset);
	vaddr += KL_PAGE_OFFSET;
	return(vaddr);
}

#endif /* VMDUMP */

/*
 * Name: vmdump_write()
 * Func: Write out memory to vmdump file.
 */
static int
vmdump_write(FILE *dfp, char *buf, int len)
{
	if (fwrite(buf, len, 1, dfp) != 1) {
		return(-1);
	} else {
		return(0);
	}
}

/*
 * Name: copy_live_to_reg_header()
 * Func: initialize new dump header (output) from live system data
 */
static int      
copy_live_to_reg_header(int flags)
{
	char *utsname;
	int entry_size;
	struct timeval h_time;

	if(!(utsname = get_utsname())){
		return(-1);
	}

	/* Need to fill in data that normally is initialized by
	 * the kernel, such as start and size of memory, etc.
	 */
	dump_header.memory_size  = memory_size;
	dump_header.memory_start = KL_PAGE_OFFSET;
	dump_header.memory_end   = KL_PAGE_OFFSET + memory_size;

	/* write dump header initial values 
	 */
	dump_header.magic_number = KL_DUMP_MAGIC_LIVE;
	gettimeofday(&h_time, NULL);
	dump_header.time.tv_sec  = h_time.tv_sec;
	dump_header.time.tv_usec = h_time.tv_usec;
	dump_header.version = KL_DUMP_VERSION_NUMBER;
	dump_header.header_size = sizeof(kl_dump_header_t);
	dump_header.page_size = KL_DUMP_PAGE_SIZE;
	dump_header.dump_level     = KL_DUMP_LEVEL_ALL;

	if( KL_UTS_LEN <
	    (entry_size = kl_member_size(NEW_UTSNAME, "sysname")) ) {
		entry_size = KL_UTS_LEN;
	}

	memcpy((void *)dump_header.utsname_sysname,
	       K_PTR(utsname, NEW_UTSNAME, "sysname"), entry_size);
	memcpy((void *)dump_header.utsname_nodename,
	       K_PTR(utsname, NEW_UTSNAME, "nodename"), entry_size);
	memcpy((void *)dump_header.utsname_release,
	       K_PTR(utsname, NEW_UTSNAME, "release"), entry_size);
	memcpy((void *)dump_header.utsname_version,
	       K_PTR(utsname, NEW_UTSNAME, "version"), entry_size);
	memcpy((void *)dump_header.utsname_machine,
	       K_PTR(utsname, NEW_UTSNAME, "machine"), entry_size);
	memcpy((void *)dump_header.utsname_domainname,
	       K_PTR(utsname, NEW_UTSNAME, "domainname"), entry_size);

	// XXX dump_header_asm.magic_number = KL_DUMP_ASM_MAGIC_NUMBER;
	// XXX dump_header_asm.version = KL_DUMP_ASM_VERSION_NUMBER;
	dump_header_asm.header_size = sizeof(kl_dump_header_asm_t);

	/* make sure the dump header isn't TOO big 
	 */
	if ((sizeof(kl_dump_header_t) + KL_DHA_SIZE)
	    > KL_DUMP_BUFFER_SIZE) {
		fprintf(KL_ERRORFP, "init_vmdump_header(): combined "
			"headers larger than KL_DUMP_BUFFER_SIZE!\n");
                kl_free_block(utsname);
		return (-1);
	}

	kl_free_block(utsname);

	return(0);
}

/*
 * Name: copy_reg_to_reg_header()
 * Func: initialize new dump header (output) from current regular 
 *       dump header (input)
 */
static int
copy_reg_to_reg_header(int flags)
{
	if(kl_get_dump_header(&dump_header))
		return -1;
	dump_header.version = KL_DUMP_VERSION_NUMBER;
	dump_header.header_size = sizeof(kl_dump_header_t);
	// XXX if(kl_get_dump_header_asm(&dump_header_asm))
	// XXX	return -1;
	return 0;
}

/*
 * Name: copy_s390sa_to_reg_header()
 * Func: initialize new dump header (output) from current s390 
 *       dump header (input)
 */
static int
copy_s390sa_to_reg_header(int flags)
{
#if defined(DUMP_ARCH_S390) || defined(DUMP_ARCH_S390X)
	int entry_size;
	char* utsname;
	kl_dump_header_s390sa_t s390_dh;

	/* get the dump header
         */
        if (lseek(MIP->core_fd, 0, SEEK_SET) < 0) {
                fprintf(KL_ERRORFP, "Error: Cannot lseek() to get the dump header "
                        "from the dump file!\n");
                return(-1);
        }
        if (read(MIP->core_fd, (char *)&s390_dh,
                sizeof(s390_dh)) != sizeof(s390_dh)) {
                        fprintf(KL_ERRORFP, "Error: Cannot read() dump header "
                                "from dump file!\n");
                return(-1);
        }

	if (KLP->host->byteorder != KLP->dump->arch.byteorder)
		kl_swap_dump_header_s390sa(&s390_dh);
	kl_s390sa_to_reg_header(&s390_dh,&dump_header);

        if(!(utsname = get_utsname())){
                return(-1);
        }

        if( KL_UTS_LEN <
            (entry_size = kl_member_size(NEW_UTSNAME, "sysname")) ) {
                entry_size = KL_UTS_LEN;
        }
 
        memcpy((void *)dump_header.utsname_sysname,
               K_PTR(utsname, NEW_UTSNAME, "sysname"), entry_size);
        memcpy((void *)dump_header.utsname_nodename,
               K_PTR(utsname, NEW_UTSNAME, "nodename"), entry_size);
        memcpy((void *)dump_header.utsname_release,
               K_PTR(utsname, NEW_UTSNAME, "release"), entry_size);
        memcpy((void *)dump_header.utsname_version,
               K_PTR(utsname, NEW_UTSNAME, "version"), entry_size);
        memcpy((void *)dump_header.utsname_domainname,
               K_PTR(utsname, NEW_UTSNAME, "domainname"), entry_size);

	kl_free_block(utsname);
	return 0;
#else
	return -1;
#endif
}

/*
 * Name: init_vmdump_header()
 * Func: set the dump header initial values
 */
static int
init_vmdump_header(int flags)
{
	int rc = 0;
	switch(MIP->core_type){
		case dev_kmem:
			rc = copy_live_to_reg_header(flags);
			break;
		case reg_core:
		case cmp_core:
			rc = copy_reg_to_reg_header(flags);
			break;
		case s390_core:
			rc = copy_s390sa_to_reg_header(flags);
			break;
		default:
			fprintf(KL_ERRORFP, "invalid dump type: %x\n", MIP->core_type);
			rc = -1;
			break;
	}

	if(rc < 0)
		goto out;

        /*
         * In the case where we're compressing the pages, mark it as such.
         */
 
        if(flags & C_LDUMP_COMPR_NONE){
                dump_header.dump_compress = KL_DUMP_COMPRESS_NONE;
        } else if(flags & C_LDUMP_COMPR_RLE){
                dump_header.dump_compress = KL_DUMP_COMPRESS_RLE;
        } else if(flags & C_LDUMP_COMPR_GZIP){
                dump_header.dump_compress = KL_DUMP_COMPRESS_GZIP;
        } else {
                fprintf(KL_ERRORFP, "compression type not supported!\n");
		rc = -1;
		goto out;
        }
out:
	return rc;
}

/*
 * Name: write_vmdump_header()
 * Func: Write out the dump header to the dump device. 
 */
static int
write_vmdump_header(FILE *dfp)
{
        /* do byte swapping for dump header */

	kl_dump_header_t dh;
	dh = dump_header;

	if (KLP->host->byteorder != KLP->dump->arch.byteorder)
		kl_swap_dump_header_reg(&dh);

	/* clear the dump page buffer 
	 */
	memset(dump_page_buf, 0, KL_DUMP_BUFFER_SIZE);

	/* copy the dump header directly into the dump page buffer 
	 */
	memcpy((void *)dump_page_buf, (const void *)&dh, 
		sizeof(kl_dump_header_t));
	memcpy((void *)(dump_page_buf+dump_header.header_size),
		(const void *)&dump_header_asm, 
		sizeof(kl_dump_header_asm_t));

	/* Set the vmdump file pointer to the start of the file
	 */
	(void)fseek(dfp, 0L, SEEK_SET);

	/* write the header out to disk 
	 */

	if (vmdump_write(dfp, (char *)dump_page_buf, KL_DUMP_BUFFER_SIZE) < 0) {
		return (-1);
	}

	/* Reset the vmdump file pointer back to the end of the file
	 */
	(void)fseek(dfp, 0L, SEEK_END); 
	return (0);
}

/*
 * Functions for checking different page types.
 * These functions are used, to filter out pages,
 * which should not be dumped in order to reduce
 * the dump size 
 */
static void set_bit(char* buf, unsigned long bit){
	if(bit > NUM_PHYSPAGES){
	#if (ARCH == s390 || ARCH == s390x)
		fprintf(KL_ERRORFP,"warning (set_bit) : page %lx out of range (0 - %"FMT64"x)\n",bit,NUM_PHYSPAGES);
	#else
		/* on ia32 we could have high addresses for PCI etc.. */
	#endif
		return;
	}
	buf[bit/8] |= (1 << (bit % 8));
}

#if WE_NEED_THIS_STUFF
static void clear_bit(char* buf, unsigned long bit){
	if(bit > NUM_PHYSPAGES*8){
	#if (ARCH == s390 || ARCH == s390x)
		fprintf(KL_ERRORFP,"warning (clear_bit): page %lx out of range (0 - %"FMT64"x)\n",bit,NUM_PHYSPAGES);
	#else
		/* on ia32 we could have high addresses for PCI etc.. */
	#endif
		return;
	}
	buf[bit/8] &= (255 ^ (1 << (bit % 8)));
}
#endif

static int test_bit(char* buf, unsigned long bit){
	if(bit > NUM_PHYSPAGES){
	#if (ARCH == s390 || ARCH == s390x)
		fprintf(KL_ERRORFP,"warning (test_bit): page %lx out of range (0 - %"FMT64"x)\n",bit,NUM_PHYSPAGES);
	#else
		/* on ia32 we could have high addresses for PCI etc.. */
	#endif
		return 2;
	}
	return (buf[bit/8] & (1 << (bit % 8)));
}

/*
 * Name: check_free_pages_2_2()
 * Func: Parse buddy allocator structures (Linux 2.2) to find
 *       all unused pages and mark them
 */

/* defines from page_alloc.c */

#define NR_MEM_LISTS 10
#define NR_MEM_TYPES 2 

#if (ARCH == alpha) // alpha is the only 64 bit system for linux 2.2
#define FREE_AREA_STRUCT_SIZE 32 /* for 64 bit systems */
#else
#define FREE_AREA_STRUCT_SIZE 16 /* for 32 bit systems */
#endif

static int check_free_pages_2_2(void)
{
	syment_t     *sp;
	char*        free_area = NULL;
	int          free_area_struct_size,type,order,nr,size;

	sp = kl_lkup_symname("free_area");
	if(!sp)
		goto failed;

	/* Find out size of free_area_struct */

	free_area_struct_size = kl_struct_len("free_area_struct");
	if(!free_area_struct_size){
		fprintf(KL_ERRORFP,"Warning: no typeinfo for free_area_struct - assuming size = 16 bytes\n");
		free_area_struct_size = FREE_AREA_STRUCT_SIZE;
	}

	/* get the two dimensional array:
	 * free_area_struct free_area[NR_MEM_TYPES][NR_MEM_LISTS]
	 */

	size = free_area_struct_size * NR_MEM_LISTS * NR_MEM_TYPES;
	free_area =  kl_alloc_block(size, K_TEMP);
	if(!free_area)
		goto failed;
	GET_BLOCK(sp->s_addr, size, free_area);
	if(KL_ERROR)
		goto failed;
	for(type = 0; type < NR_MEM_TYPES; type++) {
		for (order=0 ; order < NR_MEM_LISTS; order++) {
			kaddr_t curr_a, end_a; /* pointer to "struct page" */
			char* act_entry = free_area + type*free_area_struct_size*NR_MEM_LISTS + order * free_area_struct_size;
			curr_a = kl_kaddr(act_entry, "list_head", "next");
                        end_a  = kl_kaddr(act_entry, "list_head", "prev");
			if(((curr_a >= sp->s_addr) && (curr_a <= (sp->s_addr + size ))) || (curr_a == 0) || (end_a == 0) ){
                                        /* empty free list:            */
                                        /*   next points to list head: */
                                        /* + consistency checks:       */
                                        /*   curr_a, end_a             */
                                        continue;
			}
                        for (;;) {
				void* lh = 0;
				size_t size=0;
				unsigned long pnr,i;
 
				/* Now lets go through the free list */
				/* and mark the pages we find        */

				/* page number = (address of "struct page" - start of struct page array (MEM_MAP)) / sizeof one "struct page" */
				pnr = (curr_a - MEM_MAP)/PAGE_SZ;

				for(i = 0; i < (1UL<<order); i++){
					free_count_sum++;
					if(!test_bit(page_map,i + pnr)){
						if(dump_info_file)
							fprintf(dump_info_file,"%016"FMT64"x X %s\n",(uint64_t)((pnr + i) * KL_PAGE_SIZE),FREE_PAGE);
						set_bit(page_map,i + pnr);
						free_count++;
					}
				}
				nr++;
				if(curr_a == end_a)
					/* end of ring list */
					break;
				if(kl_get_structure(curr_a,"list_head",&size,&lh))
					goto failed;
				curr_a = kl_kaddr(lh,"list_head","next");
				if(curr_a == 0)
					  break; /* consisteny check */
			}
			if(verbose)
				fprintf(KL_ERRORFP,"order %i: %5i * %3lu pages\n",order,nr,(1UL<<order));
		}
	}
	return 0;
failed:
	return -1;
}


/*
 * Name: check_free_pages_2_4()
 * Func: Parse zoned buddy allocator structures (Linux 2.4 and 2.6) to find 
 *       all unused pages and mark them
 */

#define MAX_NR_ZONES 3

static int check_free_pages_2_4(void)
{
	syment_t     *sp;
	void*        pgdat = NULL;
	unsigned int zone_type;
	kaddr_t      pg_dat_addr;
	size_t       size = 0;
	int	     zone_struct_size;
	int	     free_area_t_size;
	int	     pg_data_t_size;
	int          max_order;

	/* Get sizes */

	pg_data_t_size   = kl_struct_len("pg_data_t");
	if(KL_LINUX_RELEASE >= LINUX_2_6_0){
		free_area_t_size = kl_struct_len("free_area");
		zone_struct_size = kl_struct_len("zone");
		max_order = kl_member_size("zone","free_area") / free_area_t_size - 1;
	}
	else {
		free_area_t_size = kl_struct_len("free_area_t");
		zone_struct_size = kl_struct_len("zone_struct");
		max_order = kl_member_size("zone_struct","free_area") / free_area_t_size - 1;
	}
	if(!zone_struct_size || !pg_data_t_size || !free_area_t_size)
		goto failed;

	/* Get pg_data_t contig_page_data */

	sp = kl_lkup_symname("contig_page_data" /* "pgdat_list" */);
	if(!sp)
		goto failed;
	pg_dat_addr = sp->s_addr;
	if(kl_get_structure(pg_dat_addr,"pg_data_t",&size,&pgdat))
		goto failed;
	
	/* Check for all zones all free_areas */

        for (zone_type = 0; zone_type < MAX_NR_ZONES; zone_type++) {
                void *head,*zone;
		unsigned long free_pages;

                zone = (char*)K_PTR(pgdat,"pg_data_t","node_zones") + zone_type * zone_struct_size;
		if(KL_LINUX_RELEASE >= LINUX_2_6_0){
			free_pages = KL_UINT(zone,"zone","free_pages");
		} else {
			free_pages = KL_UINT(zone,"zone_struct","free_pages");
		}

		if(verbose)
			fprintf(KL_ERRORFP,"\nzone: %i free pages: %lu\n",zone_type,free_pages); 
                if (free_pages) {
			int order;
                        for (order = 0; order <= max_order; order++) {
				int nr = 0;
				void* free_area;
				kaddr_t curr_a, end_a;
				if(KL_LINUX_RELEASE >= LINUX_2_6_0){
                                	free_area = (char*)(K_PTR(zone,"zone","free_area")) + order * free_area_t_size;
				} else {
                                	free_area = (char*)(K_PTR(zone,"zone_t","free_area")) + order * free_area_t_size;

				}
				if(KL_LINUX_RELEASE >= LINUX_2_6_0)
					head = K_PTR(free_area,"free_area","free_list"
);
				else
					head = K_PTR(free_area,"free_area_t","free_list");
				curr_a = kl_kaddr(head, "list_head", "next");
				end_a  = kl_kaddr(head, "list_head", "prev");
				if(((curr_a >= pg_dat_addr) && (curr_a <= (pg_dat_addr + pg_data_t_size))) || (curr_a == 0) || (end_a == 0)){
					/* empty free list:           */
					/*   next points to list head */
					/* + consistency checks:      */
					/*   curr_a, end_a            */
					continue;
				}
                                for (;;) {
					void* lh = 0;
					size_t size=0;
					unsigned long pnr,i;

					/* Now lets go through the free list */

					/* page number = (address of "struct page" - start of struct page array (MEM_MAP)) / sizeof one "struct page" */
					pnr = (curr_a - MEM_MAP)/PAGE_SZ;
                                        for(i = 0; i < (1UL<<order); i++){
						free_count_sum++;
						if(!test_bit(page_map,i + pnr)){
							if(dump_info_file)
								fprintf(dump_info_file,"%016"FMT64"x X %s\n",((uint64_t)(pnr + i) * KL_PAGE_SIZE),FREE_PAGE);
                                                	set_bit(page_map,i + pnr);
							free_count++;
						}
					}
					nr++;
					if(curr_a == end_a)
						break; /* end of list found */
					if(kl_get_structure(curr_a,"list_head",&size,&lh))
						goto failed;
					curr_a = kl_kaddr(lh,"list_head","next");
					if(curr_a == 0)
						break; /* consistency check */

                                }
				if(verbose)
					fprintf(KL_ERRORFP,"order %i: %5i * %3lu pages\n",order,nr,(1UL<<order));
                        }
		}
	}
	fprintf(KL_ERRORFP,"ok\n");
	return 0;
failed:
	fprintf(KL_ERRORFP,"failed\n");
	return -1;	
}

/*
 * Name: check_free_pages()
 * Func: Parse buddy allocator structures to find
 *       all unused pages and mark them
 */
static int check_free_pages(void)
{
        if(verbose)
                fprintf(KL_ERRORFP,"\n======================\n");
        fprintf(KL_ERRORFP,"Checking unused pages: ");
        if(verbose)
                fprintf(KL_ERRORFP,"\n======================\n");

	if (LINUX_2_2_X(KL_LINUX_RELEASE)) {
		return check_free_pages_2_2();
	} else {
		return check_free_pages_2_4();
	}
}

/*
 * Name: is_new_addr_space()
 * Func: Check if we already have processed a task with this "mm"
 *       If not, we have a new task here. If yes, we found a 
 *       thread.
 *       RETVAL: 1 - new addr space found
 *               0 - thread found
 */
static int is_new_addr_space(kaddr_t mm){
	int i,rc = 0;
	if(tasks_mm_structs_last >= tasks_mm_structs_last_alloc){
		void* ptr;
		/* need to allocate more storage */
		ptr = realloc(tasks_mm_structs,(tasks_mm_structs_last_alloc + TASKS_MMS_JUNKSZ) * sizeof(kaddr_t));
		if(!ptr){
			rc = 1;
			goto out;
		} else {
			tasks_mm_structs_last_alloc += TASKS_MMS_JUNKSZ;
			tasks_mm_structs = ptr;
		}
	}
	for(i = 0; i < tasks_mm_structs_last; i++){
		if(tasks_mm_structs[i] == mm){
			rc = 0;
			goto out;
		}
	}
	tasks_mm_structs[tasks_mm_structs_last] = mm;
	tasks_mm_structs_last++;
	rc = 1;
out:
	return rc;
}

/*
 * Name: check_user_pages()
 * Func: Parse task structures to find out all user pages
 *       and mark those pages
 */
int check_user_pages(void)
{
        kaddr_t addr, first_task;

        void *tsp;
	int  offset;

        if(verbose)
                fprintf(KL_ERRORFP,"\n======================\n");
	fprintf(KL_ERRORFP,"Checking user pages  : ");
        if(verbose)
                fprintf(KL_ERRORFP,"\n======================\n");

	if(get_first_task(&first_task)){
		goto failed;
	}

	if (KL_LINUX_RELEASE >= LINUX_2_6_0) {
		offset = kl_member_offset("task_struct","tasks");
	}

        if (!(tsp = kl_alloc_block(TASK_STRUCT_SZ, K_TEMP)))
		goto failed;

	/* loop through all task structs */

        addr = first_task;
        do {
		kaddr_t prev = 0; 
		kaddr_t mm_addr;
                if (kl_get_task_struct(addr, 2, tsp))
                        break;

		/* consitency check */
		if(prev && (kl_kaddr(tsp, "task_struct", "prev_task") != prev)){
			 fprintf(KL_ERRORFP,"task list damaged\n");
			 goto failed;			 
		}
		mm_addr = kl_kaddr(tsp, "task_struct", "mm");
		if(mm_addr == 0){
			/* Kernel task */
#if 0
			if(verbose)
				fprintf(KL_ERRORFP,"kernel task: %s\n",(char*)(K_ADDR(tsp,"task_struct","comm")));
#endif
		} else if(!is_new_addr_space(mm_addr)){
			if(verbose)
				fprintf(KL_ERRORFP,"task: %15s (thread)\n",(char*)(K_ADDR(tsp,"task_struct","comm")));
		} else {
			size_t size = 0;
			void* mm_struct = NULL;
			kaddr_t vma_addr;
			unsigned long user_mem = 0;
			unsigned long user_mem_shared = 0;
			if(verbose)
				fprintf(KL_ERRORFP,"task: %15s - ",(char*)(K_ADDR(tsp,"task_struct","comm")));
			kl_get_structure(mm_addr,"mm_struct",&size,&mm_struct);
			vma_addr = kl_kaddr(mm_struct,"mm_struct","mmap");
			while(vma_addr != 0){
				kaddr_t i,start, end;
				void* vm_area_struct = NULL;
				size = 0;
				kl_get_structure(vma_addr,"vm_area_struct",&size,&vm_area_struct);
				start = KL_UINT(vm_area_struct,"vm_area_struct","vm_start");
				end   = KL_UINT(vm_area_struct,"vm_area_struct","vm_end");

				/* now check all pages from start to end, */
				/* if they are in physical memory         */
				for(i = start; i < end; i+=KL_PAGE_SIZE){
					kaddr_t p_addr;
					KL_VIRTOP(i,mm_struct,&p_addr);
					if(!KL_ERROR){
						/* page is not swapped out */
						user_count_sum++;
	 					// printf("%x/%x\n",p_addr,i);
						// XXX test rc 2 of test_bit !!! 
						if(!test_bit(page_map,p_addr/KL_PAGE_SIZE)){
							if(dump_info_file)
								fprintf(dump_info_file,"%016"FMT64"x X %s\n",(uint64_t)p_addr,USER_PAGE);
							set_bit(page_map,p_addr/KL_PAGE_SIZE);
							user_count++;
							user_mem += KL_PAGE_SIZE;
						} else {
							user_mem_shared += KL_PAGE_SIZE;
						}
					} else {
						/* page is swapped out */
						kl_reset_error();
					}
				}
				vma_addr = kl_kaddr(vm_area_struct,"vm_area_struct","vm_next");
			}
			if(verbose)
				fprintf(KL_ERRORFP,"%8lu KB/%8lu KB (shared)\n",user_mem / 1024,user_mem_shared / 1024);
			kl_free_block(mm_struct);

		}
		prev = addr;
		if(KL_LINUX_RELEASE < LINUX_2_6_0){
			addr = kl_kaddr(tsp, "task_struct", "next_task");
		} else {
			addr = kl_kaddr(((char*)tsp) + offset, "list_head","next") - offset;
		}
        } while (addr != first_task);
        kl_free_block(tsp);

        fprintf(KL_ERRORFP,"ok\n");
        return 0;
failed:
        fprintf(KL_ERRORFP,"failed\n");
        return -1;
}	

/*
 * Name: check_cache_pages()
 * Func: Parse page structures to find out all pages, which
 *       are used from buffer cache or page cache and mark those 
 *       pages
 */
int check_cache_pages(int flags)
{
        char pp[1024];
        kaddr_t p;
        uint64_t i;
	void *pgdat = NULL;

        if(verbose)
                fprintf(KL_ERRORFP,"\n===========================\n");
        fprintf(KL_ERRORFP,"Checking Buffer/Page Cache: ");
        if(verbose)
                fprintf(KL_ERRORFP,"\n===========================\n");

	pgdat  = kl_alloc_block(kl_struct_len("pg_data_t"), K_TEMP);
        if (KL_ERROR) {
		fprintf(KL_ERRORFP,"failed\n");
                return(-1);
        }

	for (i = 0; i < NUM_PHYSPAGES; i++) {
                uint64_t buffers   = 0;
                uint64_t pagecache = 0;
		unsigned long pnr;
		if (!(p = get_pg_struct_addr(i, pgdat, 1, &pnr))) {
			continue;
		}
		if(verbose && ((i % (NUM_PHYSPAGES/8)) == 0))
			fprintf(KL_ERRORFP,"Reached page: %"FMT64"i\n",i);
		GET_BLOCK(p, PAGE_SZ, pp);
		if (KL_ERROR)
			goto failed;
                buffers = ADDR64(KL_UINT(pp, "page", "buffers"));
		if (LINUX_2_2_X(KL_LINUX_RELEASE)) {
			pagecache = ADDR64(KL_UINT(pp, "page", "inode"));
		} else if(KL_LINUX_RELEASE < LINUX_2_6_0){
			pagecache = ADDR64(KL_UINT(pp, "page", "mapping"));
		} else {
			pagecache = ADDR64(KL_UINT(pp, "page", "flags"));
			if(pagecache & PG_MAPPEDTODISK)
				pagecache = 1;
			else
				pagecache = 0;
		}
		if((flags & C_LDUMP_X_BCACHE) && buffers){
			buffer_count_sum++;
			if(!test_bit(page_map,i)){
				if(dump_info_file)
					fprintf(dump_info_file,"%016"FMT64"x X %s\n",i*KL_PAGE_SIZE,BCACHE_PAGE);
				buffer_count++;
				set_bit(page_map,i);
			}
		} 
                if((flags & C_LDUMP_X_PCACHE) && pagecache){
			pcache_count_sum++;
                        if(!test_bit(page_map,i)){
				if(dump_info_file)
					fprintf(dump_info_file,"%016"FMT64"x X %s\n",i,PCACHE_PAGE);
                                pcache_count++;
                                set_bit(page_map,i);
                        }
                }
	}
	kl_free_block(pgdat);
        fprintf(KL_ERRORFP,"ok\n");
        return 0;
failed:
	kl_free_block(pgdat);
        fprintf(KL_ERRORFP,"failed\n");
        return -1;
}

/*
 * Name: is_to_dump()
 * Func:    mem_loc: pysical address of page
 *          RETVAL:  0: page should not be dumped
 *                   1: page should be dumped
 */
static inline int is_to_dump(uint64_t mem_loc)
{
        if(test_bit(page_map,mem_loc/KL_PAGE_SIZE)){
                return 0;
        } else {
                return 1;
        }
}

/*
 * Name: vmdump()
 * Func: System dumping routine.
 */
void
vmdump(int dump_level, int flags, char* dump_target_dir)
{
	int ifd = -1, ofd = -1, count, bounds = 0;
	kl_dump_page_t dp;
	uint64_t dp_address;
	uint32_t dp_size,dp_flags;
	uint64_t mem_loc;
	uint32_t buf_loc = 0;
	uint32_t cloop = 0, psize = sizeof(kl_dump_page_t);
	int size;
	FILE *dump_file;
	char *buf;
	char dname[1024], mname[1024], kname[1024],iname[1024];
	struct stat st_info;
 
	/* Check target directory
	 */
 
	if(stat(dump_target_dir,&st_info)){
		fprintf(KL_ERRORFP,"Target directory '%s' does not exist\n",dump_target_dir);
		return;
	} else if(!S_ISDIR(st_info.st_mode)){
		fprintf(KL_ERRORFP,"Specified target directory '%s' is not a directory\n",dump_target_dir);
		return;
	}

	/* Check verbose output flag
	 */
	if(flags & C_LDUMP_VERBOSE)
		verbose  = 1;
	else
		verbose  = 0;

	/* Make sure that no exclude flags are set for live dumps
         */
	if (CORE_IS_KMEM && (flags & EXCL_FLAGS)){
		fprintf(KL_ERRORFP, "Exclude flags are not permitted for live dumps\n");
		return;
	}

	/* 
	 * Allocate and clear some memory for the dump_page_buf and buf
	 */
	buf = (void*)malloc(KL_DUMP_PAGE_SIZE);
	dump_page_buf = (void*)malloc(KL_DUMP_BUFFER_SIZE + KL_DUMP_PAGE_SIZE + psize);
	memset(dump_page_buf, 0, KL_DUMP_BUFFER_SIZE + KL_DUMP_PAGE_SIZE +
psize);

	/* Check to see if a file named dump.0 exists. If it does, 
	 * then bump the number and keep testing until a match is not 
	 * found. Use the resulting filename as the dump file name.
	 */
	while (1) {
		sprintf(dname, "%s/%s.%d", dump_target_dir, "dump", bounds);
		if ((ofd = open(dname, O_RDONLY)) == -1) {
			break;
		}
		bounds++;
		close(ofd);
	} 

	/* Copy the System.map file, adding the appropriate bounds number
	 */
	sprintf(mname, "%s/map.%d", dump_target_dir,bounds);

	/* open original System.map */
	if ((ifd = open(KL_MAP_FILE, O_RDONLY)) == -1) {
		fprintf(KL_ERRORFP, "Could not open %s for reading!\n",
			KL_MAP_FILE);
		goto cleanup;
	}
	/* open new System.map */
	if ((ofd = open(mname, O_WRONLY|O_CREAT, 0644)) == -1) {
		fprintf(KL_ERRORFP, "Could not open %s for writing!\n", mname);
		goto cleanup;
	}
	/* copy org to new */
	while((count = read(ifd, dump_page_buf, KL_DUMP_BUFFER_SIZE))) {
		if (count == -1) {
			fprintf(KL_ERRORFP, 
				"Error reading %s (%d)!\n", KL_MAP_FILE, errno);
			goto cleanup;
		} 
		if (write(ofd, dump_page_buf, count) == -1) {
			fprintf(KL_ERRORFP, "Error writing to %s (%d)\n", 
				mname, errno);
			goto cleanup;
		}
	}

	close(ifd); ifd = -1;
	close(ofd); ofd = -1;

	/* Copy the Kerntypes file, adding the appropriate bounds number
	 */
	sprintf(kname, "%s/kerntypes.%d", dump_target_dir, bounds);

	/* open original kerntypes */
	if ((ifd = open(KL_KERNTYPES_FILE, O_RDONLY)) == -1) {
		fprintf(KL_ERRORFP, "Could not open %s for reading!\n",
			KL_KERNTYPES_FILE);
		goto cleanup;
	}
	/* open new kerntypes */
	if ((ofd = open(kname, O_WRONLY|O_CREAT, 0644)) == -1) {
		fprintf(KL_ERRORFP, "Could not open %s for writing!\n", kname);
		goto cleanup;
	}
	/* copy org to new */
	while((count = read(ifd, dump_page_buf, KL_DUMP_BUFFER_SIZE))) {
		if (count == -1) {
			fprintf(KL_ERRORFP, 
				"Error reading %s (%d)!\n", KL_KERNTYPES_FILE,
				errno);
			goto cleanup;
		} 
		if (write(ofd, dump_page_buf, count) == -1) {
			fprintf(KL_ERRORFP, "Error writing to %s (%d)\n", 
				kname, errno);
			goto cleanup;
		}
	}

	close(ifd); ifd = -1;
	close(ofd); ofd = -1;

	/* Open the dump file for output
	 */
	if (!(dump_file = fopen(dname, "w"))) {
		fprintf(KL_ERRORFP, 
			"Could not open %s for writing!\n", dname);
		return;
	}
	if (fchmod(fileno(dump_file), 0664)) {
		fprintf(KL_ERRORFP,
			"Could not set file mode for %s!\n", dname);
		return;
	}

	/* Open dump_info 
	 */

	if(flags & C_LDUMP_INFO_FILE){
		sprintf(iname, "%s/dump_info.%d", dump_target_dir, bounds);
        	if (!(dump_info_file = fopen(iname, "w"))) {
                	fprintf(KL_ERRORFP,
                        	"Could not open %s for writing!\n", iname);
                	return;
        	}
        	if (fchmod(fileno(dump_info_file), 0664)) {
			fprintf(KL_ERRORFP,
				"Could not set file mode for %s!\n", iname);
			return;
		}
        }

	if (init_vmdump(flags)) {
		fprintf(KL_ERRORFP, "init_vmdump() failed!\n");
		goto cleanup;
	}


	fprintf(KL_ERRORFP, "Creating system dump...\n");
	fprintf(KL_ERRORFP, "map = %s, vmdump = %s, kerntypes = %s\n", 
		mname, dname, kname);

	/* dump out the header 
	 */
	fprintf(KL_ERRORFP, "Writing dump header ...\n");
	if (write_vmdump_header(dump_file) < 0) {
		fprintf(KL_ERRORFP, "write_vmdump_header() failed!\n");
		goto cleanup;
	}

	/* if we only want the header, return 
	 */
	if (dump_level == KL_DUMP_LEVEL_HEADER) {
		fprintf(KL_ERRORFP, "\nDump complete.\n");
		goto cleanup;
	}
        page_map = (char*)malloc(NUM_PHYSPAGES/8);
	if(!page_map){
		fprintf(KL_ERRORFP, "\nalloc failed!\n");
		goto cleanup;
	}
	memset(page_map,0,NUM_PHYSPAGES/8); 

	/* Exlude flags processing 
	 */

	if(flags & C_LDUMP_X_USER_PAGES)
		check_user_pages();
        if(flags & C_LDUMP_X_FREE_PAGES)
                check_free_pages();
        if((flags & C_LDUMP_X_BCACHE) || (flags & C_LDUMP_X_PCACHE))
                check_cache_pages(flags);

	fprintf(KL_ERRORFP, "Writing dump pages ...");

	/* get the first memory page from the start of memory
	 */
#ifdef VMDUMP
	mem_loc = dump_header.memory_start;
#else
	mem_loc = 0; /* dump_header.memory_start - KL_PAGE_OFFSET; */
#endif

	dump_header.num_dump_pages = 0;

	/* Start walking through each page of memory, dumping it out
	 * as you go. We need to fit as much as possible into a 64K 
	 * write().  So we write dump page header, then the page
	 * itself (which may be compressed), etc., etc., etc., until
	 * we can't fit another compressed page into the write buffer.
	 * We then write the entire page out to the dump device.
	 */

#ifdef VMDUMP
	while (mem_loc < memory_size + KL_PAGE_OFFSET) {
#else
	while (mem_loc < memory_size) {
#endif
#ifdef VMDUMP
		paddr_t tmp;
#endif

 
                /* see if we want to print out a '.'
                 */
        	cloop++;
        	if ((cloop % (NUM_PHYSPAGES/32) == 0)) {
                	fprintf(KL_ERRORFP, ".");
        	}
#ifdef VMDUMP
		/* make sure the address is valid in the physical space
		 */
		if (!vmdump_page_valid(mem_loc)) {
			mem_loc = vmdump_next_valid_page(mem_loc);
			if(dump_info_file){
				fprintf(dump_info_file,"%016"FMT64"x X %s\n",mem_loc,MISSING_PAGE);
			}
			continue;
		}
#endif

		/* Create the dump header. Since we are walking through
		 * virtual address space, we have to convert mem_loc to
		 * a physical address before we put it into the page 
		 * header. Since the conversion of virtual to physical 
		 * address is very architecture specific, we make a call 
		 * to kl_virtop() to take care of this.
		 */
		/* introduced tmp here to be able to use this stuff
		 * on big endian 32bit machines, too
		 */
#ifdef VMDUMP
		if (KL_VIRTOP((kaddr_t)mem_loc, NULL, &tmp)) {
			fprintf(KL_ERRORFP, 
				"Error converting to physical address (%Lx)!\n",mem_loc);
			goto cleanup;
		}
		dp_address = tmp;
#else
		dp_address = mem_loc;
#endif

                if(!is_to_dump(dp_address)){
                        mem_loc += KL_DUMP_PAGE_SIZE;
                        continue;
                }

#ifdef VMDUMP
		GET_BLOCK(mem_loc, KL_DUMP_PAGE_SIZE, buf);
#else
		kl_readmem(mem_loc, KL_DUMP_PAGE_SIZE, buf);
#endif
		if (KL_ERROR) {
			/* this can happen, if we process a dump with missing pages */
			mem_loc += KL_DUMP_PAGE_SIZE;
			kl_reset_error();
			if(dump_info_file){
				fprintf(dump_info_file,"%016"FMT64"x X %s\n",mem_loc,MISSING_PAGE);
			}
			continue;
		}

		memset(dpcpage, 0, KL_DUMP_PAGE_SIZE);

		/* get the new compressed page size 
		 */
			
		size = vmdump_compress((char *)buf, KL_DUMP_PAGE_SIZE,
				(char *)dpcpage, KL_DUMP_PAGE_SIZE);

		/* if compression failed or compressed was ineffective,
		 * we write an uncompressed page
		 */
		if (size < 0) {
			dp_flags = KL_DUMP_DH_RAW;
			dp_size  = KL_DUMP_PAGE_SIZE;
		} else {
			dp_flags = KL_DUMP_DH_COMPRESSED;
			dp_size  = size;
		}

		/*  do byte swapping if necessary
		 */

		dp.address = KL_GET_UINT64(&dp_address);
		dp.size    = KL_GET_UINT32(&dp_size);
		dp.flags   = KL_GET_UINT32(&dp_flags);

		/* copy the page header 
		 */
		memcpy((void *)(dump_page_buf + buf_loc),
			(const void *)&dp, psize);
		buf_loc += psize;

		/* copy the page of memory 
		 */
		if (dp_flags & KL_DUMP_DH_COMPRESSED) {
			/* copy the compressed page 
			 */
			memcpy((void *)(dump_page_buf + buf_loc),
				(const void *)dpcpage, dp_size);
		} else {
			/* copy directly from memory 
			 */
			memcpy((void *)(dump_page_buf + buf_loc),
				(const void *)buf, dp_size);
		}
		buf_loc += dp_size;

		/* see if we need to write out the buffer 
		 */
		if (buf_loc >= KL_DUMP_BUFFER_SIZE) {

			if (vmdump_write(dump_file, dump_page_buf,
				KL_DUMP_BUFFER_SIZE) < 0) {
					fprintf(KL_ERRORFP, 
						"dump write error!\n");
					goto cleanup;
			}

			/* After writing out a set of pages, be sure to go
			 * through and update the dump header with the new
			 * count for num_pages.  This is so lcrash knows
			 * there are at least some valid pages in the dump.
			 *
			 * This is also important for dumps that reach the
			 * end of the partition and fill up - if we don't
			 * update the num_pages at least once, we won't
			 * know how many pages are in the dump at all.
			 */
			if (write_vmdump_header(dump_file) < 0) {
				fprintf(KL_ERRORFP, "Incremental dump header "
					"update failed!\n");
				goto cleanup;
			}

			if (buf_loc > KL_DUMP_BUFFER_SIZE) {

				/* clear the dump page buffer 
				 */
				memset(dump_page_buf, 0, KL_DUMP_BUFFER_SIZE);

				/* copy the dump page buffer overrun 
				 */
				memcpy((void *)dump_page_buf,(const void *)
					(dump_page_buf + KL_DUMP_BUFFER_SIZE),
					buf_loc - KL_DUMP_BUFFER_SIZE);

				/* set the new buffer location 
				 */
				buf_loc -= KL_DUMP_BUFFER_SIZE;
			} else {
				buf_loc = 0;
			}
		}

		/* update the page count 
		 */
		dump_header.num_dump_pages++;

		/* increment to the next page 
		 */
		mem_loc += KL_DUMP_PAGE_SIZE;
	}

	/* set up the dump page header with DUMP_DH_END 
	 */
	dp_address = 0x0;
	dp_flags   = KL_DUMP_DH_END;
	dp_size    = 0x0;

	/*  do byte swapping if necessary
	 */
 
	dp.address = KL_GET_PTR(&dp_address);
	dp.size    = KL_GET_UINT32(&dp_size);
	dp.flags   = KL_GET_UINT32(&dp_flags);

	/* copy the current buffer 
	 */
	memcpy((void *)(dump_page_buf + buf_loc),(const void *)&dp, psize);

	/* increment the buffer size 
	 */
	buf_loc += psize;

	/* write out now that we have the DUMP_DH_END header in there 
	 */
	if (vmdump_write(dump_file, dump_page_buf, KL_DUMP_BUFFER_SIZE) < 0) {
		fprintf(KL_ERRORFP, " dump write error!\n");
		goto cleanup;
	}

	/* we have something left in the buffer to write out ... 
	 */
	if (buf_loc > KL_DUMP_BUFFER_SIZE) {
		/* clear the dump page buffer 
		 */
		memset(dump_page_buf, 0, KL_DUMP_BUFFER_SIZE);

		/* copy the dump page buffer remnants 
		 */
		memcpy((void *)dump_page_buf,
			(const void *)(dump_page_buf + KL_DUMP_BUFFER_SIZE),
			buf_loc - KL_DUMP_BUFFER_SIZE);

		/* set the new buffer location 
		 */
		buf_loc -= KL_DUMP_BUFFER_SIZE;

		/* write out that _last_ bit of buffer! 
		 */
		if (vmdump_write(dump_file, dump_page_buf, KL_DUMP_BUFFER_SIZE) < 0) {
			fprintf(KL_ERRORFP, " dump write error!\n");
			goto cleanup;
		}
	}

	/* terminate the dots ... 
	 */
	fprintf(KL_ERRORFP, "\n");

	/* dump out the dump header again - reset the file position! 
	 */
	if (write_vmdump_header(dump_file) < 0) {
		fprintf(KL_ERRORFP, "Final dump header update failed!\n");
		goto cleanup;
	}

	fprintf(KL_ERRORFP, "Dump complete.\n");


	if(flags & EXCL_FLAGS){
	   	fprintf(KL_ERRORFP,"\n\n===========================\n");
		fprintf(KL_ERRORFP,"Summary of excluded memory:\n");
		fprintf(KL_ERRORFP,"===========================\n");
		fprintf(KL_ERRORFP,"        (all)     (real)\n");
		fprintf(KL_ERRORFP,"user:   %7.2f (%7.2f) MB\n",((double)user_count_sum*4/1024),((double)user_count*4/1024));
		fprintf(KL_ERRORFP,"free:   %7.2f (%7.2f) MB\n",((double)free_count_sum*4/1024),((double)free_count*4/1024));
		if(KL_LINUX_RELEASE < LINUX_2_6_0){
			fprintf(KL_ERRORFP,"bcache: %7.2f (%7.2f) MB\n",((double)buffer_count_sum*4/1024),((double)buffer_count*4/1024));
		}
		fprintf(KL_ERRORFP,"pcache: %7.2f (%7.2f) MB\n",((double)pcache_count_sum*4/1024),((double)pcache_count*4/1024));

	}

cleanup:

	/* close the dump device 
	 */
	if (ifd != -1) {
		close (ifd);
	}
	if (ofd != -1) {
		close (ofd);
	}
	vmdump_close(dump_file);
	if (buf) {
		free(buf);
	}
	if (dump_page_buf) {
		free(dump_page_buf);
		dump_page_buf = NULL;
	}
	if(page_map){
		free(page_map);
		page_map = NULL;
	}
	return;
}
