/*
 * $Id: swconfig.c,v 1.1 2004/12/21 23:26:20 tjm Exp $
 *
 * This file is part of libconfig.
 * A library which provides a framework for managing system hardware
 * and software configuration information.
 *
 * Created by Silicon Graphics, Inc.
 *
 * Copyright (C) 2004 Silicon Graphics, Inc. All rights reserved.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#include <kl_lib.h>
#include <kl_swconfig.h>

/*
 * free_swcomponent()
 */
void
free_swcomponent(sw_component_t *swcp)
{
	if (swcp->sw_name) {
		kl_free_block(swcp->sw_name);
	}
	if (swcp->sw_description) {
		kl_free_block(swcp->sw_description);
	}
	kl_free_block(swcp);
}

/*
 * free_swcomponents()
 */
void
free_swcomponents(sw_component_t *head)
{
	sw_component_t *swcp, *next_swcp; 

	swcp = head;
	while(swcp) {
		next_swcp = swcp->sw_next;
		free_swcomponent(swcp);
		swcp = next_swcp;
	}
}

/*
 * add_swcomponent()
 */
int
add_swcomponent(swconfig_t *scp, sw_component_t *swcp)
{
	int ret;
	sw_component_t *s;

	if (!scp || !swcp) {
		return(1);
	}

	if (!(scp->s_swcmp_head)) {

		/* List is empty. The new item becomes the list head
		 */
		swcp->sw_open = 0;
		swcp->sw_config = scp;
		swcp->sw_next = swcp->sw_prev = (sw_component_t *)NULL;
		scp->s_swcmp_head = swcp;

	} else if (strcmp(scp->s_swcmp_head->sw_name, swcp->sw_name) > 0) {

		/* Insert the new item at the head of the list
		 */
		swcp->sw_open = 0;
		swcp->sw_config = scp;
		swcp->sw_prev = (sw_component_t *)NULL;
		swcp->sw_next = scp->s_swcmp_head;
		scp->s_swcmp_head->sw_prev = swcp;
		scp->s_swcmp_head = swcp;

	} else {

		/* Cycle through the list and find the insertion point
		 */
		s = scp->s_swcmp_head;
		while(s) {
			ret = strcmp(s->sw_name, swcp->sw_name);
			if (ret == 0) {

				/* Duplicate item -- not allowed
				 */
				return(1);

			} else if (ret > 0) {

				/* Insert new item after s
				 */
				swcp->sw_open = 0;
				swcp->sw_config = scp;
				swcp->sw_prev = s->sw_prev;
				swcp->sw_prev->sw_next = swcp;
				swcp->sw_next = s;
				s->sw_prev = swcp;
				break;

			} else if (!(s->sw_next)) {

				/* Insert new item at end of the list
				 */
				swcp->sw_open = 0;
				swcp->sw_config = scp;
				s->sw_next = swcp;
				swcp->sw_prev = s;
				swcp->sw_next = (sw_component_t *)NULL;
				break;

			}
			s = s->sw_next;
		}
		if (!(s)) {
			return(1);
		}
	}
	scp->s_swcmp_cnt++;
	return(0);
}

/*
 * first_swcomponent()
 */
sw_component_t *
first_swcomponent(sw_component_t *swcp)
{
	swconfig_t *swconfig;

	if (!swcp || !(swconfig = swcp->sw_config)) {
		return((sw_component_t *)NULL);
	}
	return(swconfig->s_swcmp_head);
}

/*
 * next_swcomponent()
 */
sw_component_t *
next_swcomponent(sw_component_t *swcp)
{
	if (swcp) {
		return(swcp->sw_next);
	}
	return((sw_component_t *)NULL);
}

/*
 * prev_swcomponent()
 */
sw_component_t *
prev_swcomponent(sw_component_t *swcp)
{
	if (swcp) {
		return(swcp->sw_prev);
	}
	return((sw_component_t *)NULL);
}

/*
 * find_swcomponent()
 */
sw_component_t *
find_swcomponent(swconfig_t *swconfig, char *name)
{
	sw_component_t *swcp;

	if (!swconfig || !name) {
		return((sw_component_t *)NULL);
	}
	swcp = swconfig->s_swcmp_head;
	while(swcp) {
		if (!strcmp(swcp->sw_name, name)) {
			return(swcp);
		}
		swcp = swcp->sw_next;
	}
	return((sw_component_t *)NULL);
}

/*
 * unlink_swcomponent()
 */
void
unlink_swcomponent(sw_component_t *swcp)
{
	swconfig_t *swconfig;

	if (!swcp || !(swconfig = swcp->sw_config)) {
		return;
	}
	if (swconfig->s_swcmp_head == swcp) {
		/* Our swcomponent is first on the list
		 */
		if ((swconfig->s_swcmp_head = swcp->sw_next)) {
			swcp->sw_next->sw_prev = (sw_component_t *)NULL;
		}
	} else {
		if ((swcp->sw_prev->sw_next = swcp->sw_next)) {
			swcp->sw_next->sw_prev = swcp->sw_prev;
		}
	}
	swcp->sw_next = swcp->sw_prev = (sw_component_t *)NULL;
	swcp->sw_config = (swconfig_t *)NULL;
	swconfig->s_swcmp_cnt--;
}

/*
 * alloc_swconfig()
 */
swconfig_t *
alloc_swconfig(int flags, int level)
{
	swconfig_t *scp;

	scp = (swconfig_t *)
		kl_alloc_block(sizeof(swconfig_t), AFLG(flags));
	scp->s_flags = flags;
	scp->s_level = level;
	return(scp);
}

/*
 * free_swconfig()
 */
void
free_swconfig(swconfig_t *scp)
{
	if (scp) {
		if (scp->s_swcmp_head) {
			free_swcomponents(scp->s_swcmp_head);
		}
		kl_free_block(scp);
	}
}

/*
 * update_swconfig()
 */
void
update_swconfig(swconfig_t *scp)
{
	int i = 1;
	sw_component_t *swcp; 

	swcp = scp->s_swcmp_head;
	while(swcp) {
		swcp->sw_seq = i++;
		swcp = swcp->sw_next;
	}
}

