/*
 * $Id: kl_dump.c,v 1.6 2005/03/02 21:38:01 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>

/* global variables
 */
struct_sizes_t struct_sizes;
off_t dump_header_offset = 0;
void *G_dump_header = NULL; 		 /* Pointer to raw dump_header */
void *G_dump_header_asm = NULL;		 /* Pointer to raw asm dump_header */
kl_dump_header_t *KL_DUMP_HEADER = NULL; /* converted dump_header */
void *KL_DUMP_HEADER_ASM = NULL; 	 /* converted asm dump_header */

/* Certain very old dumps have a different name for the LKCD control 
 * structs. To allow for more flexibility, we use global variables
 * instead of a string containing the type names. The names of the 
 * members are the same, so this should allow a single piece of code 
 * to handle new and old style dumps.
 */
#define DH_NAME_COMMON "__dump_header"
#define DHA_NAME_COMMON "__dump_header_asm"
char *dh_typename = DH_NAME_COMMON;
char *dha_typename = DHA_NAME_COMMON;

/* Dump Constants obtained from the dump header at startup. We need
 * to seed the values because they are used during live system
 * analysis when there is no dump_header to draw from. Default 16K
 * dump page size.
 */
uint64_t KL_DUMP_PAGE_SIZE = 0x4000;
uint64_t KL_DUMP_PAGE_MASK = 0xffffffffffffc000;
uint64_t KL_DUMP_PAGE_SHIFT = 14;
uint64_t KL_DUMP_BUFFER_SIZE = 0x10000;
uint64_t KL_DUMP_HEADER_SIZE = 0x10000;

/* Forward declarations
 */
int kl_setup_dumparch(void);

/* Declarations 
 */
#ifdef MAX
#undef MAX
#endif
#define MAX(a, b)  (((a) > (b)) ? (a) : (b))
#define LKCD_DH_4_2_VERSION 8

/* function definitions
 */

/*
 * kl_valid_dump_magic()
 *
 *   Any valid DUMP_MAGIC define (common and arch specific) define 
 *   needs to be added below.
 *
 *   Return value:
 *
 *   -1 == invalid magic_number
 *    0 == valid magic_number (no byte swap)
 *    1 == valid magic_number (needed byte swap)
 *
 *   Note that KL_DUMP_MAGIC_ASM is used as a case value for all arch
 *   specific dump_headers. That is because the define value for each
 *   of the individual arch dump headers is equal to "0xdeaddeadULL".
 *   If they ever contain unique magic numbers, then they will need to
 *   be included below.
 */
int
kl_valid_dump_magic(uint64_t magic)
{
	uint64_t magic_swap = kl_get_swap_uint64(&magic);

	switch (magic) {
		case KL_DUMP_MAGIC_LIVE:
		case KL_DUMP_MAGIC_NUMBER:
		case KL_DUMP_MAGIC_S390SA:
		case KL_DUMP_MAGIC_ASM:
			return(0);
	}

	switch (magic_swap) {
		case KL_DUMP_MAGIC_LIVE:
		case KL_DUMP_MAGIC_NUMBER:
		case KL_DUMP_MAGIC_S390SA:
		case KL_DUMP_MAGIC_ASM:
			return(1);
	}
	return(-1);
}

/*
 * kl_valid_header()
 */
int
kl_valid_header(void *hp)
{
	generic_dump_header_t *gdhp = (generic_dump_header_t *)hp;

	if (!gdhp || (kl_valid_dump_magic(gdhp->magic_number) < 0)) {
		return(0);
	}
	return(1);
}

/* 
 * kl_header_swap() -- Determine if header values need to be byte swapped.
 *
 *   Return value:
 *
 *   -1 == invalid magic_number
 *    0 == valid magic_number (no byte swap)
 *    1 == valid magic_number (needed byte swap)
 */
int
kl_header_swap(void *hp)
{
	generic_dump_header_t *gdhp = (generic_dump_header_t *)hp;

	return(kl_valid_dump_magic(gdhp->magic_number));
}

/* 
 * kl_header_magic()
 */
uint64_t
kl_header_magic(void *hp)
{
	int swap;
	uint64_t magic;
	generic_dump_header_t *gdhp = (generic_dump_header_t *)hp;

	if (!gdhp) {
		return(0);
	}

	/* kl_header_swap() will determine if there is a valid 
	 * dump_magic value. It will also determine if the values
	 * in the dump_header need to be byte_swapped.
	 */
	if ((swap = kl_header_swap(gdhp)) < 0) {
		return(0);
	}
	magic  = gdhp->magic_number;
	if (swap) {
		magic  = kl_get_swap_uint64(&magic);              
	}
	return(magic);
}

/*
 * kl_header_version()
 */
uint32_t
kl_header_version(void *hp)
{
	int swap;
	uint32_t version;
	generic_dump_header_t *gdhp = (generic_dump_header_t *)hp;

	if (!gdhp) {
		return(0);
	}

	/* kl_header_swap() will determine if the generic_dump_header is
	 * valid and will determine if the values it contains should be
	 * byte swapped.
	 */
	if ((swap = kl_header_swap(gdhp)) < 0) {
		return(0);
	}
	version  = gdhp->version;
	if (swap) {
		version  = kl_get_swap_uint64(&version);              
	}
	return(version);
}

/* 
 * kl_header_size()
 */
int
kl_header_size(void *hp)
{
	int swap;
	uint32_t header_size;
	generic_dump_header_t *gdhp = (generic_dump_header_t *)hp;

	if (!gdhp) {
		return(0);
	}

	/* kl_header_swap() will determine if the generic_dump_header is
	 * valid and will determine if the values it contains should be
	 * byte swapped.
	 */
	if ((swap = kl_header_swap(gdhp)) < 0) {
		return(0);
	}
	header_size  = gdhp->header_size;
	if (swap) {
		header_size  = kl_get_swap_uint32(&header_size);              
	}
	return(header_size);
}

/*
 * kl_read_header()
 */
void *
kl_read_header(int fd, void *hp)
{
	int header_size;
	void *dhp;
	generic_dump_header_t *gdhp = (generic_dump_header_t *)hp;

	if (!gdhp) {
		return(NULL);
	}

	/* Determine the size of the dump_header and then allocate a block 
	 * of memory to hold it.
	 */
	header_size = kl_header_size(gdhp);
	if (!(dhp = malloc(header_size))) {
		KL_ERROR = KLE_INVALID_DUMP_HEADER;
		return(NULL);
	}

	/* Now read the full dump_header in
	 */
	if (read(fd, dhp, header_size) != header_size) {
		free(dhp);
		KL_ERROR = KLE_INVALID_READ;
		return(NULL);
	}
	return(dhp);
}

/*
 * set_dump_offset()
 */
static int
set_dump_offset(int fd) 
{
	int tries = 0;
	uint64_t magic;

again:
	if (lseek(fd, dump_header_offset, SEEK_SET) == -1) {
		KL_ERROR = KLE_INVALID_LSEEK;
		return(1);
	}
	if (read(fd, &magic, sizeof(magic)) < sizeof(magic)) {
		KL_ERROR = KLE_INVALID_READ;
		return(1);
	} else {
		if (kl_valid_dump_magic(magic) < 0) {
			/* We first try an offset of zero. If that doesn't
			 * work, then we try the standard dump offset to 
			 * see if that is where the dump_header is located. 
			 * If that doesn't work, we bail.
			 */
			if (tries) {
				KL_ERROR = KLE_INVALID_DUMP_HEADER;
				return(1);
			}
			dump_header_offset = KL_DUMP_HEADER_OFFSET;
			tries++;
			goto again;
		}
	}
	return(0);
}

/*
 * kl_get_raw_dh()
 */
int
kl_get_raw_dh(int fd)
{
	generic_dump_header_t gdh;
	
	/* Determine the offset to the start of the dump. Note that the
	 * start of the dump_header is dump_offset + 8 (sizeof magic_number).
	 */     
	if (set_dump_offset(fd)) {
		KL_ERROR = KLE_INVALID_DUMP_HEADER;
		return(1);
	}

	/* Now read in the generic_dump_header_s struct. This will
	 * give us the actual dump_header sise. Verify that the header
	 * is valid.
	 */
	if (lseek(fd, dump_header_offset, SEEK_SET) < 0) {
		KL_ERROR = KLE_INVALID_LSEEK;
		return(1);
	}
	if ((read(fd, &gdh, sizeof(gdh)) != sizeof(gdh))) {
		KL_ERROR = KLE_INVALID_DUMP_HEADER;
		return(1);
	}
	if (kl_valid_dump_magic(gdh.magic_number) < 0) {
		KL_ERROR = KLE_INVALID_DUMP_HEADER;
		return(1);
	}

	/* Rewind to the start of the dump_header and read the full 
	 * header in.
	 */
	if (lseek(fd, dump_header_offset, SEEK_SET) < 0) {
		KL_ERROR = KLE_INVALID_LSEEK;
		return(1);
	}
	if (!(G_dump_header = kl_read_header(fd, &gdh))) {
		KL_ERROR = KLE_INVALID_DUMP_HEADER;
		return(1);
	}
	return(0);
}

/*
 * kl_get_raw_asm_dh()
 */
int
kl_get_raw_asm_dh(int fd)
{
	int header_size, header_asm_size;
	uint64_t magic;
	generic_dump_header_t gdh;

	if (!G_dump_header) {
		return(1);
	}

	header_size = kl_header_size(G_dump_header);

	/* Step over the dump header and read the dump_header_asm 
	 * in from disk...
         */
        if (lseek(fd, dump_header_offset+header_size, SEEK_SET) < 0) {
                KL_ERROR = KLE_INVALID_LSEEK;
                return(1);
        }

        /* Now read in the generic dump_header for the arch specific        
         * dump_header and make sure it's valid. 
         */
        if ((read(fd, &gdh, sizeof(gdh)) != sizeof(gdh))) {
                KL_ERROR = KLE_INVALID_DUMP_HEADER;
                return(1);
        }

	/* Verify the magic number for this dump_header...
	 */
	magic = kl_header_magic(&gdh);
	if (magic != KL_DUMP_MAGIC_ASM) {
		KL_ERROR = KLE_INVALID_DUMP_HEADER;
                return(1);
	}
	header_asm_size = kl_header_size(&gdh);
	
        /* Rewind and read in the asm dump_header.
         */
        if (lseek(fd, dump_header_offset+header_size, SEEK_SET) < 0) {
                KL_ERROR = KLE_INVALID_LSEEK;
                return(1);
        }

	G_dump_header_asm = kl_read_header(fd, &gdh);

	return(0);
}

/*
 * kl_read_dump_header()
 */
int
kl_read_dump_header(void)
{
	if (!KLP || !KLP->dump || (KL_CORE_FD == -1)) {
		KL_ERROR = KLE_INVALID_DUMP_HEADER;
		return(1);
	}
	return(kl_get_raw_dh(KL_CORE_FD));
}

/*
 * kl_read_dump_header_asm()
 */
int
kl_read_dump_header_asm(void)
{
	if (!G_dump_header) {
		return(1);
	}
	return(kl_get_raw_asm_dh(KL_CORE_FD));
}

/*
 * kl_get_common_dh()
 */
static int
kl_get_common_dh(void)
{
	kl_dump_header_t *dh;
	uint32_t version;

	if (!G_dump_header) {
		KL_ERROR = KLE_INVALID_DUMP_HEADER;
		return(1);
	}
	version = kl_header_version(G_dump_header);

	/* Check to make sure the format of the raw dump_header matches
	 * the common one. The dump headers from some older dumps may be
	 * in a slightly different format.
	 *
	 * Note that older SGI dump headers can only be detected if                                                   
	 * type information is initialized first (because the name of                                                 
	 * the dump header struct is different). If no type informaiton                                               
	 * is present, it's either a 4.1 header or a current header.                                                  
	 */                                                                                                           
	if (kl_find_type("_dump_header_s", KLT_TYPES)) {                                                              
		/* This is an older, SGI SN2 style dump header. Since the                                             
		 * layout is somewhat different than our standard one, we                                             
		 * need to convert it to the common form.                                                             
		 */                                                                                                   
		dh = get_dump_header_SN2_24X(G_dump_header);
	} else if (version < LKCD_DH_4_2_VERSION) {
		/* This is an 4.1 LKCD dump header. We need to convert it 
		 * to the common form.
		 */
		dh = get_dump_header_4_1(G_dump_header);
	} else {
		/* The dump header is current. Make a copy, so that our
		 * raw dump_header stays as is.
		 */
		int hdrsz;

		hdrsz = kl_header_size(G_dump_header);
		dh = malloc(hdrsz);
		memcpy(dh, G_dump_header, hdrsz);
	}

	/* Swap the common bytes if needed...
	 */
	if (kl_header_swap(dh) == 1) {
		kl_swap_dump_header_reg(dh);
	}
	KL_DUMP_HEADER = dh;
	return(0);
}

/*
 * kl_setup_dump_variables()
 */
void
kl_setup_dump_variables(void)
{
	kl_dump_header_t *dhp = KL_DUMP_HEADER;

        KL_DUMP_PAGE_SIZE = dhp->page_size;
	if (SN2_24X) {
		/* Hack for older SGI dumps...
		 */
		dhp->page_size = 0x4000;
	}  
	KL_DUMP_PAGE_MASK = ~((uint64_t)KL_DUMP_PAGE_SIZE-1);
        KL_DUMP_PAGE_SHIFT = 12;
        while ((KL_DUMP_PAGE_SIZE >> KL_DUMP_PAGE_SHIFT) != 1) {
                KL_DUMP_PAGE_SHIFT++;
        }
	if ((dhp->version >= 9) || (SN2_24X && (dhp->version >= 8))) {
		KL_DUMP_BUFFER_SIZE = dhp->dump_buffer_size;
	} else if (SN2_24X) {
		KL_DUMP_BUFFER_SIZE = 0x10000;
        } 
        KL_DUMP_HEADER_SIZE = KL_DUMP_BUFFER_SIZE;
}

/*
 * kl_get_dump_header() -- get common dump header of lkcd dumps and live dumps
 */
int
kl_get_dump_header(kl_dump_header_t *dh)
{
	/* first, make sure this isn't a live system
	 */
	if (CORE_IS_KMEM) {
		KL_ERROR = KLE_ACTIVE;
		return(1);
	}

	/* get the dump header
	 */
	memcpy(dh, KL_DUMP_HEADER, sizeof(kl_dump_header_t));
	return(0);
}

/*
 * kl_set_dumparch() -- tries to determine architecture of dump
 */
int
kl_set_dumparch(int dump_arch)
{
	int retries = 0;
	uint64_t magic, magic_swap;
	int is_live=0, is_lkcd=0, is_s390=0;
	int dumparch = KL_ARCH_UNKNOWN;
	kl_dump_header_t *dh;

	/* Check to see if this is a live system. If it is, then
	 * just set the dump arch the same as the host arch.
	 */
	if (CORE_IS_KMEM) {
		KLP->dump->arch.arch = KLP->host->arch;
		return(0);
	}

	if (!(dh = (kl_dump_header_t *)G_dump_header)) {
		fprintf(KL_ERRORFP,"ERROR: Invalid dump header pointer.\n");
		return(1);
	}
		
	magic = dh->magic_number;
	magic_swap = kl_get_swap_uint64(&magic);
		   
	if((magic == KL_DUMP_MAGIC_LIVE) || 
			(magic_swap == KL_DUMP_MAGIC_LIVE)) {
		++is_live; 
	} else if((magic == KL_DUMP_MAGIC_NUMBER) ||
			(magic_swap == KL_DUMP_MAGIC_NUMBER)){
		++is_lkcd; 
	} else if ((magic == KL_DUMP_MAGIC_S390SA) || 
			(magic_swap == KL_DUMP_MAGIC_S390SA)) { 
		++is_s390;      
	} else {
		fprintf(KL_ERRORFP,"Warning: "
			"Unknown magic number in dump header.\n");
	}
	if (is_live || is_lkcd) {
	
		/* Get the arch string from the dump header.
		 */
		char *arch_string;
							      
		arch_string = (char *)&dh->utsname_machine;
try_again:
		if(!(strcmp(arch_string, KL_ARCH_STR_ALPHA))){
			dumparch = KL_ARCH_ALPHA;             
		} else if(!(strncmp(arch_string, KL_ARCH_STR_ARM, 3))){     
			dumparch = KL_ARCH_ARM;               
		} else if((strlen(arch_string) == 4) &&       
			  (arch_string[0] == 'i') &&          
			  (arch_string[2] == '8') &&          
			  (arch_string[3] == '6')){           
			dumparch = KL_ARCH_I386;              
		} else if(!(strcmp(arch_string, KL_ARCH_STR_IA64))){
			dumparch = KL_ARCH_IA64;              
		} else if(!(strcmp(arch_string, KL_ARCH_STR_S390))){
			dumparch = KL_ARCH_S390;              
		} else if(!(strcmp(arch_string, KL_ARCH_STR_S390X))){
			dumparch = KL_ARCH_S390X;             
		} else if(!(strcmp(arch_string, KL_ARCH_STR_PPC64))) {      
			dumparch = KL_ARCH_PPC64;             
		} else if(!(strcmp(arch_string, KL_ARCH_STR_X86_64))) {     
			dumparch = KL_ARCH_X86_64;            
		} else if (retries) {
			if (dump_arch) {
				/* Force the dump arch to be the one
				 * passed in.
				 */
				dumparch = dump_arch;
			} else {
				fprintf(KL_ERRORFP, "Warning: Unknown dump "
					"arch: \"%s\"" " stored in dump "
					"header.\n", arch_string);
				return(1);
			}
		} else if(!(dumparch = kl_dump_arch_4_1(G_dump_header))) {
			/* For older SGI dumps, there may be a difference 
			 * in the alignment of the start of the utsname 
			 * members. This appears to be the result of a 
			 * difference in the size of the time struct 
			 * compared to how it is layed out in memory. 
			 * Adumst the arch string by four bytes and try 
			 * again.
			 */
			retries++;
			arch_string += 4;
			goto try_again;
		}                                             
	} else if (is_s390) {                                   
#if defined(DUMP_ARCH_S390) || defined(DUMP_ARCH_S390X)       
		/* get the arch_id from the kl_dump_header_s390_t struct
		 */
		kl_dump_header_s390sa_t *dha;                  
		uint32_t arch_id, arch_id_swap;               

		dha = (kl_dump_header_s390sa_t *)G_dump_header;
		arch_id = dha->arch_id;                        
		arch_id_swap = kl_get_swap_uint32(&arch_id);  
		if((arch_id == KL_DH_ARCH_ID_S390) ||         
		   (arch_id_swap == KL_DH_ARCH_ID_S390)){     
			dumparch = KL_ARCH_S390;              
		} else if ((arch_id == KL_DH_ARCH_ID_S390X) ||
			   (arch_id_swap == KL_DH_ARCH_ID_S390X)) {
			dumparch = KL_ARCH_S390X;             
		} else {                                      
			/* wrong arch_id, set error code */   
			fprintf(KL_ERRORFP, "Warning: "       
				"Unknown arch_id for s390 dump: %x\n",      
				arch_id);                     
		}                                             
#else                                                         
		dumparch = KL_ARCH_S390;                      
#endif                                                        
	}                                                     
	if (dump_arch && (dumparch != dump_arch)) {
		/* We have a mismatch */
		fprintf(KL_ERRORFP, "Warning: Dump architecture passed "
			"in (%d) does not match one from dump_header "
			"(%d)!\n",
			dump_arch, dumparch);
		KL_ARCH = dump_arch;
	} else {
		KL_ARCH = dumparch;
	}
	return(0);                                     
}

/*
 * kl_open_dump()
 */
int
kl_open_dump(void)
{
	int core_fd;

	/* Make sure we have what we need to proceed...
	 */
	if (!KLP || !KLP->dump || !KLP->dump->dump) {
		KL_ERROR = KLE_NOT_INITIALIZED;
		return(1);
	}
	if ((core_fd = open(KLP->dump->dump, 0)) == -1) {
		KL_ERROR = KLE_OPEN_ERROR|KLE_DUMP;
		return(1);
	}
	if (strcmp(KLP->dump->dump, "/dev/mem") == 0) {
		/* We're starting up on a live system
		 */
		KL_CORE_TYPE = dev_kmem;
	} else {
		/* We know it's a core dump, just not what type of
		 * dump. We will determine this a bit later on after 
		 * we read in the dump_header.
		 */
		KL_CORE_TYPE = reg_core;
	}
	KL_CORE_FD = core_fd;
	return(0);
}

/* kl_setup_dumpaccess()
 */
int
kl_setup_dumpaccess(int flags)
{
	if (CORE_IS_DUMP) {
		kl_get_common_dh();
		kl_setup_dump_variables();
	}
	if (kl_setup_dumparch()) {
		return(1);
	}
	if (kl_set_dumpaccess()) {
		return(1);
	}

	/* initialize the compression library
	 */
	if (CORE_IS_DUMP && (kl_cmpinit(KL_CORE_FD, (char *)NULL, flags) < 0)) {
		if (KL_ERROR == KLE_DUMP_HEADER_ONLY) {
			fprintf(KL_ERRORFP, "\n"
				"Warning: dump only contains a dump "
				"header.  Printing dump header:\n"
				"Note: Ignore the last four fields "
				"of \"DUMP INFORMATION\"\n");
			kl_print_dump_header_reg(KL_DUMP_HEADER);
			KL_ERROR = 0;
		} else {
			KL_ERROR = KLE_CMP_ERROR;
		}
		return(1);
	}
	return(0);
}

/* kl_init_struct_sizes()
 */
void
kl_init_struct_sizes(void)
{
	TASK_STRUCT_SZ = kl_struct_len("task_struct");
	MM_STRUCT_SZ = kl_struct_len("mm_struct");
	PAGE_SZ = kl_struct_len("page");
	MODULE_SZ = kl_struct_len("module");
	NEW_UTSNAME_SZ = kl_struct_len("new_utsname");
	SWITCH_STACK_SZ = kl_struct_len("switch_stack");
	PT_REGS_SZ = kl_struct_len("pt_regs");
}


/* 
 * kl_setup_hostinfo()
 */
int
kl_setup_hostinfo(void)
{
	if(!(KLP->host)){
		if(!(KLP->host = calloc(1, sizeof(kl_hostarch_t)))){
			KL_ERROR = KLE_NO_MEMORY;
			return(1);
		}
	}
#if defined(HOST_ARCH_ALPHA)
	KLP->host->arch=KL_ARCH_ALPHA;
	KLP->host->ptrsz=64;
	KLP->host->byteorder=KL_LITTLE_ENDIAN;
#elif defined(HOST_ARCH_I386)
	KLP->host->arch=KL_ARCH_I386;
	KLP->host->ptrsz=32;
	KLP->host->byteorder=KL_LITTLE_ENDIAN;
#elif defined(HOST_ARCH_IA64)
	KLP->host->arch=KL_ARCH_IA64;
	KLP->host->ptrsz=64;
	KLP->host->byteorder=KL_LITTLE_ENDIAN;
#elif defined(HOST_ARCH_S390)
	KLP->host->arch=KL_ARCH_S390;
	KLP->host->ptrsz=32;
	KLP->host->byteorder=KL_BIG_ENDIAN;
#elif defined(HOST_ARCH_S390X)
	KLP->host->arch=KL_ARCH_S390X;
	KLP->host->ptrsz=64;
	KLP->host->byteorder=KL_BIG_ENDIAN;
#elif defined(HOST_ARCH_ARM)
	KLP->host->arch=KL_ARCH_ARM;
	KLP->host->ptrsz=32;
	KLP->host->byteorder=KL_LITTLE_ENDIAN;
#elif defined(HOST_ARCH_PPC64)
	KLP->host->arch=KL_ARCH_PPC64;
	KLP->host->ptrsz=64;
	KLP->host->byteorder=KL_BIG_ENDIAN;
#elif defined(HOST_ARCH_X86_64)
	KLP->host->arch=KL_ARCH_X86_64;
	KLP->host->ptrsz=64;
	KLP->host->byteorder=KL_LITTLE_ENDIAN;
#else
#error "Host architecture not supported"
#endif
	return(0);
}


/* 
 * kl_setup_dumpinfo()
 */
int
kl_setup_dumpinfo(char *map, char *dump, int flags)
{
	KL_ERROR = 0;

	if(!(KLP->dump)){
		if(!(KLP->dump = calloc(1, sizeof(kl_dumpinfo_t)))){
			KL_ERROR = KLE_NO_MEMORY;
			return(1);
		}
	}

	KLP->dump->map = (char *)strdup(map);
	if (KLP->dump->map == NULL) {
		KL_ERROR = KLE_NO_MEMORY;
		return(1);
	}

	KLP->dump->dump = (char *)strdup(dump);
	if (KLP->dump->dump == NULL) {
		KL_ERROR = KLE_NO_MEMORY;
		return(1);
	}

	KL_CORE_FD = -1;

	return(0);
}

/* 
 * kl_set_dumpinfo()
 */
int
kl_set_dumpinfo(char *map, char *dump, int dump_arch, int flags)
{
	KL_ERROR = 0;

	if(kl_set_kerninfo()){
		/* XXX set error code */
		return(1);
	}

	return(0);
}


/* 
 * kl_free_dumpinfo()
 */
void
kl_free_dumpinfo(kl_dumpinfo_t *dump)
{
	free(dump->map);
	free(dump->dump);
	
	if (dump->core_fd >= 0) {
		close(dump->core_fd);
	}

	free(dump);
	dump = NULL;
}

/* 
 * kl_setup_dumparch()
 */
int
kl_setup_dumparch(void)
{
	/* set defaults for some functions */
	KLP->dump->arch.valid_physmem = kl_valid_physmem;
	KLP->dump->arch.next_valid_physaddr = kl_next_valid_physaddr;
	KLP->dump->arch.fix_vaddr = kl_fix_vaddr;
#ifdef XXX_TO_ADD
        KLP->dump->arch.kaddr_is_physical = kl_kaddr_is_physical;
        KLP->dump->arch.kaddr_is_highmem = kl_kaddr_is_highmem;
#endif
	KLP->dump->arch.init_virtop = kl_init_virtop;
	KLP->dump->arch.virtop = kl_virtop;

	switch(KL_ARCH){
#ifdef DUMP_ARCH_ALPHA
	case KL_ARCH_ALPHA:
		KL_ARCH       = KL_ARCH_ALPHA;
		KL_PTRSZ      = 64;
		KL_BYTE_ORDER = KL_LITTLE_ENDIAN;
		if((kl_set_dumparch_alpha())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_ARM
	case KL_ARCH_ARM:
		KL_ARCH	      = KL_ARCH_ARM;
		KL_PTRSZ      = 32;
		KL_BYTE_ORDER = kl_get_target_byteorder(KLP->dump->dump);
		if((kl_set_dumparch_arm())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_I386
	case KL_ARCH_I386:
		KL_ARCH       = KL_ARCH_I386;
		KL_PTRSZ      = 32;
		KL_BYTE_ORDER = KL_LITTLE_ENDIAN;
		if((kl_set_dumparch_i386())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_IA64
	case KL_ARCH_IA64:
		KL_ARCH       = KL_ARCH_IA64;
		KL_PTRSZ      = 64;
		KL_BYTE_ORDER = KL_LITTLE_ENDIAN;
		if((kl_set_dumparch_ia64())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_S390
	case KL_ARCH_S390:
		KL_ARCH       = KL_ARCH_S390;
		KL_PTRSZ      = 32;
		KL_BYTE_ORDER = KL_BIG_ENDIAN;
		if((kl_set_dumparch_s390())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_S390X
	case KL_ARCH_S390X:
		KL_ARCH       = KL_ARCH_S390X;
		KL_PTRSZ      = 64;
		KL_BYTE_ORDER = KL_BIG_ENDIAN;
		if((kl_set_dumparch_s390x())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_PPC64
        case KL_ARCH_PPC64:
                KL_ARCH       = KL_ARCH_PPC64;
                KL_PTRSZ      = 64;
                KL_BYTE_ORDER = KL_BIG_ENDIAN;
                if((kl_set_dumparch_ppc64())){
                        return(1);
                }
                break;
#endif
#ifdef DUMP_ARCH_X86_64
        case KL_ARCH_X86_64:
                KL_ARCH       = KL_ARCH_X86_64;
                KL_PTRSZ      = 64;
                KL_BYTE_ORDER = KL_LITTLE_ENDIAN;
                if((kl_set_dumparch_x86_64())){
                        return(1);
                }
                break;
#endif
	default:
		KL_ERROR = KLE_UNSUPPORTED_ARCH;
		return(1);
	}
	return(0);
}

/* 
 * kl_set_kerninfo()
 */
int
kl_set_kerninfo(void)
{
	syment_t *sp;

	kl_reset_error();

	if((sp = kl_lkup_symname("smp_num_cpus"))) {
		/* Linux 2.2 and 2.4 */
		/* XXX - adaption for ia64 needed */
		KL_NUM_CPUS = KL_READ_UINT32(sp->s_addr - KL_PAGE_OFFSET);
		if (KL_ERROR) {
			KL_NUM_CPUS = 1;
			kl_reset_error();
			goto out;
		}
	} else if((sp = kl_lkup_symname("cpu_online_map"))){
		/* Linux 2.5 */
		uint64_t map = 0;
		int i, cpu_num = 0;
		kaddr_t paddr;

		if (KL_VIRTOP(sp->s_addr, NULL, &paddr)==0) {
#ifdef DUMP_ARCH_X86_64
			if (KL_ARCH == KL_ARCH_X86_64)
				map = KL_READ_LONG(sp->s_addr - KL_START_KERNEL_map_X86_64);
			else
#endif
				map = KL_READ_LONG(sp->s_addr - KL_PAGE_OFFSET);
		}
		if (KL_ERROR) {
			KL_NUM_CPUS = 1;
			kl_reset_error();
			goto out;
		}
		for(i = 0; i < sizeof(map)*8;i++){
			if(map & 0x1UL)
				cpu_num += 1;
			map >>= 1;
		}
		KL_NUM_CPUS = cpu_num;
	} else {
		/* no smp kernel, so we have 1 cpu */
		KL_NUM_CPUS = 1;
		kl_reset_error();
	}
out:
	return(0);
}

/* 
 * kl_swap_dump_header_reg() - Byte swap all members of regular lkcd header
 */
void 
kl_swap_dump_header_reg(kl_dump_header_t* dh)
{
	dh->magic_number = kl_get_swap_uint64(&dh->magic_number);
	dh->version	 = kl_get_swap_uint32(&dh->version);
	dh->header_size  = kl_get_swap_uint32(&dh->header_size);
	dh->dump_level	 = kl_get_swap_uint32(&dh->dump_level);
	dh->page_size	 = kl_get_swap_uint32(&dh->page_size);
	dh->memory_size  = kl_get_swap_uint64(&dh->memory_size);
	dh->memory_start = kl_get_swap_uint64(&dh->memory_start);
	dh->memory_end	 = kl_get_swap_uint64(&dh->memory_end);
	dh->num_dump_pages = kl_get_swap_uint32(&dh->num_dump_pages);
	dh->time.tv_sec  = kl_get_swap_uint64(&dh->time.tv_sec);
	dh->time.tv_usec = kl_get_swap_uint64(&dh->time.tv_usec);
	dh->current_task = kl_get_swap_uint64(&dh->current_task);
	dh->dump_compress = kl_get_swap_uint32(&dh->dump_compress);
	dh->dump_flags	 = kl_get_swap_uint32(&dh->dump_flags);
	dh->dump_device  = kl_get_swap_uint32(&dh->dump_device);
}

/* 
 * kl_swap_dump_header_s390sa() - Byte swap all members of s390sa header
 */
void 
kl_swap_dump_header_s390sa(kl_dump_header_s390sa_t* dh)
{
	dh->magic_number = kl_get_swap_uint64(&dh->magic_number);
	dh->version	 = kl_get_swap_uint32(&dh->version);
	dh->header_size  = kl_get_swap_uint32(&dh->header_size);
	dh->dump_level	 = kl_get_swap_uint32(&dh->dump_level);
	dh->page_size	 = kl_get_swap_uint32(&dh->page_size);
	dh->memory_size  = kl_get_swap_uint64(&dh->memory_size);
	dh->memory_start = kl_get_swap_uint64(&dh->memory_start);
	dh->memory_end	 = kl_get_swap_uint64(&dh->memory_end);
	dh->num_pages	 = kl_get_swap_uint32(&dh->num_pages);
	dh->tod		 = kl_get_swap_uint64(&dh->tod);
	dh->cpu_id	 = kl_get_swap_uint64(&dh->cpu_id);
	dh->arch_id	 = kl_get_swap_uint32(&dh->arch_id);
	dh->build_arch_id =  kl_get_swap_uint32(&dh->build_arch_id);
}

/* 
 * kl_s390sa_to_reg_header() - copy info from s390sa header to reg lkcd header
 */
void
kl_s390sa_to_reg_header(kl_dump_header_s390sa_t* dh_s390sa, kl_dump_header_t* dh)
{
	struct timeval	   h_time;
 
	/* adjust todclock to 1970 */
	uint64_t tod = dh_s390sa->tod;
	tod -= 0x8126d60e46000000LL - (0x3c26700LL * 1000000 * 4096);
	tod >>= 12;
	h_time.tv_sec  = tod / 1000000;
	h_time.tv_usec = tod % 1000000;
 
	dh->memory_size    = dh_s390sa->memory_size;
	dh->memory_start   = dh_s390sa->memory_start;
	dh->memory_end	   = dh_s390sa->memory_end;
	dh->num_dump_pages = dh_s390sa->num_pages;
	dh->page_size	   = dh_s390sa->page_size;
	dh->dump_compress  = KL_DUMP_COMPRESS_GZIP;
	dh->dump_level	   = dh_s390sa->dump_level;
 
	sprintf(dh->panic_string,"zSeries-dump (CPUID = %16"FMT64"x)",dh_s390sa->cpu_id);
 
	if(dh_s390sa->arch_id == KL_DH_ARCH_ID_S390){
		strcpy(dh->utsname_machine,"s390");
	} else if(dh_s390sa->arch_id == KL_DH_ARCH_ID_S390X) {
		strcpy(dh->utsname_machine,"s390x");
	} else {
		strcpy(dh->utsname_machine,"unknown");
	}
 
	dh->magic_number   = KL_DUMP_MAGIC_NUMBER;
	dh->version	   = KL_DUMP_VERSION_NUMBER;
	dh->header_size    = sizeof(kl_dump_header_t);
	dh->time.tv_sec    = h_time.tv_sec;
	dh->time.tv_usec   = h_time.tv_usec;
}

/*
 * kl_dumptask()
 */
kaddr_t
kl_dumptask(void)
{
	if ((KL_ARCH == KL_ARCH_S390) || (KL_ARCH == KL_ARCH_S390X)) {
		return((kaddr_t)NULL);
	}
	return((kaddr_t)KL_DUMP_HEADER->current_task);
}

/*
 * Name: kl_print_dump_header_reg()
 * Func: Print out the lkcd regular dump header on the screen.
 */
void
kl_print_dump_header_reg(kl_dump_header_t *dh)
{
	time_t t;

	fprintf(kl_stdout,"        Dump Type: LKCD-Dump\n");
	fprintf(kl_stdout,"       Systemname: %s\n", dh->utsname_sysname);
	fprintf(kl_stdout,"         Nodename: %s\n", dh->utsname_nodename);
	fprintf(kl_stdout,"          Release: %s\n", dh->utsname_release);
	fprintf(kl_stdout,"          Version: %s\n", dh->utsname_version);
	fprintf(kl_stdout,"          Machine: %s\n", dh->utsname_machine);
	fprintf(kl_stdout,"       Domainname: %s\n", dh->utsname_domainname);
	fprintf(kl_stdout,"\n");
	fprintf(kl_stdout,"     Memory Start: 0x%"FMT64"x\n", dh->memory_start);
	fprintf(kl_stdout,"       Memory End: 0x%"FMT64"x\n", dh->memory_end);
	fprintf(kl_stdout,"      Memory Size: %"FMT64"d\n", 
		dh->memory_size*KL_PAGE_SIZE);
	t = (time_t) dh->time.tv_sec;
	fprintf(kl_stdout,"\n");
	fprintf(kl_stdout,"     Time of dump: %s", ctime(&t));
	fprintf(kl_stdout,"     Panic string: %s\n", dh->panic_string);
	fprintf(kl_stdout,"  Number of pages: %d\n", dh->num_dump_pages);
	fprintf(kl_stdout," Kernel page size: %d\n", dh->page_size);
	fprintf(kl_stdout,"   Version number: %d\n", dh->version);
	fprintf(kl_stdout,"     Magic number: 0x%"FMT64"x\n", dh->magic_number);
	fprintf(kl_stdout," Dump header size: %d\n", dh->header_size);
	fprintf(kl_stdout,"       Dump level: 0x%d\n", dh->dump_level);
}

/*
 * Name: kl_print_dump_header_s390sa()
 * Func: Print out the s390 stand alone dump header on the screen.
 */
void
kl_print_dump_header_s390sa(kl_dump_header_s390sa_t dh)
{
	char* machine;	
	char* build_machine;
	struct timeval t;

	kl_s390tod_to_timeval(dh.tod, &t);
	
	if(dh.arch_id == KL_DH_ARCH_ID_S390)
		machine = "s390 (ESA)";
	else if(dh.arch_id == KL_DH_ARCH_ID_S390X)
		machine = "s390x (ESAME)";
	else
		machine = "unknown";
	if(dh.version >= 2){
		if(dh.build_arch_id == KL_DH_ARCH_ID_S390)
			build_machine="s390 (ESA)";
		else if(dh.arch_id == KL_DH_ARCH_ID_S390X)
			build_machine="s390x (ESAME)";
		else
			build_machine="unknown";
	}
	fprintf(kl_stdout,"        Dump Type: s390 standalone dump\n");
	fprintf(kl_stdout,"          Machine: %s\n",machine);
	fprintf(kl_stdout,"           CPU ID: 0x%"FMT64"x\n", dh.cpu_id);
	fprintf(kl_stdout,"\n");
	fprintf(kl_stdout,"     Memory Start: 0x%"FMT64"x\n", dh.memory_start);
	fprintf(kl_stdout,"       Memory End: 0x%"FMT64"x\n", dh.memory_end);
	fprintf(kl_stdout,"      Memory Size: %"FMT64"d\n", dh.memory_size);
	fprintf(kl_stdout,"\n");
	fprintf(kl_stdout,"     Time of dump: %s", ctime(&t.tv_sec));
	fprintf(kl_stdout,"  Number of pages: %d\n", dh.num_pages);
	fprintf(kl_stdout," Kernel page size: %d\n", dh.page_size);
	fprintf(kl_stdout,"   Version number: %d\n", dh.version);
	fprintf(kl_stdout,"     Magic number: 0x%"FMT64"x\n", dh.magic_number);
	fprintf(kl_stdout," Dump header size: %d\n", dh.header_size);
	fprintf(kl_stdout,"       Dump level: 0x%d\n", dh.dump_level);
	if(dh.version >= 2)
		fprintf(kl_stdout,"       Build arch: %s\n", build_machine);
}

/* 
 * _kl_check_s390sa_endmarker
 */
static int 
_kl_check_s390sa_endmarker(int fh, kl_dump_header_s390sa_t* dh,
			   int swap)
{
	char em[16];
	uint64_t mem_size = dh->memory_size;
	uint64_t tod_em;
	struct timeval t;
	int rc = 0;

	/* seek to endmarker (memsize + size of dump header) */
	if(lseek(fh, mem_size + 4096,SEEK_SET) < 0){
		fprintf(KL_ERRORFP, "lseek failed! Perhaps dump not complete?\n");
		rc = -1; goto out;
	}
	/* read endmarker */
	if(read(fh, &em,sizeof(em)) != sizeof(em)){
		fprintf(KL_ERRORFP, "read failed! Perhaps dump not complete?\n");
		rc = -1; goto out;
	}
	/* check endmarker */
	if(memcmp(em,"DUMP_END",8) != 0){
		fprintf(kl_stdout,"Dump not valid! No Endmarker found!\n");
		rc = -1; goto out;
	} 
	tod_em = *((uint64_t*)(em + 8));
	if(swap)
		tod_em = kl_get_swap_uint64(&tod_em);
	if(tod_em <= dh->tod){
		fprintf(kl_stdout,"Dump not vaild! Endmarker time <= dump time!\n");
		rc = -1; goto out;
	}
	kl_s390tod_to_timeval(tod_em, &t);
	fprintf(kl_stdout," Time of dump end: %s\n", ctime(&t.tv_sec));
	fprintf(kl_stdout,"End Marker found! Dump is valid!\n");
out:
	return rc;
}

/*
 * kl_print_dump_header()
 */
int
kl_print_dump_header(const char* dump)
{
	int fh;
	uint64_t magic,magic_swap;
	char* dh = NULL;
	int bufsize;
	int rc = 0;

	bufsize= MAX(sizeof(kl_dump_header_t),sizeof(kl_dump_header_s390sa_t));

	dh = (char*)malloc(bufsize);
	fh = open(dump, O_RDONLY);
	if(fh == -1){
		fprintf(KL_ERRORFP,"Error: Cannot open dump: %s\n",dump);
		rc = -1; goto out;
	}
	if(read(fh, (void*)dh, bufsize) != bufsize){
		fprintf(KL_ERRORFP,"Error: Read failed for %s\n",dump);
		rc = -1; goto out;
	}
	magic = *(uint64_t*)dh;
	magic_swap = kl_get_swap_uint64(&magic);
	if(magic == KL_DUMP_MAGIC_S390SA){
		kl_print_dump_header_s390sa(*(kl_dump_header_s390sa_t*)dh);
		rc = _kl_check_s390sa_endmarker(fh,
						(kl_dump_header_s390sa_t*) dh,
						0);
		goto out;
	} else if(magic_swap == KL_DUMP_MAGIC_S390SA){
		kl_swap_dump_header_s390sa((kl_dump_header_s390sa_t*)dh);
		kl_print_dump_header_s390sa(*(kl_dump_header_s390sa_t*)dh);
		rc = _kl_check_s390sa_endmarker(fh,
						(kl_dump_header_s390sa_t*) dh,
						1);
		goto out;
		
	} else if((magic==KL_DUMP_MAGIC_NUMBER)||(magic==KL_DUMP_MAGIC_LIVE)){
		kl_print_dump_header_reg((kl_dump_header_t*)dh);
		goto out;
	} else if((magic_swap==KL_DUMP_MAGIC_NUMBER) ||
		  (magic_swap==KL_DUMP_MAGIC_LIVE)){
		kl_swap_dump_header_reg((kl_dump_header_t*)dh);
		kl_print_dump_header_reg((kl_dump_header_t*)dh);
		goto out;
	} else {
		fprintf(KL_ERRORFP,"Unknown Magic Number: %"FMT64"x\n",magic);
		rc = -1; goto out;
	}
out:
	if(dh)
		free(dh);
	return rc;
}	
