/*
 * $Id: kl_dump_ia64.c,v 1.5 2005/03/01 07:15:50 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>
#include <linux/stddef.h>	/*offsetof*/

/*
 * dha_num_cpus_ia64()
 */
uint32_t
dha_num_cpus_ia64(void)
{
        uint64_t off;
        uint32_t num_cpus;
	void *ptr;

        off = kl_member_offset(dha_typename, "dha_smp_num_cpus");
        ptr = (void *)((ulong)KL_DUMP_HEADER_ASM + off);
        num_cpus = KL_GET_UINT32(ptr);
	return(num_cpus);
}

/*
 * dha_current_task_ia64()
 */
kaddr_t
dha_current_task_ia64(int cpuid)
{
        uint64_t off;
        kaddr_t task_addr;
	void *ptr;

        if (cpuid < 0) {
                return(0);
        }
        off = kl_member_offset(dha_typename, "dha_smp_current_task");
	off += (cpuid * KL_NBPW);
        ptr = (void *)((ulong)KL_DUMP_HEADER_ASM + off);
        task_addr = KL_GET_UINT64(ptr);
	return(task_addr);
}

/*
 * dha_cpuid_ia64()
 */
int
dha_cpuid_ia64(kaddr_t task)
{
        int i;
        uint32_t num_cpus;
        kaddr_t task_addr;

	num_cpus = dha_num_cpus_ia64();
        for (i = 0; i < num_cpus; i++) {
		task_addr = dha_current_task_ia64(i);
                if (task == task_addr) {
                        return(i);
                }
        }
        return(-1);
}

/*
 * dha_stack_ia64()
 */
kaddr_t
dha_stack_ia64(int cpuid)
{
        uint64_t off;
        kaddr_t stack_addr;
	void *ptr;

        if (cpuid < 0) {
                return(0);
        }
        off = kl_member_offset(dha_typename, "dha_stack");
	off += (cpuid * KL_NBPW);
        ptr = (void *)((ulong)KL_DUMP_HEADER_ASM + off);
        stack_addr = KL_GET_UINT64(ptr);
        return(stack_addr);
}

/*
 * dha_stack_ptr_ia64()
 */
kaddr_t
dha_stack_ptr_ia64(int cpuid)
{
        uint64_t off;
        kaddr_t stack_ptr;
	void *ptr;

        if (cpuid < 0) {
                return(0);
        }

        off = kl_member_offset(dha_typename, "dha_stack_ptr");
	off += (cpuid * KL_NBPW);
        ptr = (void *)((ulong)KL_DUMP_HEADER_ASM + off);
        stack_ptr = KL_GET_UINT64(ptr);
        return(stack_ptr);
}

/*
 * kl_read_dump_header_ia64()
 */
int
kl_read_dump_header_ia64(void)
{
	/* first, make sure this isn't a live system
	 */
	if (CORE_IS_KMEM) {
		KL_ERROR = KLE_ACTIVE;
		return(1);
	}
	return(kl_read_dump_header_asm());
}

int
kl_set_dumparch_ia64(void)
{

        /* Get the page size and then use it to determine the 
	 * dump shift and mask values. 
         */      
	if (CORE_IS_DUMP) {
		/* Get the asm dump header for arch ia64.
		 */
		if (kl_read_dump_header_ia64()) {
			return(1);
		}

		/* We don't do any translation so just reference the 
		 * raw version.
		 */
		KL_DUMP_HEADER_ASM = G_dump_header_asm;       

		/* Now get the page size...
		 */
		KL_PAGE_SIZE = KL_DUMP_HEADER->page_size;
	} else {
		/* This is a live system.
		 * XXX -- need a way to get page size from live system
		 */
		KL_PAGE_SIZE = 0x4000;
	}
        KL_PAGE_SHIFT = 12;
        while ((KL_PAGE_SIZE >> KL_PAGE_SHIFT) != 1) { 
                KL_PAGE_SHIFT++;
        }       
        KL_PAGE_MASK = ~((uint64_t)KL_PAGE_SIZE-1);

        KL_PAGE_OFFSET  = KL_PAGE_OFFSET_IA64;
        KL_STACK_OFFSET = KL_KSTACK_SIZE_IA64;

	KLP->dump->arch.pgdshift = KL_PGDIR_SHIFT_IA64;
	KLP->dump->arch.pgdsize = KL_PGDIR_SIZE_IA64;
	KLP->dump->arch.pgdmask = KL_PGDIR_MASK_IA64;
	KLP->dump->arch.pmdshift = KL_PMD_SHIFT_IA64;
	KLP->dump->arch.pmdsize = KL_PMD_SIZE_IA64;
	KLP->dump->arch.pmdmask = KL_PMD_MASK_IA64;
	KLP->dump->arch.ptrsperpgd = KL_PTRS_PER_PGD_IA64;
	KLP->dump->arch.ptrsperpmd = KL_PTRS_PER_PMD_IA64;
	KLP->dump->arch.ptrsperpte = KL_PTRS_PER_PTE_IA64;
	KLP->dump->arch.kernelstack = kl_kernelstack_ia64;
	KLP->dump->arch.virtop = kl_virtop_ia64;
	KLP->dump->arch.mmap_virtop = kl_mmap_virtop_ia64;
	KLP->dump->arch.init_virtop = kl_init_virtop_ia64;
	KLP->dump->arch.valid_physmem = kl_valid_physmem_ia64;	
	KLP->dump->arch.next_valid_physaddr = kl_next_valid_physaddr_ia64;
	KLP->dump->arch.fix_vaddr = kl_fix_vaddr_ia64;

	return(0);
}
