/*
 * $Id: kl_alloc.c,v 1.1 2004/12/21 23:26:23 tjm Exp $
 *
 * This file is part of libutil.
 * A library which provides auxiliary functions.
 * libutil is part of lkcdutils -- utilities for Linux kernel crash dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <kl_lib.h>

alloc_functions_t alloc_functions;

/*
 * _kl_alloc_block()
 */
void *
_kl_alloc_block(int size, int flags, void *ra)
{
	void *blk;

	if (size == 0) {
		errno = EINVAL;
		return((void *)NULL);
	}
	if (KL_BLOCK_ALLOC()) {
	        blk = KL_BLOCK_ALLOC()(size, flags, ra);
	} else {
		if ((blk = (void*)malloc(size))) {
			memset(blk, 0, size);
		}
	}
	if (!blk) {
		errno = ENOMEM;
	}
	return(blk);
}

/*
 * _kl_realloc_block()
 */
void *
_kl_realloc_block(void *b, int new_size, int flags, void *ra)
{
	void * blk;


	if (!b || new_size == 0) {
		errno = EINVAL;
		return((void *)NULL);
	}

	if (KL_BLOCK_REALLOC()) {
		blk = (void *)KL_BLOCK_REALLOC()(b, new_size, flags, ra);
	} else {
		if ((blk = (void *)realloc(b, new_size))) {
			memset(blk, 0, new_size);
		}
	}
	if (!blk) {
		errno = ENOMEM;
	}
	return(blk);
}

/*
 * _kl_dup_block()
 */
void *
_kl_dup_block(void *b, int flags, void *ra)
{
	void *blk;

	if (!b) {
		errno = EINVAL;
		return((void *)NULL);
	}

	if (KL_BLOCK_DUP()) {
		blk = (void *)KL_BLOCK_DUP()(b, flags, ra);
	} else {
		/* There is no way to determine the size of a block if 
		 * it has been allocated via malloc(). Therefore return 
		 * a NULL pointer (and an error).
		 */
		errno = EPERM;
		return((void *)NULL);
	}
	if (!blk) {
		errno = ENOMEM;
	}
	return(blk);
}

/*
 * _kl_str_to_block()
 */
void *
_kl_str_to_block(char *s, int flags, void *ra)
{
	void *blk = (void *)NULL;

	if (!s) {
		errno = EINVAL;
		return((void *)NULL);
	}

	if (KL_STR_TO_BLOCK()) {
		blk = (void *)KL_STR_TO_BLOCK()(s, flags, ra);
	} else {
		int size;

		if ((size = (strlen(s) + 1))) {
			blk = _kl_alloc_block(size, flags, ra);
			memcpy(blk, s, size);
		}
	}
	if (!blk) {
		errno = ENOMEM;
	}
	return(blk);
}

/*
 * kl_free_block()
 */
void
kl_free_block(void *blk)
{
	if (!blk) {
		return;
	}
	if (KL_BLOCK_FREE()) {
		KL_BLOCK_FREE()(blk);
	} else {
		free(blk);
	}
}
