/*
 * $Id: kl_table.c,v 1.1 2004/12/21 23:26:23 tjm Exp $
 *
 * This file is part of libutil.
 * A library which provides auxiliary functions.
 * libutil is part of lkcdutils -- utilities for Linux kernel crash dumps.
 *
 * Created by Silicon Graphics, Inc.
 *
 * Copyright (C) 2003, 2004 Silicon Graphics, Inc. All rights reserved.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#include <kl_lib.h>
#include <kl_table.h>

/*
 * tbl_alloc_table()
 */
table_t *
tbl_alloc_table(int num_cols)
{
	table_t *tbl;

	tbl = (table_t *)kl_alloc_block(sizeof(table_t), K_PERM);
	tbl->num_cols = num_cols;
	tbl->col_info = (col_info_t*)
		kl_alloc_block(num_cols * sizeof(col_info_t), K_PERM);
	return(tbl);
}

/*
 * tbl_free_table()
 */
void
tbl_free_table(table_t *tbl)
{
	int i;
	tbl_row_t *row, *next_row;

	if (!tbl) {
		return;
	}
	row = tbl->rows;
	while (row) {
		next_row = row->next_row;
		if (row->columns) {
			for (i = 0; i < tbl->num_cols; i++) {
				kl_free_block(row->columns[i].src_line);
			}
			kl_free_block(row->columns);
		}
		kl_free_block(row);
		row = next_row;
	}
	if (tbl->col_info) {
		kl_free_block(tbl->col_info);
	}
	kl_free_block(tbl);
}

/*
 * tbl_init_col()
 */
int
tbl_init_col(table_t *tbl, int col_num, int col_width, int col_padd)
{
	if (col_num > tbl->num_cols) {
		return(1);
	}
	tbl->col_info[col_num].col_num = col_num;
	tbl->col_info[col_num].col_width = col_width;
	tbl->col_info[col_num].col_padd = col_padd;
	return(0);
}

/* 
 * tbl_finish_setup()
 */
void
tbl_finish_setup(table_t *tbl)
{
	int i, size = 0;

	for (i = 0; i < tbl->num_cols; i++) {
		size += tbl->col_info[i].col_width;
	}
	tbl->row_size = size + 1;
}

/*
 * tbl_add_row()
 */
tbl_row_t *
tbl_add_row(table_t *tbl)
{
	int i;
	tbl_row_t *row, *last_row;
	
	if ((last_row = tbl->rows)) {
		while (last_row->next_row) {
			last_row = last_row->next_row;
		}
	}
	row = (tbl_row_t *)kl_alloc_block(sizeof(tbl_row_t), K_PERM);
	row->tbl = tbl;
	row->row_num = tbl->num_rows;
	tbl->num_rows++;
	row->columns = (tbl_col_t *)
		kl_alloc_block(tbl->num_cols * sizeof(tbl_col_t), K_PERM);
	for (i = 0; i < tbl->num_cols; i++) {
		row->columns[i].col_info = &tbl->col_info[i];
	}
	if (last_row) {
		last_row->next_row = row;
	} else {
		tbl->rows = row;
	}
	return(row);
}

/* 
 * next_line()
 */
static char *
next_line(char *line, int width)
{
	int i;
	char *c, *last_space;

	c = line;
	last_space = (char *)NULL;
	for (i = 0; i < width; i++) {
		if (!(*c)) {
			break;
		} else if (*c == ' ') {
			last_space = c;
		}
		c++;
	}

	if (*c) {
		if ((*c != ' ') && last_space) {
			c = (char *)(last_space + 1);
		}
		return(c);
	}
	return((char *)NULL);
}

/*
 * line_count()
 */
static int
line_count(char *line, int width)
{
	int count;
	char *nline, *c;
	
	if (!line) {
		return(0);
	}
	nline = next_line(line, width);
	if (nline) {
		c = line;
		count = 0;
		while (c != nline) {
			count++;
			c++;
		}
	} else {
		count = strlen(line);
	}
	return(count);
}

/*
 * tbl_setup_col()
 */
int
tbl_setup_col(tbl_row_t *row, int col_num, int attr, char *src_line)
{
	int i, num_lines, width, padd;
	char *c;
	tbl_col_t *col;

	if (col_num > row->tbl->num_cols) {
		return(1);
	}
	col = &row->columns[col_num];
	col->src_line = (char *)kl_alloc_block(strlen(src_line) + 1, K_PERM);
	if (col->src_line) {
		strcpy(col->src_line, src_line);
	}
	col->attr = attr;

	width = COL_WIDTH(col);

	/* See if we have been requested to padd the column. We only do
	 * this when we are wrapping, justifying left or right.
	 */
	if ((padd = COL_PADD(col)) && !(col->attr == TBL_ATTR_CENTER)) {
		width -= (2 * padd);
	}
	if (!(c = col->src_line)) {
		return(0);
	}

	/* Determine exactly how many lines are neede for this
	 * src_line -- src_line is not NULL, so we know we have
	 * at least one.
	 */
	num_lines = 1;
	while ((c = next_line(c, width))) {
		num_lines++;
	}
	col->num_lines = num_lines;

	/* allocate space for the line array and then capture 
	 * the line pointers and line char counts.
	 */
	col->lines = (tbl_line_t *)
		kl_alloc_block(num_lines * sizeof(tbl_line_t), K_PERM);

	c = col->src_line;
	col->lines[0].ptr = c;
	col->lines[0].cnt = line_count(c, width);
	i = 1;
	while ((c = next_line(c, width))) {
		col->lines[i].ptr = c;
		col->lines[i].cnt = line_count(c, width);
		i++;		
	}
	if (num_lines > row->max_lines) {
		row->max_lines = num_lines;
	}
	return(0);
}

/*
 * col_size()
 */
static int
col_size(char *str, int len)
{
	int size = len;
	char *c = &str[len - 1];

	/* Strip off trailing blanks 
	 */
	while (*c == ' ') {
		size--;
		c--;
	}

	/* Now strip off leading blanks
	 */
	c = str;
	while (size && (*c == ' ')) {
		size--;
		c++;
	}
	return(size);
}

/* 
 * format_col()
 */
static void
format_col(tbl_col_t *col, int line, char **c)
{
	int i, count, len, width, padd;
	int before = 0, after, beginning = 1;
	char *p;

	width = col->col_info->col_width;
	if (col->attr == TBL_ATTR_CENTER) {
		len = col_size(col->lines[line].ptr, col->lines[line].cnt);
		before = (width - len) / 2;
	} else {
		if ((padd = COL_PADD(col))) {
			before = padd;
		}
		switch(col->attr) {
			case TBL_ATTR_WRAP:
				len = col->lines[line].cnt;
				break;

			case TBL_ATTR_LEFT:
				len = col_size(col->lines[line].ptr, 
					col->lines[line].cnt);
				break;

			case TBL_ATTR_RIGHT:
				len = col_size(col->lines[line].ptr, 
					col->lines[line].cnt);
				before += (width - len - 1 - padd);
				break;
		}
	}
	count = before;
	while (count) {
		*(*c) = ' ';
		(*c)++;
		count--;
	}
	p = col->lines[line].ptr;
	for (i = 0; i < len; i++) {
		switch (col->attr) {
			case TBL_ATTR_LEFT:
			case TBL_ATTR_RIGHT:
				if (beginning) {
					while (*p == ' ') {
						p++;
					}
					beginning = 0;
				}
				break;
		}
		*(*c) = *p;
		(*c)++;
		p++;
	}
	after = (width - len - before);
	while (after) {
		*(*c) = ' ';
		(*c)++;
		after--;
	}
}

/*
 * tbl_format_line()
 */
void
tbl_format_line(tbl_row_t *row, int line, char *str)
{
	int i, extra;
	char *c = str; 
	tbl_col_t *col;
	table_t *tbl = row->tbl;

	for (i = 0; i < tbl->num_cols; i++) {
		*c = '|';
		c++;

		col = &row->columns[i];
		if (col->num_lines > line) {
			format_col(col, line, &c);
		} else {
			extra = col->col_info->col_width;
			while (extra) {
				*c = ' ';
				c++;
				extra--;
			}
		}
	}
	*c = '|';
	c++;
	*c = 0;
}

/*
 * tbl_format_seperator()
 */
void
tbl_format_seperator(table_t *tbl, char *str)
{
	int i, j, width;
	char *c = str;

	for (i = 0; i < tbl->num_cols; i++) {
		width = tbl->col_info[i].col_width;
		*c = '+';
		c++;
		for (j = 0; j < width; j++) {
			*c = '-';
			c++;
		}
	}
	*c = '+';
	c++;
	*c = 0;
}
