<?php
/*
$Id: modules.inc,v 1.168 2006/10/26 17:49:50 gruberroland Exp $

  This code is part of LDAP Account Manager (http://www.sourceforge.net/projects/lam)
  Copyright (C) 2003 - 2006  Roland Gruber

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


*/

/**
* Interface between modules and other parts of LAM.
*
* @package modules
* @author Tilo Lutz
* @author Michael Duergner
* @author Roland Gruber
*/

/** LDAP caches */
include_once("cache.inc");
/** some helper functions */
include_once("account.inc");
/** parent class of account modules */
include_once("baseModule.inc");
/** access to LDAP server */
include_once("ldap.inc");
/** lamdaemon functions */
include_once("lamdaemon.inc");
/** security functions */
include_once("security.inc");

/**
* This includes all module files.
*/
$modulesINC_dirname = substr(__FILE__, 0, strlen(__FILE__) - 12) . "/modules";
$modulesINC_dir = dir($modulesINC_dirname);
// get module names.
while ($entry = $modulesINC_dir->read())
if ((substr($entry, strlen($entry) - 4, 4) == '.inc') && is_file($modulesINC_dirname . '/'.$entry)) {
	include_once($modulesINC_dirname . '/'.$entry);
}

/**
* Returns the alias name of a module
*
* @param string $name the module name
* @param string $scope the account type ("user", "group", "host")
* @return string alias name
*/
function getModuleAlias($name, $scope) {
	$module = new $name($scope);
	return $module->get_alias();
}

/**
* Returns true if the module is a base module
*
* @param string $name the module name
* @param string $scope the account type ("user", "group", "host")
* @return boolean true if base module
*/
function is_base_module($name, $scope) {
	$module = new $name($scope);
	return $module->is_base_module();
}

/**
* Returns the LDAP filter used by the account lists
*
* @param string $scope the account type ("user", "group", "host")
* @return string LDAP filter
*/
function get_ldap_filter($scope) {
	$mods = $_SESSION['config']->get_AccountModules($scope);
	$filters = array();
	$orFilter = '';
	for ($i = 0; $i < sizeof($mods); $i++) {
		$module = new $mods[$i]($scope);
		$modinfo = $module->get_ldap_filter();
		if (isset($modinfo['or'])) $filters['or'][] = $modinfo['or'];
		if (isset($modinfo['and'])) $filters['and'][] = $modinfo['and'];
	}
	// build OR filter
	if (sizeof($filters['or']) == 1) {
		$orFilter = $filters['or'][0];
	}
	elseif (sizeof($filters['or']) > 1) {
		$orFilter = "(|" . implode("", $filters['or']) . ")";
	}
	// add built OR filter to AND filters
	if ($orFilter != '') $filters['and'][] = $orFilter;
	// collapse AND filters
	if (sizeof($filters['and']) < 2) return $filters['and'][0];
	else return "(&" . implode("", $filters['and']) . ")";
}

/**
* Returns a list of LDAP attributes which can be used to form the RDN.
*
* The list is already sorted by the priority given by the nodules.
*
* @param string $scope account type (user, group, host)
* @return array list of LDAP attributes
*/
function getRDNAttributes($scope) {
	$mods = $_SESSION['config']->get_AccountModules($scope);
	$return = array();
	$attrs_low = array();
	$attrs_normal = array();
	$attrs_high = array();
	for ($i = 0; $i < sizeof($mods); $i++) {
		// get list of attributes
		$module = new $mods[$i]($scope);
		$attrs = $module->get_RDNAttributes();
		$keys = array_keys($attrs);
		// sort attributes
		for ($k = 0; $k < sizeof($keys); $k++) {
			switch ($attrs[$keys[$k]]) {
				case "low":
					$attrs_low[] = $keys[$k];
					break;
				case "normal":
					$attrs_normal[] = $keys[$k];
					break;
				case "high":
					$attrs_high[] = $keys[$k];
					break;
				default:
					$attrs_low[] = $keys[$k];
					break;
			}
		}
	}
	// merge arrays
	$return = array_values(array_unique($attrs_high));
	for ($i = 0; $i < sizeof($attrs_normal); $i++) {
		if (!in_array($attrs_normal[$i], $return)) $return[] = $attrs_normal[$i];
	}
	for ($i = 0; $i < sizeof($attrs_low); $i++) {
		if (!in_array($attrs_low[$i], $return)) $return[] = $attrs_low[$i];
	}
	return $return;
}

/**
* Returns a hash array (module name => dependencies) of all module dependencies
*
* "dependencies" contains an array with two sub arrays: depends, conflicts
* <br>The elements of "depends" are either module names or an array of module names (OR-case).
* <br>The elements of conflicts are module names.
*
* @param string $scope the account type (user, group, host)
* @return array dependencies
*/
function getModulesDependencies($scope) {
	$mods = getAvailableModules($scope);
	for ($i = 0; $i < sizeof($mods); $i++) {
		$module = new $mods[$i]($scope);
		$return[$mods[$i]] = $module->get_dependencies();
	}
	return $return;
}


/**
* Checks if there are missing dependencies between modules.
*
* @param array $selected selected module names
* @param array $deps module dependencies
* @return mixed false if no misssing dependency was found,
* otherwise an array of array(selected module, depending module) if missing dependencies were found
*/
function check_module_depends($selected, $deps) {
	$ret = array();
	for ($m = 0; $m < sizeof($selected); $m++) {  // check selected modules
		for ($i = 0; $i < sizeof($deps[$selected[$m]]['depends']); $i++) {  // check dependencies of module
			// check if we have OR-combined modules
			if (is_array($deps[$selected[$m]]['depends'][$i])) {
				// one of the elements is needed
				$found = false;
				$depends = $deps[$selected[$m]]['depends'][$i];
				for ($d = 0; $d < sizeof($depends); $d++) {
					if (in_array($depends[$d], $selected)) {
						$found = true;
						break;
					}
				}
				if (! $found) {
					// missing dependency, add to return value
					$ret[] = array($selected[$m], implode(" || ", $depends));
				}
			}
			else {
				// single dependency
				if (! in_array($deps[$selected[$m]]['depends'][$i], $selected)) {
					// missing dependency, add to return value
					$ret[] = array($selected[$m], $deps[$selected[$m]]['depends'][$i]);
				}
			}
		}
	}
	if (sizeof($ret) > 0) return $ret;
	else return false;
}

/**
* Checks if there are conflicts between modules
*
* @param array $selected selected module names
* @param array $deps module dependencies
* @return boolean false if no conflict was found,
* otherwise an array of array(selected module, conflicting module) if conflicts were found
*/
function check_module_conflicts($selected, $deps) {
	$ret = array();
	for ($m = 0; $m < sizeof($selected); $m++) {
		for ($i = 0; $i < sizeof($deps[$selected[$m]]['conflicts']); $i++) {
			if (in_array($deps[$selected[$m]]['conflicts'][$i], $selected)) {
				$ret[] = array($selected[$m], $deps[$selected[$m]]['conflicts'][$i]);
			}
		}
	}
	if (sizeof($ret) > 0) return $ret;
	else return false;
}

/**
* Returns an array with all available user module names
*
* @param string $scope account type (user, group, host)
* @return array list of possible modules
*/
function getAvailableModules($scope) {
	$dirname = substr(__FILE__, 0, strlen(__FILE__) - 12) . "/modules";
	$dir = dir($dirname);
	$return = array();
	// get module names.
	while ($entry = $dir->read())
		if ((substr($entry, strlen($entry) - 4, 4) == '.inc') && is_file($dirname . '/'.$entry)) {
			$entry = substr($entry, 0, strpos($entry, '.'));
			$temp = new $entry($scope);
			if ($temp->can_manage()) $return[] = $entry;
		}
	return $return;
}

/**
* Returns the elements for the profile page.
*
* @param string $scope account type (user, group, host)
* @return array profile elements
*/
function getProfileOptions($scope) {
	$mods = $_SESSION['config']->get_AccountModules($scope);
	$return = array();
	for ($i = 0; $i < sizeof($mods); $i++) {
		$module = new $mods[$i]($scope);
		$return[$mods[$i]] = $module->get_profileOptions();
	}
	return $return;
}

/**
* Checks if the profile options are valid
*
* @param string $scope account type (user, group, host)
* @param array $options hash array containing all options (name => array(...))
* @return array list of error messages
*/
function checkProfileOptions($scope, $options) {
	$mods = $_SESSION['config']->get_AccountModules($scope);
	$return = array();
	for ($i = 0; $i < sizeof($mods); $i++) {
		$module = new $mods[$i]($scope);
		$temp = $module->check_profileOptions($options);
		$return = array_merge($return, $temp);
	}
	return $return;
}

/**
* Returns a hash array (module name => elements) of all module options for the configuration page.
*
* @param array $scopes hash array (module name => array(account types))
* @return array configuration options
*/
function getConfigOptions($scopes) {
	$return = array();
	$modules = array_keys($scopes);
	for ($i = 0; $i < sizeof($modules); $i++) {
		$m = new $modules[$i]('none');
		$return[$modules[$i]] = $m->get_configOptions($scopes[$modules[$i]], $scopes);
	}
	return $return;
}

/**
* Returns a hash array (module name => descriptions) containing descriptions shown on configuration pages.
*
* The returned array has the format array('legend' => array('posixAccount' => '...', ...), descriptions => array('option1' => '...', ...)).
* <br> The "legend" value is used as text for the fieldset, the descriptions are used when the configuration is printed.
*
* @return array configuration descriptions
*/
function getConfigDescriptions() {
	$return = array('legend' => array(), 'descriptions' => array());
	$modules = array();
	$types = $_SESSION['config']->get_ActiveTypes();
	for ($i = 0; $i < sizeof($types); $i++) {
		$modules = array_merge($modules, getAvailableModules($types[$i]));
	}
	$modules = array_merge(getAvailableModules('user'), getAvailableModules('group'), getAvailableModules('host'));
	$modules = array_values(array_unique($modules));
	for ($i = 0; $i < sizeof($modules); $i++) {
		$m = new $modules[$i]('none');
		$desc = $m->get_configDescriptions();
		$return['legend'][$modules[$i]] = $desc['legend'];
		$return['descriptions'] = array_merge($return['descriptions'], $desc['descriptions']);
	}
	return $return;
}

/**
* Checks if the configuration options are valid
*
* @param array $scopes hash array (module name => array(account types))
* @param array $options hash array containing all options (name => array(...))
* @return array list of error messages
*/
function checkConfigOptions($scopes, $options) {
	$return = array();
	$modules = array_keys($scopes);
	for ($i = 0; $i < sizeof($modules); $i++) {
		$m = new $modules[$i]('none');
		$errors = $m->check_configOptions($scopes[$modules[$i]], $options);
		$return = array_merge($return, $errors);
	}
	return $return;
}

/**
* Returns a help entry from an account module.
*
* @param string $helpID help identifier
* @param string $module module name
* @return array help entry
*/
function getHelp($module,$helpID,$scope='') {
	$moduleObject = new $module((($scope != '') ? $scope : 'none'));
	return $moduleObject->get_help($helpID);
}

/**
* Returns a list of available PDF entries.
*
* @param string $scope account type (user, group, host)
* @return array PDF entries
*/
function getAvailablePDFFields($scope) {
	$mods = $_SESSION['config']->get_AccountModules($scope);
	$return = array();
	for ($i = 0; $i < sizeof($mods); $i++) {
		$module = new $mods[$i]($scope);
		$return[$mods[$i]] = $module->get_pdfFields();
	}
	$return['main'] = array('dn');
	return $return;
}

/**
* Returns an array containing all input columns for the file upload.
*
* Syntax:
* <br> array(
* <br>  string: name,  // fixed non-translated name which is used as column name (should be of format: <module name>_<column name>)
* <br>  string: description,  // short descriptive name
* <br>  string: help,  // help ID
* <br>  string: example,  // example value
* <br>  boolean: required  // true, if user must set a value for this column
* <br> )
*
* @param string $scope account type
* @return array column list
*/
function getUploadColumns($scope) {
	$mods = $_SESSION['config']->get_AccountModules($scope);
	$return = array();
	for ($i = 0; $i < sizeof($mods); $i++) {
		$module = new $mods[$i]($scope);
		$return[$mods[$i]] = $module->get_uploadColumns();
	}
	return $return;
}

/**
* This function builds the LDAP accounts for the file upload.
*
* If there are problems status messages will be printed automatically.
*
* @param string $scope account type
* @param array $data array containing one account in each element
* @param array $ids array(<column_name> => <column number>)
* @return mixed array including accounts or false if there were errors
*/
function buildUploadAccounts($scope, $data, $ids) {
	// build module order
	$unOrdered = $_SESSION['config']->get_AccountModules($scope);
	$ordered = array();
	$predepends = array();
	// get dependencies
	for ($i = 0; $i < sizeof($unOrdered); $i++) {
		$mod = new $unOrdered[$i]($scope);
		$predepends[$unOrdered[$i]] = $mod->get_uploadPreDepends();
	}
	// first all modules without predepends can be ordered
	for ($i = 0; $i < sizeof($unOrdered); $i++) {
		if (sizeof($predepends[$unOrdered[$i]]) == 0) {
			$ordered[] = $unOrdered[$i];
			unset($unOrdered[$i]);
			$unOrdered = array_values($unOrdered);
			$i--;
		}
	}
	$unOrdered = array_values($unOrdered);  // fix indexes
	// now add all modules with fulfilled dependencies until all are in order
	while (sizeof($unOrdered) > 0) {
		$newRound = false;
		for ($i = 0; $i < sizeof($unOrdered); $i++) {
			$deps = $predepends[$unOrdered[$i]];
			$depends = false;
			for ($d = 0; $d < sizeof($deps); $d++) {
				if (in_array($deps[$d], $unOrdered)) {
					$depends = true;
					break;
				}
			}
			if (!$depends) {  // add to order if dependencies are fulfilled
				$ordered[] = $unOrdered[$i];
				unset($unOrdered[$i]);
				$unOrdered = array_values($unOrdered);
				$newRound = true;
				break;
			}
		}
		if ($newRound) continue;
		// this point should never be reached, LAM was unable to find a correct module order
		StatusMessage("ERROR", "Internal Error: Unable to find correct module order.", "");
		return false;
	}
	// give raw data to modules
	$errors = array();
	$partialAccounts = array();
	for ($i = 0; $i < sizeof($data); $i++) $partialAccounts[$i]['objectClass'] = array();
	for ($i = 0; $i < sizeof($ordered); $i++) {
		$module = new $ordered[$i]($scope);
		$errors = $module->build_uploadAccounts($data, $ids, $partialAccounts);
		if (sizeof($errors) > 0) {
			array_unshift($errors, array("INFO", _("Displayed account numbers start at \"0\". Add 2 to get the row in your spreadsheet."), ""));
			$errors[] = array("ERROR", _("Upload was stopped after errors in %s module!"), "", array($module->get_alias()));
			break;
		}
	}
	if (sizeof($errors) > 0) {
		for ($i = 0; (($i < sizeof($errors)) || ($i > 49)); $i++) call_user_func_array("StatusMessage", $errors[$i]);
		return false;
	}
	else return $partialAccounts;
}

/**
* This function executes one post upload action.
*
* @param string $scope account type
* @param array $data array containing one account in each element
* @param array $ids array(<column_name> => <column number>)
* @param array $failed list of accounts which were not created successfully
* @return array current status
* <br> array (
* <br>  'status' => 'finished' | 'inProgress'
* <br>  'module' => <name of active module>
* <br>  'progress' => 0..100
* <br>  'errors' => array (<array of parameters for StatusMessage>)
* <br> )
*/
function doUploadPostActions($scope, $data, $ids, $failed) {
	// check if function is called the first time
	if (! isset($_SESSION['mass_postActions']['remainingModules'])) {
		// make list of remaining modules
		$moduleList = $_SESSION['config']->get_AccountModules($scope);
		$_SESSION['mass_postActions']['remainingModules'] = $moduleList;
	}
	$activeModule = $_SESSION['mass_postActions']['remainingModules'][0];
	// initialize temporary variable
	if (!isset($_SESSION['mass_postActions'][$activeModule])) {
		$_SESSION['mass_postActions'][$activeModule] = array();
	}
	// let first module do one post action
	$module = new $activeModule($scope);
	$return = $module->doUploadPostActions($data, $ids, $failed, $_SESSION['mass_postActions'][$activeModule]);
	// remove active module from list if already finished
	if ($return['status'] == 'finished') {
		unset($_SESSION['mass_postActions']['remainingModules'][0]);
		$_SESSION['mass_postActions']['remainingModules'] = array_values($_SESSION['mass_postActions']['remainingModules']);
	}
	// update status and return back to upload page
	$return['module'] = $activeModule;
	if (sizeof($_SESSION['mass_postActions']['remainingModules']) > 0) {
		$return['status'] = 'inProgress';
	}
	else {
		$return['status'] = 'finished';
	}
	return $return;
}

/**
* Returns true if the module is a base module
*
* @return array required extensions
*/
function getRequiredExtensions() {
	$extList = array();
	$scopes = $_SESSION['config']->get_ActiveTypes();
	for ($i = 0; $i < sizeof($scopes); $i++) {
		$mods = $_SESSION['config']->get_AccountModules($scopes[$i]);
		for ($m = 0; $m < sizeof($mods); $m++) {
			$module = new $mods[$m]($scopes[$i]);
			$ext = $module->getRequiredExtensions();
			for ($e = 0; $e < sizeof($ext); $e++) {
				if (!in_array($ext[$e], $extList)) $extList[] = $ext[$e];
			}
		}
	}
	return $extList;
}

/**
* Takes a list of meta-HTML elements and prints the equivalent HTML output.
*
* @param string $module Name of account module
* @param array $input List of meta-HTML elements
* @param array $values List of values which override the defaults in $input (name => value)
* @param boolean $restricted If true then no buttons will be displayed
* @param integer $tabindex Start value of tabulator index for input fields
* @param integer $tabindexLink Start value of tabulator index for links
* @param string $scope Account type
* @return array List of input field names and their type (name => type)
*/
function parseHtml($module, $input, $values, $restricted, &$tabindex, &$tabindexLink, $scope) {
	$ret = array();
	if (is_array($input)) {
		echo "<table>\n";
		for ($i=0; $i<count($input); $i++) {	// $i = row number
			// Draw column
			echo "<tr>\n";
			for ($j=0; $j<count($input[$i]); $j++ ) {	// $j = column number
				// Draw cell
				echo "<td";
				if (isset($input[$i][$j]['td']['align'])) echo " align=\"" . $input[$i][$j]['td']['align'] . "\"";
				if (isset($input[$i][$j]['td']['valign'])) echo " valign=\"" . $input[$i][$j]['td']['valign'] . "\"";
				if (isset($input[$i][$j]['td']['colspan'])) echo " colspan=\"" . $input[$i][$j]['td']['colspan'] . "\"";
				if (isset($input[$i][$j]['td']['rowspan'])) echo " rowspan=\"" . $input[$i][$j]['td']['rowspan'] . "\"";
				if (isset($input[$i][$j]['td']['width'])) echo " width=\"" . $input[$i][$j]['td']['width'] . "\"";
				echo ">\n";
				switch ($input[$i][$j]['kind']) {
					// plain text
					case 'text':
						if (isset($input[$i][$j]['text'])) echo $input[$i][$j]['text'];
						break;
					// input fields
					case 'input':
						$type = $input[$i][$j]['type'];
						if ($restricted && (($type == "submit") || ($type == "reset") || ($type == "file"))) break;	// no buttons in restricted mode
						$output = "<input";
						if ($input[$i][$j]['name']!='') $output .= ' name="' . $input[$i][$j]['name'] . '"';
						if ($type != '') $output .= ' type="' . $type . '"';
						if (isset($input[$i][$j]['size']) && ($input[$i][$j]['size'] != '')) $output .= ' size="' . $input[$i][$j]['size'] . '"';
						if (isset($input[$i][$j]['maxlength']) && ($input[$i][$j]['maxlength'] != '')) $output .= ' maxlength="' . $input[$i][$j]['maxlength'] . '"';
						// checkbox
						if ($type == "checkbox") {
							if (isset($values[$input[$i][$j]['name']])) {
								if ($values[$input[$i][$j]['name']][0] == "true") $output .= ' checked';
							}
							elseif ($input[$i][$j]['checked']) $output .= ' checked';
						}
						// other input element
						else {
							if (isset($values[$input[$i][$j]['name']])) {
								$output .= ' value="' . $values[$input[$i][$j]['name']][0] . '"';
							}
							elseif (isset($input[$i][$j]['value']) && $input[$i][$j]['value']!='') $output .= ' value="' . $input[$i][$j]['value'] . '"';
						}
						if (isset($input[$i][$j]['disabled']) && ($input[$i][$j]['disabled'] == true)) $output .= ' disabled';
						// Show taborder
						else {
							$output .= " tabindex=$tabindex";
							$tabindex++;
						}
						$output .= ">";
						echo $output;
						$ret[$input[$i][$j]['name']] = $type;	// save type
						break;
					// text area
					case 'textarea':
						echo "<textarea";
						if (isset($input[$i][$j]['name'])) echo ' name="' . $input[$i][$j]['name'] . '"';
						if (isset($input[$i][$j]['readonly']) && (isset($input[$i][$j]['readonly']) === true)) echo ' readonly';
						echo ' cols="' . $input[$i][$j]['cols'] . '"';
						echo ' rows="' . $input[$i][$j]['rows'] . '"';
						echo ">";
						if (isset($values[$input[$i][$j]['name']])) echo $values[$input[$i][$j]['name']];
						else echo $input[$i][$j]['value'];
						echo "</textarea>";
						break;
					// inner fieldset
					case 'fieldset':
						echo "<fieldset>\n";
						if ($input[$i][$j]['legend']!='') echo "<legend>" . $input[$i][$j]['legend'] . "</legend>\n";
						parseHtml($module, $input[$i][$j]['value'], $values, $restricted, $tabindex, $tabindexLink, $scope);
						echo "</fieldset>\n";
						break;
					// selection
					case 'select':
						if (! isset($input[$i][$j]['size'])) $input[$i][$j]['size'] = 1;	// correct size if needed
						if (isset($input[$i][$j]['multiple'])) {
							echo "<select name=\"" . $input[$i][$j]['name'] . '[]"';
							echo ' multiple';
							$ret[$input[$i][$j]['name']] = 'multiselect';	// save type
						}
						else {
							echo "<select name=\"" . $input[$i][$j]['name'] . '"';
							$ret[$input[$i][$j]['name']] = 'select';	// save type
						}
						echo ' size="' . $input[$i][$j]['size'] . '"';
						// Show taborder
						echo " tabindex=$tabindex";
						$tabindex++;
						echo ">\n";
						// init option fields
						if (!is_array($input[$i][$j]['options'])) $input[$i][$j]['options'] = array ( $input[$i][$j]['options'] );
						if (isset($input[$i][$j]['options_selected'])) {
							if (!is_array($input[$i][$j]['options_selected'])) {
								// one selected element
								$input[$i][$j]['options_selected'] = array ( $input[$i][$j]['options_selected'] );
							}
						}
						else {
							$input[$i][$j]['options_selected'] = array();
						}
						if (isset($values[$input[$i][$j]['name']])) $input[$i][$j]['options_selected'] = $values[$input[$i][$j]['name']];
						// merge both option arrays and sort them.
						$options = $input[$i][$j]['options'];
						if (!isset($input[$i][$j]['descriptiveOptions']) || ($input[$i][$j]['descriptiveOptions'] === false)) {
							// merge both option arrays and sort them.
							$options = array_merge($input[$i][$j]['options'], $input[$i][$j]['options_selected'] );
							$options = array_unique($options);
						}
						if (!isset($input[$i][$j]['noSorting']) || !$input[$i][$j]['noSorting']) {
							sort($options);
						}
						foreach ($options as $option) {
							if (isset($input[$i][$j]['descriptiveOptions']) && ($input[$i][$j]['descriptiveOptions'] === true)) {
								if (in_array($option[0], $input[$i][$j]['options_selected'])) echo "<option value=\"" . $option[0] . "\" selected>" . $option[1] . "</option>\n";
								else echo "<option value=\"" . $option[0] . "\">" . $option[1] . "</option>\n";
							}
							elseif ($option!='') {
								if (in_array($option, $input[$i][$j]['options_selected'])) echo "<option selected>" . $option . "</option>\n";
								else echo "<option>" . $option . "</option>\n";
							}
						}
						echo "</select>\n";
						break;
					// sub table
					case 'table':
						$retTemp = parseHtml($module, $input[$i][$j]['value'], $values, $restricted, $tabindex, $tabindexLink, $scope);
						$ret = array_merge($ret, $retTemp);
						break;
					// help link
					case 'help':
						$helpPath = "../";
						if (is_file("./help.php")) $helpPath = "";
						echo "<a href=" . $helpPath . "help.php?module=$module&amp;HelpNumber=". $input[$i][$j]['value'] . "&amp;scope=" . $scope . " target=\"help\" tabindex=$tabindexLink>";
						echo "<img src=\"../$helpPath/graphics/help.png\" alt=\"" . _('Help') . "\" title=\"" . _('Help') . "\">";
						echo "</a>\n";
						$tabindexLink++;
						break;
					// status message
					case 'message':
						StatusMessage($input[$i][$j]['type'], $input[$i][$j]['headline'], $input[$i][$j]['text']);
						break;
					// image
					case 'image':
						echo "<img ";
						if (isset($input[$i][$j]['path'])) echo 'src="' . $input[$i][$j]['path'] . '" ';
						if (isset($input[$i][$j]['width'])) echo 'width="' . $input[$i][$j]['width'] . '" ';
						if (isset($input[$i][$j]['height'])) echo 'height="' . $input[$i][$j]['height'] . '" ';
						if (isset($input[$i][$j]['alt'])) echo 'alt="' . $input[$i][$j]['alt'] . '" ';
						echo ">\n";
						break;
					// error, unknown type
					default:
						echo "Unrecognized type: " . $input[$i][$j]['kind'] . "\n";
						break;
				}
				echo "</td>\n";
			}
			echo "</tr>\n";
		}
	}
	echo "</table>\n";
	return $ret;
}


/**
* This class includes all modules and attributes of an account.
*
* @package modules
*/
class accountContainer {
	/**
	* Constructor
	*
	* @param string $type account type
	* @param string $base key in $_SESSION where this object is saved
	*/
	function accountContainer($type, $base) {
		/* Set the type of account. Valid
		* types are: user, group, host
		*/
		// Check input variable

		if (!is_string($type)) trigger_error('Argument of accountContainer must be string.', E_USER_ERROR);
		if (!is_string($base)) trigger_error('Argument of accountContainer must be string.', E_USER_ERROR);
		$this->type = $type;
		$this->base = $base;
		// Set startpage
		$this->current_page=0;
		$this->subpage='attributes';
		$this->isNewAccount = false;
		return 0;
		}

	/**
	* Array of all used attributes
	* Syntax is attribute => array ( objectClass => MUST or MAY, ...)
	*/
	var $attributes;

	/**
	* This variale stores the account type.
	* Currently "user", "group" and "host" are supported.
	*/
	var $type;

	/** This is an array with all module objects */
	var $module;

	/** DN suffix of the account */
	var $dn;

	/** DN suffix of account when it was loaded */
	var $dn_orig;

	/** RDN attribute of this account */
	var $rdn;

	/** original LDAP attributes when account was loaded from LDAP */
	var $attributes_orig;

	/** Module order */
	var $order;

	/** Name of accountContainer variable in session */
	var $base;

	/** This variable stores the name of the currently displayed page */
	var $current_page;

	/** This variable is set to the pagename of a subpage if it should be displayed */
	var $subpage;

	/** True if this is a newly created account */
	var $isNewAccount;


	/**
	* Returns the accout type of this object (e.g. user, group, host).
	*
	* @return string account type
	*/
	function get_type() {
		return $this->type;
		}

	/**
	* This function is called when the user clicks on any button on the account pages.
	* It prints the HTML code of each account page.
	*/
	function continue_main() {
		$result = array();
		if ($this->subpage=='') $this->subpage='attributes';
		if (isset($_POST['form_main_reset'])) {
			$result = $this->load_account($this->dn_orig);
		}
		else {
			if ($this->current_page==0) {
				if ($this->subpage=='attributes') {
					// change dn
					if (isset($_POST['suffix']) && ($_POST['suffix'] != '')) $this->dn = $_POST['suffix'];
					// change RDN
					if (isset($_POST['rdn'])) $this->rdn = $_POST['rdn'];

					// load profile
					if (isset($_POST['selectLoadProfile']) && isset($_POST['loadProfile'])) {
						$profile = loadAccountProfile($_POST['selectLoadProfile'], $this->type);
						// pass profile to each module
						$modules = array_keys($this->module);
						foreach ($modules as $module) $this->module[$module]->load_profile($profile);
						if (isset($profile['ldap_rdn'][0])) {
							if (in_array($profile['ldap_rdn'][0], getRDNAttributes($this->type))) {
								$this->rdn = $profile['ldap_rdn'][0];
							}
						}
						if (isset($profile['ldap_suffix'][0])) {
							$this->dn = $profile['ldap_suffix'][0];
						}
					}
					// save account
					if (isset($_POST['create'])) {
						$errors = $this->save_account();
						if (sizeof($errors) > 0) {
							$result = $errors;
							$stopProcessing = true;
						}
						else $this->subpage = 'finish';
					}
				}
				if ($this->subpage=='finish') {
					if (isset($_POST['createagain'])) {
						// open fresh account page
						unset($_SESSION[$this->base]);
						metaRefresh("edit.php?type=" . $this->type);
						exit();
					}
					if (isset($_POST['backmain'])) {
						// Return to account list
						unset($_SESSION[$this->base]);
						metaRefresh("../lists/list.php?type=" . $this->type);
						exit;
					}
					if (isset($_POST['outputpdf'])) {
						// Create / display PDf-file
						createModulePDF(array($_SESSION[$this->base]), $_POST['pdfStructure']);
						exit;
					}
				}
			}
			else $result = call_user_func(array(&$this->module[$this->order[$this->current_page]], 'process_'.$this->subpage));
		}
		// change to next page
		$errorsOccured = false;
		if (is_array($result)) {  // messages were returned, check for errors
			for ($i = 0; $i < sizeof($result); $i++) {
				if ($result[$i][0] == 'ERROR') {
					$errorsOccured = true;
					break;
				}
			}
		}
		if (!$errorsOccured) {
			// go to subpage of current module
			$postKeys = array_keys($_POST);
			for ($p = 0; $p < sizeof($postKeys); $p++) {
				if (is_string($postKeys[$p]) && (strpos($postKeys[$p], 'form_subpage_' . $this->order[$this->current_page]) === 0)) {
					$temp = substr($postKeys[$p], strlen($this->order[$this->current_page]) + 14);
					$temp = explode('_', $temp);
					if (sizeof($temp) == 2) {
						$this->subpage = $temp[0];
					}
				}
			}
			// change module page if requested
			if (isset($_POST['form_main_main'])) {
				$this->current_page = 0;
				$this->subpage='attributes';
			}
			else {
				for ($i=1; $i<count($this->order); $i++ ) {
					if (isset($_POST['form_main_'.$this->order[$i]])) {
						if ($this->module[$this->order[$i]]->module_ready()) {
							$this->current_page = $i;
							$this->subpage='attributes';
						}
						else {
							StatusMessage('ERROR', _('The module %s is not yet ready.'),
								_('Please enter the account information on the other pages first.'),
								array($this->module[$this->order[$i]]->get_alias()));
						}
					}
				}
			}
		}
		// Write HTML-Code
		echo $_SESSION['header'];
		echo "<title>LDAP Account Manager</title>\n";
		echo "<link rel=\"stylesheet\" type=\"text/css\" href=\"../../style/layout.css\">\n";
		echo "<link rel=\"stylesheet\" type=\"text/css\" href=\"../../style/type_" . $this->type . ".css\">\n";
		echo "</head><body>\n";
		echo "<form enctype=\"multipart/form-data\" action=\"edit.php\" method=\"post\">\n";
		// Display error-messages
		if (is_array($result)) {
			for ($i=0; $i<sizeof($result); $i++) {
				call_user_func_array("StatusMessage", $result[$i]);
			}
			if (isset($stopProcessing) && ($stopProcessing == true)) {
				echo "</body>\n";
				echo "</html>\n";
				return 0;
			}
		}
		if ($this->current_page==0) {
			if ($this->subpage=='attributes')  {
				$modules = array_keys($this->module);
				$disabled = false;
				$incompleteModules = array();
				foreach ($modules as $module) {
					if (!$this->module[$module]->module_complete()) {
						$disabled = true;
						$incompleteModules[] = $this->module[$module]->get_alias();
					}
				}
				if (sizeof($incompleteModules) > 0) {
					StatusMessage('INFO', _('Some required information is missing'),
						sprintf(_('Please set up all required attributes on page: %s'), implode(", ", $incompleteModules)));
				}
			}
		}
		// Create left module-menu
		echo "<table border=0 width=\"100%\">\n<tr valign=\"top\"><td width=\"15%\">";

		echo "<fieldset class=\"".$this->type."edit\"><legend><b>";
			echo _('Please select page:');
			echo "</b></legend><center>\n";
		// calculate button width
		$buttonWidth = round(0.8 * strlen(utf8_decode(_("Reset changes"))));
		for ($b = 0; $b < sizeof($this->order); $b++) {
			$tempWidth = round(0.8 * strlen(utf8_decode($this->order[$b])));
			if ($buttonWidth < $tempWidth) $buttonWidth = $tempWidth;
		}
		$x=0;
		// print "main" button
		echo "<input style=\"width:" . $buttonWidth ."em;margin:2px;\" name=\"form_main_main\" type=\"submit\" value=\"";
		echo _('Main');
		echo "\" tabindex=$x";
		if ($this->subpage == 'finish') echo " disabled";
		echo ">\n<br>";
		$x++;
		// Loop for module
		// $x is used to count up tabindex
		for ($i=1; $i<count($this->order); $i++ ) {
			$buttonStatus = $this->module[$this->order[$i]]->getButtonStatus();
			// skip hidden buttons
			if ($buttonStatus == 'hidden') continue;
			// print normal button
			echo "<input style=\"width:" . $buttonWidth ."em;margin:2px;\" name=\"form_main_".$this->order[$i]."\" type=\"submit\" value=\"";
			echo $this->module[$this->order[$i]]->get_alias();
			echo "\" tabindex=$x";
			if (($buttonStatus == 'disabled') || $this->subpage == 'finish') echo " disabled";
			echo ">\n<br>";
			$x++;
		}
		if ($this->dn_orig!='') {
			echo "<br><input style=\"width:" . $buttonWidth ."em;margin:2px;\" name=\"form_main_reset\" type=\"submit\" value=\"" . _('Reset changes') . "\"";
			if ($this->subpage == 'finish') echo " disabled";
			echo "><br>\n";
		}
		echo "</center></fieldset>\n";
		echo "</td>\n<td>";
		if ($this->current_page==0) {
			echo "<fieldset class=\"".$this->type."edit\"><legend><b>";
			echo _('Main');
			echo "</b></legend>\n";
			}
		else {
			echo "<fieldset class=\"".$this->type."edit\"><legend><b>";
				echo $this->module[$this->order[$this->current_page]]->get_alias();
				echo "</b></legend>\n";
			}
		// display html-code from mdule
		$return = array();
		if ($this->current_page == 0) {
			if ($this->subpage == 'finish') {
				// Show success message
				if ($this->dn_orig == '') {
					$text = _("Account was created successfully.");
				}
				else {
					$text = _("Account was modified successfully.");
				}
				$return[] = array(
					0 => array('kind' => 'message', 'type' => 'INFO', 'headline' =>  _('LDAP operation successful.'), 'text' => $text,
						'td' => array('colspan' => 3)));
				$return[] = array(
					0 => array('kind' => 'text', 'text' => '', 'td' => array('colspan' => 3)));
				$return[] = array(
					0 => array('kind' => 'input', 'type' => 'submit', 'name' => 'createagain', 'value' => _('Create another account')),
					1 => array('kind' => 'input', 'type' => 'submit', 'name' => 'backmain', 'value' => _('Back to account list')));
				$return[] = array(0 => array('kind' => 'text', 'text' => '&nbsp;')); // empty line
				$return[] = array(0 => array('kind' => 'text', 'text' => '&nbsp;')); // empty line
				$pdfStructures = getPDFStructureDefinitions($this->type);
				$return[] = array(
					0 => array('kind' => 'fieldset', 'legend' => '<b>PDF</b>', 'td' => array('colspan' => 2), 'value' => array(array(
						0 => array('kind' => 'select', 'name' => 'pdfStructure', 'options' => $pdfStructures),
						1 => array('kind' => 'input', 'type' => 'submit', 'name' => 'outputpdf', 'value' => _('Create PDF file')),
						2 => array('kind' => 'help', 'value' => '403'))
					)));
			}
			else {
				// loop through all suffixes
				$rootsuffix = $_SESSION['config']->get_Suffix($this->type);
				foreach ($_SESSION['ldap']->search_units($rootsuffix) as $suffix) {
					if ($this->dn == $suffix) $options_selected = $suffix;
					$suffixes[] = $suffix;
				}
				$return[] = array(
					0 => array('kind' => 'text', 'text' => _('Suffix')),
					1 => array('kind' => 'select', 'name' => 'suffix', 'options' => $suffixes, 'options_selected' => array($options_selected), 'noSorting' => true),
					2 => array ('kind' => 'help', 'value' => '361'));
				// RDN selection
				$rdnlist = getRDNAttributes($this->type);
				$rdnSelected = array();
				if ($this->rdn != '') {
					$rdnSelected[] = $this->rdn;
				}
				else {
					$rdnSelected[] = $rdnlist[0];
				}
				$return[] = array(
					0 => array('kind' => 'text', 'text' => _('RDN identifier')),
					1 => array('kind' => 'select', 'name' => 'rdn', 'options' => $rdnlist, 'options_selected' => $rdnSelected, 'noSorting' => true),
					2 => array ('kind' => 'help', 'value' => '301'));

				// Get list of profiles
				$profilelist = getAccountProfiles($this->type);
				if (count($profilelist)!=0) {
					$return[] = array(
						0 => array('kind' => 'text', 'text' => _("Load profile")),
						1 => array('kind' => 'table', 'value' => array(0 => array(
							0 => array('kind' => 'select', 'name' => 'selectLoadProfile', 'options' => $profilelist),
							1 => array('kind' => 'input', 'type' => 'submit', 'name' => 'loadProfile', 'value' => _('Load profile'))
							))),
						2 => array('kind' => 'help', 'value' => '401')
					);
				}
				$return[] = array(0 => array('kind' => 'text', 'text' => '&nbsp;')); // empty line
				if ($this->dn_orig!='') $text = _('Modify account');
					else $text = _('Create account');
				$return[] = array ( 0 => array ( 'kind' => 'text', 'text' => $text ),
					1 => array ( 'kind' => 'input', 'type' => 'submit', 'name' => 'create', 'value' => $text, 'disabled' => $disabled),
					2 => array ('kind' => 'help', 'value' => '402'));
			}
		}
		else $return = call_user_func(array($this->module[$this->order[$this->current_page]], 'display_html_'.$this->subpage));
		$y = 5000;
		$z = 10000;
		parseHtml($this->order[$this->current_page], $return, array(), false, $y, $z, $this->type);
		// Display rest of html-page
		echo "</fieldset>\n";
		if ($this->current_page != "main") echo "<br>*" . _('required');
		echo "</td></tr></table>\n";
		echo "</form>\n";
		echo "</body>\n";
		echo "</html>\n";
		return 0;
	}

	/**
	* This function checks which LDAP attributes have changed while the account was edited.
	*
	* @param array $attributes list of current LDAP attributes
	* @param array $orig list of old attributes when account was loaded
	* @return array an array which can be passed to $this->saveAccount()
	*/
	function save_module_attributes($attributes, $orig) {
		$toadd = array();
		$tomodify = array();
		$torem = array();
		$notchanged = array();
		// get list of all attributes
		$attr_names = array_keys($attributes);
		$orig_names = array_keys($orig);
		// find deleted attributes (in $orig but no longer in $attributes)
		for ($i = 0; $i < sizeof($orig_names); $i++) {
			if (!isset($attributes[$orig_names[$i]])) {
				$torem[$orig_names[$i]] = $orig[$orig_names[$i]];
			}
		}
		// find changed attributes
		for ($i=0; $i<count($attr_names); $i++) {
			// find deleted attributes
			for ($j=0; $j<count($orig[$attr_names[$i]]); $j++) {
				if (is_array($attributes[$attr_names[$i]])) {
					if (!in_array($orig[$attr_names[$i]][$j], $attributes[$attr_names[$i]])) {
						if (($orig[$attr_names[$i]][$j]!='') && ($attr_names[$i]!='objectClass')) $torem[$attr_names[$i]][] = $orig[$attr_names[$i]][$j];
					}
				}
				else if (($orig[$attr_names[$i]][$j]!='') && ($attr_names[$i]!='objectClass')) $torem[$attr_names[$i]][] = $orig[$attr_names[$i]][$j];
			}
			// find new attributes
			for ($j=0; $j<count($attributes[$attr_names[$i]]); $j++) {
				if (is_array($orig[$attr_names[$i]])) {
					if (!in_array($attributes[$attr_names[$i]][$j], $orig[$attr_names[$i]]))
						if ($attributes[$attr_names[$i]][$j]!='') {
							$toadd[$attr_names[$i]][] = $attributes[$attr_names[$i]][$j];
						}
				}
				else if ($attributes[$attr_names[$i]][$j]!='') $toadd[$attr_names[$i]][] = $attributes[$attr_names[$i]][$j];
			}
			// find unchanged attributes
			if (is_array($orig[$attr_names[$i]]) && is_array($attributes[$attr_names[$i]])) {
				for ($j=0; $j<count($attributes[$attr_names[$i]]); $j++) {
					if (($attributes[$attr_names[$i]][$j] != '') && in_array($attributes[$attr_names[$i]][$j], $orig[$attr_names[$i]])) {
						$notchanged[$attr_names[$i]][] = $attributes[$attr_names[$i]][$j];
					}
				}
			}
		}
		// create modify with add and remove
		$attributes2 = array_keys($toadd);
		for ($i=0; $i<count($attributes2); $i++) {
			if (isset($torem[$attributes2[$i]]))
				if ((count($toadd[$attributes2[$i]]) > 0) && (count($torem[$attributes2[$i]]) > 0)) {
					// found attribute which should be modified
					$tomodify[$attributes2[$i]] = $toadd[$attributes2[$i]];
					// merge unchanged values
					if (isset($notchanged[$attributes2[$i]])) {
						$tomodify[$attributes2[$i]] = array_merge($tomodify[$attributes2[$i]], $notchanged[$attributes2[$i]]);
						unset($notchanged[$attributes2[$i]]);
					}
					// remove old add and remove commands
					unset($toadd[$attributes2[$i]]);
					unset($torem[$attributes2[$i]]);
				}
			}
		if (count($toadd)!=0) $return[$this->dn]['add'] = $toadd;
		if (count($torem)!=0) $return[$this->dn]['remove'] = $torem;
		if (count($tomodify)!=0) $return[$this->dn]['modify'] = $tomodify;
		if (count($notchanged)!=0) $return[$this->dn]['notchanged'] = $notchanged;
		return $return;
		}

	/**
	* Loads an LDAP account with the given DN.
	*
	* @param string $dn the DN of the account
	* @return array error messages
	*/
	function load_account($dn) {
		$this->module = array();
		$modules = $_SESSION['config']->get_AccountModules($this->type);
		$search = substr($dn, 0, strpos($dn, ','));
		$result = @ldap_search($_SESSION['ldap']->server(), $dn, $search);
		if (!$result) {
			return array(array("ERROR", _("Unable to load LDAP entry:") . " " . $dn, ldap_error($_SESSION['ldap']->server())));
		}
		$entry = @ldap_first_entry($_SESSION['ldap']->server(), $result);
		if (!$entry) {
			return array(array("ERROR", _("Unable to load LDAP entry:") . " " . $dn, ldap_error($_SESSION['ldap']->server())));
		}
		$this->dn = substr($dn, strpos($dn, ',')+1);
		$this->dn_orig = $dn;
		// extract RDN
		$this->rdn = split("=", substr($dn, 0, strpos($dn, ',')));
		$this->rdn = $this->rdn[0];
		$attr = ldap_get_attributes($_SESSION['ldap']->server(), $entry);
		// remove 'count' entries and numerical entries
		for ($i = 0; $i < count($attr); $i++) {
			if (isset($attr[$i])) unset($attr[$i]);
		}
		$attrNames = array_keys($attr);
		for ($i = 0; $i < sizeof($attrNames); $i++) unset($attr[$attrNames[$i]]['count']);
		unset($attr['count']);
		// fix spelling errors
		$attr = $this->fixLDAPAttributes($attr, $modules);
		// get binary attributes
		$binaryAttr = array('jpegPhoto');
		for ($i = 0; $i < sizeof($binaryAttr); $i++) {
			if (isset($attr[$binaryAttr[$i]][0])) {
				$binData = ldap_get_values_len($_SESSION['ldap']->server(), $entry, $binaryAttr[$i]);
				unset($binData['count']);
				$attr[$binaryAttr[$i]] = $binData;
			}
		}
		// save original attributes
		$this->attributes_orig = $attr;

		foreach ($modules as $module) {
			if (!isset($this->module[$module])) {
				$this->module[$module] = new $module($this->type);
				$this->module[$module]->init($this->base);
			}
			$this->module[$module]->load_attributes($attr);
			}

		// sort module buttons
		$this->sortModules();
		return array();
	}

	/**
	 * Fixes spelling errors in the attribute names.
	 *
	 * @param array $attributes LDAP attributes
	 * @param array $modules list of active modules
	 * @return array fixed attributes
	 */
	function fixLDAPAttributes($attributes, $modules) {
		if (!is_array($attributes)) return $attributes;
		$keys = array_keys($attributes);
		// get correct object class names, aliases and attributes
		$objectClasses = array();
		$aliases = array();
		$ldapAttributesTemp = array();
		foreach ($modules as $module) {
			$moduleObj = new $module($this->type);
			$objectClasses = array_merge($objectClasses, $moduleObj->getManagedObjectClasses());
			$aliases = array_merge($aliases, $moduleObj->getLDAPAliases());
			$ldapAttributesTemp = array_merge($ldapAttributesTemp, $moduleObj->getManagedAttributes());
		}
		// build lower case attribute names
		$ldapAttributes = array();
		for ($i = 0; $i < sizeof($ldapAttributesTemp); $i++) {
			$ldapAttributes[strtolower($ldapAttributesTemp[$i])] = $ldapAttributesTemp[$i];
			unset($ldapAttributes[$i]);
		}
		$ldapAttributesKeys = array_keys($ldapAttributes);
		// convert alias names to lower case (for easier comparison)
		$aliasKeys = array_keys($aliases);
		for ($i = 0; $i < sizeof($aliasKeys); $i++) {
			if ($aliasKeys[$i] != strtolower($aliasKeys[$i])) {
				$aliases[strtolower($aliasKeys[$i])] = $aliases[$aliasKeys[$i]];
				unset($aliases[$aliasKeys[$i]]);
				$aliasKeys[$i] = strtolower($aliasKeys[$i]);
			}
		}
		// fix object classes and attributes
		for ($i = 0; $i < sizeof($keys); $i++) {
			// check object classes
			if (strtolower($keys[$i]) == 'objectclass') {
				// fix object class attribute
				if ($keys[$i] != 'objectClass') {
					$temp = $attributes[$keys[$i]];
					unset($attributes[$keys[$i]]);
					$attributes['objectClass'] = $temp;
				}
				// fix object classes
				for ($attrClass = 0; $attrClass < sizeof($attributes['objectClass']); $attrClass++) {
					for ($modClass = 0; $modClass < sizeof($objectClasses); $modClass++) {
						if (strtolower($attributes['objectClass'][$attrClass]) == strtolower($objectClasses[$modClass])) {
							if ($attributes['objectClass'][$attrClass] != $objectClasses[$modClass]) {
								unset($attributes['objectClass'][$attrClass]);
								$attributes['objectClass'][] = $objectClasses[$modClass];
							}
							break;
						}
					}
				}
			}
			else {
				// fix aliases
				if (in_array(strtolower($keys[$i]), $aliasKeys)) {
					$attributes[$aliases[strtolower($keys[$i])]] = $attributes[$keys[$i]];
					unset($attributes[$keys[$i]]);
				}
				// fix attribute names
				elseif (in_array(strtolower($keys[$i]), $ldapAttributesKeys)) {
					if ($keys[$i] != $ldapAttributes[strtolower($keys[$i])]) {
						$attributes[$ldapAttributes[strtolower($keys[$i])]] = $attributes[$keys[$i]];
						unset($attributes[$keys[$i]]);
					}
				}
			}
		}
		return $attributes;
	}

	/**
	* This function will prepare the object for a new account.
	*/
	function new_account() {
		$this->isNewAccount = true;
		$modules = $_SESSION['config']->get_AccountModules($this->type);
		foreach ($modules as $module) {
			$this->module[$module] = new $module($this->type);
			$this->module[$module]->init($this->base);
		}
		// sort module buttons
		$this->sortModules();
		$profile = loadAccountProfile('default', $this->type);
		// pass profile to each module
		$modules = array_keys($this->module);
		foreach ($modules as $module) $this->module[$module]->load_profile($profile);
		if (isset($profile['ldap_rdn'][0])) {
			if (in_array($profile['ldap_rdn'][0], getRDNAttributes($this->type))) {
				$this->rdn = $profile['ldap_rdn'][0];
			}
		}
		if (isset($profile['ldap_suffix'][0])) {
			$this->dn = $profile['ldap_suffix'][0];
		}
		return 0;
		}

	/**
	* This function will save an account to the LDAP database.
	*
	* @return array list of status messages if any errors occured
	*/
	function save_account() {
		$errors = array();
		$ldapUser = $_SESSION['ldap']->decrypt_login();
		$ldapUser = $ldapUser[0];
		$module = array_keys ($this->module);
		$attributes = array();
		// load attributes
		foreach ($module as $singlemodule) {
			// load changes
			$temp = $this->module[$singlemodule]->save_attributes();
			if (!is_array($temp)) $temp = array();
			// merge changes
			$DNs = array_keys($temp);
			// *** fixme don't include references
			if (is_array($temp)) $attributes = array_merge_recursive($temp, $attributes);
			for ($i=0; $i<count($DNs); $i++) {
				$ops = array_keys($temp[$DNs[$i]]);
				for ($j=0; $j<count($ops); $j++) {
					$attrs = array_keys($temp[$DNs[$i]][$ops[$j]]);
					for ($k=0; $k<count($attrs); $k++)
						$attributes[$DNs[$i]][$ops[$j]][$attrs[$k]] = array_unique($attributes[$DNs[$i]][$ops[$j]][$attrs[$k]]);
					}
				}
			}
		// Complete dn with RDN attribute
		$search = $this->rdn;
		$added = false;
		foreach ($attributes as $DN) {
			if (isset($DN['modify'][$search][0]) && !$added) {
				$attributes[$search.'='.$DN['modify'][$search][0].','.$this->dn] = $attributes[$this->dn];
				unset ($attributes[$this->dn]);
				$this->dn = $search.'='.$DN['modify'][$search][0].','.$this->dn;
				$added = true;
				}
			if (isset($DN['add'][$search][0]) && !$added) {
				$attributes[$search.'='.$DN['add'][$search][0].','.$this->dn] = $attributes[$this->dn];
				unset ($attributes[$this->dn]);
				$this->dn = $search.'='.$DN['add'][$search][0].','.$this->dn;
				$added = true;
				}
			if (isset($DN['notchanged'][$search][0]) && !$added) {
				$attributes[$search.'='.$DN['notchanged'][$search][0].','.$this->dn] = $attributes[$this->dn];
				unset ($attributes[$this->dn]);
				$this->dn = $search.'='.$DN['notchanged'][$search][0].','.$this->dn;
				$added = true;
				}
			}
		// Add old dn if dn hasn't changed
		if (!$added) {
			$attributes[$this->dn_orig] = $attributes[$this->dn];
			unset ($attributes[$this->dn]);
			$this->dn = $this->dn_orig;
			}
		// Set to true if an real error has happened
		$stopprocessing = false;
		if (strtolower($this->dn) != strtolower($this->dn_orig)) {
			// move existing DN
			if ($this->dn_orig!='') {
				// merge attributes together
				$attr = array();
				if (is_array($attributes[$this->dn]['add'])) $attr = array_merge_recursive($attr, $attributes[$this->dn]['add']);
				if (is_array($attributes[$this->dn]['modify'])) $attr = array_merge_recursive($attr, $attributes[$this->dn]['modify']);
				// add unchanged attributes if not already set
				if (is_array($attributes[$this->dn]['notchanged'])) {
					$notChangedKeys = array_keys($attributes[$this->dn]['notchanged']);
					for ($i = 0; $i < sizeof($notChangedKeys); $i++) {
						if (!isset($attr[$notChangedKeys[$i]])) {
							$attr[$notChangedKeys[$i]] = $attributes[$this->dn]['notchanged'][$notChangedKeys[$i]];
						}
					}
				}
				// add attributes which are not controlled by modules from original account
				$attrNames = array_keys($this->attributes_orig);
				for ($i = 0; $i < sizeof($attrNames); $i++) {
					if (!isset($attr[$attrNames[$i]])) $attr[$attrNames[$i]] = $this->attributes_orig[$attrNames[$i]];
				}
				// add missing object classes
				for ($i = 0; $i < sizeof($this->attributes_orig['objectClass']); $i++) {
					if (!in_array($this->attributes_orig['objectClass'][$i], $attr['objectClass'])) {
						$attr['objectClass'][] = $this->attributes_orig['objectClass'][$i];
					}
				}
				$success = ldap_add($_SESSION['ldap']->server(), $this->dn, $attr);
				if ($success) {
					logNewMessage(LOG_NOTICE, '[' . $ldapUser .'] Created DN: ' . $this->dn);
					$success = ldap_delete($_SESSION['ldap']->server(), $this->dn_orig);
					if (!$success) {
						logNewMessage(LOG_ERR, '[' . $ldapUser .'] Unable to delete DN: ' . $this->dn_orig . ' (' . ldap_err2str(ldap_errno($_SESSION['ldap']->server())) . ').');
						$errors[] = array('ERROR', sprintf(_('Was unable to delete DN: %s.'), $this->dn_orig), ldap_error($_SESSION['ldap']->server()));
						$stopprocessing = true;
					}
					else {
						logNewMessage(LOG_NOTICE, '[' . $ldapUser .'] Deleted DN: ' . $this->dn_orig);
					}
				}
				if (!$success) {
					logNewMessage(LOG_ERR, '[' . $ldapUser .'] Unable to create DN: ' . $this->dn . ' (' . ldap_err2str(ldap_errno($_SESSION['ldap']->server())) . ').');
					$errors[] = array('ERROR', sprintf(_('Was unable to create DN: %s.'), $this->dn), ldap_error($_SESSION['ldap']->server()));
					$stopprocessing = true;
				}
			}
			// create complete new dn
			else {
				$attr = array();
				if (is_array($attributes[$this->dn]['add'])) $attr = array_merge_recursive($attr, $attributes[$this->dn]['add']);
				if (is_array($attributes[$this->dn]['notchanged'])) $attr = array_merge_recursive($attr, $attributes[$this->dn]['notchanged']);
				if (is_array($attributes[$this->dn]['modify'])) $attr = array_merge_recursive($attr, $attributes[$this->dn]['modify']);
				$success = ldap_add($_SESSION['ldap']->server(), $this->dn, $attr);
				if (!$success) {
					logNewMessage(LOG_ERR, '[' . $ldapUser .'] Unable to create DN: ' . $this->dn . ' (' . ldap_err2str(ldap_errno($_SESSION['ldap']->server())) . ').');
					$errors[] = array('ERROR', sprintf(_('Was unable to create DN: %s.'), $this->dn), ldap_error($_SESSION['ldap']->server()));
					$stopprocessing = true;
				}
				else {
					logNewMessage(LOG_NOTICE, '[' . $ldapUser .'] Created DN: ' . $this->dn);
				}
				// lamdaemon commands for the new account
				if (!$stopprocessing) {
					$DN = $attributes[$this->dn];
					if (is_array($DN['lamdaemon']['command'])) {
						$result = lamdaemon($DN['lamdaemon']['command']);
						// Error somewhere in lamdaemon
						if (is_array($result)) {
							foreach ($result as $singleresult) {
								$singleresult = explode(",", $singleresult);
								if (is_array($singleresult)) {
									if ($singleresult[0] == 'ERROR') {
										$stopprocessing = true;
										$temparray[0] = $singleresult[0];
										$temparray[1] = _($singleresult[1]);
										$temparray[2] = _($singleresult[2]);
										$errors[] = $temparray;
									}
								}
							}
						}
					}
				}
			}
			unset($attributes[$this->dn]);
		}
		$DNs = array_keys($attributes);
		for ($i=0; $i<count($DNs); $i++) {
			if (!$stopprocessing) {
				// modify attributes
				if (isset($attributes[$DNs[$i]]['modify']) && !$stopprocessing) {
					$success = @ldap_mod_replace($_SESSION['ldap']->server(), $DNs[$i], $attributes[$DNs[$i]]['modify']);
					if (!$success) {
						logNewMessage(LOG_ERR, '[' . $ldapUser .'] Unable to modify attribtues from DN: ' . $DNs[$i] . ' (' . ldap_err2str(ldap_errno($_SESSION['ldap']->server())) . ').');
						$errors[] = array('ERROR', sprintf(_('Was unable to modify attribtues from DN: %s.'), $DNs[$i]), ldap_error($_SESSION['ldap']->server()));
						$stopprocessing = true;
					}
					else {
						logNewMessage(LOG_NOTICE, '[' . $ldapUser .'] Modified DN: ' . $DNs[$i]);
					}
				}
				// add attributes
				if (isset($attributes[$DNs[$i]]['add']) && !$stopprocessing) {
					$success = @ldap_mod_add($_SESSION['ldap']->server(), $DNs[$i], $attributes[$DNs[$i]]['add']);
					if (!$success) {
						logNewMessage(LOG_ERR, '[' . $ldapUser .'] Unable to add attribtues to DN: ' . $DNs[$i] . ' (' . ldap_err2str(ldap_errno($_SESSION['ldap']->server())) . ').');
						$errors[] = array('ERROR', sprintf(_('Was unable to add attribtues to DN: %s.'), $DNs[$i]), ldap_error($_SESSION['ldap']->server()));
						$stopprocessing = true;
					}
					else {
						logNewMessage(LOG_NOTICE, '[' . $ldapUser .'] Modified DN: ' . $DNs[$i]);
					}
				}
				// remove attributes
				if (isset($attributes[$DNs[$i]]['remove']) && !$stopprocessing) {
					$success = @ldap_mod_del($_SESSION['ldap']->server(), $DNs[$i], $attributes[$DNs[$i]]['remove']);
					if (!$success) {
						logNewMessage(LOG_ERR, '[' . $ldapUser .'] Unable to delete attribtues from DN: ' . $DNs[$i] . ' (' . ldap_err2str(ldap_errno($_SESSION['ldap']->server())) . ').');
						$errors[] = array('ERROR', sprintf(_('Was unable to remove attribtues from DN: %s.'), $DNs[$i]), ldap_error($_SESSION['ldap']->server()));
						$stopprocessing = true;
					}
					else {
						logNewMessage(LOG_NOTICE, '[' . $ldapUser .'] Modified DN: ' . $DNs[$i]);
					}
				}
			}
		}

		if (!$stopprocessing) {
			foreach ($attributes as $DN) {
				if (isset($DN['lamdaemon']['command']) && is_array($DN['lamdaemon']['command'])) {
					$result = lamdaemon($DN['lamdaemon']['command']);
					// Error somewhere in lamdaemon
					if (isset($result) && is_array($result)) {
						foreach ($result as $singleresult) {
							$singleresult = explode(",", $singleresult);
							if (is_array($singleresult)) {
								if ($singleresult[0] == 'ERROR') {
									$stopprocessing = true;
									$temparray[0] = $singleresult[0];
									$temparray[1] = _($singleresult[1]);
									$temparray[2] = _($singleresult[2]);
									$errors[] = $temparray;
								}
							}
						}
					}
				}
			}
		}
		$_SESSION['cache']->refresh_cache(true);
		return $errors;
	}

	/**
	 * Returns a list of possible PDF entries for this account.
	 *
	 * @return list of PDF entries (array(<name> => <PDF lines>))
	 */
	function get_pdfEntries() {
		$return = array();
		while(($current = current($this->module)) != null) {
			$return = array_merge($return,$current->get_pdfEntries());
			next($this->module);
		}
		$return = array_merge($return,array('main_dn' => array('<block><key>' . _('DN') . '</key><value>' . $this->dn . '</value></block>')));
		return $return;
	}

	/**
	* Sorts the module buttons for the account page.
	*/
	function sortModules() {
		$order = array();
		$modules = array_keys($this->module);
		$depModules = array();
		for ($i = 0; $i < sizeof($modules); $i++) {
			// insert waiting modules
			for ($w = 0; $w < sizeof($depModules); $w++) {
				$dependencies = $this->module[$depModules[$w]]->get_dependencies($this->type);
				$dependencies = $dependencies['depends'];
				$everything_found = true;
				for ($d = 0; $d < sizeof($dependencies); $d++) {
					if (!in_array($dependencies[$d], $order)) {
						$everything_found = false;
						break;
					}
				}
				// inser after depending module
				if ($everything_found) {
					$order[] = $depModules[$w];
					unset($depModules[$w]);
					$depModules = array_values($depModules);
					$w--;
				}
			}
			// check next module
			$dependencies = $this->module[$modules[$i]]->get_dependencies($this->type);
			if (is_array($dependencies['depends'])) {
				$everything_found = true;
				$dependencies = $dependencies['depends'];
				for ($d = 0; $d < sizeof($dependencies); $d++) {
					if (!in_array($dependencies[$d], $order)) {
						$everything_found = false;
						break;
					}
				}
				// remove module if dependencies are not satisfied
				if (!$everything_found) {
					$depModules[] = $modules[$i];
					unset($modules[$i]);
					$modules = array_values($modules);
					$i--;
				}
				else {
					$order[] = $modules[$i];
				}
			}
			else {
				$order[] = $modules[$i];
			}
		}
		// add modules which could not be sorted (e.g. because of cyclic dependencies)
		if (sizeof($depModules) > 0) {
			for ($i = 0; $i < sizeof($depModules); $i++) $order[] = $depModules[$i];
		}
		array_unshift($order, 'main');
		$this->order = $order;
	}

	/**
	* Encrypts sensitive data before storing in session.
	*
	* @return array list of attributes which are serialized
	*/
	function __sleep() {
		// encrypt data
		$this->attributes = $_SESSION['ldap']->encrypt(serialize($this->attributes));
		$this->attributes_orig = $_SESSION['ldap']->encrypt(serialize($this->attributes_orig));
		$this->module = $_SESSION['ldap']->encrypt(serialize($this->module));
		// save all attributes
		return array_keys(get_object_vars($this));
	}

	/**
	* Decrypts sensitive data after accountContainer was loaded from session.
	*/
	function __wakeup() {
		// decrypt data
		$this->attributes = unserialize($_SESSION['ldap']->decrypt($this->attributes));
		$this->attributes_orig = unserialize($_SESSION['ldap']->decrypt($this->attributes_orig));
		$this->module = unserialize($_SESSION['ldap']->decrypt($this->module));
	}

}
?>
