/*
 *  Copyright 1994-2012 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "constants.h"
#include "brandom.h"
#include "pictfader.h"
#include "pictures.h"
#include "globals.h"


PictFader_t *
PictFader_new(const u_short size)
{
  PictFader_t *pf = xcalloc(1, sizeof(PictFader_t));

  pf->on = 0;
  pf->cur = Picture8_new();
  if (pictures != NULL)
    pf->dst = pictures->pics[0];
  pf->fader = Fader_new(BUFFSIZE);
  pf->shf = Shuffler_new(size);
  Shuffler_set_mode(pf->shf, BS_RANDOM);
#ifdef XDEBUG
  Shuffler_verbose(pf->shf);
#endif

  PictFader_set(pf);

  return pf;
}


void
PictFader_delete(PictFader_t *pf)
{
  Picture8_delete(pf->cur);
  Fader_delete(pf->fader);
  Shuffler_delete(pf->shf);
  xfree(pf);
}


void
PictFader_init(PictFader_t *pf)
{
  Fader_t *fader = pf->fader;
  u_long i;
  const Buffer8_t *src = pf->cur->buff;
  const Buffer8_t *dst = pf->dst->buff;
  
  Fader_init(fader);
  for (i = BUFFSIZE; i--; ) { /* i < BUFFSIZE; i++) { */
    /* delta values for fading */
    fader->delta[i] = (long)(
			      ((float)dst->buffer[i] - (float)src->buffer[i])
			      / (float)(fader->max)
			      * MFACTOR);
    
    /* initial values for fading */
    /*fader->tmp[i] = (float)src->buffer[i];*/
    fader->tmp[i] = (u_long)src->buffer[i]*MFACTOR;
  }
  Fader_start(fader);
}


void
PictFader_run(PictFader_t *pf)
{
  Fader_t *fader = pf->fader;
  const u_long elapsed = Fader_elapsed(fader);

  Fader_start(fader);
  fader->faded += elapsed;

  if (fader->faded < fader->max) {
    /* now do some fading stuff #~{@ */
    /* we spent (elapsed) msecs */
    Pixel_t *ptr = pf->cur->buff->buffer;
/*     float   *tmp = fader->tmp; */
/*     float   *delta = fader->delta; */
    u_long   *tmp = fader->tmp;
    long   *delta = fader->delta;
    u_long i;

    for (i = BUFFSIZE; i--; ptr++, tmp++, delta++)
      /**ptr = (Pixel_t)(*tmp += (elapsed * *delta));*/
    *ptr = (Pixel_t)((*tmp += (elapsed * *delta))/MFACTOR);
  } else {
    /* we're done */
    fader->fading = 0;
    /* copy, just in case */
    Picture8_copy(pf->dst, pf->cur);
  }
}


void
PictFader_set(PictFader_t *pf)
{
  Fader_t *fader = pf->fader;

  pf->dst = pictures->pics[fader->target];

  if (pf->dst->name != NULL) {
    if (libbiniou_verbose)
      printf("[i] Using picture '%s'\n", pf->dst->name);
  } else
    xerror("Picture without name, WTF #@!\n");

  if (pf->cur != NULL && pf->dst->id != pf->cur->id) {
    if (fader->fade)
      PictFader_init(pf);
    else
      pf->cur = pf->dst;

    fader->fading = fader->fade;
  }
}


void
PictFader_prev(PictFader_t *pf)
{
  DEC(pf->fader->target, pf->shf->size);
  PictFader_set(pf);
}


void
PictFader_next(PictFader_t *pf)
{
  INC(pf->fader->target, pf->shf->size);
  PictFader_set(pf);
}


void
PictFader_random(PictFader_t *pf)
{
  pf->fader->target = Shuffler_get(pf->shf);
  PictFader_set(pf);
}


int
PictFader_ring(const PictFader_t *pf)
{
  const Fader_t *fader = pf->fader;

  return (fader->fading && (b_timer_elapsed(fader->timer)*MFACTOR));
}
