;;; @(#) dict.el --- An emacs interface to dict

;;; Copyright (C) 1998 Shenghuo Zhu
;;; Author: Shenghuo Zhu <zsh@cs.rochester.edu>
;;; Created: Aug 5 1998
;;; Time-stamp: <Tue Sep 29 00:37:18 EDT 1998 zsh>
;;; $Revision: 0.2 $
;;; Keywords: dictionary

;;; This file is not part of GNU Emacs, but the same permissions
;;; apply.
;;;
;;; GNU Emacs is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published
;;; by the Free Software Foundation; either version 2, or (at your
;;; option) any later version.
;;;
;;; GNU Emacs is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.

;;; Commentary:
;;;
;;; Q: What is dict?
;;; A: http://www.dict.org/ 

;;; Installation:
;;;
;;; (autoload 'dict "dict" "Dictionary")
;;; (autoload 'dict-at-point "dict" "Dictionary at point")
;;; (global-set-key [(control c) (return)] 'dict-at-point)
;;; (global-set-key [(control c) ??] 'dict)
;;; (setq dict-font-lock t)

;;; Code:

(defcustom dict-buffer-name "*Dict*"
  "The buffer name of dict"
  :type 'string
  :group 'dict)

(defcustom dict-name "dict"
  "Process name of \"dict\""
  :type 'string
  :group 'dict)

(defcustom dict-program "dict"
  "Program name of \"dict\". 
It should be a path to \"dict\", if \"dict\" is not on the $PATH"
  :type 'string
  :group 'dict)

(defcustom dict-options nil
  "Options for the program \"dict\""
  :group 'dict
)

(defcustom dict-font-lock nil
  "*Non-nil (and non-null) means Dict buffers will use font-lock-mode."
  :type 'boolean
  :group 'dict)

(defcustom dict-default-coding-system nil
  "Default coding system for dict"
  :type 'symbol
  :group 'dict)

(defun dict-process-sentinel (process event)
  (let ((buf (process-buffer process)))
    (save-excursion
      (set-buffer buf)
      (if (and dict-default-coding-system
	       (fboundp 'decode-coding-region))
	  (decode-coding-region (point-min) (point-max) 
				dict-default-coding-system))
      (setq buffer-read-only t)
      (set-window-start (get-buffer-window buf) (point-min)))))

(defun dict (word &optional arg)
  "Look up word in dictionary"
  (interactive "sWord: \nP")
  (let ((process-connection-type nil)
	(buf (get-buffer-create dict-buffer-name))
	eval-line proc default-process-coding-system)
    ;; default-process-coding-system does not work in XEmacs 20.4
    (if (get-buffer-process buf)
	(delete-process buf))
    (save-excursion
      (set-buffer buf)
      (if (eq major-mode 'dict-mode) nil
	(dict-mode))
      (setq buffer-read-only nil)
      (erase-buffer))
    (setq proc (eval (append (list 'start-process dict-name buf 
				   dict-program)
			     (if (stringp dict-options)
				 (split-string dict-options)
			       dict-options) 
			     (if arg
				 (list "-m"))
			     (list word))))
    (set-process-sentinel proc 'dict-process-sentinel)
    (display-buffer buf)))

(defun dict-at-point ()
  "Look up word under or before the cursor"
  (interactive)
  (cond ((eq major-mode 'dict-mode)
	 (let ((cpt (point)) pt line-beginning line-end)
	   (if (fboundp 'line-beginning-position)
	       (setq line-beginning (line-beginning-position)
		     line-end (line-end-position))
	     (save-excursion 
	       (forward-line 0)
	       (setq line-beginning (point))
	       (forward-line 1)
	       (setq line-end (point))))
	   (if (and (re-search-backward "[\\{\\}]" line-beginning t)
		    (equal (buffer-substring (point) 
					     (setq pt (1+ (point)))) "{")
		    (re-search-forward "\\}" line-end t))
	       (dict (buffer-substring pt (point)))
	     (goto-char cpt)
	     (dict (current-word)))))
	((if (boundp 'mark-active) mark-active
	   zmacs-region-active-p)
	 (dict (buffer-substring (region-beginning) (region-end))))
	(t (dict (current-word)))))

(defun dict-at-mouse (event)
  "Look up word under or before the position of the mouse click."
  (interactive "e")
  (mouse-set-point event)
  (dict-at-point))

;; Create mode-specific tables.
(defvar dict-mode-syntax-table nil 
  "Syntax table used while in dict mode.")

(if dict-mode-syntax-table
    ()              ; Do not change the table if it is already set up.
  (setq dict-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?\" ".   " dict-mode-syntax-table)
  (modify-syntax-entry ?\\ ".   " dict-mode-syntax-table)
  (modify-syntax-entry ?' "w   "  dict-mode-syntax-table))

(defvar dict-mode-map nil)   ; Create a mode-specific keymap.

(if dict-mode-map ()
  (setq dict-mode-map (make-sparse-keymap))
  (define-key dict-mode-map [?q] 'bury-buffer)
  (define-key dict-mode-map [??] 'dict)
  (define-key dict-mode-map [(return)] 'dict-at-point)
  (define-key dict-mode-map [(mouse-2)] 'dict-at-mouse))

(defun dict-mode ()  
  "Sets up dict functionality in the current buffer."
  (interactive)  
  (kill-all-local-variables)  
  (use-local-map dict-mode-map)     ; This provides the local keymap.  
  (setq mode-name "Dict")           ; This name goes into the mode line.  
  (setq major-mode 'dict-mode)      ; This is how describe-mode
				    ;   finds the doc string to print.  
  (set-syntax-table dict-mode-syntax-table)  
  (setq buffer-read-only t)
  (and dict-font-lock (functionp 'font-lock-mode) (font-lock-mode 1))
  (run-hooks 'dict-mode-hook))      

(if (functionp 'font-lock-add-keywords)
    (font-lock-add-keywords 
     'dict-mode '(("\\{\\([^\\}]*\\)\\}" (1 font-lock-reference-face)))))

(provide 'dict)

;;; end of dict.el


