/***********************************************************************************

	Copyright (C) 2007-2009 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <string>
#include <cstdlib>
#include <unistd.h>

#include "lifeograph.hpp"


using namespace LIFEO;


// CONSTRUCTOR
Lifeograph::Lifeograph ( int argc, char *argv[] )
	:	m_dialog_password( NULL ), m_dialog_tageditor( NULL ),
	m_dialog_chaptereditor( NULL ), m_dialog_export( NULL ), m_textbufferdiary( NULL ),
	m_menu_treeviewentries( NULL ), m_menu_tag( NULL ), m_menu_link( NULL ),
	m_programpath( argv[ 0 ] ), m_loginstatus( LOGGED_OUT ), m_option_idletime_max( 90 ),
	m_option_autologout( true ), m_option_bg_path( "" ), m_entrycount( 0 ),
	m_internaloperation( 0 ), m_flag_entrychanged( false ),
	m_ptr2listitem_cur( NULL ), m_ptr2listitem_prev( NULL )
{
	m_filebuttonrecent = new LIFEO::FilebuttonRecent( _( "Select or Create a Diary" ) );
	m_textbufferdiary = new TextbufferDiary( &m_undomanager, &m_database );
	m_window = new Gtk::Window;
	colrec_tag = new ColrecTag;	// global variable

	// CONFIGURATION
	m_gconfclient = Gnome::Conf::Client::get_default_client();
	m_flag_firsttimeuser = ! read_conf();

	// COMMANDLINE OPTIONS
	bool  force_welcome( false );
	for( int i = 1; i < argc; i++ )
	{
		if( ! strcmp( argv[ i ], "--open" ) || ! strcmp( argv[ i ], "-o" ) )
		{
			if( i < argc )
			{
				if( Glib::file_test( argv[ ++i ], Glib::FILE_TEST_EXISTS ) )
					m_filebuttonrecent->add_recent( argv[ i ] );
			}
			else
				Error( "No path provided" );
		}
		else
		if( ! strcmp( argv[ i ], "--purge-conf" ) )
		{
			if( m_gconfclient->dir_exists( CONFIG_PATH ) )
			{
				std::vector< Gnome::Conf::Entry > vector_entries =
						m_gconfclient->all_entries( CONFIG_PATH );
				for(	std::vector< Gnome::Conf::Entry >::iterator begin =
							vector_entries.begin();
						begin != vector_entries.end();
						begin++ )
					m_gconfclient->unset( ( *begin ).get_key() );
				// RECENT FILES
				vector_entries = m_gconfclient->all_entries( CONFIG_PATH + "/recent" );
				for(	std::vector< Gnome::Conf::Entry >::iterator begin =
							vector_entries.begin();
						begin != vector_entries.end();
						begin++ )
					m_gconfclient->unset( ( *begin ).get_key() );

				std::cout << "Purged gconf keys related to Lifeograph." << std::endl;
			}
			else
				std::cout << "No gconf key found." << std::endl;

			delete m_window;
			m_window = NULL;
			return;
		}
		else
		if( ! strcmp( argv[ i ], "--set-bg" ) )
		{
			if( i < argc )
				m_option_bg_path = argv[ ++i ];
			else
				Error( "No path provided" );
		}
		else
		if( ! strcmp( argv[ i ], "--force-welcome" ) )
		{
			force_welcome = true;
		}
		else
		if( Glib::file_test( argv[ i ], Glib::FILE_TEST_EXISTS ) )
		{
			m_filebuttonrecent->add_recent( argv[ i ] );
		}
	}

	// PROGRAM-WIDE VARIABLE INITIALIZATIONS
	Gtk::AboutDialog::set_url_hook( &open_url );
	Gtk::AboutDialog::set_email_hook( &mail_to );

	m_treestore_entries = Gtk::TreeStore::create( m_colrec_entry );
	m_liststore_tagtable = Gtk::ListStore::create( *colrec_tag );
	m_liststore_tagcombo = Gtk::ListStore::create( *colrec_tag );

	m_treestore_entries->set_sort_func(
			m_colrec_entry.m_id, sigc::mem_fun( *this, &Lifeograph::sort_bydate ) );
	m_treestore_entries->set_sort_func(
			m_colrec_entry.m_info, sigc::mem_fun( *this, &Lifeograph::sort_bytitle ) );

	// BACKGROUND PICTURE
	if( m_option_bg_path == "none" )	// disable bg picture
		;
	else
	if( ! Glib::file_test( m_option_bg_path, Glib::FILE_TEST_EXISTS ) )
	{
		// try to use the default bg picture:
		m_option_bg_path = PIXMAPDIR "/lifeograph/backgrounds/bg.png";
		if( ! Glib::file_test( m_option_bg_path, Glib::FILE_TEST_EXISTS ) )
		{
			Error error1( "No background!" );
			m_option_bg_path.clear();
		}
	}
	if( ! m_option_bg_path.empty() && m_option_bg_path != "none" )
		m_pixbuf_background = Gdk::Pixbuf::create_from_file( m_option_bg_path );

	// RECENT FILES
	if( m_filebuttonrecent->get_recentcount() > 0 )
	{
		m_database.set_path( m_filebuttonrecent->get_filename() );
		// update label and tooltip:
		m_filebuttonrecent->update_filename();
	}

	// INTERFACE INITIALIZATION
	if( m_flag_firsttimeuser || force_welcome )
		draw_welcomescreen();
	else
		draw_loginscreen();
	
	// SIGNALS
	LinkEntry::signal_activated().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_textview_link_activated ) );
	m_filebuttonrecent->signal_selection_changed().connect(
			sigc::mem_fun( *this, &Lifeograph::handle_filebutton_db_selectionchanged ) );
	m_filebuttonrecent->signal_create_file().connect(
			sigc::mem_fun( *this, &Lifeograph::handle_create_new_diary ) );
	m_textbufferdiary->signal_changed().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_textview_entry_changed ) );
	m_textbufferdiary->signal_link_needs_checking().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_textbuffer_link_needs_checking ) );
	m_textbufferdiary->signal_title_changed().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_textbuffer_title_changed ) );
	m_window->signal_delete_event().connect(
			sigc::mem_fun( *this, &Lifeograph::handle_event_delete ) );
}

void
Lifeograph::run( void )
{
	if( m_window )
		Gtk::Main::run( *m_window );
}

bool
Lifeograph::read_conf( void )
{
	bool result = true;

	// READ CONF
	if( m_gconfclient->dir_exists( CONFIG_PATH ) )
	{
		// TEMPORARY VALUE HOLDERS
		int int1, int2;
		std::string string;

		// DIARY PATHS
		if( m_gconfclient->dir_exists( CONFIG_PATH + "/recent" ) )
		{
			for( int i = LIFEO::MAX_RECENT_FILE_COUNT - 1; i >= 0; i-- )
			{
				string =
						m_gconfclient->get_string(
								Glib::ustring::compose(	"%1/recent/%2",
														CONFIG_PATH,
														i ) );
				if( string.size() > 0 )
					m_filebuttonrecent->add_recent( string );
			}
		}

		// WINDOW GEOMETRY
		int1 = m_gconfclient->get_int( CONFIG_PATH + "/width" );
		int2 = m_gconfclient->get_int( CONFIG_PATH + "/height" );
		if( int1 > 0 && int2 > 0 )
			m_window->set_default_size( int1, int2 );
		else
			m_window->set_default_size( 500, -1 );

		int1 = m_gconfclient->get_int( CONFIG_PATH + "/position_x" );
		int2 = m_gconfclient->get_int( CONFIG_PATH + "/position_y" );
		if( int1 > 0 && int2 > 0 )
			m_window->move( int1, int2 );

		// BACKGROUND PICTURE
		m_option_bg_path = m_gconfclient->get_string(
				CONFIG_PATH + "/bg_picture_welcome" );

		// IDLE TIME
		int1 = m_gconfclient->get_int( CONFIG_PATH + "/max_idletime" );
		// limit to sane values:
		if( int1 >= 30 && int1 <= 3600 )
		{
			m_option_idletime_max = int1;
			m_option_autologout = m_gconfclient->get_bool( CONFIG_PATH + "/auto_logout" );
		}
	}
	else
	{
		result = false;
	}

	return result;
}

bool
Lifeograph::write_conf( void )
{
	int x, y;

	m_window->get_size( x, y );
	m_gconfclient->set( CONFIG_PATH + "/width", x );
	m_gconfclient->set( CONFIG_PATH + "/height", y );
	m_window->get_position( x, y );
	m_gconfclient->set( CONFIG_PATH + "/position_x", x );
	m_gconfclient->set( CONFIG_PATH + "/position_y", y );
	m_gconfclient->set( CONFIG_PATH + "/pane_position", m_hpaned_main->get_position() );
	m_gconfclient->set( CONFIG_PATH + "/max_idletime", m_option_idletime_max );
	m_gconfclient->set( CONFIG_PATH + "/auto_logout", m_option_autologout );
	if( ! m_option_bg_path.empty() )
		m_gconfclient->set( CONFIG_PATH + "/bg_picture_welcome", m_option_bg_path );

	std::list< std::string >::iterator iter =
			m_filebuttonrecent->get_recentlist().begin();
	for( int i = 0; i < m_filebuttonrecent->get_recentcount(); i++ )
	{
		m_gconfclient->set(	Glib::ustring::compose( "%1/recent/%2", CONFIG_PATH, i ),
							*iter );
		iter++;
	}

	return true;    // reserved
}

bool
Lifeograph::handle_event( GdkEvent* )
{
	if( m_secondsremaining < LOGOUT_COUNTDOWN )
		update_title();
	// restart
	m_secondsremaining = LOGOUT_COUNTDOWN;
	m_connection_timeout.disconnect();
	m_connection_timeout = Glib::signal_timeout().connect_seconds(
			sigc::mem_fun( *this, &Lifeograph::handle_idle ),
			m_option_idletime_max - LOGOUT_COUNTDOWN );

	return false;
}

bool
Lifeograph::handle_event_expose( GdkEventExpose *event )
{
	m_window->get_window()->draw_pixbuf(
		m_window->get_style()->get_bg_gc( Gtk::STATE_NORMAL ),
		m_pixbuf_background,
		0, 0, 0, 0, -1, -1,
		Gdk::RGB_DITHER_NONE, 0, 0 );

	if( m_window->get_child() )
		m_window->propagate_expose( *m_window->get_child(), event );

	return true;
}

bool
Lifeograph::handle_event_delete( GdkEventAny* )
{
	if( m_loginstatus != LOGGED_IN )
		return false;

	return( ! save_database() );
}

bool
Lifeograph::handle_idle()
{
	if( m_secondsremaining > 0 )
	{
		m_window->set_title( Glib::ustring::compose(
				_( "<<<%1 SECONDS TO LOG OUT>>>" ), m_secondsremaining ) );
		m_connection_timeout = Glib::signal_timeout().connect_seconds(
				sigc::mem_fun( *this, &Lifeograph::handle_idle ), 1 );
		m_secondsremaining--;
		return false;
	}

	if( m_dialog_password )
		m_dialog_password->hide();
	if( m_dialog_export )
		m_dialog_export->hide();
	if( m_dialog_tageditor )
		m_dialog_tageditor->hide();
	if( m_dialog_chaptereditor )
		m_dialog_chaptereditor->hide();
	m_loginstatus = LOGGED_TIME_OUT;
	logout( true );
	return false;
}

bool
Lifeograph::handle_backup( void )
{
	sync_entry_current();
	return m_database.write_backup();
}

void
Lifeograph::update_login_file( void )
{
	m_entry_password->set_sensitive( false );
	m_button_opendiary->set_sensitive( false );

	// is this still necessary with FILECHOOSERBUTTONRECENT?:
	if( Glib::file_test( m_filebuttonrecent->get_filename(), Glib::FILE_TEST_IS_DIR ) )
		return;

	m_database.set_path( m_filebuttonrecent->get_filename() );
	Result result = m_database.read_header();
	if( result == SUCCESS )
	{
		if( m_database.is_encrypted() )
		{
			m_entry_password->set_sensitive( true );
			if( m_loginstatus == LOGGED_TIME_OUT )
				m_label_startupmessage->set_text( _( STRING::ENTER_PASSWORD_TIMEOUT ) );
			else
				m_label_startupmessage->set_text( _( STRING::ENTER_PASSWORD ) );
			m_button_opendiary->set_sensitive( m_entry_password->get_text_length() > 0 );
		}
		else
		{
			m_label_startupmessage->set_text( _( STRING::DB_IS_NOT_ENCRYPTED ) );
			m_button_opendiary->set_sensitive( true );
		}
	}
	else
	if( result == INCOMPATIBLE_FILE )
	{
		m_label_startupmessage->set_text( _( STRING::INCOMPATIBLE_DB ) );
	}
	else
	if( result == CORRUPT_FILE )
	{
		m_label_startupmessage->set_text( _( STRING::CORRUPT_DB ) );
	}
	else
	{
		m_label_startupmessage->set_text( _( STRING::FAILED_TO_OPEN_DB ) );
	}

	// FOCUS ADJUSTMENT
	if( m_database.is_encrypted() )
		m_entry_password->grab_focus();
	else
		m_button_opendiary->grab_focus();
}

void
Lifeograph::handle_button_quit_clicked( void )
{
	Gtk::Main::quit();
}

void
Lifeograph::handle_button_about_clicked( void )
{
	std::list< Glib::ustring > authors;
	authors.push_back( "Ahmet Öztürk <aoz_2@yahoo.com>" );

	Gtk::AboutDialog *aboutdialog = new Gtk::AboutDialog;
	aboutdialog->set_name( PROGRAM_NAME );
	aboutdialog->set_version( PROGRAM_VERSION_STRING );
	if( Glib::file_test( m_programpath + ".svg", Glib::FILE_TEST_EXISTS ) )
	{
		aboutdialog->set_logo( Gdk::Pixbuf::create_from_file( m_programpath + ".svg" ) );
	}
	aboutdialog->set_copyright( _( "Copyright (C) 2007-2010 Ahmet Öztürk" ) );
	aboutdialog->set_comments( _( STRING::SLOGAN ) );
	aboutdialog->set_website( "https://launchpad.net/lifeograph/" );
	aboutdialog->set_website_label( _( "Launchpad Page" ) );
	aboutdialog->set_authors( authors );
	aboutdialog->set_license(
			"Lifeograph is licensed under GNU Public License v3" "\n\n"
			"You should have received a copy of the GNU General Public License\n"
			"along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>" );
	aboutdialog->set_transient_for( *m_window );
	aboutdialog->run();
	delete aboutdialog;
}

void
Lifeograph::handle_filebutton_db_selectionchanged( void )
{
	if( m_internaloperation ) return;

	update_login_file();
}

void
Lifeograph::handle_entry_password_changed( void )
{
	if( m_database.is_path_set() )
		m_button_opendiary->set_sensitive( m_entry_password->get_text_length() > 0 );
}

void
Lifeograph::handle_create_new_diary( void )
{
	if( initialize_new_database() )
	{
		m_database.add_today();
		draw_editscreen();
		update_treeview_entries();
		show_row( m_row_cur, UI_BOTH );
		m_database.set_passphrase( "" );	//TODO: this shouldn't be needed at all
		m_loginstatus = LOGGED_IN;
	}
}

void
Lifeograph::handle_button_opendb_clicked( void )
{
	if( m_database.m_flag_old )
	{
		Gtk::MessageDialog messagedialog( *m_window,
											"",
											false,
											Gtk::MESSAGE_WARNING,
											Gtk::BUTTONS_CANCEL,
											true );
		messagedialog.set_message(
			_( "Are You Sure You Want to Upgrade The Diary?" ) );
		messagedialog.set_secondary_text(
			_( "You are about to open an old diary that "
			   "has to be upgraded to the new format. \n\n"
			   "It is STRONGLY recommended to back up any diary "
			   "before upgrading.\n\n"
			   "After you upgrade a diary, you won't be able to open it "
			   "in a previous version of Lifeograph." ) );
		messagedialog.add_button( _( "Upgrade The Diary" ), Gtk::RESPONSE_ACCEPT );

		if( messagedialog.run() !=  Gtk::RESPONSE_ACCEPT )
		{
			return;
		}
	}
	if( m_database.is_encrypted() )
	{
		std::string t_string = m_entry_password->get_text();
		m_database.set_passphrase( t_string );
	}

	switch( m_database.read_body() )
	{
		case EMPTY_DATABASE:
			m_database.add_today();
			// continues:
		case SUCCESS:
			draw_editscreen();
			update_treeview_entries();
			show_row( m_row_cur, UI_BOTH );
			m_loginstatus = LOGGED_IN;
			break;
		case WRONG_PASSWORD:
			m_label_startupmessage->set_text( _( STRING::WRONG_PASSWORD ) );
			m_entry_password->set_text( "" );
			m_entry_password->set_sensitive( false );
			sleep( 1 );
			m_entry_password->set_sensitive( true );
			m_entry_password->grab_focus();
			break;
		case FAILURE:
			m_label_startupmessage->set_text( _( STRING::INCOMPATIBLE_DB ) );
			m_database.clear();	// clear partially read content if any
			break;
		default:
			break;
	}
}

void
Lifeograph::handle_toolbutton_previous( void )
{
	Gtk::TreeRow row_prev = m_row_cur;
	if( m_row_cur[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry )
	{
		if( row_prev == ( *m_row_cur.parent() ).children().begin() )
		{
			row_prev = *( m_row_cur.parent() );
			if( m_treestore_entries->children().size() > 1 )
			{
				do
				{
					if( row_prev == m_treestore_entries->children().begin() )
					{
						row_prev = *( m_treestore_entries->children().end() );
					}
					row_prev--;
				}
				while( row_prev.children().size() < 1 );
			}
	
			row_prev = *( row_prev.children().end() );
		}
		row_prev--;
	}
	else
//	if( m_row_cur[ m_colrec_entry.m_type ] == IT_Folder )
	{
		if( row_prev == m_treestore_entries->children().begin() )
		{
			row_prev = *( m_treestore_entries->children().end() );
		}
		row_prev--;
	}

	show_row( row_prev, UI_BOTH );
}

void
Lifeograph::handle_toolbutton_next( void )
{
	Gtk::TreeRow row_next = m_row_cur;
	if( m_row_cur[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry )
	{
		if( ++row_next == ( *m_row_cur.parent() ).children().end() )
		{
			row_next = *( m_row_cur.parent() );
			if( m_treestore_entries->children().size() > 1 )
			{
				do
				{
					if( ++row_next == m_treestore_entries->children().end() )
						row_next = *( m_treestore_entries->children().begin() );
				}
				while( row_next.children().size() < 1 );
			}
	
			row_next = *( row_next.children().begin() );
		}
	}
	else
//	if( m_row_cur[ m_colrec_entry.m_type ] == IT_Folder )
	{
		if( ++row_next == m_treestore_entries->children().end() )
			row_next = *( m_treestore_entries->children().begin() );
	}

	show_row( row_next, UI_BOTH );
}

void
Lifeograph::handle_entry_filter_changed( void )
{
	if( m_internaloperation ) return;
	// current entry must be closed here or else it loses...
	// ...changes since it was selected:
	sync_entry_current();

	const std::string filterstring( m_entry_filter->get_text().lowercase() );
	if( filterstring.size() > 0 )
	{
		m_database.set_filter( filterstring );
		m_entry_filter->set_icon_sensitive( Gtk::ENTRY_ICON_SECONDARY, true );
		m_textbufferdiary->set_searchstr( filterstring );
		update_treeview_entries();
		m_entry_filter->set_tooltip_markup(
				Glib::ustring::compose(
						_( "Found in <b>%1</b> of <b>%2</b> entrie(s)" ),
						m_entrycount, m_database.get_size() ) );
	}
	else
	{
		m_database.remove_filter();
		m_entry_filter->set_icon_sensitive( Gtk::ENTRY_ICON_SECONDARY, false );
		m_textbufferdiary->set_search( false );
		update_treeview_entries();
		m_entry_filter->set_tooltip_text( _( "Enter text to be filtered" ) );
	}

	if( m_entrycount > 0 )
	{
		if( m_ptr2listitem_cur != m_textbufferdiary->get_entry() )
		{
			show_row( m_row_cur, UI_BOTH );
			m_iconview_tags->show();
		}
		else
		{
			present_entryrow( m_row_cur );
			m_textbufferdiary->reparse();
		}
	}
	else
	{
		m_iconview_tags->hide();
		m_textviewdiary->set_sensitive( false );

		if( filterstring.size() > 0 )
		{
			m_internaloperation++;
			m_textbufferdiary->set_richtext( _( "No matches found" ) );
			m_hbox_tagtools->hide();
			m_internaloperation--;
		}
	}

	if( filterstring.size() > 0 && m_entrycount > 0 )
	{
		m_toolbutton_findprevious->show();
		m_toolbutton_findnext->show();
		m_hbox_replace->show();

		m_textbufferdiary->select_searchstr_next();
	}
	else
	{
		m_toolbutton_findprevious->hide();
		m_toolbutton_findnext->hide();
		m_hbox_replace->hide();
	}

	update_calendar();
}

void
Lifeograph::handle_entry_filter_clear( Gtk::EntryIconPosition,
									   const GdkEventButton* )
{
	m_entry_filter->set_text( "" );
}

void
Lifeograph::handle_matchprevious( void )
{
	if( m_entrycount < 1 )
		return;

	if( m_row_cur[ m_colrec_entry.m_type ] != ColrecEntry::IT_Entry )
		// all folders must have at least one child:
		show_row( * m_row_cur.children().begin() );

	if( ! m_textbufferdiary->select_searchstr_previous() )
	{
		handle_toolbutton_previous();
		// put cursor to the end
		m_textbufferdiary->select_range(	m_textbufferdiary->end(),
											m_textbufferdiary->end() );
		m_textbufferdiary->select_searchstr_previous();
	}
}

void
Lifeograph::handle_matchnext( void )
{
	if( m_entrycount < 1 )
		return;

	if( m_row_cur[ m_colrec_entry.m_type ] != ColrecEntry::IT_Entry )
		// all folders must have at least one child:
		show_row( * m_row_cur.children().begin(), UI_BOTH );

	if( ! m_textbufferdiary->select_searchstr_next() )
	{
		handle_toolbutton_next();
		m_textbufferdiary->select_searchstr_next();
	}
}

void
Lifeograph::replace_match( void )
{
	if( ! m_textbufferdiary->get_has_selection() )
		handle_matchnext();
	if( m_textbufferdiary->get_has_selection() )
	{
		m_textbufferdiary->erase_selection();
		m_textbufferdiary->insert_at_cursor( m_entry_replace->get_text() );
		handle_matchnext();
	}
}

void
Lifeograph::replace_allmatches( void )
{
	m_database.replace_filter( m_entry_replace->get_text() );
	// update current entry:
	Entry *ptr2entry = static_cast< Entry* >( m_ptr2listitem_cur );
	m_textbufferdiary->set_richtext( ptr2entry );
}

void
Lifeograph::handle_button_backup_clicked( void )
{
	m_dialog_export = new DialogExport( &m_database );
	m_dialog_export->set_transient_for( *m_window );
	run_dialog( m_dialog_export );

	delete m_dialog_export;
	m_dialog_export = NULL;
}

void
Lifeograph::handle_autologout_toggled( void )
{
	m_option_autologout = m_menuitem_logoutonidle->get_active();

	if( m_database.is_encrypted() )
	{
		if( m_option_autologout )
		{
			m_connection_event = m_window->signal_event().connect(
					sigc::mem_fun( *this, &Lifeograph::handle_event ) );
		}
		else
		{
			m_connection_timeout.disconnect();
			m_connection_event.disconnect();
		}
	}
}

void
Lifeograph::handle_treeview_entries_selectionchanged( void )
{
	if( m_internaloperation ) return;

	Glib::RefPtr< Gtk::TreeSelection > selection = m_treeview_entries->get_selection();
	if( selection->count_selected_rows() > 0 )
	{
		Gtk::TreeRow row = ( *selection->get_selected() );

		show_row( row, UI_CAL );
	}
	// if the list is not empty, one row has to be selected
	else
	if( m_entrycount > 0 )
	{
		show_listitem( m_database.get_entry_first(), UI_CAL );
	}
}

void
Lifeograph::handle_treeview_entries_event_buttonpress( GdkEventButton *event )
{
	if( event->button != 3 )
		return;

	Glib::RefPtr< Gtk::TreeSelection > selection =
			m_treeview_entries->get_selection();

	if( selection->count_selected_rows() <= 0 )
		return;

	// treeview issues selection changed signal after button press signal.
	// this causes context menus to be linked to the previously selected item
	// when a row is selected with right click.
	// to get around this  we use a somewhat dirty hack and select the row
	// from within the button press event handler
	// p.s.: i dunno but there might be a simpler way of doing this
	Gtk::TreeModel::Path path;
	Gtk::TreeViewColumn  *column;
	int i, j;
	if( ! m_treeview_entries->get_path_at_pos(
			(int) event->x, (int) event->y, path, column, i, j ) )
		return;

	m_treeview_entries->set_cursor( path );
	// end of dirty hack

	Gtk::TreeRow row = *( selection->get_selected() );

	if( ! (	row[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry ||
			row[ m_colrec_entry.m_type ] == ColrecEntry::IT_Tagfolder ) )
		return;

	if( m_menu_treeviewentries )
	{
		delete m_menu_treeviewentries;
		m_menu_treeviewentries = NULL;
	}

	m_menu_treeviewentries = new Gtk::Menu;
	Gtk::Image *imageDismiss = Gtk::manage(
		new Gtk::Image( Gtk::Stock::DELETE, Gtk::ICON_SIZE_MENU ) );
	Gtk::Image *imageEdit = Gtk::manage(
		new Gtk::Image( Gtk::Stock::EDIT, Gtk::ICON_SIZE_MENU ) );

	if( row[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry )
	{
		Gtk::Image *image_favored = Gtk::manage(
			new Gtk::Image( m_pixbuf_favored ) );
		m_menu_treeviewentries->items().push_back(
				Gtk::Menu_Helpers::ImageMenuElem( _( "_Dismiss" ), *imageDismiss,
				sigc::bind( sigc::mem_fun( *this, &Lifeograph::delete_entry ), row ) ) );

		Entry *entry = get_treerow_ptr_entry( row );
		m_menu_treeviewentries->items().push_back(
			Gtk::Menu_Helpers::ImageMenuElem(
				entry->is_favored() ?
						_( "Remove from _Favorites" ) : _( "Add to _Favorites" ),
				*image_favored,
				sigc::bind( sigc::mem_fun(	*this,
											&Lifeograph::toggle_entryfavoredness ),
				row ) ) );
	}
	else	// IT_Tagfolder
	{
		m_menu_treeviewentries->items().push_back(
			Gtk::Menu_Helpers::ImageMenuElem( _( "_Edit Tag" ), *imageEdit,
					sigc::bind( sigc::mem_fun( *this, &Lifeograph::edit_tag ),
					get_treerow_ptr_tag( row ) ) ) );
		m_menu_treeviewentries->items().push_back(
			Gtk::Menu_Helpers::ImageMenuElem( _( "_Delete Tag" ), *imageDismiss,
					sigc::bind( sigc::mem_fun( *this, &Lifeograph::delete_tag ),
					get_treerow_ptr_tag( row ) ) ) );
	}

	m_menu_treeviewentries->accelerate( *m_window );
	m_menu_treeviewentries->popup( event->button, event->time );
}

void
Lifeograph::handle_treeview_sortcolumnchanged()
{
	// FIXME: dirty hack: this signal is issued before list is actually sorted
	// so we delay the evaluation a little bit..
	Glib::signal_timeout().connect_once(
			sigc::mem_fun( *this, &Lifeograph::update_toolbar ), 100 );
}

void
Lifeograph::handle_calendar_doubleclicked( void )
{
	guint year, month, day;

	m_calendar->get_date( year, month, day );

	Entry *entry = m_database.create_entry(
			combine_date( year, month + 1, day ) );

	sync_entry_current();	// synchronize current entry before the list update
	update_treeview_entries();
	update_calendar();

	show_listitem( entry, UI_LIST );

	// this is the only way i could find to give focus to textview
	Glib::signal_timeout().connect_once(
			sigc::mem_fun( m_textviewdiary, &Gtk::Widget::grab_focus ), 200 );
}

void
Lifeograph::handle_calendar_dayselected( void )
{
	if( m_internaloperation ) return;

	guint year, month, day;
	m_calendar->get_date( year, month, day );
	Entry *entry = m_database.get_entry_nextinday(
			combine_date( year, month + 1, day ),
			get_entry_current() );

	if( entry )
		show_listitem( entry, UI_LIST );
}

void
Lifeograph::handle_calendar_monthchanged( void )
{
	if( m_internaloperation ) return;

	update_calendar();
}

void
Lifeograph::handle_textview_entry_changed( void )
{
	if( m_internaloperation ) return;

	m_flag_entrychanged = true;

	if( m_hbox_tagtools->is_visible() )
		m_hbox_tagtools->hide();
}

void
Lifeograph::handle_textview_link_activated( Date date )
{
	Entryvector *entries = m_database.get_entries( date );

	if( entries->size() > 1 )
	{
		if( m_menu_link )
			m_menu_link->items().clear();
		else
			m_menu_link = new Gtk::Menu;

		for( Entryiterv iter = entries->begin(); iter != entries->end(); iter++ )
		{
			m_menu_link->items().push_back(
					Gtk::Menu_Helpers::MenuElem(
							( *iter )->get_title(),
							sigc::bind(
									sigc::mem_fun( *this, &Lifeograph::show_listitem ),
									*iter, UI_BOTH ) ) );
		}

		m_menu_link->accelerate( *m_window );
		m_menu_link->popup( 0, gtk_get_current_event_time() );
	}
	else
	if( entries->size() == 1 )
		show_listitem( entries->at( 0 ), UI_BOTH );
	else
	{
		Entry *entry = m_database.create_entry( date );
		update_treeview_entries();
		show_listitem( entry, UI_BOTH );

	}
	
	delete entries;
}

LinkStatus
Lifeograph::handle_textbuffer_link_needs_checking( Date date )
{
	Entry *ptr2entry = m_database.get_entry( date );
	if( ptr2entry == NULL )
		return LS_ENTRY_UNAVAILABLE;
	else
	if( ptr2entry == m_ptr2listitem_cur )
		return LS_CYCLIC;
	else
		return LS_OK;
}

void
Lifeograph::handle_textbuffer_title_changed( const Glib::ustring& title )
{
	if( m_internaloperation ) return;

	m_row_cur[ m_colrec_entry.m_info ] = title;

	update_title();
}

bool
Lifeograph::handle_hboxtagtools_enternotify( GdkEventCrossing *event )
{
	if( event->mode == GDK_CROSSING_NORMAL )
		m_hbox_tagtools->show();
	return false;
}

void
Lifeograph::handle_removetags( void )
{
	std::list< Gtk::TreeModel::Path > selected_items =
			m_iconview_tags->get_selected_items();
	Entry *ptr2entry = static_cast< Entry* >( m_ptr2listitem_cur );
	for( std::list< Gtk::TreeModel::Path >::iterator iter = selected_items.begin();
		 iter != selected_items.end();
		 iter++ )
	{
		Gtk::TreeRow row = *m_liststore_tagtable->get_iter( *iter );
		ptr2entry->get_tags().remove_tag( row[ colrec_tag->m_ptr2tag ] );
	}
	update_tags();
	if( m_database.get_entryliststyle() == LIST_BYTAG )
	{
		update_treeview_entries();
		present_entryrow( m_row_cur );
	}
}

void
Lifeograph::handle_iconviewtags_buttonpress( GdkEventButton *event )
{
	if( event->button != 3 )
		return;

	std::list< Gtk::TreeModel::Path > selected_items =
			m_iconview_tags->get_selected_items();

	if( selected_items.empty() )
		return;
	else
	if( m_liststore_tagtable->get_iter( selected_items.front() )->get_value(
				colrec_tag->m_ptr2tag ) == NULL )
		return;

	if( m_menu_tag )
	{
		delete m_menu_tag;
		m_menu_tag = NULL;
	}

	m_menu_tag = new Gtk::Menu;
	Gtk::Image *image_remove = Gtk::manage(
		new Gtk::Image( Gtk::Stock::REMOVE, Gtk::ICON_SIZE_MENU ) );

	m_menu_tag->items().push_back(
			// TRANSLATORS: label of the remove-selected-tags-from-current-entry
			// menu item
			Gtk::Menu_Helpers::ImageMenuElem( _( "_Remove from Entry" ), *image_remove,
					sigc::mem_fun( *this, &Lifeograph::handle_removetags ) ) );

	m_menu_tag->accelerate( *m_window );
	m_menu_tag->popup( event->button, event->time );
}

void
Lifeograph::handle_iconviewtags_keyrelease( GdkEventKey *event )
{
	if( event->keyval != GDK_Delete )	// the cheapest check first
		return;

	std::list< Gtk::TreeModel::Path > selected_items =
			m_iconview_tags->get_selected_items();

	if( selected_items.empty() )
		return;
	else
	if( m_liststore_tagtable->get_iter( selected_items.front() )->get_value(
				colrec_tag->m_ptr2tag ) == NULL )
		return;

	handle_removetags();
}

void
Lifeograph::handle_tagwidget_activated( void )
{
	Entry *ptr2entry = static_cast< Entry* >( m_ptr2listitem_cur );
	Tag *tag = Tag::create( m_tagwidget->get_active_text() );
	if( ptr2entry->get_tags().checkfor_member( tag ) )
		ptr2entry->get_tags().remove_tag( tag );
	else
		ptr2entry->get_tags().add_tag( tag );

	update_tags();
	m_tagwidget->get_entry()->set_text( "" );
	if( m_database.get_entryliststyle() == LIST_BYTAG )
	{
		update_treeview_entries();
		present_entryrow( m_row_cur );
	}
}

void
Lifeograph::handle_entryliststyle_changed( void )
{
	if( m_internaloperation ) return;

	m_database.set_entryliststyle(
			m_combobox_liststyle->get_active_row_number() );

	update_entryliststyle();
}

void
Lifeograph::handle_undo( void )
{
	m_undomanager.undo();
}

void
Lifeograph::handle_redo( void )
{
	m_undomanager.redo();
}

bool
Lifeograph::initialize_new_database( void )
{
	DialogSaveDiary *dialogsavediary = new DialogSaveDiary;
	dialogsavediary->set_current_folder( Glib::get_home_dir() );
	dialogsavediary->set_do_overwrite_confirmation( true );
	bool result;

	if( dialogsavediary->run() == Gtk::RESPONSE_ACCEPT )
	{
		m_database.set_path( dialogsavediary->get_filename() );
		// TODO: check for validity of file?
		result = true;
	}
	else
		result = false;

	delete dialogsavediary;
	return result;
}

bool
Lifeograph::save_database( void )
{
	sync_entry_current();
	//TODO: may there be a case when it is required to save only the database
	//		not the configuration?
	write_conf();
	return( m_database.write() == SUCCESS );
}

void
Lifeograph::logout( bool opt_save )
{
	if( opt_save )
	{
		// files added to recent list here: for newly created files, files that...
		// ...are saved under different names/dirs (a future save as option):
		m_filebuttonrecent->set_filename( m_database.get_path() );
		if( ! save_database() )
		{
			Gtk::MessageDialog messagedialog(	*m_window,
												"",
												false,
												Gtk::MESSAGE_WARNING,
												Gtk::BUTTONS_OK,
												true );
			messagedialog.set_message( _( STRING::CANNOT_WRITE ) );
			messagedialog.set_secondary_text( _( STRING::CANNOT_WRITE_SUB ) );

			messagedialog.run();
		}
	}
	else
	{
		Gtk::MessageDialog messagedialog(	*m_window,
											"",
											false,
											Gtk::MESSAGE_WARNING,
											Gtk::BUTTONS_CANCEL,
											true );
		messagedialog.set_message(
			_( "Are you sure you want to log out without saving?" ) );
		messagedialog.set_secondary_text( Glib::ustring::compose(
			// TRANSLATORS: please keep the \"%1.~backup~\" part intact
			_(	"Your changes will be backed up in \"%1.~backup~\". "
				"If you exit normally, your diary is saved automatically." ),
			m_database.m_path ) );
		messagedialog.add_button( _( "_Log out without Saving" ), Gtk::RESPONSE_ACCEPT );

		if( messagedialog.run() !=  Gtk::RESPONSE_ACCEPT )
		{
			return;
		}
		// back up changes
		sync_entry_current();
		m_database.write_backup();
	}
	if( m_loginstatus == LOGGED_IN )
		m_loginstatus = LOGGED_OUT;
	m_database.clear();
	m_textbufferdiary->reset();
	m_connection_event.disconnect();
	m_connection_timeout.disconnect();
	m_connection_backup.disconnect();
	m_ptr2listitem_cur = NULL;
	m_ptr2listitem_prev = NULL;
	draw_loginscreen();
}

void
Lifeograph::show_row( const Gtk::TreeRow& row, Updateindex ui )
{
	m_internaloperation++;

	m_undomanager.clear();

	sync_entry_current();

	if( m_ptr2listitem_cur != row[ m_colrec_entry.m_ptr ] )
	{
		m_ptr2listitem_prev = m_ptr2listitem_cur;
		m_ptr2listitem_cur = row[ m_colrec_entry.m_ptr ];
		m_row_cur = row;
	}

	if( row[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry )
	{
		Entry *ptr2entry = static_cast< Entry* >( m_ptr2listitem_cur );
		m_textbufferdiary->set_richtext( ptr2entry );

		if( ui & UI_CAL )
		{
			// FIXME: strangely, the only way i figured out to avoid...
			// ...day invalidation warning is first selecting a day...
			// ...which is in every month:
			m_calendar->select_day( 1 );
			m_calendar->select_month( ptr2entry->get_month() - 1, ptr2entry->get_year() );
			update_calendar();
			m_calendar->select_day( ptr2entry->get_day() );
		}

		// INTERFACE UPDATE
		update_tags();
		m_textviewdiary->set_sensitive( true );
		m_iconview_tags->set_sensitive( true );
	}
	else
	{
		// strip pango markup from the folder name:
		// (an adaptation of the C algorithm at: http://www.gtkforums.com/about1458.html )
		Glib::RefPtr< Glib::Regex > regex = Glib::Regex::create(
				"<(.|\n)*?>",
				Glib::REGEX_DOTALL | Glib::REGEX_OPTIMIZE,
				Glib::RegexMatchFlags( 0 ) );

		m_textbufferdiary->set_richtext(
				// TRANSLATORS: %1 is the title of the folder and
				// %2 is the number of the entries in it
				Glib::ustring::compose( _("%1\n\n%2 Entries"),
						regex->replace_literal(	row[ m_colrec_entry.m_id ], 0, "",
												Glib::RegexMatchFlags( 0 ) ),
						row.children().size() ) );

		m_liststore_tagtable->clear();
		m_textviewdiary->set_sensitive( false );
		m_iconview_tags->set_sensitive( false );
	}

	if( ui & UI_LIST )
		present_entryrow( m_row_cur );

	m_hbox_tagtools->hide();
	m_tagwidget->get_entry()->set_text( "" );
	update_toolbar();
	update_title();

	m_internaloperation--;
}

void
Lifeograph::show_today( void )
{
	Entry *entry_today = m_database.get_entry_today();

	if( entry_today == NULL )
	{
		sync_entry_current();
		entry_today = m_database.add_today();
		update_treeview_entries();
		show_listitem( entry_today, UI_BOTH );
	}
	else
	if( entry_today != m_ptr2listitem_cur )
	{
		show_listitem( entry_today, UI_BOTH );
	}
}

inline void
Lifeograph::show_listitem( const void *ptr2listitem, Updateindex ui )
{
	Gtk::TreeRow row;
	if( get_entryrow( row, ptr2listitem ) )
		show_row( row, ui );
}

void
Lifeograph::show_empty( void )
{
	m_textbufferdiary->set_richtext( "" );
	m_textviewdiary->set_sensitive( false );
	m_window->set_title( _( "Empty Diary" ) );
	m_action_previous_match->set_sensitive( false );
	m_action_next_match->set_sensitive( false );
	m_hbox_tagtools->hide();
	m_iconview_tags->hide();
}

void
Lifeograph::go_back( void )
{
	if( m_ptr2listitem_prev )
		show_listitem( m_ptr2listitem_prev, UI_BOTH );
}

inline void
Lifeograph::sync_entry_current( void )
{
	if( m_entrycount > 0 )
		if( m_flag_entrychanged )
			if( m_row_cur[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry )
			{
				Entry *ptr2entry = static_cast< Entry* >( m_ptr2listitem_cur );
				ptr2entry->set_text( m_textbufferdiary->get_text() );
				m_flag_entrychanged = false;
			}
}

void
Lifeograph::present_entryrow( const Gtk::TreeRow &row )
{
	m_internaloperation++;

//	m_treeview_entries->collapse_all();
	// disabled: causes flickering, will be re-enabled after we start to keep...
	// ...pointers to previous entries (browse history)
	if( row[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry )
		m_treeview_entries->expand_to_path( m_treestore_entries->get_path( row ) );
	m_treeview_entries->get_selection()->select( row );
	m_treeview_entries->scroll_to_row( m_treestore_entries->get_path( row ) );

	m_internaloperation--;
}

void
Lifeograph::present_currentrow( void )
{
	if( m_entrycount > 0 )
		present_entryrow( m_row_cur );
}

void
Lifeograph::delete_entry( Gtk::TreeRow &row )
{
	if( m_database.m_entries.size() > 0 )
	{
		Gtk::MessageDialog message(	*m_window, "",
									false, Gtk::MESSAGE_WARNING,
									Gtk::BUTTONS_NONE, true );
		message.set_message( _( "Are you sure, you want to dismiss?" ) );
		message.set_secondary_text( _( "This operation cannot be undone!" ) );
		message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
		message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

		if( message.run() != Gtk::RESPONSE_ACCEPT )
			return;

		m_internaloperation++;
		m_flag_entrychanged = false;

		m_database.delete_entry( get_treerow_ptr_entry( row ) ); // delete entry

		update_calendar();
		update_treeview_entries();

		if( m_entrycount == 0 )
			show_empty();
		else
			show_row( m_row_cur, UI_BOTH );

		m_internaloperation--;
	}
}

void
Lifeograph::toggle_entryfavoredness( Gtk::TreeRow &row )
{
	Entry *entry = get_treerow_ptr_entry( row );

	if( entry->is_favored() )
	{
		entry->make_unfavored();
		row[ m_colrec_entry.m_icon ] = m_pixbuf_unfavored;
	}
	else
	{
		entry->make_favored();
		row[ m_colrec_entry.m_icon ] = m_pixbuf_favored;
	}
}

void
Lifeograph::edit_tag( Tag *tag )
{
	m_dialog_tageditor = new DialogTageditor( tag );

	if( run_dialog( m_dialog_tageditor ) == Gtk::RESPONSE_OK )
	{
		tag->set_name( m_dialog_tageditor->get_tagname() );

		Gtk::TreeRow row_selected =
			*m_treeview_entries->get_selection()->get_selected();
		if( get_treerow_ptr_tag( row_selected ) == tag )
		{
			row_selected[ m_colrec_entry.m_id ] = m_dialog_tageditor->get_tagname();
			show_row( row_selected );
		}
	}
	delete m_dialog_tageditor;
	m_dialog_tageditor = NULL;
}

void
Lifeograph::delete_tag( Tag *tag )
{
	Gtk::MessageDialog message(	*m_window, "",  false,
								Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
	message.set_message( _( "Are you sure, you want to delete selected tag?" ) );
	message.set_secondary_text( _( "This operation cannot be undone!" ) );
	message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	message.add_button( Gtk::Stock::DELETE, Gtk::RESPONSE_ACCEPT );
	if( message.run() != Gtk::RESPONSE_ACCEPT )
		return;

	m_database.delete_tag( tag );
	update_treeview_entries();
	if( m_entrycount )
		show_row( m_row_cur, UI_BOTH );
}

void
Lifeograph::update_treeview_entries( void )
{
	m_internaloperation++;

	Gtk::TreeRow	row_folder, row_entry;
	Entryiter		iter_entry = m_database.m_entries.begin();

	m_treestore_entries->clear();
	m_entrycount = 0;

	// GROUP BY TAG
	if( m_database.get_entryliststyle() == LIST_BYTAG )
	{
		typedef std::pair< Tag*, Gtk::TreeRow >	PairTagRow;
		typedef std::map< Tag*, Gtk::TreeRow >	MapTagRow;

		MapTagRow						map_tag_row;
		Gtk::TreeRow					row_untagged;
		Tag::Tagpool::const_iterator	iter_tag;

		// ***********************************************************************
		// TODO: this part will be removed after a tag editor is introduced
		if( ! m_database.is_filter_set() )
		{
			for(	iter_tag = Tag::get_pool().begin();
					iter_tag != Tag::get_pool().end();
					iter_tag++ )
			{
				row_folder = *m_treestore_entries->append();
				// ptr must be first:
				row_folder[ m_colrec_entry.m_ptr ] = ( *iter_tag );
				row_folder[ m_colrec_entry.m_type ] = ColrecEntry::IT_Tagfolder;
				row_folder[ m_colrec_entry.m_id ] =
						Glib::Markup::escape_text( ( *iter_tag )->get_name() );
				//row_folder[ m_colrec_entry.m_info ] = // TODO..
				row_folder[ m_colrec_entry.m_icon ] = m_pixbuf_folder;
				map_tag_row.insert( PairTagRow( *iter_tag, row_folder ) );
			}
		}
		// ***********************************************************************

		for(	iter_entry = m_database.m_entries.begin();
				iter_entry != m_database.m_entries.end();
				++iter_entry )
		{
			if( ( *iter_entry )->get_filtered_out() == false )
			{
				m_entrycount++;

				if( ( *iter_entry )->get_tags().empty() )
				{
					if( ! row_untagged )
					{
						row_untagged = *m_treestore_entries->prepend();
						// ptr must be first:
						row_untagged[ m_colrec_entry.m_ptr ] = &m_address_folderuntagged;
						row_untagged[ m_colrec_entry.m_type ] =
								ColrecEntry::IT_Untaggedfolder;
						row_untagged[ m_colrec_entry.m_id ] = _( "<i>Untagged</i>" );
						//row_untagged[ m_colrec_entry.m_info ] = // TODO..
						row_untagged[ m_colrec_entry.m_icon ] = m_pixbuf_folder;
					}
					row_entry = *m_treestore_entries->append( row_untagged.children() );
					// ptr must be first:
					row_entry[ m_colrec_entry.m_ptr ] = ( *iter_entry );
					row_entry[ m_colrec_entry.m_type ] = ColrecEntry::IT_Entry;
					row_entry[ m_colrec_entry.m_date_long ] = ( *iter_entry )->get_date();
					row_entry[ m_colrec_entry.m_id ] = ( *iter_entry )->get_date_str();
					row_entry[ m_colrec_entry.m_info ] = ( *iter_entry )->get_title();
					row_entry[ m_colrec_entry.m_icon ] =
							( *iter_entry )->is_favored() ?
									m_pixbuf_favored : m_pixbuf_unfavored;
					continue;
				}
				for(	iter_tag = ( *iter_entry )->get_tags().begin();
						iter_tag != ( *iter_entry )->get_tags().end();
						++iter_tag )
				{
					// select row (inserts a new row if it does not exist)
					MapTagRow::iterator iter_pair = map_tag_row.find( *iter_tag );
					if( iter_pair != map_tag_row.end() )
						row_folder = iter_pair->second;
					else
					{
						row_folder = *m_treestore_entries->append();
						map_tag_row.insert( PairTagRow( *iter_tag, row_folder ) );

						// ptr must be first:
						row_folder[ m_colrec_entry.m_ptr ] = ( *iter_tag );
						row_folder[ m_colrec_entry.m_type ] = ColrecEntry::IT_Tagfolder;
						row_folder[ m_colrec_entry.m_id ] =
								Glib::Markup::escape_text( ( *iter_tag )->get_name() );
						//row_folder[ m_colrec_entry.m_info ] = // TODO..
						row_folder[ m_colrec_entry.m_icon ] = m_pixbuf_folder;
					}
					row_entry = *m_treestore_entries->append( row_folder.children() );
					row_entry[ m_colrec_entry.m_ptr ] = ( *iter_entry );
					row_entry[ m_colrec_entry.m_type ] = ColrecEntry::IT_Entry;
					row_entry[ m_colrec_entry.m_date_long ] = ( *iter_entry )->get_date();
					row_entry[ m_colrec_entry.m_id ] = ( *iter_entry )->get_date_str();
					row_entry[ m_colrec_entry.m_info ] = ( *iter_entry )->get_title();
					row_entry[ m_colrec_entry.m_icon ] =
							( *iter_entry )->is_favored() ?
									m_pixbuf_favored : m_pixbuf_unfavored;
				}
			}
		}
	}
	// GROUP BY DATE
	else
	if( m_database.get_entryliststyle() == LIST_BYDATE )
	{
		unsigned int date_last = 0xFFFFFFF;

		for( ; iter_entry != m_database.m_entries.end(); iter_entry++ )
		{
			if( ( *iter_entry )->get_filtered_out() == true )
				continue;

			if( date_last != ( *iter_entry )->get_yearmonth() )
			{
				date_last = ( *iter_entry )->get_yearmonth();
				row_folder = *( m_treestore_entries->append() );
				// TODO: add a dummy pointer for years
				row_folder[ m_colrec_entry.m_ptr ] = NULL;
				row_folder[ m_colrec_entry.m_type ] = ColrecEntry::IT_Datefolder;
				row_folder[ m_colrec_entry.m_date_long ] = make_year( date_last );
				row_folder[ m_colrec_entry.m_id ] = ( *iter_entry )->get_yearmonth_str();
				//row_folder[ m_colrec_entry.m_info ] = // TODO..
				row_folder[ m_colrec_entry.m_icon ] = m_pixbuf_folder;
			}

			row_entry = *( m_treestore_entries->append( row_folder.children() ) );
			row_entry[ m_colrec_entry.m_ptr ] = ( *iter_entry );
			row_entry[ m_colrec_entry.m_type ] = ColrecEntry::IT_Entry;
			row_entry[ m_colrec_entry.m_date_long ] = ( *iter_entry )->get_date();
			row_entry[ m_colrec_entry.m_id ] = ( *iter_entry )->get_day_str();
			row_entry[ m_colrec_entry.m_info ] = ( *iter_entry )->get_title();
			row_entry[ m_colrec_entry.m_icon ] =
				( *iter_entry )->is_favored() ? m_pixbuf_favored : m_pixbuf_unfavored;

			m_entrycount++;
		}
	}
	// GROUP BY CHAPTER
	else
	if( m_database.get_entryliststyle() > LIST_BYTAG )
	{
		Chapterset						*chapterset;
		Chapterset::iterator			iter_chapter, iter_chapter_end;
		Chapterset::Set::const_iterator iter_chapterset =
											m_database.get_chaptersets()->begin();

		std::advance( iter_chapterset, m_database.get_entryliststyle() - LIST_BYTAG - 1 );
		chapterset = ( *iter_chapterset );
		iter_chapter_end = chapterset->end();

		for( Chapterset::const_iterator iter = chapterset->begin(); ; iter++ )
		{
			if( iter == iter_chapter_end || ! ( *iter )->is_initialized() )
			{
				if( iter_entry != m_database.m_entries.end() )
				{
					row_folder = *m_treestore_entries->append();
					// ptr must be first:
					row_folder[ m_colrec_entry.m_ptr ] = &m_address_folderearlier;
					row_folder[ m_colrec_entry.m_type ] = ColrecEntry::IT_Earlier;
					//row_folder[ m_colrec_entry.m_date_long ] = 0;
					row_folder[ m_colrec_entry.m_id ] = _("<i>Earlier Entries</i>");
					//row_folder[ m_colrec_entry.m_info ] = // TODO..
					row_folder[ m_colrec_entry.m_icon ] = m_pixbuf_folder;
				}
				else
					break;
			}
			else
			if( ( *iter )->is_initialized() )
			{
				row_folder = *m_treestore_entries->append();
				// ptr must be first:
				row_folder[ m_colrec_entry.m_ptr ] = ( *iter );
				row_folder[ m_colrec_entry.m_type ] = ColrecEntry::IT_Chapterfolder;
				row_folder[ m_colrec_entry.m_date_long ] = ( *iter )->get_date();
				row_folder[ m_colrec_entry.m_id ] =
								Glib::Markup::escape_text( ( *iter )->get_name() );
				//row_folder[ m_colrec_entry.m_info ] = // TODO..
				row_folder[ m_colrec_entry.m_icon ] = m_pixbuf_folder;
			}
			else	// skip this round for entries if no folder is added
				continue;

			for( ; iter_entry != m_database.m_entries.end(); iter_entry++ )
			{
				if(	row_folder[ m_colrec_entry.m_date_long ] >
					( *iter_entry )->get_date() )
					break;
				if( ( *iter_entry )->get_filtered_out() == true )
					continue;

				row_entry = *( m_treestore_entries->append( row_folder.children() ) );
				// ptr must be first:
				row_entry[ m_colrec_entry.m_ptr ] = ( *iter_entry );
				row_entry[ m_colrec_entry.m_type ] = ColrecEntry::IT_Entry;
				row_entry[ m_colrec_entry.m_date_long ] = ( *iter_entry )->get_date();
				row_entry[ m_colrec_entry.m_id ] = ( *iter_entry )->get_date_str();
				row_entry[ m_colrec_entry.m_info ] = ( *iter_entry )->get_title();
				row_entry[ m_colrec_entry.m_icon ] =
					( *iter_entry )->is_favored() ? m_pixbuf_favored : m_pixbuf_unfavored;

				m_entrycount++;
			}

			if( iter == iter_chapter_end )
				break;
		}
	}

	// RESTORE m_row_cur
	if( m_entrycount > 0 )
	{
		if( m_ptr2listitem_cur )
		{
			if( ! get_entryrow( m_row_cur, m_ptr2listitem_cur ) )
				m_ptr2listitem_cur = NULL;
		}
		if( m_ptr2listitem_cur == NULL )
		{
			get_entryrow( m_row_cur, m_database.get_entry_first() );
			m_ptr2listitem_cur = m_row_cur[ m_colrec_entry.m_ptr ];
		}
	}

	m_internaloperation--;
}

void
Lifeograph::update_toolbar()
{
	bool flag_sensitive =
			m_row_cur[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry ?
					m_entrycount > 1 : m_treestore_entries->children().size() > 1;

	m_action_previous_list->set_sensitive( flag_sensitive );
	m_action_next_list->set_sensitive( flag_sensitive );
}

void
Lifeograph::update_calendar( void )
{
	guint year, month, day;

	m_calendar->get_date( year, month, day );
	Entryiter itr_entry;
	m_calendar->clear_marks();

	for(	itr_entry = m_database.m_entries.begin();
			itr_entry != m_database.m_entries.end();
			itr_entry++ )
	{
		if( ( *itr_entry )->get_filtered_out() ) continue;

		if( ( *itr_entry )->get_year() == year )
		{
			if( ( *itr_entry )->get_month() == month + 1 )
			{
				m_calendar->mark_day( ( *itr_entry )->get_day() );
			}
			else
			if( ( *itr_entry )->get_month() < month + 1 )
				break;
		}
		else
		if( ( *itr_entry )->get_year() < year )
			break;
	}
}

// ATTENTION: this function has to be called after textbuffer is updated:
void
Lifeograph::update_title( void )
{
	if( m_row_cur[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry )
	{
		Entry *entry = static_cast< Entry* >( m_ptr2listitem_cur );
		m_window->set_title(
				entry->get_date_str() +
				": " +
				m_textbufferdiary->get_title() );
	}
	else
		m_window->set_title( m_textbufferdiary->get_title() );
}

void
Lifeograph::update_tags( void )
{
	m_internaloperation++;

	Gtk::TreeRow row;
	m_liststore_tagtable->clear();
	m_liststore_tagcombo->clear();

	Entry *ptr2entry = static_cast< Entry* >( m_ptr2listitem_cur );

	for(	Tag::Tagpool::const_iterator iter = Tag::get_pool().begin();
			iter != Tag::get_pool().end();
			iter++ )
	{
		if( ptr2entry->get_tags().checkfor_member( *iter ) )
		{
			row = *( m_liststore_tagtable->append() );
			row[ colrec_tag->m_name ] =
					Glib::Markup::escape_text( ( *iter )->get_name() );
		}
		else
		{
			row = *( m_liststore_tagcombo->append() );
			row[ colrec_tag->m_name ] = ( *iter )->get_name();
		}

		row[ colrec_tag->m_icon ] = m_pixbuf_tag;
		row[ colrec_tag->m_ptr2tag ] = ( *iter );
	}

	if( ptr2entry->get_tags().size() < 1 )
	{
		row = *( m_liststore_tagtable->append() );
		row[ colrec_tag->m_name ] = _( "<i>Untagged Entry</i>" );
		row[ colrec_tag->m_ptr2tag ] = NULL;
	}

	m_internaloperation--;
}

void
Lifeograph::update_combobox_entryliststyle( void )
{
	m_internaloperation++;

	m_combobox_liststyle->clear_items();
	m_combobox_liststyle->append_text( _( "Group by Date" ) );
	m_combobox_liststyle->append_text( _( "Group by Tag" ) );

	for(	Chapterset::Set::const_iterator iter =
					m_database.get_chaptersets()->begin();
			iter != m_database.get_chaptersets()->end();
			iter++ )
	{
		m_combobox_liststyle->append_text(
				Glib::ustring::compose(	_( "Group by Chapters: %1" ),
										( *iter )->get_name() ) );
	}

	m_combobox_liststyle->set_active( m_database.get_entryliststyle() );

	m_internaloperation--;
}

void
Lifeograph::update_entryliststyle( void )
{
	m_internaloperation++;

	sync_entry_current();

	update_treeview_entries();
	present_currentrow();

	m_treeview_entries->columns_autosize();

	m_internaloperation--;
}

void
Lifeograph::focus_filter( void )
{
	m_entry_filter->grab_focus();
}

void
Lifeograph::focus_tag( void )
{
	m_hbox_tagtools->show();
	m_tagwidget->get_entry()->grab_focus();
}

// wraps Dialog::run() to add activity reporting
int
Lifeograph::run_dialog( Gtk::Dialog *dialog )
{
	if( m_database.is_encrypted() && m_option_autologout )
	{
		m_connection_event.disconnect();
		m_connection_event = dialog->signal_event().connect(
				sigc::mem_fun( *this, &Lifeograph::handle_event ) );
	}

	int result = dialog->run();

	if( m_database.is_encrypted() && m_option_autologout )
	{
		m_connection_event.disconnect();
		m_connection_event = m_window->signal_event().connect(
				sigc::mem_fun( *this, &Lifeograph::handle_event ) );
	}

	return result;
}

void
Lifeograph::start_dialog_password()
{
	if( ! m_dialog_password )
	{
		try
		{
			m_dialog_password = new DialogPassword(	m_database.get_passphrase(),
			// TRANSLATORS: this is the title of the password dialog
													_( "Change Password" ) );
		}
		catch( ... )
		{
			// TODO: throw
			return;
		}
		m_dialog_password->set_transient_for( *m_window );
		m_dialog_password->show_all();
		
		run_dialog( m_dialog_password );

		if( m_dialog_password->is_passphrase_set() )
		// TODO: if( run_dialog( m_dialog_password ) )
		{
			m_database.set_passphrase( m_dialog_password->get_passphrase() );
		}

		delete m_dialog_password;
		m_dialog_password = NULL;
	}
}

void
Lifeograph::start_dialog_chaptereditor( void )
{
	m_dialog_chaptereditor = new DialogChaptereditor( &m_database );

	run_dialog( m_dialog_chaptereditor );

	if( m_dialog_chaptereditor->is_groups_changed() )
	{
		if(	m_database.get_entryliststyle() >=
			2 + m_database.get_chaptersets()->size() )
		{
			m_database.set_entryliststyle( 0 );
		}

		update_combobox_entryliststyle();
		update_entryliststyle();
	}
	else
	if( m_database.get_entryliststyle() > LIST_BYTAG )
	//~ if( dialog_chaptereditor->is_chapters_changed() )
	{
		update_treeview_entries();
		present_entryrow( m_row_cur );
	}

	delete m_dialog_chaptereditor;
	m_dialog_chaptereditor = NULL;
}

int
Lifeograph::sort_bydate(	const Gtk::TreeModel::iterator &itr1,
							const Gtk::TreeModel::iterator &itr2 )
{
	ColrecEntry::ItemType itemtype1 = ( *itr1 )[ m_colrec_entry.m_type ];
	ColrecEntry::ItemType itemtype2 = ( *itr2 )[ m_colrec_entry.m_type ];

	// untagged folder is always the first, earlier entries is always the last:
	if(	itemtype1 == ColrecEntry::IT_Untaggedfolder ||
		itemtype2 == ColrecEntry::IT_Earlier )
		return -1;
	else
	if(	itemtype2 == ColrecEntry::IT_Untaggedfolder ||
		itemtype1 == ColrecEntry::IT_Earlier )
		return 1;

	if(	m_database.get_entryliststyle() == LIST_BYTAG &&
		itemtype1 != ColrecEntry::IT_Entry )
	{
		if( ( *itr1 )[ m_colrec_entry.m_id ] > ( *itr2 )[ m_colrec_entry.m_id ] )
			return 1;
		else
		if( ( *itr1 )[ m_colrec_entry.m_id ] < ( *itr2 )[ m_colrec_entry.m_id ] )
			return -1;
		else
			return 0;
	}

	if(	( *itr1 )[ m_colrec_entry.m_date_long ] >
		( *itr2 )[ m_colrec_entry.m_date_long ] )
		return -1;
	else
	if(	( *itr1 )[ m_colrec_entry.m_date_long ] <
		( *itr2 )[ m_colrec_entry.m_date_long ] )
		return 1;
	else
		return 0;
}

// sort by favoredness:
int
Lifeograph::sort_bytitle(	const Gtk::TreeModel::iterator &itr1,
							const Gtk::TreeModel::iterator &itr2 )
{
	// only compare entries (not folders)
	if( ( *itr1 )[ m_colrec_entry.m_type ] != ColrecEntry::IT_Entry ||
		( *itr2 )[ m_colrec_entry.m_type ] != ColrecEntry::IT_Entry )
		return 0;

	Entry *entry1 = get_treerow_ptr_entry( *itr1 );
	Entry *entry2 = get_treerow_ptr_entry( *itr2 );

	if( entry1->is_favored() > entry2->is_favored() )
		return -1;
	else
	if( entry1->is_favored() < entry2->is_favored() )
		return 1;
	else
	{
		if( entry1->get_date() > entry2->get_date() )
			return -1;
		else
		if( entry1->get_date() < entry2->get_date() )
			return 1;
		else
			return 0;
	}

}

bool
Lifeograph::get_entryrow( Gtk::TreeRow &row, const void *ptr )
{
	Gtk::TreeIter iter;
	for(	iter = m_treestore_entries->children().begin();
			iter != m_treestore_entries->children().end();
			iter++ )
	{
		if( ptr == ( *iter )[ m_colrec_entry.m_ptr ] )
		{
			row = ( *iter );
			return true;
		}
		else
		{
			for(	Gtk::TreeIter itr2 = ( *iter )->children().begin();
					itr2 != ( *iter )->children().end();
					itr2++ )
			{
				if( ptr == ( *itr2 )[ m_colrec_entry.m_ptr ] )
				{
					row = ( *itr2 );
					return true;
				}
			}
		}
	}
	row = *( m_treestore_entries->children().begin() );
	return false;
}

inline Entry*
Lifeograph::get_entry_current( void )
{
	if( m_entrycount < 1 )
		return NULL;
	else
	if( m_row_cur[ m_colrec_entry.m_type ] == ColrecEntry::IT_Entry )
		return( static_cast< Entry* >( m_ptr2listitem_cur ) );
	else
		return NULL;
}

inline
Entry*
Lifeograph::get_treerow_ptr_entry( const Gtk::TreeRow &row )
{
	return( static_cast< Entry* >( (void*) row[ m_colrec_entry.m_ptr ] ) );
}

inline Tag*
Lifeograph::get_treerow_ptr_tag( const Gtk::TreeRow &row )
{
	return( static_cast< Tag* >( (void*) row[ m_colrec_entry.m_ptr ] ) );
}

