/***********************************************************************************

    Copyright (C) 2007-2012 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <cmath>
#include <cairomm/context.h>

#include "widget_tag_list.hpp"
#include "helpers.hpp"
#include "lifeograph.hpp"


using namespace LIFEO;


// TAG WIDGET ======================================================================================
TagWidget::TagWidget( BaseObjectType* cobject, const Glib::RefPtr<Gtk::Builder>& b )
:   EntryClear( cobject, b ), m_completion( Gtk::EntryCompletion::create() )
{
    m_completion->set_popup_completion( true );
    m_completion->set_popup_single_match( true );
    m_completion->set_match_func(
            sigc::mem_fun( *this, &TagWidget::compare_text ) );

    m_liststore = Gtk::ListStore::create( *PanelExtra::colrec );
    m_completion->set_model( m_liststore );
    m_completion->set_text_column( PanelExtra::colrec->name );

    set_completion( m_completion );
}

bool
TagWidget::compare_text( const Glib::ustring& text,
                         const Gtk::TreeModel::const_iterator& iter )
{
    Glib::ustring tagname = iter->get_value( PanelExtra::colrec->name );
    if( text.size() == 1 )
        return( Glib::Unicode::tolower( tagname[ 0 ] ) ==
                Glib::Unicode::tolower( text[ 0 ] ) );
    else
        return( tagname.lowercase().find( text.lowercase() ) != std::string::npos );
}

void
TagWidget::populate( const Tagset* set )
{
    m_liststore->clear();

    Gtk::TreeRow row;

    for( Tagset::const_iterator iter = set->begin();
         iter != set->end();
         ++iter )
    {
        row = *( m_liststore->append() );
        row[ PanelExtra::colrec->name ] = ( *iter )->get_name();
    }
}

void
TagWidget::populate()
{
    m_liststore->clear();

    Gtk::TreeRow row;

    for( PoolTags::const_iterator iter = Diary::d->get_tags()->begin();
         iter != Diary::d->get_tags()->end();
         ++iter )
    {
        row = *( m_liststore->append() );
        row[ PanelExtra::colrec->name ] = ( *iter ).second->get_name();
    }
}

// TAG LIST WIDGET =================================================================================
WidgetTagList::WidgetTagList( BaseObjectType* cobject, const Glib::RefPtr< Gtk::Builder >& )
:   DrawingArea( cobject ), m_ptr2entry( NULL ), m_pos_x( MARGIN ), m_pos_y( MARGIN ),
    m_hovered_tag( NULL ), m_flag_editable( true )
{
    set_events( Gdk::BUTTON_PRESS_MASK | Gdk::BUTTON_RELEASE_MASK |
                Gdk::POINTER_MOTION_MASK | Gdk::LEAVE_NOTIFY_MASK | Gdk::SCROLL_MASK );

    // DEFAULT FONT
    Gtk::Button* button = new Gtk::Button;
    button->show();
    m_font_system = Cairo::ToyFontFace::create( button->get_style_context()->get_font().to_string(),
                                                Cairo::FONT_SLANT_NORMAL,
                                                Cairo::FONT_WEIGHT_NORMAL );
    m_color_text_default = button->get_style_context()->get_color( Gtk::STATE_FLAG_ACTIVE );
    delete button;

    // TAG ICON
    //ICON_SIZE = Lifeograph::icons->tag_16->get_width(); // we assume that icon is a square
    /*Cairo::Format format( image_ptr_->get_has_alpha() ?
            Cairo::FORMAT_ARGB32 : Cairo::FORMAT_RGB24 );*/
    /*m_image_surface = Cairo::ImageSurface::create( Cairo::FORMAT_ARGB32,
                                                   ICON_SIZE, ICON_SIZE );
    m_image_context = Cairo::Context::create( m_image_surface );
    Gdk::Cairo::set_source_pixbuf( m_image_context, Lifeograph::icons->tag_theme_16, 0.0, 0.0 );
    m_image_context->paint();*/

    m_image_surface_add = Cairo::ImageSurface::create( Cairo::FORMAT_ARGB32,
                                                       ICON_SIZE, ICON_SIZE );
    m_image_context = Cairo::Context::create( m_image_surface_add );
    Gdk::Cairo::set_source_pixbuf(
            m_image_context,
            render_icon_pixbuf( Gtk::Stock::ADD, Gtk::ICON_SIZE_MENU ),
            0.0, 0.0 );
    m_image_context->paint();
}

void
WidgetTagList::set_entry( Entry* ptr2entry )
{
    m_ptr2entry = ptr2entry;
    m_pos_x = MARGIN;

    if( Glib::RefPtr< Gdk::Window > window = get_window() )
        window->invalidate( false );
}

void
WidgetTagList::update()
{
    if( Glib::RefPtr< Gdk::Window > window = get_window() )
        window->invalidate( false );
}

bool
WidgetTagList::on_scroll_event( GdkEventScroll* event )
{
    /*if( m_points )
    {
        if( event->direction == GDK_SCROLL_UP && m_col_start > 0 )
            m_col_start--;
        else
        if( event->direction == GDK_SCROLL_DOWN &&
            m_col_start < ( m_points->size() - m_col_count ) )
            m_col_start++;
        else
            return true;

        get_window()->invalidate( false );
    }*/
    return true;
}

bool
WidgetTagList::on_button_press_event( GdkEventButton* event )
{
    if( m_ptr2entry == NULL )
        return false;
    if( event->button == 1 && m_hovered_tag != NULL )
    {
        if( m_hovered_tag == &m_add_tag_item )
        {
            m_signal_add_tag.emit();
            PRINT_DEBUG( "Add Tag Item Pressed" );
        }
        else
        {
            m_signal_tag_selected.emit( m_hovered_tag );
            PRINT_DEBUG( "Pressed tag: " + m_hovered_tag->get_name() );
        }

        //m_flag_button_pressed = true;
    }

    return true;
}

/*bool
WidgetTagList::on_button_release_event( GdkEventButton* event )
{
    if( event->button == 1 )
    {
        m_flag_button_pressed = false;
    }

    return true;
}*/

bool
WidgetTagList::on_motion_notify_event( GdkEventMotion* event )
{
    if( m_ptr2entry != NULL )
    {

        /*if( m_flag_button_pressed )
        {
            int col_start = ( event->x / m_width ) * m_points->size() - m_col_count / 2;
            if( col_start > int( m_points->size() - m_col_count ) )
                col_start = m_points->size() - m_col_count;
            else
            if( col_start < 0 )
                col_start = 0;

            if( col_start != ( int ) m_col_start )
            {
                m_col_start = col_start;
                get_window()->invalidate( false );
            }
        }*/
        const Tag *hovered_tag( NULL );
        for( TagItemList::iterator iter = m_items.begin(); iter != m_items.end(); ++iter )
        {
            if( iter->xl < event->x && iter->xr > event->x &&
                iter->yl < event->y && iter->yr > event->y )
            {
                hovered_tag = iter->tag;
                break;
            }
        }

        if( hovered_tag != m_hovered_tag )
        {
            m_hovered_tag = hovered_tag;
            get_window()->invalidate( false );  // TODO: clip
        }
    }

    return Gtk::DrawingArea::on_motion_notify_event( event );
}

bool
WidgetTagList::on_leave_notify_event( GdkEventCrossing* event )
{
    /*if( m_points )
    {
        m_flag_pointer_hovered = false;
        get_window()->invalidate( false );  // TODO: limit to scrollbar only
    }*/

    return true;
}

void
WidgetTagList::on_size_allocate( Gtk::Allocation& allocation )
{
    Gtk::Widget::on_size_allocate( allocation );
    m_width = allocation.get_width();
    m_height = allocation.get_height();
}

void
WidgetTagList::add_item( const Cairo::RefPtr< Cairo::Context >& cr,
                         const Glib::ustring &label,
                         const Tag *tag )
{
    TagItem titem;
    Cairo::TextExtents te;

    if( tag->get_has_own_theme() )
    {

        m_font_theme.clear();    // is this necessary?
        m_font_theme = Cairo::ToyFontFace::create( tag->get_theme()->font.get_family(),
                                                   Cairo::FONT_SLANT_NORMAL,
                                                   Cairo::FONT_WEIGHT_NORMAL );
        cr->set_font_face( m_font_theme );
    }
    else
        cr->set_font_face( m_font_system );

    cr->get_text_extents( label, te );
    if( m_pos_x + ICON_SIZE + LABEL_OFFSET + te.width + MARGIN > m_width )
    {
        m_pos_x = MARGIN;
        m_pos_y += ( ICON_SIZE + VSPACING );
    }

    titem.tag = tag;
    titem.xl = m_pos_x - ITEM_BORDER;
    titem.xr = m_pos_x + te.width + HALF_HEIGHT;
    titem.yl = m_pos_y - ITEM_BORDER;
    titem.yr = titem.yl + ITEM_HEIGHT;

    if( tag == &m_add_tag_item )
        titem.xr += ( ICON_SIZE + LABEL_OFFSET );

    m_items.push_back( titem );

    // BACKGROUND
    if( !( tag != m_hovered_tag && tag == &m_add_tag_item ) )
    {
        int width( titem.xr - titem.xl - HALF_HEIGHT );

        cr->move_to( titem.xl + 0.5, titem.yl + 0.5 );
        cr->rel_line_to( width, 0 );
        cr->rel_line_to( HALF_HEIGHT, HALF_HEIGHT );
        cr->rel_line_to( HALF_HEIGHT * -1, HALF_HEIGHT );
        cr->rel_line_to(-width, 0);
        cr->close_path();

        if( tag == m_hovered_tag && m_flag_editable )
            cr->set_line_width( 2.0 );

        if( tag->get_has_own_theme() )
        {
            Gdk::Cairo::set_source_rgba( cr, tag->get_theme()->color_base );
            cr->fill_preserve();

            Gdk::Cairo::set_source_rgba( cr, tag->get_theme()->color_highlight );
            cr->stroke_preserve();

            std::vector< double > dashes;
            dashes.push_back( 10.0 );
            dashes.push_back( 10.0 );

            cr->set_dash( dashes, 0.0 );

            Gdk::Cairo::set_source_rgba( cr, tag->get_theme()->color_heading );
            cr->stroke();
            cr->unset_dash();
        }
        else
        {
            Gdk::Cairo::set_source_rgba( cr, m_color_text_default );
            cr->stroke();
        }

        if( tag == m_hovered_tag && m_flag_editable )
            cr->set_line_width( 1.0 );
    }

    // ICON
    if( tag == &m_add_tag_item )
    {
        cr->set_source( m_image_surface_add, m_pos_x, m_pos_y );
        cr->rectangle( m_pos_x, m_pos_y, ICON_SIZE, ICON_SIZE );
        cr->clip();
        cr->paint();
        cr->reset_clip();

        m_pos_x += ( ICON_SIZE + LABEL_OFFSET );
    }

    // LABEL
    if( tag->get_has_own_theme() )
        Gdk::Cairo::set_source_rgba( cr, tag->get_theme()->color_text );
    else
        Gdk::Cairo::set_source_rgba( cr, m_color_text_default );

    cr->move_to( m_pos_x + 0.5, m_pos_y + TEXT_HEIGHT + 0.5 );
    cr->show_text( label );

    m_pos_x += ( te.width + HALF_HEIGHT + HSPACING );
}

bool
WidgetTagList::on_draw( const Cairo::RefPtr< Cairo::Context >& cr )
{
    if( m_ptr2entry == NULL )
        return false;

    /* TODO
    if( event )
    {
        // clip to the area indicated by the expose event so that we only
        // redraw the portion of the window that needs to be redrawn
        cr->rectangle( event->area.x, event->area.y,
                event->area.width, event->area.height );
        cr->clip();
    }*/

    m_pos_x = m_pos_y = MARGIN;
    m_items.clear();
    cr->set_font_size( TEXT_HEIGHT );
    cr->set_line_width( 1.0 );
    cr->set_line_join( Cairo::LINE_JOIN_ROUND );

    const Tagset &ptr2tagset = m_ptr2entry->get_tags();

    if( ptr2tagset.empty() && !m_flag_editable )
    {
        cr->set_font_face( m_font_system );
        Gdk::Cairo::set_source_rgba( cr, m_color_text_default );
        cr->move_to( m_pos_x, m_pos_y + TEXT_HEIGHT );
        cr->show_text( _( "Not tagged" ) );
    }

    for( Tagset::const_iterator iter = ptr2tagset.begin();
         iter != ptr2tagset.end();
         ++iter )
    {
        add_item( cr, ( *iter )->get_name(), *iter );
    }

    if( m_flag_editable )
    {
        add_item( cr, _( "Add Tag" ), &m_add_tag_item );
    }

    set_size_request( -1, m_pos_y + TEXT_HEIGHT + MARGIN );

    return true;
}
