/*!!***************************************************************************
 *!! FILE NAME
 *!!	$Source: /var/cvs/pub/repository/LinCVS/src/logtree.cpp,v $
 *!!
 *!! AUTHOR
 *!!	$Author: joseh $
 *!!
 *!! DATE, REVISION AND STATE
 *!!	$Date: 2001/11/04 13:01:54 $
 *!!	$Revision: 1.9 $
 *!!	$State: Exp $
 *!!
 *!! DESCRIPTION
 *!!
 *!!	Copyright (C) 2001 Jose Hernandez.
 *!!
 *!!    Heavilly based on cervisia's code by Bernd Gehrmann and others.
 *!!    Copyright (C) 1999 Bernd Gehrmann.  bernd@physik.hu-berlin.de
 *!!
 *!!	This program is free software; you can redistribute it and/or modify
 *!!	it under the terms of the GNU General Public License as published by
 *!!	the Free Software Foundation; either version 2, or (at your option)
 *!!	any later version.
 *!!
 *!!	This program is distributed in the hope that it will be useful,
 *!!	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *!!	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *!!	GNU General Public License for more details.
 *!!
 *!!**************************************************************************/


#include "config.h"
#include "ac_system_defs.h"

#include <iostream>
#include <qpainter.h>
#include <qapplication.h>
#include <qtooltip.h>

#include "logtree.h"


static const int BORDER = 8;
static const int INSPACE = 3;

static bool static_initialized = false;
static int  static_width;
static int  static_height;


QColor LogTreeView::SelColor [2];


class DynamicTip : public QToolTip
{
public:
	DynamicTip(LogTreeView *parent) : QToolTip(parent) {}

protected:
	virtual void maybeTip(const QPoint &pos);
};


class LogTreeItem
{
public:
	LogTreeItem();
	~LogTreeItem();

	QString rev;
	QString date;
	QString author;
	QString state;
	QString comment;
	QString tagcomment;
	QString BranchTagList;
	QString RegularTagList;
	QString branchpoint;
	bool firstonbranch;
	bool followed;
	bool branched;
	int row;
	int col;
	bool selected;
	unsigned char selection; 
};


class LogTreeConnection
{
public:
	LogTreeItem *start;
	LogTreeItem *end;
};



LogTreeItem::LogTreeItem()
	: firstonbranch(false), followed(false), branched(false),
	row(0), col(0), selected(false), selection(0)
{
}


LogTreeItem::~LogTreeItem()
{
}


void LogTreeView::setSelectionColor (const unsigned char selection, const QColor &color)
{
	SelColor [selection] = color;
}

	
QColor LogTreeView::getSelectionColor (const unsigned char selection) const
{
	return SelColor [selection];
}


LogTreeView::LogTreeView(QWidget *parent, const char *name)
    : QTableView(parent, name), bEnableBranchTags(false),
      bEnableRegularTags(false), bEnableCompactMode(true)
{
    if (!static_initialized) {
	static_initialized = true;
	QFontMetrics fm( fontMetrics() );
	static_width = fm.width("123456789") + 2*BORDER + 2*INSPACE;
	static_height = 2*fm.height() + 2*BORDER + 3*INSPACE;

	// Set default selection color to black
	for (int i=0; i<2; i++)
		setSelectionColor (i, QColor("black"));
    }

    setNumCols(0);
    setNumRows(0);
    setAutoUpdate(false);
    setTableFlags( Tbl_autoVScrollBar|Tbl_autoHScrollBar|
		   Tbl_smoothVScrolling | Tbl_smoothHScrolling );
    setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
    setBackgroundMode(PaletteBase);

    setMouseTracking(true);
    setFocusPolicy(ClickFocus);
    
    setCellWidth(0);
    setCellHeight(0);

    currentRow = -1;
    currentCol = -1;

    items.setAutoDelete(true);
    connections.setAutoDelete(true);

    (void) new DynamicTip(this);
}


LogTreeView::~LogTreeView ()
{
}


QString LogTreeView::findCollision (int row, int col) const
{
    QListIterator<LogTreeItem> it(items);
    QString branch ("");

    for (it.toFirst(); it.current(); ++it)
	if ((it.current()->row == row) && (it.current()->col == col))
	{
	    branch = it.current()->rev;
	    branch.replace (QRegExp ("\\.[0-9]+$"), "");
	}

    return (branch);
}


void LogTreeView::shiftBranch (QString branch)
{
    // Move branches to the right
    QListIterator<LogTreeItem> it(items);

    if (!branch.isEmpty())
    {
	for (it.toFirst(); it.current(); ++it)
	    if (it.current()->rev.startsWith(branch))
	    {
		shiftBranch (findCollision (it.current()->row, it.current()->col + 1));

		if (++it.current()->col >= numCols())
		    setNumCols (it.current()->col + 1);
		
		if (it.current()->rev.contains (QRegExp("\\.1$")))
		    shiftBranch (findCollision (it.current()->row + 1, it.current()->col));
	    }
    }
}


void LogTreeView::addRevision(const QString &rev, const QString &author,
			      const QString &date, const QString &state,
			      const QString &comment,
			      const QString &BranchTagList,
			      const QString &RegularTagList,
			      const QString &tagcomment)

{
    LogTreeItem *item = new LogTreeItem;
    QString revision, branch;
    int pos;

    revision = rev;
    branch = "";

    // find branch
    if ((pos = revision.findRev('.')) != -1)
    {
	revision.truncate(pos);
	if ((pos = revision.findRev('.')) != -1)
	{
	    // e. g. for rev = 1.1.2.3 we have
	    // branch = 1.1.2, revision = 1.1
	    branch = revision;
	    revision.truncate(pos);
	}
    }

    item->rev = rev;
    item->author = author;
    item->state = state;
    item->comment = comment;
    item->tagcomment = tagcomment;
    item->date = date;
    item->BranchTagList = BranchTagList;
    item->RegularTagList = RegularTagList;
    item->branchpoint = revision;
    item->selected = false;

    if (branch.isEmpty())
    {
	// Most probably we are on the trunk
	item->firstonbranch = false;
	item->row = numRows();
	item->col = 0;
	items.append(item);

	/* Set the number of columns to 1 as CVS log
	 * reports all the revisions on the main trunk
	 * before moving on to branches.
	 */
	setNumRows (numRows()+1);
	setNumCols (1);
	return;
    }
    
    // look whether we have revisions on this branch
    // shift them up
    int row = -1, col = -1;
    QListIterator<LogTreeItem> it(items);
    for (; it.current(); ++it)
    {
	    if (it.current()->rev.startsWith(branch))
	    {
		    it.current()->firstonbranch = false;
		    row = it.current()->row;
		    col = it.current()->col;
		    it.current()->row--;

		    /* Are we inserting at the top of the widget?
		     * If so, we need to shift all the rows down.
		     */
		    if (row == 0)
		    {
			    QListIterator<LogTreeItem> it2(items);
			    for (; it2.current(); ++it2)
				    it2.current()->row++;
			    setNumRows(numRows()+1);
			    row = 1;
		    }
	    }
    }

    if (row == -1)
    {
	/* Ok, so we must open a new branch
	 * Let's find the branch point
	 */
	QListIterator<LogTreeItem> it3(items);
	for (it3.toLast(); it3.current(); --it3)
	{
	    if (revision == it3.current()->rev)
	    {
		row = it3.current()->row-1;
		col = it3.current()->col+1;

		if (bEnableCompactMode)
		{
			/* Display a compacted log tree so we can get
			 * more stuff on less screen.  This mode is
			 * basically a slightly smarted layout algorithm
			 * that attempts to position branches in a
			 * space saving way.
			 */
			shiftBranch (findCollision (row, col));
			shiftBranch (findCollision (row + 1, col));

			if (col >= numCols())
				setNumCols (col + 1);
		}
		else
		{
			/* Use the non-compacted mode.  This is a quicker
			 * layout algorithm but uses more screen.  This
			 * tends to clutter the screen if there is a large
			 * number of revisions.  In essense, this code
			 * just shifts right any pre-existing branches
			 * before creating the new one, this way there is
			 * no need to worry about positional collisions.
			 */
			QListIterator<LogTreeItem> it4(items);

			for (it4.toFirst(); it4.current(); ++it4)
				if (it4.current()->col >= col)
					it4.current()->col++;

			setNumCols (numCols() + 1);
		}

		if (row == -1)
		{
		    QListIterator<LogTreeItem> it5(items);
		    for (; it5.current(); ++it5)
			it5.current()->row++;
		    setNumRows(numRows()+1);
		    row = 0;
		}
		break;
	    }
	}
    }

    item->firstonbranch = true;
    item->row = row;
    item->col = col;
    items.append(item);
}


void LogTreeView::collectConnections()
{
    QListIterator<LogTreeItem> it(items);
    for (; it.current(); ++it)
	{
            QString rev = it.current()->rev;
            
            QListIterator<LogTreeItem> it2(items);
            for (it2=it,++it2; it2.current(); ++it2)
                if (it2.current()->branchpoint == rev &&
                    it2.current()->firstonbranch)
                    {
                        LogTreeConnection *conn = new LogTreeConnection;
                        conn->start = it.current();
                        conn->end = it2.current();
                        connections.append(conn);
                    }
        }
}


void LogTreeView::setSelectedPair (QString selectionA, QString selectionB)
{
	bool selected, refresh = false;
	unsigned char selection;

	for (QListIterator<LogTreeItem> it(items); it.current(); ++it)
	{
		if (selectionA == it.current()->rev)
		{
			selected = true;
			selection = 0;
		}
		else if (selectionB == it.current()->rev)
		{
			selected = true;
			selection = 1;
		}
		else
		{
			selected = false;
			selection = 0;
		}

		if ((selected != it.current()->selected) ||
		    (selection != it.current()->selection))
		{
			it.current()->selected = selected;
			it.current()->selection = selection;

			// mark for refresh if there are any changes
			refresh = true;
		}
	}

	repaint (refresh);
}


QSize LogTreeView::sizeHint() const
{
    return QSize (2 * static_width, 3 * static_height);
}


void LogTreeView::setupPainter(QPainter *p)
{
    p->setBackgroundColor(colorGroup().base());
}


void LogTreeView::paintCell(QPainter *p, int row, int col)
{
    bool followed, branched;
    LogTreeItem *item;

    branched = false;
    followed = false;
    item = 0;
    
    QListIterator<LogTreeItem> it(items);
    for(; it.current(); ++it)
        {
            int itcol = it.current()->col;
            int itrow = it.current()->row;
            if (itrow == row-1 && itcol == col)
                followed = true;
            if (itrow == row && itcol == col)
                item = it.current();
        }
    QListIterator<LogTreeConnection> it2(connections);
    for (; it2.current(); ++it2)
        {
            int itcol1 = it2.current()->start->col;
            int itcol2 = it2.current()->end->col;
            int itrow = it2.current()->start->row;
            if (itrow == row && itcol1 <= col && itcol2 > col)
                branched = true;
        }

    p->fillRect (0, 0, cellWidth(col), cellHeight(row), colorGroup().base());
    p->setPen (colorGroup().text());

    if (item)
	    paintRevisionCell (p, item, followed, branched);

    else
	    if (followed || branched)
		    paintConnector(p, row, col, followed, branched);
}


void LogTreeView::paintConnector (QPainter *p, int row, int col,
				  bool followed, bool branched)
{
    int midx = colWidths[col] / 2;
    int midy = rowHeights[row] / 2;

    p->drawLine(0, midy, branched ? colWidths[col] : midx, midy);
    if (followed)
        p->drawLine(midx, midy, midx, 0);
}


void LogTreeView::paintRevisionCell (QPainter *p, LogTreeItem *item,
				     bool followed, bool branched)
{
    QFontMetrics fm(p->fontMetrics());
    
    QSize r1 = fm.size(AlignCenter, item->author);
    QSize r2 = fm.size(AlignCenter, item->rev);
    QSize r3 = fm.size(AlignCenter, item->BranchTagList);
    QSize r4 = fm.size(AlignCenter, item->RegularTagList);
    
    int boxwidth, boxheight;
    boxwidth = QMAX(static_width-2*BORDER, QMAX(r1.width(), r2.width()));
    boxheight = r1.height() + r2.height() + 3*INSPACE;
    
    if ((bEnableBranchTags) && (!item->BranchTagList.isEmpty()))
    {
	boxwidth = QMAX(boxwidth, r3.width());
	boxheight += r3.height() + INSPACE;
    }

    if ((bEnableRegularTags) && (!item->RegularTagList.isEmpty()))
    {
	boxwidth = QMAX(boxwidth, r4.width());
	boxheight += r4.height() + INSPACE;
    }

    boxwidth += 2*INSPACE;
    
    int x = (colWidths[item->col] - boxwidth) / 2;
    int midx = colWidths[item->col] / 2;
    int y = (rowHeights[item->row] - boxheight) / 2;
    int midy = rowHeights[item->row] / 2;
    
    // Connectors
    if (followed)
        p->drawLine(midx, 0, midx, y);
    
    if (branched)
        p->drawLine(midx + boxwidth / 2, midy, colWidths[item->col], midy);
    
    p->drawLine(midx, y + boxheight, midx, rowHeights[item->row]);
    
    /* Draw the box itself.  Revisions marked as dead by cvs are
     * a special case because we can't run normal CVS operations
     * on them.
     */
    if (item->state == "dead")
    {
	    p->setPen (QPen(black, 0, DotLine));
	    p->drawRoundRect(x, y, boxwidth, boxheight, 10, 10);
    }
    else if (item->selected)
    {
	    int h, s, v;
	    QColor FillColor (getSelectionColor(item->selection));
	    FillColor.hsv (&h, &s, &v);
	    p->fillRect (x, y, boxwidth, boxheight, FillColor);
	    p->setPen ((v > 128) ? QColor("black") : QColor("white"));
    }
    else
	    p->drawRoundRect(x, y, boxwidth, boxheight, 10, 10);
    
    x += INSPACE;
    y += INSPACE;
    boxwidth -= 2*INSPACE;
    
    p->drawText(x, y, boxwidth, boxheight, AlignHCenter, item->author);
    y += r1.height() + INSPACE;
    
    if ((bEnableRegularTags) && (!item->RegularTagList.isEmpty()))
    {
	QFont font(p->font());
	QFont underline(font);
            
	underline.setUnderline(true);
	p->setFont(underline);
	p->drawText(x, y, boxwidth, boxheight, AlignHCenter, item->RegularTagList);
	p->setFont(font);
	y += r4.height() + INSPACE;
    }

    if ((bEnableBranchTags) && (!item->BranchTagList.isEmpty()))
    {
	QFont font(p->font());
	QFont underline(font);
            
	underline.setUnderline(true);
	p->setFont(underline);
	p->drawText(x, y, boxwidth, boxheight, AlignHCenter, item->BranchTagList);
	p->setFont(font);
	y += r3.height() + INSPACE;
    }

    p->drawText(x, y, boxwidth, boxheight, AlignHCenter, item->rev);
}


/*!!***************************************************************************
 *!! FUNCTION NAME
 *!!    LogTreeView::mouseDoubleClickEvent
 *!!
 *!! PARAMETERS
 *!!    QMouseEvent *e
 *!!
 *!! RETURN VALUE
 *!!    none
 *!!
 *!! GLOBALS ACCESSED
 *!!    ?
 *!!
 *!! DESCRIPTION
 *!!    Double clicking on a revision has the effect of bringing up a file
 *!!    view window.  The exception to this are dead revisions.  
 *!!**************************************************************************/

void LogTreeView::mouseDoubleClickEvent (QMouseEvent *e)
{
	if (e->button() == LeftButton)
	{
		int row = findRow (e->pos().y());
		int col = findCol (e->pos().x());

		QListIterator<LogTreeItem> it(items);
		for (; it.current(); ++it)
			if ((it.current()->row == row) &&
			    (it.current()->col == col) &&
			    (it.current()->state != "dead"))
			{
				emit revisionDblClicked (it.current()->rev);
				break;
			}
	}
}


void LogTreeView::mousePressEvent(QMouseEvent *e)
{
	if ((e->button() == MidButton) ||
	    (e->button() == LeftButton) ||
	    (e->button() == RightButton))
	{
		int row = findRow( e->pos().y() );
		int col = findCol( e->pos().x() );

		QListIterator<LogTreeItem> it(items);
		for(; it.current(); ++it)
			if ((it.current()->row == row) &&
			    (it.current()->col == col) &&
			    (it.current()->state != "dead"))
			{
				emit revisionClicked (it.current()->rev,
						      (e->button() == MidButton)
						      || (e->button() == RightButton));
				break;
			}
	}
}


void LogTreeView::recomputeCellSizes ()
{
    // Fill with default
    int v = static_width;
    colWidths.fill(v, numCols());
    v = static_height;
    rowHeights.fill(v, numRows());
    
    QFontMetrics fm(fontMetrics());

    // Compute maximum for each column and row
    QListIterator<LogTreeItem> it(items);
    for (; it.current(); ++it)
        {
            LogTreeItem *item = it.current();

            QSize r1 = fm.size(AlignCenter, item->rev);
	    QSize r2 = fm.size(AlignCenter, item->author);

            int boxwidth = QMAX(r1.width(), r2.width());
            int boxheight = r1.height() + r2.height() + 3*INSPACE;
            
            if ((bEnableBranchTags) && (!item->BranchTagList.isEmpty()))
	    {
		QSize r3 = fm.size(AlignCenter, item->BranchTagList);
		boxwidth = QMAX(boxwidth, r3.width());
		boxheight += r3.height() + INSPACE;
	    }

            if ((bEnableRegularTags) && (!item->RegularTagList.isEmpty()))
	    {
		QSize r4 = fm.size(AlignCenter, item->RegularTagList);
		boxwidth = QMAX(boxwidth, r4.width());
		boxheight += r4.height() + INSPACE;
	    }

            boxwidth += 2*INSPACE;
            
            colWidths[item->col] = QMAX(colWidths[item->col], boxwidth + 2*BORDER);
            rowHeights[item->row] = QMAX(rowHeights[item->row], boxheight + 2*BORDER);
	}
    
    setAutoUpdate(true);
    updateTableSize();
    update();
}


int LogTreeView::cellWidth(int col)
{
    if (col < 0 || col >= (int)colWidths.size())
        return 0;
    
    return colWidths[col];
}


int LogTreeView::cellHeight(int row)
{
    if (row < 0 || row >= (int)rowHeights.size())
        return 0;
    
    return rowHeights[row];
}


void LogTreeView::toggleBranchTags (bool enable)
{
    bEnableBranchTags = enable;
    recomputeCellSizes();
    repaint (true);
}


void LogTreeView::toggleRegularTags (bool enable)
{
    bEnableRegularTags = enable;
    recomputeCellSizes();
    repaint (true);

}


void LogTreeView::lookupTip (const QPoint &p, QRect *r, QString *s)
{
	int row = findRow (p.y());
	int col = findCol (p.x());
    
	*r = QRect(0, 0, -1, -1);
	*s = "";

	for (QListIterator<LogTreeItem> it(items); it.current(); ++it)
		if ((it.current()->row == row) && (it.current()->col == col))
		{
			QString text = "<qt><b>";
			text += it.current()->rev;
			text += "</b>&nbsp;&nbsp;";
			text += it.current()->author;
			text += "&nbsp;&nbsp;<b>";
			text += it.current()->date;
			text += "</b>";
			QStringList list2 = QStringList::split("\n", it.current()->comment);
			QStringList::Iterator it2;
			for (it2 = list2.begin(); it2 != list2.end(); ++it2)
                        {
				text += "<br>";
				text += (*it2);
                        }
			if (!it.current()->tagcomment.isEmpty())
                        {
				text += "<i>";
				QStringList list3 = QStringList::split("\n", it.current()->tagcomment);
				QStringList::Iterator it3;
				for (it3 = list3.begin(); it3 != list3.end(); ++it3)
                                {
					text += "<br>";
					text += (*it3);
                                }
				text += "</i>";
                        }
			text += "</qt>";

			int left; colXPos(col, &left); //!!!left += cellWidth(col);
			int top; rowYPos(row, &top);
			int width = cellWidth(col);
			int height = cellHeight(row);

			r->setRect(left, top, width, height);
			*s = text;
			return;
		}
}


void DynamicTip::maybeTip(const QPoint &pos)
{
	if (!parentWidget()->inherits("LogTreeView"))
		return;

	LogTreeView *logtree = static_cast<LogTreeView*>(parentWidget());

	QRect r;
	QString s;
	logtree->lookupTip(pos, &r, &s);
	if (r.isValid())
		tip( r, s );
}
